/*
 *  Hearts - events.c
 *  Copyright 2006 Sander Marechal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "events.h"

/* New game menu item and button */
void on_new_event(GtkWidget *widget, gpointer data)
{
	game_new();
	draw_playingarea();
}

/* Restart menu item and button */
void on_restart_event(GtkWidget *widget, gpointer data)
{
	game_restart_round();
	draw_playingarea();
}

/* Show scores menu item */
void on_scores_event(GtkWidget *widget, gpointer data)
{
	GtkWidget *score = glade_xml_get_widget(xml, "score");
	gtk_widget_show (score);
}

/*
 * menu/toolbar events
 */

/* Quit menu item */
void on_quit_event(GtkWidget *widget, gpointer data)
{
	gtk_main_quit();
	return;
}

/* Undo menu item and button */
void on_undo_event(GtkWidget *widget, gpointer data)
{
	return;
}

/* Redo game menu item and button */
void on_redo_event(GtkWidget *widget, gpointer data)
{
	return;
}

/* Open the help file */
void on_help(GtkWidget *widget, gpointer data)
{
	gnome_help_display("gnome-hearts.xml", NULL, NULL);
}

/* Open the preferences help file */
void on_preferences_help(GtkWidget *widget, gpointer data)
{
	gnome_help_display("gnome-hearts.xml", "hearts-prefs", NULL);
}

/* About menu item */
void on_about_event(GtkImageMenuItem *widget, gpointer data)
{
	GtkWidget *about;
	about = glade_xml_get_widget (xml, "about");
	gtk_widget_show (about);

	return;
}

/* Fullscreen */
void on_fullscreen_event(GtkWidget *widget, gpointer data)
{
	GtkWidget *window = glade_xml_get_widget(xml, "hearts");
	GdkWindowState state = gdk_window_get_state(window->window);
	
	if (state & GDK_WINDOW_STATE_FULLSCREEN)
	{
		gtk_widget_show(glade_xml_get_widget(xml, "fullscreen"));
		gtk_widget_set_sensitive(glade_xml_get_widget(xml, "fullscreen"), TRUE);
		gtk_widget_hide(glade_xml_get_widget(xml, "leave_fullscreen"));
		gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen"), FALSE);
		gtk_widget_hide(glade_xml_get_widget(xml, "leave_fullscreen_button"));
		gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen_button"), FALSE);
		gtk_window_unfullscreen((GtkWindow*)glade_xml_get_widget(xml, "hearts"));
	}
	else
	{
		gtk_widget_hide(glade_xml_get_widget(xml, "fullscreen"));
		gtk_widget_set_sensitive(glade_xml_get_widget(xml, "fullscreen"), FALSE);
		gtk_widget_show(glade_xml_get_widget(xml, "leave_fullscreen"));
		gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen"), TRUE);
		gtk_widget_show(glade_xml_get_widget(xml, "leave_fullscreen_button"));
		gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen_button"), TRUE);
		gtk_window_fullscreen((GtkWindow*)glade_xml_get_widget(xml, "hearts"));
	}
}

void on_leave_fullscreen_event(GtkWidget *widget, gpointer data)
{
	gtk_widget_show(glade_xml_get_widget(xml, "menubar"));
	gtk_widget_show(glade_xml_get_widget(xml, "fullscreen"));
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "fullscreen"), TRUE);
	gtk_widget_hide(glade_xml_get_widget(xml, "leave_fullscreen"));
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen"), FALSE);
	gtk_widget_hide(glade_xml_get_widget(xml, "leave_fullscreen_button"));
	gtk_widget_set_sensitive(glade_xml_get_widget(xml, "leave_fullscreen_button"), FALSE);
	gtk_window_unfullscreen((GtkWindow*)glade_xml_get_widget(xml, "hearts"));
}

/* Toolbar toggle menu item */
void on_toolbar_toggled(GtkCheckMenuItem *widget, gpointer data)
{
	GtkWidget *toolbar;
	toolbar = glade_xml_get_widget(xml, "toolbar");

	if (gtk_check_menu_item_get_active(widget))
		gtk_widget_show_all(toolbar);
	else
		gtk_widget_hide_all(toolbar);
	
	return;
}

/* Statusbar toggle menu item */
void on_statusbar_toggled(GtkCheckMenuItem *widget, gpointer data)
{
	GtkWidget *statusbar;
	statusbar = glade_xml_get_widget(xml, "statusbar");

	if (gtk_check_menu_item_get_active(widget))
		gtk_widget_show_all(statusbar);
	else
		gtk_widget_hide_all(statusbar);
	
	return;
}

/*
 * playingarea events
 */

/* deal with mouse clicks */
gboolean button_press_event(GtkWidget *widget, GdkEventButton *event)
{
	Card *card, *active;
	gint i;
	
	if (event->button != 1)
		return TRUE;
	
	card = active = NULL;
	
	switch (game_state)
	{
		case GAME_PASS_CARDS:
			if (game_pass > 0 && game_pass_cards(widget))
			{
				game_set_status(_("Click somewhere to continue the game."));
				game_state = GAME_RECIEVE_CARDS;
				break;
			}
			/* Don't break here. You can deselect cards instead of passing them as well */
			
		case GAME_SELECT_CARDS:
			/* if there are 3 cards selected, cards can be passed */
			if (game_select_cards(widget) == 3)
			{
				gchar *name = NULL, *message = NULL;
				switch ((user + game_pass) % 4)
				{	
					case NORTH: name = game_rules.north_ai->str; break;
					case EAST:  name = game_rules.east_ai->str;  break;
					case WEST:  name = game_rules.west_ai->str;  break;
				}
				
				message = g_strdup_printf(_("Click on the hand of %s to pass the cards."), name);
				game_set_status(message);
				g_free(message);
				
				game_state = GAME_PASS_CARDS;
			}
			else
				game_state = GAME_SELECT_CARDS;
			break;
			
		case GAME_RECIEVE_CARDS:
			/* start the round */
			cards_deck_deselect(game_deck);
			game_set_status(_("Click on the card you wish to play."));
			game_state = GAME_PLAY;
			game_open_round();
			
			/* redraw the screen */
			draw_playingarea();
			break;
			
		case GAME_PLAY:
			/* see if we're done playing this trick */
			if (game_trick.num_played == 4)
			{
				/* show the tricks to the AI's */
				player_trick_end(player[NORTH], &game_trick);
				player_trick_end(player[EAST],  &game_trick);
				player_trick_end(player[WEST],  &game_trick);
				
				/* start the next trick in this game */
				game_hearts_broken = game_hearts_broken || trick_num_suit(&game_trick, CARDS_HEARTS);
				game_trick_winner = trick_get_winner(&game_trick);
				player[game_trick_winner]->score_round += trick_get_score(&game_trick, game_rules.ruleset);
				player[game_trick_winner]->point_cards += trick_num_point_cards(&game_trick);
				trick_reset(&game_trick);

				/* is this also the end of the round or game? */
				if (game_end_test())
					return TRUE;
				
				/* play random cards untill it's the user's turn */
				i = game_trick_winner;
				while (i != user)
				{
					player_play(player[i], &game_trick);
					i = (i + 1) % 4;
				}
				
				/* redraw the screen */
				game_set_status(_("Click on the card you wish to play."));
				draw_playingarea();
				return TRUE;
			}
			
			/* loop through the hand to find the active card */
			card = cards_hand_get_active(player[user]->hand);
			if (card != NULL)
				player_play_card(player[user], card, &game_trick);
			else
				return TRUE;

			/* play random cards untill the trick is over */
			i = user;
			while (game_trick.num_played < 4)
			{
				i = (i + 1) % 4;
				player_play(player[i], &game_trick);
			}
			
			/* redraw the screen */
			game_set_status(_("Click somewhere to continue the game."));
			draw_playingarea();
			break;
		
		case GAME_ROUND_END:
			game_new_round();
			draw_playingarea();
			break;
		
		case GAME_END:
			game_new();
			draw_playingarea();
			break;
		
		default:
			g_assert_not_reached();
	}

	return TRUE;
}

/* follow the cursor around, checking for active cards */
gboolean motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
{
	Card *card;
	gint cx, cy, x, y, width, height, offset_x, offset_y, y_adj;
	gboolean update = FALSE;
	
	/* hinst are used so we don't fall behind the user */
	if (event->is_hint)
		gtk_widget_get_pointer(widget, &cx, &cy);
	else
	{
		cx = event->x;
		cy = event->y;
	}

	/* are we in the south_hand area? */
	offset_x = cards_image->width  / 4;
	offset_y = cards_image->height / 4;

	cards_hand_get_area(player[user]->hand, widget, cards_image, &x, &y, &width, &height, TRUE);	
	
	if (cx < x || cx >= x + width || cy < y || cy >= y + height)
	{
		card = cards_hand_get_active(player[user]->hand);
		if (card && card->active)
		{
			card->active = FALSE;
			update = TRUE;
		}
	}
	else
	{
		/* find the currently active card */
		Card *current_active = cards_hand_get_active(player[user]->hand);
		Card *new_active = NULL;
		GList *i = NULL;
		
		/* loop through the hand and mark the new active card */
		for (i = player[user]->hand->list; i; i = i->next)
		{
			card = (Card*)i->data;

			/* calculate the card's onscreen position */
			if (card->selected)
				y_adj = y;
			else
				y_adj = y + offset_y;
			
			/* is the cursor on the card? */
			if (cx >= x && cx < x + cards_image->width && cy >= y_adj && cy < y_adj + cards_image->height)
			{
				/* update the card */
				if (game_state != GAME_PLAY || game_is_valid_card(card, player[user]->hand, &game_trick))
				{
					card->active = TRUE;
					new_active = card;
				}
				
				/* mark previous three cards as non-active since this card covers it */
				gint count = 0; 
				GList *j = i;
				while (g_list_previous(j) != NULL && count < 3)
				{
					count++;
					j = g_list_previous(j);
					card = (Card*)j->data;
					card->active = FALSE;
				}
			}
			else
				card->active = FALSE;
			
			x += offset_x;
		}
		
		if (current_active != new_active)
			update = TRUE;
	}
	
	if (update)
	{
		x = (widget->allocation.width - width) / 2;
		cards_hand_render(player[user]->hand, cards_image, backbuffer, backbuffer_gc, x, y);
		gtk_widget_queue_draw_area(widget, x, y, x + width, y + height);
	}
	
	return TRUE;
}

/* Called when we need to draw part of the table */
gboolean on_expose_event(GtkWidget *widget, GdkEventExpose *event) 
{
	if (backbuffer == NULL)
		on_configure_event(widget, NULL);
	
	/* Draw the exposed part from the background buffer to the screen */
	gdk_draw_drawable (widget->window, widget->style->black_gc,
		backbuffer, event->area.x, event->area.y,
		event->area.x, event->area.y,
		event->area.width, event->area.height);

  return FALSE;
}

/* Called on e.g. creation and resizes */
gboolean on_configure_event(GtkWidget *widget, gpointer data)
{
	gint width, height;

	width = widget->allocation.width;
	height = widget->allocation.height;
	
	/* Remove the existing background buffer */
 	if (backbuffer)
		g_object_unref(backbuffer);

	/* Create a new buffer */
	backbuffer = gdk_pixmap_new(widget->window,
		width,
		height,
		-1);

	/* resize the background and cards */
	background_set_size(width, height);
	cards_image_set_size(cards_image, width / 8, height / 5);

	draw_playingarea();
	return TRUE;
}

/*
 * score table events
 */

/* next round */
void on_score_ok_clicked (GtkButton *widget, gpointer *data)
{
	GtkWidget *score;
	score = glade_xml_get_widget (xml, "score");
	gtk_widget_hide_all (score);

	game_new_round();
	
	draw_playingarea();
	return;
}

/* new game */
void on_score_new_clicked (GtkButton *widget, gpointer *data)
{
	GtkWidget *score;
	score = glade_xml_get_widget (xml, "score");
	gtk_widget_hide_all (score);

	game_new();
	
	draw_playingarea();
	return;
}

/*
 * preferences events
 */

/* Show preferences dialog */
void on_preferences_event(GtkWidget *widget, gpointer data)
{
	gtk_widget_show_all(glade_xml_get_widget(xml, "preferences"));
	return;
}

/* update the ruleset */
void on_ruleset_changed	(GtkComboBox *widget, gpointer *data)
{
	gint i = gtk_combo_box_get_active(widget);
	
	if (i != -1)
		game_rules.ruleset = i;
	else
		game_rules.ruleset = RULESET_STANDARD;

	game_new();
}

/* update the clubs_lead radio */
void on_clubs_lead_changed (GtkToggleButton *widget, gpointer *data)
{
	game_rules.clubs_lead = gtk_toggle_button_get_active(widget);
	game_new();
}

/* update the hearts_broken radio */
void on_hearts_break_changed (GtkToggleButton *widget, gpointer *data)
{
	game_rules.hearts_break = gtk_toggle_button_get_active(widget);
	game_new();
}

/* update the no_blood radio */
void on_no_blood_changed (GtkToggleButton *widget, gpointer *data)
{
	game_rules.no_blood = gtk_toggle_button_get_active(widget);
	game_new();
}

/* load new north_ai */
void on_north_ai_changed (GtkComboBox *widget, gpointer *data)
{
	GtkLabel *label = (GtkLabel*)glade_xml_get_widget(xml, "north_name");
	gchar* text = gtk_combo_box_get_active_text(widget);
	gtk_label_set_text(label, text);
	g_string_assign(game_rules.north_ai, text);
	g_free(text);

	game_new();
}

/* load new east_ai */
void on_east_ai_changed (GtkComboBox *widget, gpointer *data)
{
	GtkLabel *label = (GtkLabel*)glade_xml_get_widget(xml, "east_name");
	gchar* text = gtk_combo_box_get_active_text(widget);
	gtk_label_set_text(label, text);
	g_string_assign(game_rules.east_ai, text);
	g_free(text);

	game_new();
}

/* load new west_ai */
void on_west_ai_changed (GtkComboBox *widget, gpointer *data)
{
	GtkLabel *label = (GtkLabel*)glade_xml_get_widget(xml, "west_name");
	gchar* text = gtk_combo_box_get_active_text(widget);
	gtk_label_set_text(label, text);
	g_string_assign(game_rules.west_ai, text);
	g_free(text);

	game_new();
}

/* load a new background image */
void on_background_changed (GtkFileChooser *widget, gpointer *data)
{
	if (widget == NULL)
		return;

	gchar *path = gtk_file_chooser_get_filename(widget);

	if (path == NULL)
		return;

	background_load(path);

	GtkWidget *playingarea_widget = glade_xml_get_widget(xml, "playingarea");
	on_configure_event(playingarea_widget, NULL);

	g_string_assign(game_background, path);
	g_free(path);
}

void on_background_tile_toggled (GtkToggleButton *widget, gpointer *data)
{
	game_background_tiled = gtk_toggle_button_get_active(widget);
	background_set_tiled(game_background_tiled);

	GtkWidget *playingarea_widget = glade_xml_get_widget(xml, "playingarea");
	on_configure_event(playingarea_widget, NULL);
}

/* load a new cards_image */
void on_card_style_changed (GtkComboBox *widget, gpointer *data)
{
	gchar *card = gtk_combo_box_get_active_text(widget);
	gchar *path = g_hash_table_lookup(game_card_styles, card);
	
	cards_image_free(cards_image);
	cards_image = cards_image_from_file(path);
	
	GtkWidget *playingarea_widget = glade_xml_get_widget(xml, "playingarea");
	on_configure_event(playingarea_widget, NULL);

	g_string_assign(game_card_style, card);
	g_free(card);
}

/* hide the preferences */
void on_prefs_ok_clicked (GtkButton *widget, gpointer *data)
{
	GtkWidget *pref;
	GKeyFile *cfg;
	gchar *cfg_file, *ruleset = NULL;
	gchar *cfg_path;
	
	/* set the new config */
	cfg = g_key_file_new();
	cfg_path = g_strconcat(g_get_home_dir(), "/.gnome-hearts.cfg", NULL);
	g_assert(g_key_file_load_from_file(cfg, cfg_path, G_KEY_FILE_KEEP_COMMENTS, NULL));
	
	switch (game_rules.ruleset)
	{
		case RULESET_STANDARD: 	  ruleset = "standard";		break;
		case RULESET_OMNIBUS: 	  ruleset = "omnibus"; 		break;
		case RULESET_OMNIBUS_ALT: ruleset = "omnibus_alt";  break;
		case RULESET_SPOT_HEARTS: ruleset = "spot_hearts";  break;
	}
	g_key_file_set_string(cfg, "hearts", "ruleset", ruleset);
	
	g_key_file_set_boolean(cfg, "hearts", "clubs_lead",   game_rules.clubs_lead);
	g_key_file_set_boolean(cfg, "hearts", "hearts_break", game_rules.hearts_break);
	g_key_file_set_boolean(cfg, "hearts", "no_blood", 	  game_rules.no_blood);
	
	g_key_file_set_string(cfg, "hearts", "north_ai", game_rules.north_ai->str);
	g_key_file_set_string(cfg, "hearts", "east_ai",  game_rules.east_ai->str);
	g_key_file_set_string(cfg, "hearts", "west_ai",  game_rules.west_ai->str);

	g_key_file_set_string(cfg, "hearts", "card_style",  game_card_style->str);
	g_key_file_set_string(cfg, "hearts", "background_image", game_background->str);
	g_key_file_set_boolean(cfg, "hearts", "background_tiled", game_background_tiled);
	
	cfg_file = g_key_file_to_data(cfg, NULL, NULL);
	g_key_file_free(cfg);
	
	/* write it to the cfg file */
	GIOChannel *file;
	
	file = g_io_channel_new_file(cfg_path, "w", NULL);
	g_io_channel_write_chars(file, cfg_file, -1, NULL, NULL);
	g_io_channel_unref(file);
	g_free(cfg_path);
	
	/* hide the preferences dialog */
	pref = glade_xml_get_widget(xml, "preferences");
	gtk_widget_hide_all(pref);

	/* display the new game started by changed prefs */
	draw_playingarea();
}

/* hide the preferences */
gboolean prefs_delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	on_prefs_ok_clicked(NULL, NULL);
	return TRUE;
}

/**
 * hint events
 */

/* Hint menu item and button */
void on_hint_event(GtkWidget *widget, gpointer data)
{
	gchar *hint = NULL;
	
	/* create a lua_State if there isn't one yet */
	if (game_hint_lua_state == NULL)
	{
		game_hint_lua_state = lua_state_init();

		gchar* path = g_strconcat(PACKAGE_DATA_DIR"/gnome-hearts/scripts/stock_ai.lua", NULL);
		if (lua_dofile(game_hint_lua_state, path) != 0)
		{
			printf(_("Error: %s"), lua_tostring(game_hint_lua_state, -1));
			g_assert_not_reached();
		}
		g_free(path);
		
		/* set the player ID (offset by one because lua tables start at 1) */
		lua_pushnumber(game_hint_lua_state, user + 1);
		lua_setglobal(game_hint_lua_state, "me");
	}

	hint = game_get_hint();
	
	GtkLabel *hint_text = (GtkLabel*)glade_xml_get_widget(xml, "hint_text");
	gtk_label_set_text(hint_text, hint);
	gtk_widget_show_all(glade_xml_get_widget(xml, "hint"));
	
	g_free(hint);
		
	return;
}

void on_hide_ok_clicked (GtkButton *widget, gpointer *data)
{
	gtk_widget_hide_all(glade_xml_get_widget(xml, "hint"));
	return;
}

/* hide the hint dialog */
gboolean hide_hint_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_widget_hide_all(glade_xml_get_widget(xml, "hint"));
	return TRUE;
}
