/*
 * PPP dialer applet for the GNOME panel
 *
 *  Copyright (c) 1999 by Cody A. Russell
 *  Written by Cody Russell (bratsche@dfw.net or car0969@gamma2.uta.edu)
 *  Graphics by Dietmar Heil (Dietmar.Heil@unidui.uni-duisburg.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTCIULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <gnome.h>
#include <applet-widget.h>
#include <signal.h>
#include "libgnome_ppp/gnome-ppp.h"
#include "ppp-applet.h"

#ifndef GNOMELOCALEDIR
#define GNOMELOCALEDIR "/usr/share/locale"
#endif /* GNOMELOCALEDIR */

void draw_sprite(gpointer data);

static void load_sprites(Dialer *dialer)
{
   GdkImlibImage *im = gdk_imlib_load_image(gnome_unconditional_pixmap_file(IMAGE_FILENAME));
   
   gdk_imlib_render(im, im->rgb_width, im->rgb_height);
   dialer->image = im;
   dialer->pixmap = gdk_imlib_move_image(im);
   dialer->mask = gdk_imlib_move_mask(im);
   
   im = gdk_imlib_load_image(gnome_unconditional_pixmap_file(IMAGE_FILENAME_FILLED));
   gdk_imlib_render(im, im->rgb_width, im->rgb_height);
   dialer->image_filled = im;
   dialer->pixmap_filled = gdk_imlib_move_image(im);
   dialer->mask_filled = gdk_imlib_move_image(im);
}

static gint expose_event(GtkWidget *widget, GdkEventExpose *evt, gpointer d)
{
   GdkGC *gc;
   Dialer *dialer = (Dialer *) d;

   if(!dialer->back_pixmap)
      dialer->back_pixmap = gdk_pixmap_new(dialer->drawing_area->window,
                               dialer->drawing_area->allocation.width,
                               dialer->drawing_area->allocation.height,
                               -1);

   draw_sprite(dialer);
   gc = dialer->drawing_area->style->fg_gc[GTK_STATE_NORMAL];

   gdk_draw_pixmap(widget->window,
                   gc,
                   dialer->back_pixmap,
                   evt->area.x, evt->area.y,
                   evt->area.x, evt->area.y,
                   evt->area.width, evt->area.height);

   return FALSE;
}

static void dialer_destroy(GtkWidget *widget, gpointer data)
{
   Dialer *dialer = (Dialer *) data;

   gtk_timeout_remove(dialer->timer_id);
}

void draw_sprite(gpointer data)
{
   GdkGC *gc;
   Dialer *dialer = (Dialer *) data;

   gc = dialer->drawing_area->style->bg_gc[GTK_STATE_NORMAL];

   gdk_draw_rectangle(dialer->back_pixmap,
                      gc,
                      TRUE, 0, 0,
                      dialer->drawing_area->allocation.width,
                      dialer->drawing_area->allocation.height);

   if ( dialer->filled )
     {
       gdk_draw_pixmap(dialer->back_pixmap,
		       gc,
		       dialer->pixmap_filled,
		       0,
		       0,
		       0,
		       0,
		       WIDGET_WIDTH,
		       WIDGET_HEIGHT);
     }
   else
     {
       gdk_draw_pixmap(dialer->back_pixmap,
		       gc,
		       dialer->pixmap,
		       dialer->frame_num * WIDGET_WIDTH,
		       0,
		       0,
		       0,
		       WIDGET_WIDTH,
		       WIDGET_HEIGHT);
     }
}

gint paint(gpointer data)
{
   GdkRectangle update_area;
   Dialer *dialer = (Dialer *) data;
 
   gdk_draw_rectangle(dialer->back_pixmap,
                      dialer->drawing_area->style->black_gc,
                      TRUE,
                      0, 0,
                      dialer->drawing_area->allocation.width,
                      dialer->drawing_area->allocation.height);
   draw_sprite(dialer);
   update_area.x = 0;
   update_area.y = 0;
   update_area.width = dialer->drawing_area->allocation.width;
   update_area.height = dialer->drawing_area->allocation.height;

   gtk_widget_draw(dialer->drawing_area, &update_area);

   if(dialer->frame_num == 0 && dialer->stop == TRUE)
   {
      gtk_timeout_remove(dialer->timer_id);
   }
   else
   {
      dialer->frame_num++;
      if(dialer->frame_num >= dialer->total_frames)
      {
         dialer->frame_num = 0;
         if(dialer->stop == TRUE)
            gtk_timeout_remove(dialer->timer_id);
      }
   }

   return 1;
}

void start_animation(Dialer *dialer)
{
   dialer->stop = FALSE;
   dialer->filled = FALSE;
   dialer->timer_id = gtk_timeout_add(dialer->animation_speed, paint, dialer);
}

void stop_animation(Dialer *dialer)
{
   dialer->stop = TRUE;
   dialer->filled = FALSE;
   paint(dialer);
}

void stop_animation_filled(Dialer *dialer)
{
  dialer->stop = TRUE;
  dialer->filled = TRUE;
}

static Dialer *malloc_dialer(GtkWidget *applet)
{
   Dialer *dialer;
   gint x;

   dialer = g_malloc(sizeof(Dialer));

   dialer->account = NULL;
   dialer->state = PPP_DISCONNECTED;
   dialer->timer_id = -1;
   dialer->applet = applet;
   dialer->menu = gtk_menu_new();
   dialer->account_list = account_list();
   dialer->frame_num = 0;
   dialer->total_frames = 36;
   dialer->back_pixmap = NULL;
   dialer->pixmap = NULL;
   dialer->pixmap_filled = NULL;
   dialer->mask = NULL;
   dialer->stop = FALSE;
   dialer->filled = FALSE;
   dialer->property_box = NULL;
   dialer->account = NULL;
   if((dialer->animation_speed = gnome_config_get_int("/gnome-ppp/applet/animation_speed")) == 0)
      dialer->animation_speed = 100;
   load_sprites(dialer);

   return dialer;
}

static void free_dialer(Dialer *dialer)
{
   g_free(dialer);
}

static void create_dialer_applet(Dialer *dialer)
{
   GtkWidget *frame;
   GtkWidget *vbox;
   GtkWidget *separator;
   GtkWidget *menu_item;
   GtkWidget *hbox;

   vbox = gtk_vbox_new(TRUE, 0);

   dialer->drawing_area = gtk_drawing_area_new();
   gtk_drawing_area_size(GTK_DRAWING_AREA(dialer->drawing_area),
                         WIDGET_WIDTH, WIDGET_HEIGHT);
   gtk_object_set_data(GTK_OBJECT(dialer->drawing_area), "dialer", dialer);
   gtk_box_pack_start(GTK_BOX(vbox), dialer->drawing_area, TRUE, TRUE, 0);

   gtk_signal_connect(GTK_OBJECT(dialer->drawing_area),
                      "button_press_event",
                      GTK_SIGNAL_FUNC(button_press),
                      dialer);
   gtk_signal_connect(GTK_OBJECT(dialer->drawing_area),
                      "expose_event",
                      GTK_SIGNAL_FUNC(expose_event),
                      dialer);
   gtk_widget_set_events(dialer->drawing_area, GDK_EXPOSURE_MASK |
                        GDK_BUTTON_PRESS_MASK);

   gtk_widget_show(dialer->drawing_area);

   gtk_signal_connect(GTK_OBJECT(dialer->menu), "deactivate",
	 	      GTK_SIGNAL_FUNC(hide_widgets),
                      NULL);
   menu_item = gtk_menu_item_new_with_label("Raise PPP Link");
   gtk_widget_set_sensitive(menu_item, FALSE);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		      GTK_SIGNAL_FUNC(link_raise_cb),
		      dialer);
   gtk_widget_show(menu_item);
   gtk_object_set_data(GTK_OBJECT(dialer->menu), "dialer", dialer);
   gtk_object_set_data(GTK_OBJECT(dialer->menu), "raise_item",
		       menu_item);
   gtk_menu_append(GTK_MENU(dialer->menu), menu_item);

   menu_item = gtk_menu_item_new_with_label("Lower PPP Link");
   gtk_widget_set_sensitive(menu_item, FALSE);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		      GTK_SIGNAL_FUNC(link_lower_cb),
		      dialer);
   gtk_widget_show(menu_item);
   gtk_object_set_data(GTK_OBJECT(dialer->menu), "lower_item",
		       menu_item);
   gtk_menu_append(GTK_MENU(dialer->menu), menu_item);

   build_submenu(dialer);

   applet_widget_register_stock_callback(APPLET_WIDGET(dialer->applet),
					"about",
					GNOME_STOCK_MENU_ABOUT,
					_("About..."),
					about_cb,
					dialer);
   applet_widget_register_stock_callback(APPLET_WIDGET(dialer->applet),
                                         "properties",
                                         GNOME_STOCK_MENU_PROP,
                                         _("Properties..."),
                                         properties,
                                         dialer);

   gtk_widget_show(vbox);
   applet_widget_add(APPLET_WIDGET(dialer->applet), vbox);

   gtk_widget_show(dialer->applet);
}

static void select_account_cb(GtkWidget *widget, Account *account)
{
   GtkWidget *tmp;

   Dialer *dialer = gtk_object_get_data(GTK_OBJECT(widget), "dialer");
   dialer->account = account;
   tmp = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(dialer->menu),
                                           "raise_item");
   gtk_widget_set_sensitive(tmp, TRUE);
}

static void add_submenu_item(gpointer data, gpointer user_data)
{
   GtkWidget *menu_item;
   Account *account = (Account *) data;
   Dialer *dialer = (Dialer *) user_data;

   menu_item = gtk_menu_item_new_with_label(account->name->str);
   gtk_object_set_data(GTK_OBJECT(menu_item), "dialer",
                       (gpointer) dialer);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		     GTK_SIGNAL_FUNC(select_account_cb),
		     account);

   gtk_widget_show(menu_item);
   gtk_menu_append(GTK_MENU(dialer->submenu), menu_item);
}

static void build_submenu(Dialer *dialer)
{
   GtkWidget *menu;
   GtkWidget *menu_item;
   GtkWidget *submenu;
   gint length;

   length = g_list_length(dialer->account_list);
   if(length != 1)
   {
      submenu = gtk_menu_new();
      dialer->submenu = submenu;
      menu_item = gtk_menu_item_new_with_label("Select Account");
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu);
      gtk_menu_append(GTK_MENU(dialer->menu), menu_item);
      gtk_widget_show(menu_item);

      if(length > 1)
      {
         g_list_foreach(dialer->account_list, add_submenu_item, dialer);
      }
      else
      {
         menu_item = gtk_menu_item_new_with_label("No available hosts to dial");
         gtk_widget_set_sensitive(menu_item, FALSE);
         gtk_widget_show(menu_item);
         gtk_menu_append(GTK_MENU(dialer->submenu), menu_item);
      }
   }
   else if(length == 1)
   {
      dialer->account = g_list_nth_data(dialer->account_list, 0);
      menu_item = gtk_object_get_data(GTK_OBJECT(dialer->menu), "raise_item");
      gtk_widget_set_sensitive(menu_item, TRUE);
   }
}

static void dialog_destroy_cb(GtkWidget *widget, gpointer data)
{
   Dialer *dialer = (Dialer *) data;

   dialer->input_dialog = NULL;
}

static void input_dialog_cb(gchar *str, gpointer data)
{
   Dialer *dialer;

   dialer = (Dialer *) data;
   connect_send(dialer->account, str, TRUE);
}

static void set_raise_active(Dialer *dialer, gboolean active)
{
   GtkWidget *tmp;

   tmp = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(dialer->menu),
					   "raise_item");
   gtk_widget_set_sensitive(tmp, active);
   tmp = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(dialer->menu),
					   "lower_item");
   gtk_widget_set_sensitive(tmp, ! active );
}

static void ppp_message_handler(Account *account, PPPMessageType msg,
                                gchar *text, gpointer data)
{
   Dialer *dialer = (Dialer *) data;
   gint update_tooltip = FALSE;

   switch(msg)
   {
      case PPP_DISCONNECTED:
         dialer->state = PPP_DISCONNECTED;
         update_tooltip = TRUE;
	 set_raise_active(dialer, TRUE);
        break;
      case PPP_IN_PROGRESS:
         dialer->state = PPP_IN_PROGRESS;
         update_tooltip = TRUE;
         break;
      case PPP_CONNECTED:
         dialer->state = PPP_CONNECTED;
         update_tooltip = TRUE;
	 stop_animation_filled(dialer);
         break;
      case PPP_ERROR:
         gnome_error_dialog(_(text));
	 lower_link(dialer);
         break;
      case PPP_DEBUG:
         break;
      case PPP_PROMPT_INPUT:
         if(dialer->input_dialog)
            g_error("dialer->input_dialog != NULL");

         dialer->input_dialog = gnome_request_dialog(
                                  TRUE,
                                  _(text),
                                  "",
                                  100,
                                  (GnomeStringCallback) input_dialog_cb,
                                  dialer,
                                  NULL);
         gtk_signal_connect(GTK_OBJECT(dialer->input_dialog),
                            "destroy",
                            GTK_SIGNAL_FUNC(dialog_destroy_cb),
                            dialer);
         break;
      default:
         g_error("unknown callback message");
   }

   return;
}

static int lower_link(Dialer *dialer)
{
   set_raise_active(dialer, TRUE);

   connect_stop(dialer->account);
   account_unlock(dialer->account);
   stop_animation(dialer);

   return 0;
}

static int raise_link(Dialer *dialer)
{
   if(account_is_locked(dialer->account))
   {
      gnome_error_dialog(_("That account is already in use.\n"));
   }
   else
   {
      account_lock(dialer->account);

      set_raise_active(dialer, FALSE);

      if(connect_start(dialer->account, ppp_message_handler, dialer))
      {
         start_animation(dialer);
      }
   }

   return 0;
}

static void menu_position_func(GtkMenu *menu, gint *x, gint *y,
                               gpointer data)
{
   GtkWidget *button;
   Dialer *dialer;
   int wx, wy;
   int ax, ay;

   button = GTK_WIDGET(data);
   dialer = gtk_object_get_data(GTK_OBJECT(menu), "dialer");
   *x = *y = 0;
   gdk_window_get_origin(button->window, &wx, &wy);

   switch(applet_widget_get_panel_orient(APPLET_WIDGET(dialer->applet)))
   {
      case ORIENT_UP:
         ay = GTK_WIDGET(menu)->requisition.height;
         *x = wx;
         *y = wy - ay;
         break;

      case ORIENT_DOWN:
         ay = GTK_WIDGET(menu)->requisition.height;
         *x = wx;
         *y = wy + WIDGET_HEIGHT;
         break;

      case ORIENT_RIGHT:
         ax = GTK_WIDGET(menu)->requisition.width;
         *x = wx + WIDGET_WIDTH;
         *y = wy;
         break;

      case ORIENT_LEFT:
         ax = GTK_WIDGET(menu)->requisition.width;
         *x = wx - ax;
         *y = wy;
         break;

      default:
         g_error("Unknown panel orientation!\n");
   }
}

static void hide_widgets(GtkWidget *widget, GtkWidget *toggle)
{
   /* Remove this soon */
   return;
}

static void link_raise_cb(GtkWidget *widget, Dialer *dial)
{
   raise_link(dial);
}

static void link_lower_cb(GtkWidget *widget, Dialer *dialer)
{
   lower_link(dialer);
}

static void about_cb(AppletWidget *widget, gpointer data)
{
   GtkWidget *about;
   Dialer *dialer = (Dialer *) data;

   static const gchar *authors[] = {
     "Cody Russell (car0969@gamma2.uta.edu or bratsche@dfw.net)",
     "Graphics by Dietmar Heil (Dietmar.Heil@unidui.uni-duisburg.de)",
     NULL
   };

   dialer->about =
      gnome_about_new(_("GNOME PPP Dialer Applet"), VERSION,
			"(C) 1999 Cody Russell",
			authors,
                        _("An unnecessarily nice looking applet for GNOME's "
                        "panel for easy control of PPP network links.\n"
                        "This program is free software; you can redistribute "
                        "it and/or modify it under the terms of the GNU "
                        "General Public License as published by the Free "
                        "Software Foundation; either version 2, or (at your "
                        "option) any later version."),
			NULL);
  gtk_widget_show(dialer->about);
}

static void prop_apply_cb(GtkWidget *widget, gint button, gpointer data)
{
   Dialer *dialer = (Dialer *) data;

   dialer->animation_speed = dialer->tmp_speed;
   gnome_config_set_int("/gnome-ppp/applet/animation_speed", dialer->animation_speed);

   return;
}

static void prop_destroy_cb(GtkWidget *widget, gpointer data)
{
   Dialer *dialer = (Dialer *) data;

   gtk_widget_destroy(dialer->property_box);
   dialer->property_box = NULL;
}

static void speed_adjustment_cb(GtkAdjustment *adj, gpointer data)
{
   Dialer *dialer = (Dialer *) data;

   dialer->tmp_speed = (gint) adj->value;
   gnome_property_box_changed(GNOME_PROPERTY_BOX(dialer->property_box));
}

static void properties(AppletWidget *applet, gpointer data)
{
   GtkWidget *hbox;
   GtkWidget *vbox;
   GtkWidget *slider;
   GtkWidget *radio;
   GtkWidget *animation_select;
   GtkWidget *x_size;
   GtkWidget *y_size;
   GtkWidget *frame;
   GtkWidget *label;
   GtkWidget *hscale;
   GSList *group;
   GtkWidget *box2;
   GtkObject *adjust;

   static GnomeHelpMenuEntry help = {
          NULL, "ppp-applet-properties"
   };
   Dialer *dialer = (Dialer *) data;

   help.name = gnome_app_id;

   if(dialer->property_box != NULL)
   {
      gdk_window_raise(dialer->property_box->window);
      return;
   }

   dialer->property_box = gnome_property_box_new();
   gtk_window_set_title(GTK_WINDOW(dialer->property_box),
                        _("PPP Applet Properties"));
   gtk_signal_connect(GTK_OBJECT(dialer->property_box), "apply",
                      GTK_SIGNAL_FUNC(prop_apply_cb), dialer);
   gtk_signal_connect(GTK_OBJECT(dialer->property_box), "destroy",
                      GTK_SIGNAL_FUNC(prop_destroy_cb), dialer);
   gtk_signal_connect(GTK_OBJECT(dialer->property_box), "help",
                      GTK_SIGNAL_FUNC(gnome_help_pbox_display), &help);
   gtk_widget_show(dialer->property_box);

   hbox = gtk_hbox_new(FALSE, 0);
   gtk_widget_show(hbox);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
   gtk_widget_show(vbox);

   box2 = gtk_vbox_new(FALSE, 0);
   frame = gtk_frame_new(_("Timer display"));
   radio = gtk_radio_button_new_with_label(NULL, _("None"));
   gtk_box_pack_start(GTK_BOX(box2), radio, TRUE, TRUE, 0);
   gtk_widget_show(radio);
   group = gtk_radio_button_group(GTK_RADIO_BUTTON(radio));
   radio = gtk_radio_button_new_with_label(group, _("Tooltip"));
   gtk_box_pack_start(GTK_BOX(box2), radio, TRUE, TRUE, 0);
   gtk_widget_show(radio);
   radio = gtk_radio_button_new_with_label(
               gtk_radio_button_group(GTK_RADIO_BUTTON(radio)),
               _("Label"));
   gtk_box_pack_start(GTK_BOX(box2), radio, TRUE, TRUE, 0);
   gtk_widget_show(radio);
   gtk_container_add(GTK_CONTAINER(frame), box2);
   gtk_widget_show(box2);
   gtk_widget_show(frame);
   gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   frame = gtk_frame_new(_("Animation Speed"));
   gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
   gtk_widget_show(frame);
   adjust = gtk_adjustment_new(50.0, 1.0, 300.0, 1.0, 1.0, 0.0);
   gtk_signal_connect(adjust, "value_changed",
                      GTK_SIGNAL_FUNC(speed_adjustment_cb), dialer);
   hscale = gtk_hscale_new(GTK_ADJUSTMENT(adjust));
   gtk_container_add(GTK_CONTAINER(frame), hscale);
   gtk_scale_set_digits(GTK_SCALE(hscale), 0);
   gtk_widget_show(hscale);
   gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
   gtk_widget_show(vbox);

   label = gtk_label_new(_("PPP Applet"));
   gnome_property_box_append_page(GNOME_PROPERTY_BOX(dialer->property_box),
                                  hbox, label);
}

static gint button_press(GtkWidget *widget, GdkEventButton *evt)
{
   Dialer *dialer = gtk_object_get_data(GTK_OBJECT(widget), "dialer");

   if(evt->button == 1)
      gtk_menu_popup(GTK_MENU(dialer->menu), 0, 0,
                     (GtkMenuPositionFunc) menu_position_func,
                     GTK_OBJECT(dialer->drawing_area),
                     evt->button, evt->time);

   return TRUE;
}

gint main(gint argc, gchar **argv)
{
   GtkWidget *applet;
   Dialer *dialer;
   gchar *goad_id;

   bindtextdomain(PACKAGE, GNOMELOCALEDIR);
   textdomain(PACKAGE);

   applet_widget_init ("PPP_Applet", VERSION, argc, argv, NULL, 0, NULL);
   gnomeppplib_init();
   account_load();

   goad_id = (gchar *) goad_server_activation_id();
   if(goad_id && !strcmp(goad_id, "PPP_Applet"))
   {
      applet = applet_widget_new("PPP_Applet");
      if(!applet)
         g_error("Cannot create applet!\n");

      dialer = malloc_dialer(applet);
      dialer->account_list = account_list();
      create_dialer_applet(dialer);
   }

   applet_widget_gtk_main();
   free_dialer(dialer);

   return 0;
}
