/**
 ** mp3info.c - handles loading of song length information
 **
 ** Some code taken from mpg123 and splay.
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gnomp3.h"
#include "mp3info.h"
#include "mp3list.h"
#include <pthread.h>
/*
 * Since determining the play time of an mp3 involves opening the mp3 file and
 * reading its headers, this can be a slow process. As a result songs times
 * to be determined are queued so that they can be processed in the backgroud
 * while the rest of the application can be used.
 */

pthread_t mp3info_thread;

/* 
 * Verify that the header given is a valid mp3 header 
 */
int mp3info_header_check(unsigned long head)
{
	if ((head & 0xffe00000) != 0xffe00000)
		return FALSE;
	if (!((head >> 17) & 3))
		return FALSE;
	if (((head >> 12) & 0xf) == 0xf)
		return FALSE;
	if (!((head >> 12) & 0xf))
		return FALSE;
	if (((head >> 10) & 0x3) == 0x3)
		return FALSE;
	if (((head >> 19) & 1) == 1 && ((head >> 17) & 3) == 3 && ((head >> 16) & 1) == 1)
		return FALSE;
	if ((head & 0xffff0000) == 0xfffe0000)
		return FALSE;
	
	return TRUE;
}

void mp3info_extract_id3(MP3 *mp3, FILE *fp)
{
    char buff[200];
    char title[31]={0,};
    char artist[31]={0,};
    char album[31]={0,};
    char year[5]={0,};
    char comment[31]={0,};
    unsigned char *genre_number = &buff[127];

    if( fseek(fp, -128, SEEK_END ) == -1 ){
	return;
    }
    fread(buff, sizeof(char), sizeof(buff), fp);
    
    if(buff[0] == 'T' && buff[1] == 'A' && buff[2] == 'G'){
	strncpy(title,   &buff[3],  30); g_strstrip(title);
	strncpy(artist,  &buff[33], 30); g_strstrip(artist);
	strncpy(album,   &buff[63], 30); g_strstrip(album);
	strncpy(year,    &buff[93], 4);  g_strstrip(year);
	strncpy(comment, &buff[97], 30); g_strstrip(comment);
	
	mp3->id3_title   = g_strdup(title);
	mp3->id3_artist  = g_strdup(artist);
	mp3->id3_album   = g_strdup(album);
	mp3->id3_year    = g_strdup(year);
	mp3->id3_comment = g_strdup(comment);
	mp3->id3_genre   = *genre_number;
    
	mp3->display_name_from_id3 = g_strdup_printf("%s - %s", artist,title); 
    }
    //else printf("<>{}[] %s\n", mp3->filename );
}

/* 
 * load the mp3 header information from a file. mp3info->path must be set to 
 * pointt to the mp3 file.
 * RETURNS: 0 on success, else -1
 */
int mp3info_get(MP3 *mp3)
{
    FILE *fp;
    unsigned long header;
//  struct mp3info 
    int version;
    int layer;
    int frequency;
    int bitrate;
    int filesize;
    
    int bitrateindex, freq;
    int padding, framesize;
    int totalframes;
    char msg[256];
    struct stat st;
    
    
    // For header
    int frequencies[2][3]= {
	{44100,48000,32000}, // MPEG 1
	{22050,24000,16000}  // MPEG 2
    };
  
    int bitrates[2][3][15]={
	// MPEG 1
	{{0,32,64,96,128,160,192,224,256,288,320,352,384,416,448},
	 {0,32,48,56,64,80,96,112,128,160,192,224,256,320,384},
	 {0,32,40,48,56,64,80,96,112,128,160,192,224,256,320}},
	
	// MPEG 2
	{{0,32,48,56,64,80,96,112,128,144,160,176,192,224,256},
	 {0,8,16,24,32,40,48,56,64,80,96,112,128,144,160},
	 {0,8,16,24,32,40,48,56,64,80,96,112,128,144,160}}
    };

    mp3->play_time = -1;
    mp3->file_time = 0;

    fp = fopen( mp3->filename, "r" );
    if(!fp){
	snprintf( msg, 256, "Could not open mp3 '%s'", mp3->filename );
	perror(msg);
	return -1;
    }

    if( fstat(fileno(fp), &st) != -1 ){
	mp3->file_time = st.st_mtime;
    }

    mp3info_extract_id3(mp3, fp);

    /* find out the file size */
    fseek( fp, 0, SEEK_END );
    filesize = ftell(fp);
    rewind(fp);
    
    header = getc(fp) << 24;
    header |= getc(fp) << 16;
    header |= getc(fp) << 8;
    header |= getc(fp);
    
    while (!mp3info_header_check(header)){
	header <<= 8;
	header |= getc(fp);
	
	if( feof(fp) ){
	    fclose(fp);
	    return -1;
	}
    }
    
    
    layer = 4 - ((header >> 17) & 3);
    version = (header & (1 << 19)) ? 0x0 : 0x1;
    version++;
    
    if( layer < 1 || layer > 3 || version < 1 || version > 2){
	fclose(fp);
	printf("%s: -- Bad data in header (invalid mpeg version or layer)\n", 
	       mp3->filename);
	printf("Version: %d Layer: %d\n", version, layer);
	return -1;
    }
    
    padding = ((header >> 9) & 0x1);  
    freq = ((header >> 10) & 0x3);
    
    bitrateindex = ((header >> 12) & 0xf);
    
    if( bitrateindex < 1 || bitrateindex > 14 || freq < 0 ||freq > 2){
	fclose(fp);
	printf("%s: -- Bad data in header (invalid bitrate or frequency)\n",
	       mp3->filename);
	return -1;
    }
    
    bitrate = bitrates[version-1][layer-1][bitrateindex];
    frequency = frequencies[version-1][freq];
    
    framesize = (144000 * bitrate) / (  (frequency)<<(version-1) );
    if(padding)framesize++;
    
    totalframes = ( filesize + framesize - 1) / framesize;
    mp3->play_time = (totalframes * framesize) / (bitrate * 125);
    
    fclose(fp);
    
    return 0;
}

/*
 * Determines if the given song has been added to a list, and if it has, 
 * updates display of the playtime for that clist.
 */
void mp3info_set_times(MP3 *mp3)
{
    //printf("mp3->filename = %s .... %d <><> %d\n", mp3->filename, mp3->play_time, mp3->row_dirlist  );
    if(gnomp3.use_id3 && mp3->display_name_from_id3)
	mp3->display_name = mp3->display_name_from_id3;
    else
	mp3->display_name = mp3->display_name_from_filename;

    if( mp3->row_playlist >= 0 ){
	gtk_clist_set_text( GTK_CLIST(gnomp3.play_clist), mp3->row_playlist, 1,
			    mp3->display_name);
	gtk_clist_set_text( GTK_CLIST(gnomp3.play_clist), mp3->row_playlist, 2,
			    mp3list_build_time(mp3));
    }
    if( mp3->row_alllist >= 0 ){
	gtk_clist_set_text( GTK_CLIST(gnomp3.all_clist), mp3->row_alllist, 0, 
			    mp3->display_name);
	gtk_clist_set_text( GTK_CLIST(gnomp3.all_clist), mp3->row_alllist, 1, 
			    mp3list_build_time(mp3));
    }
    if( mp3->row_dirlist >= 0 ){
	gtk_clist_set_text( GTK_CLIST(gnomp3.dir_clist), mp3->row_dirlist, 0, 
			    mp3->display_name);
	gtk_clist_set_text( GTK_CLIST(gnomp3.dir_clist), mp3->row_dirlist, 1, 
			    mp3list_build_time(mp3));
    }
    if( mp3->row_timelist >= 0 ){
	gtk_clist_set_text( GTK_CLIST(gnomp3.time_clist), mp3->row_timelist, 0,
 			    mp3->display_name);
	gtk_clist_set_text( GTK_CLIST(gnomp3.time_clist), mp3->row_timelist, 1,
 			    mp3list_build_time(mp3));
    }
    if( mp3->row_songlist){
	gtk_ctree_node_set_text( GTK_CTREE(gnomp3.song_ctree), mp3->row_songlist, 0, mp3->display_name );
	gtk_ctree_node_set_text( GTK_CTREE(gnomp3.song_ctree), mp3->row_songlist, 1, mp3list_build_time(mp3) );
    }
}

/*
 * Finds mp3s in the global list that have not had their play_time set.
 * It also matches the condition "cond", which is why it is written as a macro 
 */
#define mp3info_process_list(cond) \
for(ptr = mp3list; ptr; ptr = ptr->next){ \
    mp3 = ptr->data; \
    if( mp3->play_time == 0 && cond ){ \
	mp3info_get(mp3); \
/* FIXME cache timeout */ \
    } \
}

void *mp3info_thread_func(void *arg)
{
    GList *ptr = mp3list;
    MP3 *mp3;

    while(1){
	/* do the playlist first */
	mp3info_process_list( mp3->row_playlist >= 0 );
	mp3info_process_list( mp3->row_dirlist >= 0 );
	mp3info_process_list( mp3->row_alllist >= 0 );
	sleep(1);
    }

}

/*
 * timeout to display mp3 playtimes in the clists
 */
int mp3info_display_timeout(gpointer data)
{
    static GList *current = NULL;

    if(!current)
	current = mp3list;
    if(!current)
	return TRUE;

    mp3info_set_times(current->data);
    current = current->next;
    return TRUE;
}

void mp3info_start()
{
    pthread_create(&mp3info_thread, NULL, mp3info_thread_func, NULL);
    gtk_timeout_add(5, mp3info_display_timeout, NULL);
}





