#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Instance.py 7065 2005-02-25 14:00:35Z kilo $

"""
Creates a "Form" instance.
"""


__all__ = ['Instance']

from wxPython.wx import *

from gnue.designer.forms import properties as formProperties
from gnue.designer.base.Instance import BaseInstance
from gnue.designer.base import PopupMenu
from gnue.designer.base.ObjectList import ObjectList
from WizardRunner import WizardRunner
from Debugger import DebugSession
from Incubator import Incubator
from events.Events import registerEvents

# Wizards
import wizards

# Tool support...
from LayoutEditor import LayoutEditor
from TreeView import TreeView
from PropertyEditor import PropertyEditor
from EventEditor import EventEditor
from BlockEditor import BlockEditor
from gnue.designer.base.tools.TriggerEditor import TriggerEditor
from gnue.designer.base.tools.DataSourceEditor import DataSourceEditor
##from gnue.designer.base.tools.SchemaViewer import SchemaViewer
from gnue.designer.base.tools.SchemaNavigator import SchemaNavigator

from gnue.designer.base.PopupMenu import ObjectMenu

from gnue.forms import GFInstance
from gnue.forms import GFForm, GFParser, GFObjects, GFLibrary
from gnue.forms.GFObjects.GFBlock import GFBlock
from gnue.common.logic import GTrigger
from gnue.common.apps import GDebug
from gnue.common.datasources import GDataSource

class Instance(BaseInstance, GFInstance.GFInstance):

  def __init__(self, app, *args, **params):

    self.incubator = Incubator(self)
    self.properties = formProperties
    self.wizardRunner = WizardRunner
    self.debugLevel = '1'

    # TODO: Can we make it so the GFInstance is not
    # TODO: part of the main Instance class, but an
    # TODO: attribute such as Instance.formInstance?

    GFInstance.GFInstance.__init__(self, app,
                                   app.connections,
                                   None, disableSplash=1)

    # For GFInstance support, iirc
    self._pages = []

    # Build a list of positionable widgets
    self._positionableList = []
    elements = self.incubator.elements
    for tag in elements.keys():
      try:
        if elements['Positionable']:
          self._positionableList.append(tag)
      except KeyError:
        pass


    BaseInstance.__init__(self, app, *args, **params)

    self.registerEventListeners({
                       'ObjectModified' : self.__onModifyObject,
                       'ObjectDeleted' :  self.__onDeleteObject,
                       'Forms:RunForm': self.__onRunForm,
                       'Forms:RequestDebugLevel': self.__onSetDebugLevel
                      })

    # Add our events/* stuff
    registerEvents(self)

    # Convenience list to keep track of datasources and blocks
    # TODO: Anything that needs either of these two
    # TODO: should call getObjectList directly.
    # TODO: Left for historical reasons.
    self.datasources = self.getObjectList('datasource')
    self.blocks = self.getObjectList('block')


  def loadBuffer(self, buffer):
    form = GFParser.loadFile (buffer, self, initialize=0)
##    if not hasattr(form,'name') or not form.name:
##      form.name = 'form'
    return form


  def loadEmpty(self, style=None):
    form = GFForm.GFForm()
    options = GFObjects.GFOptions(form)
    form.title = _("Untitled Form")
    form.height = 12
    GFObjects.GFLogic(form)
    layout = GFObjects.GFLayout(form)
    layout.Char__width = 40
    layout.Char__height = 12
    GFObjects.GFPage(layout)
    return form


  def inventoryObject(self, object):
    if object._type == 'GFForm':
      object._blockMap = {}
    elif object._type == 'GFLogic':
      object._parent._logic = object
    elif object._type == 'GFLayout':
      object._parent._layout = object
      # Yes, you guessed it! More layout mgmt hackery...
      object._xmlchildnamespaces = {'Char':'GNUe:Layout:Char'}
    elif object._type == 'GFBlock':
      if self.rootObject._blockMap.has_key(object.name):
        raise "Warning: Multiple blocks with name %s" % object.name
      self.rootObject._blockMap[object.name] = object
      object._fieldMap = {}
    elif object._type == 'GFField':
      object.findParentOfType('GFBlock')._fieldMap[object.name] = object
    elif object._type == 'GFEntry':
      object._block = self.rootObject._blockMap[object.block]
      object._field = object._block._fieldMap[object.field]
    elif isinstance(object, GFObjects.GFPage) and \
        object._parent == self.rootObject:
      self._pages.append(object)

    object._popupMenu = ObjectMenu(self, object)


  def __onModifyObject(self, event):

    object = event.object

    #
    # Maintain all those fun dicts/lists created in inventoryObject()
    #
    old = event.old
    if object._type == 'GFBlock':
      try:
        del self.rootObject._blockMap[old['name']]
      except KeyError:
        pass
      self.rootObject._blockMap[object.name] = object

    elif object._type == 'GFField':
      block = object.findParentOfType('GFBlock')
      try:
        del block._fieldMap[old['name']]
      except KeyError:
        pass
      block._fieldMap[object.name] = object


  def __onDeleteObject(self, event):
    object = event.object
    if object._type == 'GFPage':
      self._pages.remove(object)

    # Maintain all those fun dicts/lists created in inventoryObject()
    if object._type == 'GFBlock':
      del self.rootObject._blockMap[object.name]
    elif object._type == 'GFField':
      del object.findParentOfType('GFBlock')._fieldMap[object.name]
    elif isinstance(object, GFObjects.GFPage) and \
        object._parent == self.rootObject:
      del self._pages[object]

    object._popupMenu = ObjectMenu(self, object)


  def createTools(self):
    self.addTool('visualEditor',   _('Layout Editor'), LayoutEditor,
                      hotkey='F11', menuGroup=301)
    self.addTool('propertyEditor', _('Property Inspector'), PropertyEditor,
                      hotkey='F10', menuGroup=301)
    self.addTool('treeEditor',     _('Object Navigator'), TreeView,
                      hotkey='F8', menuGroup=311)

    self.addTool('schemaNavigator',_('Schema Navigator'), SchemaNavigator,
                      menuGroup=301)
    self.addTool('datasourceEditor',_('Data Source Editor'), DataSourceEditor,
                      hotkey='F9', menuGroup=311)
    self.addTool('blockEditor',   _('Block Editor'), BlockEditor,
                      hotkey='F5', menuGroup=311)
    #self.addTool('fieldEditor',   _('Field Editor'), FieldEditor,
    #                  hotkey='F6', menuGroup=201)

    self.addTool('eventEditor',    _('Trigger Mappings'), EventEditor,
                      hotkey='F7', menuGroup=321)
    self.addTool('triggerEditor',  _('Trigger Editor'), TriggerEditor,
                      hotkey='F12', menuGroup=321)

  def createWizards(self):
    self.loadWizards(wizards)

  # Instances can add to the primary toolbar
  def initToolBar(self):
    self.toolbar.addToolbarMapping(';Forms:RunForm,tb_exec;')


  def initMenu(self):
    # Add the [sub]menus
    for location, text, grouping in (
       ('Modify|Arrange Items',_('&Arrange Items'), 501),):

      self.menubar.addMenu(location, text, grouping)

    for location, event, text, hotkey, help, grouping, canToggle in (
       ('View', 'LayoutEditor:ZoomIn', _("Zoom In"), None, _("Zoom in"), 11.1, 0),
       ('View', 'LayoutEditor:ZoomOut', _("Zoom Out"), None, _("Zoom out"), 11.2, 0),
       ('View', 'LayoutEditor:ShowFocus', _("Show Focus Order"), None, _("Show the navigation order on the layout editor"), 21, 1),

       ('Tools','Forms:RunForm', _("Run Form"), None, _("Display and run the current form"), 101.1, 0),
       ('Tools','Forms:RequestDebugLevel', _("Change debugging level..."), None, _("Change the debugging output level for the Run Form option"), 101.2, 0),

       ('Modify|Align Items','Forms:AlignLeft', _("&Align Left Edges"), None, _("Horizontally align the currently selected items along their left edges"),101.1, 0),
       ('Modify|Align Items','Forms:AlignRight', _("&Align Right Edges"), None, _("Horizontally align the currently selected items along their right edges"),101.2, 0),
       ('Modify|Align Items','Forms:AlignHCenter', _("&Align Horizontal Center"), None, _("Horizontally align the currently selected items along their centers"),101.2, 0),
       ('Modify|Align Items','Forms:AlignTop', _("&Align Top Edges"), None, _("Vertically align the currently selected items along their top edges"),111.1, 0),
       ('Modify|Align Items','Forms:AlignBottom', _("&Align Bottom Edges"), None, _("Vertically align the currently selected items along their bottom edges"),111.2, 0),
       ('Modify|Align Items','Forms:AlignVCenter', _("&Align Vertical Center"), None, _("Vertically align the currently selected items along their centers"),111.3, 0),
       ('Modify|Align Items','Forms:SmartAlign', _("&Smart Align Labels/Fields"), None, _("Align Labels and fields into a block formation, with all fields left aligned and all labels left aligned."),121.1, 0),

       ('Modify','LayoutEditor:FocusOrder', _("&Set Focus Order Manually"), None, _("Set the focus order by hand"),201.2, 0),
       ('Modify','LayoutEditor:AutoArrange', _("&Reset Focus Order by Position"), None, _("Automatically set the focus order based on an items position"),201.3, 0)):
      self.menubar.addAction(location, text, event,
                      grouping, canDisable=1, canToggle=canToggle,
                      icon=None, hotkey=hotkey, help=help)



  def preSave(self):


    # Get rid of form.name = '__main__'
    try:
      if self.rootObject.name == '__main__':
        del self.rootObject.name
    except AttributeError:
      pass

    # Do a little rearranging of objects

    options = []
    imports = []
    databases = []
    datasources = []
    triggers = []
    logic = []
    layout = []
    other = []

    for child in self.rootObject._children:
      if isinstance(child, GFObjects.GFOptions):
        options.append(child)
      elif isinstance(child, GFLibrary.GFImport):
        imports.append(child)
      elif isinstance(child, GDataSource.GConnection):
        databases.append(child)
      elif isinstance(child, GFObjects.GFDataSource) or \
        (isinstance(child, GFLibrary.GFImportItem) and
         child._type == 'GFimport-datasource'):
        datasources.append(child)
      elif isinstance(child, GTrigger.GTrigger) or \
        (isinstance(child, GFLibrary.GFImportItem) and
         child._type == 'GFimport-trigger'):
        triggers.append(child)
      elif isinstance(child, GFObjects.GFLayout) or \
        (isinstance(child, GFLibrary.GFImportItem) and
         child._type == 'GFimport-layout'):
        layout.append(child)
      elif isinstance(child, GFObjects.GFLogic) or \
        (isinstance(child, GFLibrary.GFImportItem) and
         child._type == 'GFimport-logic'):
        logic.append(child)
      else:
        other.append(child)


    # Do a little sanity check before saving...
    if len(logic[0]._children):
      valid = 1
    else:
      valid = 0  # No blocks! :(

    if not valid:
      if wxMessageDialog(self,
          _('The form definition you are about to save\n' + \
          'does not appear to be a workable definition.' + \
          '\n\nIf you save an incomplete definition, you\n' + \
          'may not be able to reopen it in Designer.\n\nSave anyway?'),
             _("Incomplete Form Definition"), wxYES_NO|wxICON_QUESTION ).ShowModal() == wxID_NO:
        return 1

    # Reorder the children so items of same class are grouped
    self.rootObject._children = []
    for child in options:
      self.rootObject._children.append(child)
    for child in imports:
      self.rootObject._children.append(child)
    for child in databases:
      self.rootObject._children.append(child)
    for child in datasources:
      self.rootObject._children.append(child)
    for child in triggers:
      self.rootObject._children.append(child)
    for child in logic:
      self.rootObject._children.append(child)
    for child in layout:
      self.rootObject._children.append(child)
    for child in other:
      self.rootObject._children.append(child)

  # Used by TemplateParser to build a wizard.current dict
  def buildWizardCurrentDict(self):
    baseForm = self._currentObject.findParentOfType('GFForm')
    return {'form': baseForm,
            'logic': baseForm._logic,
            'layout': baseForm._layout,
            'page': self.visualEditor.page,
            'block': self.visualEditor.block,
            'object': self._currentObject}


  def __onSetDebugLevel(self, event):
    """Opens a dialog to let the user set the debug level
    before running a form inside Designer."""
    tmpLevel = self.debugLevel
    tmpRez = ''
    tmpMessage = _("Please enter debugging level - integer numbers [0..n]")
    tmpCaption = _("Choose debug level")
    tmpRez = wxGetTextFromUser(tmpMessage, tmpCaption, tmpLevel, NULL)
    if (tmpRez != ''):
      self.debugLevel = tmpRez


  def __onRunForm (self, event):
    #TODO: This does not work... But everyone tried this first...
    #TODO: No, not just you. I mean EVERYONE!!!
    GDebug.setDebug(str(self.debugLevel), "")
    DebugSession(self)
    #~ wxMessageDialog(self,
          #~ _('Running a form within GNUe Designer is\n' + \
          #~ 'not implemented yet.' + \
          #~ '\n\nPlease save the form and run it using\n' + \
          #~ 'the GNUe Forms tool (gnue-forms).'),
             #~ _("NOT IMPLEMENTED"), wxOK|wxICON_EXCLAMATION ).ShowModal()


