/*
**  MailboxManagerController.m
**
**  Copyright (C) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**  Author: Jonathan B. Leffert <jonathan@leffert.net> 
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "MailboxManagerController.h"

#import "GNUMail.h"
#import "GNUMailConstants.h"
#import "IMAPMailWindowController.h"
#import "LocalInboxWindowController.h"

#ifndef MACOSX
#import "MailboxManager.h"
#endif

#import "MailWindowController.h"
#import "NewMailboxPanelController.h"
#import "PasswordPanelController.h"
#import "Utilities.h"

#import <Pantomime/IMAPCacheManager.h>
#import <Pantomime/IMAPFolder.h>
#import <Pantomime/IMAPStore.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/LocalStore.h>
#import <Pantomime/Message.h>

static MailboxManagerController *singleInstance = nil;

// 
// Here's how it does work:
//
// allFolders -> allLocalFolders
//
//            -> IMAP_NSDICTIONARY -> key = name
//                                 -> value = NSArray of folder names
//
//            -> IMAP_NSDICTIONARY -> key = name
//                                 -> value = NSArray of folder names
//
@implementation MailboxManagerController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  MailboxManager *theWindow;
  
  theWindow = 
    [[MailboxManager alloc] initWithContentRect: NSMakeRect(200,200,400,435)
			    styleMask: (NSClosableWindowMask |
					NSTitledWindowMask |
					NSMiniaturizableWindowMask |
					NSResizableWindowMask)
			    backing: NSBackingStoreBuffered
			    defer: YES];

  self = [super initWithWindow: theWindow];

  [theWindow layoutWindow];
  [theWindow setDelegate: self];

  // We link our outlets
  transfer = [theWindow transfer];
  outlineView = [theWindow outlineView];
  openOnStartup = [theWindow openOnStartup];

  RELEASE(theWindow);
#endif

  // We set the title of our window (causing it to be loaded under OS X)
  [[self window] setTitle: _(@"Mailboxes")];

  // We alwas expand our root item
  [outlineView expandItem: allFolders];
  
  // We now select and expand the 'Local' folder if there's no IMAP folders defined
  if ( [allFolders count] == 1 )
    {
      [outlineView expandItem: allLocalFolders];
      [outlineView selectRow: [outlineView rowForItem: allLocalFolders]
		   byExtendingSelection: NO];
    }

  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"MailboxManagerController: -dealloc");

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ReceivingValuesHaveChanged
    object: nil];
  
  [super dealloc];
}

//
// Datasource/delegate methods for the outline view
//

- (id) outlineView: (NSOutlineView *) outlineView
	     child: (int) index
	    ofItem: (id) item
{
  
  // root object
  if ( !item )
    {
      return allFolders;
    }
  
  if ( [item isKindOfClass: [NSArray class]] )
    {
      return [item objectAtIndex: index];
    }
  
  if ( [item isKindOfClass: [NSDictionary class]] )
    {
      return [[item allValues] objectAtIndex: 0];
    }
  
  return nil;
}


//
//
//
- (BOOL) outlineView: (NSOutlineView *) outlineView
    isItemExpandable: (id) item
{
  if ( item == allFolders )
    {
      return YES;
    }

  if ( [allFolders containsObject: item] )
    {
      return YES;
    }

  return NO;
}


//
//
//
- (int)        outlineView: (NSOutlineView *) outlineView 
    numberOfChildrenOfItem: (id) item
{
  // Root, always one element
  if ( !item )
    {
      return 1;
    }
  
  // Children of our root, the Local folder and all the IMAP folders
  if ( item == allFolders )
    {
      return [allFolders count];
    }

  if ( [item isKindOfClass: [NSArray class]] )
    {
      return [(NSArray *)item count];
    }
  
    if ( [item isKindOfClass: [NSDictionary class]] )
      {
	return [(NSArray *)[[item allValues] objectAtIndex: 0] count];
      }
  
  return 0;
}

//
//
//
- (id)         outlineView: (NSOutlineView *) outlineView 
 objectValueForTableColumn: (NSTableColumn *) tableColumn 
		    byItem: (id) item
{
  if ( [[[tableColumn headerCell] stringValue] isEqual: _(@"Mailbox")] )
    {
      if ( item == allFolders )
	{
	  return _(@"All Mailboxes");
	}
      
      if ( item == allLocalFolders )
	{
	  return _(@"Local");
	}

      // We verify if we must deal with a local folder
      if ( [allLocalFolders containsObject: item] )
	{
	  return (NSString *)item;
	}
      // It's an IMAP store or an IMAP folder!
      else
	{
	  // It's an IMAP store
	  if ( [allFolders containsObject: item] )
	    {
	      NSDictionary *allValues;
	      NSString *aString;
	      
	      aString = [[item allKeys] objectAtIndex: 0];
	      allValues = [[[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"] objectForKey: aString];
	      
	      return [NSString stringWithFormat: @"%@ @ %@",
			       [allValues objectForKey: @"USERNAME"],
			       aString];
	    }
	  // It's an IMAP folder!
	  else
	    {
	      // FIXME - We only return Inbox for now...
	      return [item objectAtIndex: 0];
	    }
	}
    }
  else if ( [[[tableColumn headerCell] stringValue] isEqual: _(@"Size")] )
    {
      if ( item != allLocalFolders && item != allFolders )
	{
      	  return @"0 KB";
      	}
    }  
  else if ( [[[tableColumn headerCell] stringValue] isEqual: _(@"Messages")] )
    {
      if ( item != allLocalFolders && item != allFolders )
	{
      	  return @"0 (0 unread)";
      	}
    }
  
 
  return nil;
}


//
// Delegate methods
//
- (void) outlineViewSelectionIsChanging: (NSNotification *) aNotification
{
  NSMutableArray *foldersToOpen;
  NSString *aString;
  id item;
  
  int row, level;

  NSLog(@"openOnStartupClicked... %d", [openOnStartup state]);
  
  row = [outlineView selectedRow];

  // We verify if we have at least one selected row
  if ( row < 0 )
    {
      [openOnStartup setState: NSOffState];
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];

  // We verify that we have not more than a selected row and that we haven't
  // selected a store
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 2 )
    {
      [openOnStartup setState: NSOffState];
      return;
    }

   // We create our mutable array from all folders to open
  foldersToOpen = [NSMutableArray arrayWithArray: [[NSUserDefaults standardUserDefaults] 
						    objectForKey: @"FOLDERS_TO_OPEN"]];

  // We verify if we have selected a local mailbox
  if ( [allLocalFolders containsObject: item] )
    { 
      aString = [NSString stringWithFormat: @"local://%@/%@",
			  [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
			  (NSString *)item];
    }
  // The selected mailbox is an IMAP mailbox
  else
    {
      NSString *serverName;

      serverName = [self _serverNameForSelectedItem: item];
      
      // FIXME, we must support multiple IMAP folders (for item)
      aString = [NSString stringWithFormat: @"imap://%@/%@", serverName, [item objectAtIndex: 0]];
    }
  
  // We update the state of our button
  if ( [foldersToOpen containsObject: aString] )
    {
	  [openOnStartup setState: NSOnState];
    }
  else
    {
      [openOnStartup setState: NSOffState];
    }
}


//
//
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  // We do nothing for now...
  NSLog(@"Closing the MailboxWindow...");
}


//
//
//
- (void) windowDidLoad
{
  NSLog(@"MailboxManager window loaded!");
 
  // We initialize our array containing all Stores and we load of folders
  allFolders = [[NSMutableArray alloc] init];
  [self reloadAllFolders];

#ifdef MACOSX
  [outlineView setDoubleAction: @selector(open:)];
  [outlineView reloadData];
#endif

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(receivingValuesHaveChanged:)
    name: ReceivingValuesHaveChanged
    object: nil];
}


//
// action methods
//
- (IBAction) openOnStartupClicked: (id) sender
{
  NSMutableArray *foldersToOpen;
  NSString *aString;
  id item;
  
  int row, level;

  NSLog(@"openOnStartupClicked... %d", [openOnStartup state]);
  
  row = [outlineView selectedRow];

  // We verify if we have at least one selected row
  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];

  // We verify that we have not more than a selected row and that we haven't
  // selected a store
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 2 )
    {
      NSBeep();
      return;
    }

  // We create our mutable array from all folders to open
  foldersToOpen = [NSMutableArray arrayWithArray: [[NSUserDefaults standardUserDefaults] 
						    objectForKey: @"FOLDERS_TO_OPEN"]];

  // We verify if we have selected a local mailbox
  if ( [allLocalFolders containsObject: item] )
    {
      aString = [NSString stringWithFormat: @"local://%@/%@",
			  [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
			  (NSString *)item];
    }
  // The selected mailbox is an IMAP mailbox
  else
    {
      NSString *serverName;

      serverName = [self _serverNameForSelectedItem: item];
      
      // FIXME, we must support multiple IMAP folders (for item)
      aString = [NSString stringWithFormat: @"imap://%@/%@", serverName, [item objectAtIndex: 0]];
    }

  // We now add/remove the folder depending on the state of our button
  if ( [openOnStartup state] == NSOnState )
    {
      if ( ![foldersToOpen containsObject: aString] )
	{
	  [foldersToOpen addObject: aString];
	}
    }
  else
    {
      [foldersToOpen removeObject: aString];
    }

  // We save our FOLDERS_TO_OPEN value in the user's defaults
  [[NSUserDefaults standardUserDefaults] setObject: foldersToOpen
					 forKey: @"FOLDERS_TO_OPEN"];
  [[NSUserDefaults standardUserDefaults] synchronize];
}


//
//
//
- (IBAction) open: (id) sender
{
  NSString *aString;
  id item;

  int row, level;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  //
  // We must verify that:
  //
  // a) we have at least one selected row
  // b) we haven't selected our root, or a store (local or IMAP)
  //
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 2 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox to open!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  
  // We verify if it's a local folder
  if ( [allLocalFolders containsObject: item] )
    { 
      // We get our folder name (that's simply the item value)
      aString = (NSString *) item;
      
      [self openLocalFolderWithName: aString];
    }
  // It's an IMAP folder...
  else
    {
      NSString *serverName;

      // We get our folder name (that's simply the item value)
      // FIXME (for multiple IMAP folders)
      aString = (NSString *) [item objectAtIndex: 0];
      
      // We find which IMAP store contains our selected item
      serverName = [self _serverNameForSelectedItem: item];
      
      [self openIMAPFolderWithName: aString
	    serverName: serverName];
    }
}


//
//
//
- (IBAction) transfer: (id) sender
{
  LocalStore *localStore;
  LocalFolder *localFolder;
  Folder *aFolder;

  MailWindowController *mailWindowController;
  NSString *theFolderName;
  
  int nbOfTransferredMessages = 0;
  NSArray *selectedMessages;
  id aWindow = nil;

  id item;

  int row, level;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 2 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox\nto transfer this E-Mail to!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }

  // FIXME
  // Currently, we don't allow the user to transfer messages TO IMAP mailboxes
  if ( ![allLocalFolders containsObject: item] )
    {
      NSRunInformationalAlertPanel(_(@"Feature not implemented!"),
				   _(@"This feature is going to be implemented soon."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }

  mailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];

  if ( !mailWindowController ) 
    {
      NSBeep();
      return;
    }
  
  // We sync our popup selection
  [transfer synchronizeTitleAndSelectedItem];

  // We get our local store
  localStore = [(GNUMail *)[NSApp delegate] localStore];

  // We verify if we aren't trying to transfer to the current mbox! //
  aFolder = [mailWindowController folder];
  
  theFolderName = (NSString *)item;

  if ( [[aFolder name] isEqualToString: theFolderName] )
    {
      NSRunInformationalAlertPanel(_(@"Transfer error!"),
				   _(@"You cannot transfer a message inside the same mailbox!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;      
    }
  
  // We get a reference to our folder, w/o parsing it if it's not already open.
  localFolder = (LocalFolder *)[localStore folderForName: theFolderName];

  // Our folder is opened, let's get it from the list of opened windows.
  if (! localFolder )
    {
      aWindow = [Utilities windowForFolderName: theFolderName];
      localFolder = (LocalFolder *)[[aWindow windowController] folder];
    }
  
  
  // We transfer all the selected messages  
  selectedMessages = [mailWindowController selectedMessages];
  
  if ( selectedMessages )
    {
      Message *aMessage;
      NSData *aData;
      int i;
      
      for (i = 0; i < [selectedMessages count]; i++)
	{
	  aMessage = [selectedMessages objectAtIndex: i];
	  aData = [aMessage rawSource];
	  
	  if ( aData )
	    {
	      Flags *theFlags;
	      
	      // We get our flags but we remove the TRANSFERRED flag from them
	      theFlags = [[aMessage flags] copy];
	      [theFlags remove: TRANSFERRED];

	      // We transfer it to our new folder
	      [localFolder appendMessageFromRawSource: aData
			   flags: theFlags];

	      RELEASE(theFlags);

	      // We verify if we flag it as TRANSFERRED or not (in case of a
	      // copy operation).
	      if ( [transfer indexOfSelectedItem] == 0 )
		{
		  [[aMessage flags] add: TRANSFERRED];
		  nbOfTransferredMessages++;
		}
	    }
	}
    }

  // We need to reload our tableView's data if we transferred some messages
  if ( nbOfTransferredMessages > 0 )
    {
      // We force the reload of our cache in Folder
      [[mailWindowController folder] updateCache];
      [mailWindowController tableViewShouldReloadData];
    }

  // If the folder was open, we refresh its view. Otherwise, we just close it.
  if ( aWindow )
    {
      [[aWindow delegate] tableViewShouldReloadData];
    }
  else
    {
      [localFolder close];
    }
}


//
//
//
- (IBAction) create: (id) sender
{
  NewMailboxPanelController *theController;
  id item;

  int row, level;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level != 1  )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid root where to create this new mailbox."),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  
  // FIXME
  // Currently, we don't allow the user to create new IMAP mailboxes
  if ( item != allLocalFolders )
    {
      NSRunInformationalAlertPanel(_(@"Feature not implemented!"),
				   _(@"This feature is going to be implemented soon."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }
  
  // We create our NewMailboxPanelController object. It'll be automatically deallocated when the 
  // window will be closed.
  theController = [[NewMailboxPanelController alloc] initWithWindowNibName: @"NewMailboxPanel"];
  [theController setMailboxManagerController: self];

  [NSApp runModalForWindow: [theController window]];
}


//
//
//
- (IBAction) delete: (id) sender
{
  id item;

  int choice, row, level;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 2 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"Please select the mailbox you would like to delete."),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  
  // FIXME
  // Currently, we don't allow the use to delete IMAP mailboxes
  if ( ![allLocalFolders containsObject: item] )
    {
      NSRunInformationalAlertPanel(_(@"Feature not implemented!"),
				   _(@"This feature is going to be implemented soon."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }

  choice = NSRunAlertPanel(_(@"Delete..."),
			   _(@"Are you sure you want to delete this mailbox?"),
			   _(@"Cancel"),  // default
			   _(@"Yes"),     // alternate
			   nil);
  
  if ( choice == NSAlertAlternateReturn )
    {
      LocalStore *localStore;
      
      // We get our global local store
      localStore = [(GNUMail *)[NSApp delegate] localStore];
	  
      if ( [localStore folderForNameIsOpen: item] )
	{
	  NSRunAlertPanel(_(@"Error while deleting!"),
  			  _(@"You can't delete an opened mailbox! Close it first."),
  			  _(@"OK"),   // default
  			  NULL,       // alternate
  			  NULL); 
	  return;
	}
      else
	{
	  if ( [localStore deleteFolderWithName: item] )
	    {
	      [self reloadAllFolders];
	      [outlineView reloadData];
	      [outlineView setNeedsDisplay: YES];
	      [outlineView selectRow: [outlineView rowForItem: allLocalFolders]
	      		   byExtendingSelection: NO];

	    }
	  else
	    {
	      NSRunInformationalAlertPanel(_(@"Error!"),
					   _(@"The mailbox delete operation failed."),
					   _(@"OK"),
					   NULL,
					   NULL,
					   NULL);
	    }
	  
	}
    }
}


//
//
//
- (IBAction) rename: (id) sender
{
  NewMailboxPanelController *theController;
  LocalStore *localStore;
  
  id item;
  int row, level;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 2 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox to rename!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  
  // FIXME
  // Currently, we don't allow the user to rename IMAP mailboxes
  if ( ![allLocalFolders containsObject: item] )
    {
      NSRunInformationalAlertPanel(_(@"Feature not implemented!"),
				   _(@"This feature is going to be implemented soon."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }
     
  // We verify that the mailbox we want to rename is not open
  // We get our global local store
  localStore = [(GNUMail *)[NSApp delegate] localStore];
  
  if ( [localStore folderForNameIsOpen: item] )
    {
      NSRunAlertPanel(_(@"Error while renaming!"),
		      _(@"You can't rename an opened mailbox! Close it first."),
		      _(@"OK"),   // default
		      NULL,       // alternate
		      NULL); 
      return;
    }

  
  // We create our NewMailboxPanelController object. It'll be automatically 
  // deallocated when the window will be closed.
  theController = [[NewMailboxPanelController alloc] initWithWindowNibName: @"NewMailboxPanel"];
  [theController setMailboxManagerController: self];
  [theController setName: item];
  [theController setPath: [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];

  [NSApp runModalForWindow: [theController window]];
}


//
//
//
- (void) receivingValuesHaveChanged: (id) sender
{
  [self reloadAllFolders];
  [outlineView reloadData];
}


//
//
//
- (NSOutlineView *) outlineView
{
  return outlineView;
}

//
// class methods
//

+ (id) singleInstance
{
  if ( ! singleInstance )
    {
      singleInstance = [[self alloc] initWithWindowNibName: @"MailboxManager"];
    }
  return singleInstance;
}


//
// Other methods
//

//
//
//
- (void) openLocalFolderWithName: (NSString *) theFolderName
{
  MailWindowController *mailWindowController;
  
  LocalStore *localStore;
  LocalFolder *aFolder;
  
  NSModalSession aModalSession = 0;
  id panel = nil;
  
  // We get out local store and our folder.
  localStore = [(GNUMail *)[NSApp delegate] localStore];
  
  
  // We first verify if the folder is still valid. For example, it could have been
  // deleted (the file).
  if (! [[NSFileManager defaultManager] 
	  fileExistsAtPath: [[localStore path] stringByAppendingPathComponent: theFolderName]] )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"The local mailbox %@ does not exist!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   theFolderName);
      return;
    }

  // If the folder is already open, we show our alert panel
  // and we return
  if ( [localStore folderForNameIsOpen: theFolderName] )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"This local mailbox (%@) is already open!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   theFolderName);
      return;
    }
  
  // We show our 'wait panel', if we need to.
  if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
    {
      panel = NSGetInformationalAlertPanel(_(@"Please wait..."),
					   _(@"Opening the mailbox. Please wait."),
					   NULL,
					   NULL,
					   NULL);
      
      aModalSession = [NSApp beginModalSessionForWindow: panel];
      [NSApp runModalSession: aModalSession];
    }
  
  // The folder is not open, so we get it!       
  aFolder = [localStore folderForName: theFolderName];
  
  // No memory leaks are created here since MailWindow AUTORELEASE itself
  // in MailWindow: - windowWillClose
  if ( [theFolderName isEqualToString: [[NSUserDefaults standardUserDefaults]
					 objectForKey: @"INBOXFOLDERNAME"]] )
    {
      mailWindowController = [[LocalInboxWindowController alloc] initWithWindowNibName: @"LocalInboxWindow"];
    }
  else
    {  
      mailWindowController = [[MailWindowController alloc] initWithWindowNibName: @"MailWindow"];
    }
      
  // We now set the folder to show or hide our deleted messages from the folder
  // Showing them is the default behaviour.
  if ( ! [[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] ||
       [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] intValue] == NSOffState)
    {
      [aFolder setShowDeleted: YES];
      [[NSApp delegate] updateShowOrHideDeletedMenuItem: YES];
    }
  else
    {
      [aFolder setShowDeleted: NO];
      [[NSApp delegate] updateShowOrHideDeletedMenuItem: NO];
    }
  
  // We set the folder
  [mailWindowController setFolder: aFolder];
  
  // We close our 'wait panel'
  if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
    {
      [NSApp endModalSession: aModalSession];
      [panel close];
      NSReleaseAlertPanel(panel);
    }
  
  // And we show the window..
  NSLog(@"Showing the window...");   
  [[mailWindowController window] orderFrontRegardless];
}


//
//
//
- (void) openIMAPFolderWithName: (NSString *) theFolderName
                     serverName: (NSString *) theServerName
{
  NSNumber *serverTypeValue, *portValue;
  NSDictionary *allValues;
  
  NSString *username, *password;

  // We begin by searching in our RECEIVING values for the right server.
  // Now, let's get all the receiving values
  allValues = [[[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"]
		objectForKey: theServerName];
  
  serverTypeValue = [allValues objectForKey: @"SERVERTYPE"];
  portValue =  [allValues objectForKey: @"PORT"];
	  
  // We use the default IMAP port if it's not defined.
  if ( !portValue )
    {
      portValue = [NSNumber numberWithInt: 143];
    }
  
  // We get our username
  username = [allValues objectForKey: @"USERNAME"];
  
  // We get our password. We could prompt for it in case we need to.
  password = [self _passwordForServerName: theServerName];
  
  if ( password )
    {		
      // FIXME... IMAPStore is NEVER dealloced
      IMAPStore *aStore;
      
      aStore = [[IMAPStore alloc] initWithName: theServerName
				  port: [portValue intValue]];
      
      if ( !aStore )
	{
	  NSRunInformationalAlertPanel(_(@"Error!"),
				       _(@"Unable to communicate with the IMAP server (%@)."),
				       _(@"OK"),
				       NULL,
				       NULL,
				       theServerName);
	}
      else
	{
	  // We verify if the authentication is successful
	  if ( [aStore authenticateWithUsername: username
		       password: password] )
	    {
	      IMAPCacheManager *anIMAPCacheManager;
	      IMAPMailWindowController *imapMailWindowController;
	      
	      IMAPFolder *aFolder;

	      NSModalSession aModalSession = 0;
	      id panel = nil;
	      
	      // We get our cache manager for this server / folder
	      anIMAPCacheManager = [IMAPCacheManager imapCacheObjectsFromDiskWithPath:
						       [NSString stringWithFormat: @"%@/IMAPCache_%@_%@",
								 GNUMailUserLibraryPath(),
								 theServerName,
								 theFolderName]];
	      
	      if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
		{
		  panel = NSGetInformationalAlertPanel(_(@"Please wait..."), 
						       _(@"Connecting to the IMAP server. Please wait."),
						       NULL,
						       NULL,
						       NULL);
		  
		  aModalSession = [NSApp beginModalSessionForWindow: panel];
		  [NSApp runModalSession: aModalSession];
		}
	      
	      aFolder = (IMAPFolder *)[aStore folderForName: theFolderName
					      withIMAPCacheManager: anIMAPCacheManager];
	      
	      imapMailWindowController = [[IMAPMailWindowController alloc] initWithWindowNibName: @"IMAPMailWindow"];
	      
	      // We now set the folder to show or hide our deleted messages from the folder
	      // Showing them is the default behaviour.
	      if ( ! [[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] ||
		   [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] intValue] == NSOffState)
		{
		  [aFolder setShowDeleted: YES];
		  [[NSApp delegate] updateShowOrHideDeletedMenuItem: YES];
		}
	      else
		{
		  [aFolder setShowDeleted: NO];
		  [[NSApp delegate] updateShowOrHideDeletedMenuItem: NO];
		}
	      
	      // We set the folder
	      [imapMailWindowController setFolder: aFolder];
	      
	      // We close our 'wait panel'
	      if ( [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWWAITPANELS"] intValue] == NSOnState )
		{
		  [NSApp endModalSession: aModalSession];
		  [panel close];
		  NSReleaseAlertPanel(panel);
		}
	      
	      [[imapMailWindowController window] orderFrontRegardless];
	    }
	  else
	    {
	      NSRunInformationalAlertPanel(_(@"Error!"),
					   _(@"Invalid password %@ @ %@."), 
					   _(@"OK"),
					   NULL,
					   NULL,
					   username, theServerName);
	      [aStore close];
	      RELEASE(aStore);
	    }
	} // else of if ( !aStore ) ...
    } // if (password )
}

//
//
//
- (void) reloadAllFolders
{
  // We remove all our elements
  [allFolders removeAllObjects];

  // We always add our local folder
  allLocalFolders = [[[[[NSApp delegate] localStore] folderEnumerator] allObjects]
		      sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];

  [allFolders addObject: allLocalFolders];  

  // We verify if the RECEIVING preferences have been defined.
  if ( [[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] )
    {
      NSEnumerator *allServers;
      NSDictionary *allValues;
      id aKey;

      allServers = [[[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"] keyEnumerator];	      
      
      // We build a correct subset of all our IMAP servers defined in RECEIVING
      while ( (aKey = [allServers nextObject]) )
	{
	  allValues = [[[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"]
			objectForKey: aKey];
	  
	  if ( [[allValues objectForKey: @"SERVERTYPE"] intValue] == IMAP )
	    {
	      [allFolders addObject: [NSDictionary dictionaryWithObject: [NSArray arrayWithObject: @"Inbox"]
						   forKey: aKey]];
	    }
	}
    }
}

@end

//
// Private methods
//
@implementation MailboxManagerController (Private)

//
//
//
- (NSString *) _passwordForServerName: (NSString *) theName
{
  NSDictionary *allReceivingValues;
  NSString *password;

  allReceivingValues = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] objectForKey: theName];

  // We verify in the user defaults
  password =  [Utilities decryptPassword: [allReceivingValues objectForKey:@"PASSWORD"]
			 withKey: theName];
 
  // If we must prompt for the password
  if (! password )
    {
      PasswordPanelController *theController; 
      int result;
      
      theController = [[PasswordPanelController alloc] initWithWindowNibName: @"PasswordPanel"];
      [[theController window] setTitle: theName];
      
      result = [NSApp runModalForWindow: [theController window]];
      
      // If the user has entered a password...
      if (result == NSRunStoppedResponse)
	{
	  password = [theController password];
	}
      else
	{
	  password = nil;
	}
      
      RELEASE(theController);
    }
  
  return password;
}


//
//
//
- (NSString *) _serverNameForSelectedItem: (id) theItem
{
  int i;
  
  for (i = 1; i < [allFolders count]; i++)
    {
      id dict;
      
      dict = [allFolders objectAtIndex: i];
      
      // FIXME (for multiple IMAP folders)
      if ( [[dict allValues] objectAtIndex: 0] == theItem )
	{
	  return [[dict allKeys] objectAtIndex: 0];
	}
    }
  
  return nil;
}

@end
