/*
**  ViewingViewController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "ViewingViewController.h"

#include "Constants.h"
#include "HeadersWindowController.h"

#ifndef MACOSX
#include "ViewingView.h"
#endif

static ViewingViewController *singleInstance = nil;

@implementation ViewingViewController

//
//
//
- (id) initWithNibName: (NSString *) theName
{
  self = [super init];

#ifdef MACOSX
  
  if ( ![NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSDebugLog(@"Fatal error occured while loading the ViewingView nib file");
      AUTORELEASE(self);
      return nil;
    }

  RETAIN(view);

#else
  // We link our view
  view = [[ViewingView alloc] initWithParent: self];
  [view layoutView];

  // We link our outlets
  highlightURLs = ((ViewingView *)view)->highlightURLs;
  hideDeletedMessages = ((ViewingView *)view)->hideDeletedMessages;
  hideReadMessages = ((ViewingView *)view)->hideReadMessages;
  
  matrix = ((ViewingView *)view)->matrix;
  doubleClickPopUpButton = ((ViewingView *)view)->doubleClickPopUpButton;
#endif

  // We initialize our array containing all our headers
  _values.shownHeaders = [[NSMutableArray alloc] init];

  // We get our defaults for this panel
  [self initializeFromDefaults];

  return self;
}


//
//
//
- (void) dealloc
{
  singleInstance = nil;
  
  RELEASE(_values.shownHeaders);
  RELEASE(view);

  [super dealloc];
}


//
// action methods
//
- (IBAction) headersButtonClicked: (id) sender
{
  HeadersWindowController *aHeadersWindowController;
  int result;

  aHeadersWindowController = [[HeadersWindowController alloc]
			       initWithWindowNibName: @"HeadersWindow"];

  [aHeadersWindowController setShownHeaders: _values.shownHeaders];
  [aHeadersWindowController setShowAllHeadersButtonState: _values.showAllHeadersButtonState];
  
  result = [NSApp runModalForWindow: [aHeadersWindowController window]];
  
  // We must update our preferences
  if (result == NSRunStoppedResponse)
    {
      [_values.shownHeaders removeAllObjects];
      [_values.shownHeaders addObjectsFromArray: [aHeadersWindowController shownHeaders]];
      _values.showAllHeadersButtonState = [aHeadersWindowController showAllHeadersButtonState];
    }

  // We release our controller
  RELEASE(aHeadersWindowController);
}


//
// access methods
//
- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];
  
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"viewing" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"Viewing");
}

- (NSView *) view
{
  return view;
}

- (BOOL) hasChangesPending
{
  return YES;
}


//
//
//
- (void) initializeFromDefaults
{
  NSArray *anArray, *shownTableColumns;
  int i;

  // Let's begin with our HIDE_DELETED_MESSAGES preference value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] )
    {
      [hideDeletedMessages setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] intValue] ];
    }
  else
    {
      [hideDeletedMessages setState: NSOnState];
    }

  // We decode our HIDE_READ_MESSAGES preference value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_READ_MESSAGES"] )
    {
      [hideReadMessages setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_READ_MESSAGES"] intValue] ];
    }
  else
    {
      [hideReadMessages setState: NSOffState];
    }

  // We decode our HIGHLIGHT_URL preference value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] )
    {
      [highlightURLs setState: [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] intValue] ];
    }
  else
    {
      [highlightURLs setState: NSOffState];
    }

  // We decode our SHOWALLHEADERS value
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] )
    {
      _values.showAllHeadersButtonState = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] intValue];
    }
  else
    {
      _values.showAllHeadersButtonState = NSOffState;
    }
    
  // We decode our double-click action
  // We will have stored stored the tag of the cell, so we turn that cell on.
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"DOUBLECLICKACTION"] )
    {
      int iVal;
      
      iVal = [[NSUserDefaults standardUserDefaults] integerForKey: @"DOUBLECLICKACTION"];
      [doubleClickPopUpButton selectItemAtIndex: iVal];
    }
  else
    {
      [doubleClickPopUpButton selectItemAtIndex: 1];
    }


  // We now load the headers we want to show
  anArray = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNHEADERS"];
  
  if ( anArray )
    {
      [_values.shownHeaders addObjectsFromArray: anArray];
    }

  // We now select the right items from our matrix
  shownTableColumns = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNTABLECOLUMNS"];
  
  if ( shownTableColumns )
    {
      for (i = 0; i < [shownTableColumns count]; i++)
	{
	  NSString *column;
	  
	  column = [shownTableColumns objectAtIndex: i];
	  
	  if ( [column isEqualToString: @"Status"] )
	    {
	      [[matrix cellWithTag: 0] setState: NSOnState];
	    }
	  // FIXME: In 1.1.2, we renamed Id -> Number. Eventually remove Id.
	  else if ( [column isEqualToString: @"Id"] || 
		    [column isEqualToString: @"Number"] )
	    {
	      [[matrix cellWithTag: 1] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Date"] )
	    {
	      [[matrix cellWithTag: 2] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"From"] )
	    {
	      [[matrix cellWithTag: 3] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Subject"] )
	    {
	      [[matrix cellWithTag: 4] setState: NSOnState];
	    }
	  else if ( [column isEqualToString: @"Size"] )
	    {
	      [[matrix cellWithTag: 5] setState: NSOnState];
	    }
	}
    }
  else
    {
      [matrix selectAll: nil];
    }
}


//
//
//
- (void) saveChanges
{
  NSMutableArray *aMutableArray;
  int iVal;

  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: [hideDeletedMessages state]]
					 forKey: @"HIDE_DELETED_MESSAGES"];

  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: [hideReadMessages state]]
					 forKey: @"HIDE_READ_MESSAGES"];
  
  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: [highlightURLs state]]
					 forKey: @"HIGHLIGHT_URL"];
  
  [[NSUserDefaults standardUserDefaults] setObject: [NSNumber numberWithInt: _values.showAllHeadersButtonState]
					 forKey: @"SHOWALLHEADERS"];
  
  // store the doubleclick action
  iVal = [doubleClickPopUpButton indexOfSelectedItem];
  [[NSUserDefaults standardUserDefaults] setInteger: iVal
  					 forKey: @"DOUBLECLICKACTION"];
  
  [[NSUserDefaults standardUserDefaults] setObject: _values.shownHeaders
					 forKey: @"SHOWNHEADERS"];

  // We update our list of shown headers. The order is important.
  aMutableArray = [[NSMutableArray alloc] init];

  if ( [[matrix cellWithTag: 0] state] == NSOnState )
    {
      [aMutableArray addObject: @"Status"];
    }

  if ( [[matrix cellWithTag: 1] state] == NSOnState )
    {
      [aMutableArray addObject: @"Number"];
    }

  if ( [[matrix cellWithTag: 2] state] == NSOnState )
    {
      [aMutableArray addObject: @"Date"];
    }
  
  if ( [[matrix cellWithTag: 3] state] == NSOnState )
    {
      [aMutableArray addObject: @"From"];
    }
  
  if ( [[matrix cellWithTag: 4] state] == NSOnState )
    {
      [aMutableArray addObject: @"Subject"];
    }
  
  if ( [[matrix cellWithTag: 5] state] == NSOnState )
    {
      [aMutableArray addObject: @"Size"];
    }

  [[NSUserDefaults standardUserDefaults] setObject: aMutableArray
					 forKey: @"SHOWNTABLECOLUMNS"];
  
  RELEASE(aMutableArray);

  // FIXME - do not post if nothing has changed
  [[NSNotificationCenter defaultCenter]
    postNotificationName: TableColumnsHaveChanged
    object: nil
    userInfo: nil];
}


//
// class methods
//
+ (id) singleInstance
{
  if ( !singleInstance )
    {
      singleInstance = [[ViewingViewController alloc] initWithNibName: @"ViewingView"];
    }

  return singleInstance;
}

@end
