/*
**  EditWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**          Jonathan B. Leffert <jonathan@leffert.net>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "EditWindowController.h"

#include "AddressBookController.h"
#include "AutoCompletingTextField.h"

#ifndef MACOSX
#include "EditWindow.h"
#endif

#include "ExtendedTextView.h"
#include "ExtendedAttachmentCell.h"
#include "Filter.h"
#include "FilterManager.h"
#include "GNUMail.h"
#include "GNUMail+TaskManager.h"
#include "GNUMail/GNUMailBundle.h"
#include "Constants.h"
#include "LabelWidget.h"
#include "MailboxManagerController.h"
#include "MailWindowController.h"
#include "MimeTypeManager.h"
#include "MimeType.h"
#include "Task.h"
#include "Utilities.h"

#include <Pantomime/Charset.h>
#include <Pantomime/Constants.h>
#include <Pantomime/InternetAddress.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/MimeMultipart.h>
#include <Pantomime/MimeUtility.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/Part.h>
#include <Pantomime/URLName.h>
#include <Pantomime/Flags.h>


#include <math.h>

//
//
//
@implementation EditWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{  
  NSDictionary *allAccounts;
#ifndef MACOSX
  EditWindow *theEditWindow;
#endif
  
  allAccounts = [Utilities allEnabledAccounts];

  // We first verify if we have at least one transport agent defined
  if ( !allAccounts || [allAccounts count] == 0 )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must have at least one transport agent defined and enabled.\nSee Preferences -> Account."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
      
      AUTORELEASE(self);
      return nil;
    }
  

#ifdef MACOSX
  self = [super initWithWindowNibName: windowNibName];
  
  // We create our mutable array and our mutable dictionary allowing
  // us to dynamically add toolbar items.
  allowedToolbarItemIdentifiers = [[NSMutableArray alloc] initWithObjects: 
							    NSToolbarSeparatorItemIdentifier,
							  NSToolbarSpaceItemIdentifier,
							  NSToolbarFlexibleSpaceItemIdentifier,
							  NSToolbarCustomizeToolbarItemIdentifier, 
							  @"send",
							  @"insert",
							  @"add_cc",
							  @"add_bcc",
							  @"addresses",
							  nil];
  
  additionalToolbarItems = [[NSMutableDictionary alloc] init];
  
#else
  
  theEditWindow = [[EditWindow alloc] initWithContentRect: NSMakeRect(50,75,750,520)
				   styleMask: (NSClosableWindowMask|NSTitledWindowMask|
					       NSMiniaturizableWindowMask|NSResizableWindowMask)
				   backing: NSBackingStoreRetained
				   defer: NO];
    
  self = [super initWithWindow: theEditWindow];
  [theEditWindow layoutWindow];
  [theEditWindow setDelegate: self];
  
  // We link our outlets
  subjectText = theEditWindow->subjectText;
  toText = theEditWindow->toText;
  ccText = theEditWindow->ccText;
  bccText = theEditWindow->bccText;
  subjectLabel = theEditWindow->subjectLabel;
  toLabel = theEditWindow->toLabel;
  ccLabel = theEditWindow->ccLabel;
  bccLabel = theEditWindow->bccLabel;
  sizeLabel = theEditWindow->sizeLabel;
  scrollView = theEditWindow->scrollView;
  textView = theEditWindow->textView;
  send = theEditWindow->send;
  insert = theEditWindow->insert;
  addBcc = theEditWindow->addBcc;
  addCc = theEditWindow->addCc;
  addresses = theEditWindow->addresses;
  accountPopUpButton = theEditWindow->accountPopUpButton;
  transportMethodPopUpButton = theEditWindow->transportMethodPopUpButton;

  RELEASE(theEditWindow);
#endif

  // We set our window title and edited attribute
  [[self window] setTitle: @""];
  [[self window] setDocumentEdited: NO];
  
  // We initialize our variables
  [self setShowCc: NO];
  [self setShowBcc: NO];
  [self setUnmodifiedMessage: nil];
  [self setSignaturePosition: SIGNATURE_END];

  // We load our accessory views
  [self _loadAccessoryViews];

  // We load our accounts and our transport methods
  [Utilities loadAccountsInPopUpButton: accountPopUpButton  select: nil];
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];

  // We set some ivars
  forceEndEditingOfTextField = 0;
  previousSignatureValue = nil;

  [self accountSelectionHasChanged: nil];
  [self _loadCharset];

  // We finally set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"EditWindow"];
  [[self window] setFrameUsingName: @"EditWindow"];
  
  // Set the data sources and delegates for the address text fields
  [toText setCommaDelimited:YES];
  [toText setDataSource: self];
  [toText setDelegate: self];

  [ccText setCommaDelimited:YES];
  [ccText setDataSource: self];
  [ccText setDelegate: self];

  [bccText setCommaDelimited:YES];
  [bccText setDataSource: self];
  [bccText setDelegate: self];
  
  // Enable spellchecking, if needed
  if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"ENABLE_SPELL_CHECKING"] )
    {
      [textView setContinuousSpellCheckingEnabled: YES];
    }

  // Allow undo
  [textView setAllowsUndo: YES];
  [textView setImportsGraphics: YES];

  // Set updateColors
  updateColors = YES;
  
  // Set the sizes for the scroll bars
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"SCROLLER_SIZE"] == NSRegularControlSize )
    {
      [[scrollView verticalScroller] setControlSize: NSRegularControlSize];
      [[scrollView horizontalScroller] setControlSize: NSRegularControlSize];
    }
  else
    {
      [[scrollView verticalScroller] setControlSize: NSSmallControlSize];
      [[scrollView horizontalScroller] setControlSize: NSSmallControlSize];
    }
    
  // We the font to a fixed pitch, if we need to. Otherwise, we set it to the font used to view the parts
  // in the MailWindow/MessageViewWindow's textview.
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"USE_FIXED_FONT_FOR_TEXT_PLAIN_MESSAGES"] == NSOnState )
    {
      [textView setFont: [Utilities fontFromFamilyName: 
				      [[NSUserDefaults standardUserDefaults] 
					objectForKey: @"PLAIN_TEXT_MESSAGE_FONT_NAME"]
				    trait: NSFixedPitchFontMask
				    size: [[NSUserDefaults standardUserDefaults]
					    floatForKey: @"PLAIN_TEXT_MESSAGE_FONT_SIZE"]]];
    }
  else
    {
      [textView setFont: [Utilities fontFromFamilyName: [[NSUserDefaults standardUserDefaults] 
							  objectForKey: @"MESSAGE_FONT_NAME"]
				    trait: NSUnboldFontMask
				    size: [[NSUserDefaults standardUserDefaults] 
					    floatForKey: @"MESSAGE_FONT_SIZE"]]];
    }


  // We add our observer to update our size label if the size of the text view has changed
  [[NSNotificationCenter defaultCenter] addObserver: self
					selector: @selector (_updateSizeLabel) 
					name: @"NSViewFrameDidChangeNotification" 
					object: textView];

  // misc
  editingDraftMessage = NO;
  [[self window] setInitialFirstResponder: toText];

  return self;
}


//
//
//
#ifdef MACOSX
- (void) awakeFromNib
{
  NSToolbar *aToolbar;
  
  aToolbar = [[NSToolbar alloc] initWithIdentifier: @"EditWindowToolbar"];
  [aToolbar setDelegate: self];
  [aToolbar setAllowsUserCustomization: YES];
  [aToolbar setAutosavesConfiguration: YES];
  [[self window] setToolbar: aToolbar];

  RELEASE(aToolbar);
}
#endif


//
//
//
- (void) dealloc
{
  NSDebugLog(@"EditWindowController: -dealloc");
  
  // We remove our observer from our two notifications
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: AccountsHaveChanged
    object: nil];
  
  [[NSNotificationCenter defaultCenter] 
    removeObserver: self
    name: @"NSViewFrameDidChangeNotification" 
    object: textView];

  TEST_RELEASE(message);
  TEST_RELEASE(unmodifiedMessage);
  TEST_RELEASE(previousSignatureValue);
  TEST_RELEASE(charset);

#ifdef MACOSX
  RELEASE(send);
  RELEASE(insert);
  RELEASE(addCc);
  RELEASE(addBcc);
  RELEASE(addresses);
  RELEASE(allowedToolbarItemIdentifiers);
  RELEASE(additionalToolbarItems);
#endif

  RELEASE(addressCompletionCandidates);

  [super dealloc];
}


//
// Implementation of the AddressTaker protocol.
//
- (void) takeToAddress: (NSArray *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: toText];
  [self controlTextDidChange: [NSNotification notificationWithName: @"" object: toText]];
}


//
//
//
- (void) takeCcAddress: (NSArray *) theAddress
{
  if ( ![self showCc] )
    {
      [self showCc: self];
    }
  
  [Utilities appendAddress: theAddress
	     toTextField: ccText];
  [self controlTextDidChange: [NSNotification notificationWithName: @"" object: ccText]];
}


//
//
//
- (void) takeBccAddress: (NSArray *) theAddress
{
  if ( ![self showBcc] )
    {
      [self showBcc: self];
    }
  
  [Utilities appendAddress: theAddress
	     toTextField: bccText];
  [self controlTextDidChange: [NSNotification notificationWithName: @"" object: bccText]];
}


//
// action methods
//
- (IBAction) insertFile: (id) sender
{
  NSOpenPanel *oPanel;
  
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection:YES];
  
#ifdef MACOSX
  [oPanel beginSheetForDirectory: [GNUMail currentWorkingPath] file:nil types:nil
	  modalForWindow: [self window] 
	  modalDelegate: self
	  didEndSelector: @selector(_openPanelDidEnd: returnCode: contextInfo:)
	  contextInfo: nil];
#else
  [self _openPanelDidEnd: oPanel 
	returnCode: [oPanel runModalForDirectory: [GNUMail currentWorkingPath]  file: nil  types: nil]
	contextInfo: nil];
#endif
}


//
//
//
- (IBAction) showBcc: (id) sender
{    
  [self setShowBcc: ![self showBcc]];
  [[[self window] contentView] setNeedsDisplay: YES];
}


//
//
//
- (IBAction) showCc: (id) sender
{ 
  [self setShowCc: ![self showCc]];
  [[[self window] contentView] setNeedsDisplay: YES];
}


//
// 
//
- (IBAction) accountSelectionHasChanged: (id) sender
{
  NSDictionary *theAccount;
  NSString *aString;
  NSRange aRange;
  int i;

  // We synchronize our selection from the popup
  [accountPopUpButton synchronizeTitleAndSelectedItem];

  theAccount = [[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]];
  
  for (i = 0; i < [transportMethodPopUpButton numberOfItems]; i++)
    {
      if ( [[[theAccount objectForKey: @"SEND"] objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_MAILER )
	{
	  aString = [[theAccount objectForKey: @"SEND"] objectForKey: @"MAILER_PATH"];
	}
      else
	{
	  aString = [[theAccount objectForKey: @"SEND"] objectForKey: @"SMTP_HOST"];
	}

      aRange = [[transportMethodPopUpButton itemTitleAtIndex: i] rangeOfString: aString];

      if ( aRange.length )
	{
	  [transportMethodPopUpButton selectItemAtIndex: i];
	  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];
	  [self _replaceSignature];
	  return;
	}
    }
  
  [transportMethodPopUpButton selectItemAtIndex: 0];
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];
  [self _replaceSignature];
}


//
//
//
- (IBAction) sendMessage: (id) sender
{
  NSDictionary *allValues;
  Task *aTask;
  int op;

  // We first try to initialize our message with all the information 
  if ( ![self updateMessageContentFromTextView] )
    {
      return;
    }

  // We sync our popup
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];
  [accountPopUpButton synchronizeTitleAndSelectedItem];

  // We get our transport method type
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForTransportMethodItemTitle: [transportMethodPopUpButton titleOfSelectedItem]]] 
		objectForKey: @"SEND"];
										
  if ( [[allValues objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP )
    {
      op = SEND_SMTP;
    }
  else
    {
      op = SEND_SENDMAIL;
    }
  
  aTask = [[Task alloc] init];
  [aTask setOp: op];
  [aTask setMessage: [self message]];
  [aTask setKey: [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]];
  [aTask setSendingKey: [Utilities accountNameForTransportMethodItemTitle: [transportMethodPopUpButton titleOfSelectedItem]]];
  [[NSApp delegate] addTask: aTask];
  RELEASE(aTask);

  // FIXME - That should go in the task manager.
  // We send the notification to inform our MailWindowController that the replied message has been
  // sucessfully sent so we can set the flag to 'ANSWERED'.
  if ( [self unmodifiedMessage] )
    {
      [[NSNotificationCenter defaultCenter]
	postNotificationName: ReplyToMessageWasSuccessful
	object: nil
	userInfo: [NSDictionary dictionaryWithObject: [self unmodifiedMessage] 
				forKey: @"Message"]];
    }

  // If this message is already in the drafts folder, set the "deleted" flag 
  // of the original message.
  if ( [self editingDraftMessage] )
    {
      Flags *theFlags;
      
      theFlags = [[[self message] flags] copy];
      [theFlags add: DELETED];
      [[self message] setFlags: theFlags];
      RELEASE(theFlags);
      
      // We post our notifications
      [[NSNotificationCenter defaultCenter] postNotificationName: SelectionOfMessageHasChanged
					    object: nil
					    userInfo: nil];
      
      [[NSNotificationCenter defaultCenter] postNotificationName: ReloadMessageList
					    object: nil
					    userInfo: nil];
    }
  
  
  // Since we're done, we close our window.
  [self close];
}


//
//
//
- (void) controlTextDidEndEditing: (NSNotification *) aNotification
{
  NSControl *theControl = [aNotification object];

  if ( theControl == toText ||
       theControl == ccText ||
       theControl == bccText )
    {
      NSArray *allRecipientsFromString;
      NSMutableArray *allRecipients;
      NSString *aString;
      int i;
      
      aString = [[theControl stringValue] stringByTrimmingWhiteSpaces];
      
      if ( [aString length] == 0 )
        {
	  return;
        }
      
      allRecipientsFromString = [self _recipientsFromString: aString];
      allRecipients = [NSMutableArray array];
        
      for (i = 0; i < [allRecipientsFromString count]; i++)
        {
	  ABSearchElement *aSearchElement;
	  NSArray *allMembers;
	  NSString *aValue;
	  
	  aValue = [allRecipientsFromString objectAtIndex: i];

	  aSearchElement = [ABGroup searchElementForProperty: kABGroupNameProperty
				    label: nil
				    key: nil
				    value: aValue
				    comparison: kABPrefixMatchCaseInsensitive];

	  allMembers = [[[[ABAddressBook sharedAddressBook] recordsMatchingSearchElement: aSearchElement]
			  lastObject] members];

  	  if ( [allMembers count] )
	    {
  	      int j;

  	      for (j = 0; j < [allMembers count]; j++)
		{
		  [allRecipients addObject: [[allMembers objectAtIndex: i] formattedValue]];
		}
	    }
	  // person
	  else
            {
	      [allRecipients addObject: aValue];
            }
        }
        
      [theControl setStringValue: [allRecipients componentsJoinedByString: @", "]];
    }
}


//
//
//
- (BOOL)isACompletion:(NSString *)candidate
{
  return [[self allCompletionsForPrefix:candidate] containsObject:candidate];
}


//
//
//
- (NSString * ) completionForPrefix: (NSString *) thePrefix
{
  NSString *completion;
  NSArray *allCompletions;

  allCompletions = [self allCompletionsForPrefix: thePrefix];

  completion = nil;

  if ([allCompletions count] > 0)
    {
      completion = [allCompletions objectAtIndex:0];
    }

  return completion;
}


//
//
//
- (NSArray *) allCompletionsForPrefix: (NSString *) thePrefix
{
  NSArray *searchResults; NSMutableArray *retval;
  int i;

  searchResults = [[AddressBookController singleInstance] addressesWithPrefix: thePrefix];
  retval = [NSMutableArray arrayWithCapacity: [searchResults count]];

  for(i=0; i<[searchResults count]; i++)
    {
      if ( [[searchResults objectAtIndex: i] isKindOfClass: [ABGroup class]] )
	{
	  [retval addObject: [[searchResults objectAtIndex: i] valueForProperty: kABGroupNameProperty]];
	}
      else
	{
	  [retval addObjectsFromArray: [[searchResults objectAtIndex: i] formattedValuesForPrefix: thePrefix]];
	}
    }
  
  return retval;
}


//
//
//
- (void) controlTextDidChange: (NSNotification *) aNotification
{
  if ( [aNotification object] == subjectText )
    {
      if ( [[subjectText stringValue] length] > 0 )
	{
	  [[self window] setTitle: [subjectText stringValue]];
	}
      else
	{
	  [[self window] setTitle: _(@"(no subject)")];
	}
    }
  else if ( [aNotification object] == toText ||
	    [aNotification object] == ccText ||
	    [aNotification object] == bccText )
    {
      if ( [[toText stringValue] length] == 0 &&
	   [[ccText stringValue] length] == 0 &&
	   [[bccText stringValue] length] == 0)
        {
	  [send setEnabled: NO];
        }
      else
        {
	  [send setEnabled: YES];
        }
    }
      
  [[self window] setDocumentEdited: YES];
}


//
//
//
- (BOOL) shouldChangeTextInRange: (NSRange) affectedCharRange 
	       replacementString: (NSString *) replacementString
{
  NSString *aString;

  aString = [[[textView textStorage] string] substringWithRange: affectedCharRange];
  
  if ( (([replacementString length] > 0) && 
	(([replacementString characterAtIndex: 0] =='\n') ||
	 ([replacementString characterAtIndex: 0] =='>')))
       ||
       (([aString length] > 0) && ([aString characterAtIndex: 0] == '>')) )
    {
      updateColors = YES;
      affectedRangeForColors = NSMakeRange(affectedCharRange.location, [replacementString length]);
    }
  
  return YES;
}


//
//
//
- (void) textDidChange: (NSNotification *) aNotification
{
  [[self window] setDocumentEdited: YES];
  [self _updateSizeLabel];

  if ( updateColors )
    {
      NSTextStorage *aTextStorage;

      aTextStorage = [textView textStorage];
      
      if ( [aTextStorage length] > 1 )
	{
	  NSAttributedString *aMutableAttributedString;
	  NSRange aRange;

	  aRange = [[aTextStorage string] lineRangeForRange: affectedRangeForColors];

	  if ( aRange.length )
	    {
	      aMutableAttributedString = [aTextStorage attributedSubstringFromRange: aRange];
	      [aTextStorage replaceCharactersInRange: aRange 
			    withAttributedString: [Utilities _quoteMessageContentFromAttributedString: 
							       aMutableAttributedString]];
	    }
	}
      
      updateColors = NO;
    }
}

//
//
//
- (BOOL) windowShouldClose: (id) sender
{
  if ( [[self window] isDocumentEdited])
    {
#ifdef MACOSX
      NSBeginAlertSheet(_(@"Closing..."),
			_(@"Cancel"),                                        // defaultButton
			_(@"Save in Drafts"),                                // alternateButton
			_(@"No"),                                            // otherButton
			[self window],
			self,                                                // delegate
			@selector(_sheetDidEnd:returnCode:contextInfo:),     // didEndSelector
			@selector(_sheetDidDismiss:returnCode:contextInfo:), // didDismissSelector
			nil,                                                 // contextInfo
		      _(@"Would you like to save this message in the Drafts folder?"));

      return NO;
#else
      int choice;
      
      choice = NSRunAlertPanel(_(@"Closing..."),
			       _(@"Would you like to save this message in the Drafts folder?"),
			       _(@"Cancel"),         // default
			       _(@"Save in Drafts"), // alternate
			       _(@"No"));            // other return
      
      // We don't want to close the window
      if ( choice == NSAlertDefaultReturn )
	{
	  return NO;
	}
      // Yes we want to close it, and we also want to save the message to the Drafts folder.
      else if ( choice == NSAlertAlternateReturn )
	{
	  // We append the message to the Drafts folder. 
	  [[MailboxManagerController singleInstance] saveMessageInDraftsFolderForController: self];
	}
#endif
    }
  
  return YES;
}


//
//
//
- (void) windowWillClose: (NSNotification *) theNotification
{ 
  if ( [GNUMail lastAddressTakerWindowOnTop] == self )
    {
      [GNUMail setLastAddressTakerWindowOnTop: nil];
    }
      
  [[NSApp delegate] setEnableSaveInDraftsMenuItem: NO];
  
  // We remove our window from our list of opened windows
  [GNUMail removeEditWindow: [self window]];

  AUTORELEASE(self);
}


//
//
//
- (void) windowDidBecomeMain: (NSNotification *) theNotification
{
#ifndef MACOSX
  [[[self window] contentView] setNeedsDisplay: YES];
#endif

  [GNUMail setLastAddressTakerWindowOnTop: self];
  [[NSApp delegate] setEnableSaveInDraftsMenuItem: YES];
  
  // Make the "Deliver" menu item deliver the message that is composed
  // in this window.
  [[[NSApp delegate] deliverMenuItem] setTarget: self];
  [[[NSApp delegate] deliverMenuItem] setAction: @selector(sendMessage:)];
}


//
//
//
- (void) windowDidResignMain: (NSNotification *) theNotification
{
  // We disable our Deliver menu item.
  [[[NSApp delegate] deliverMenuItem] setTarget: nil];
  [[[NSApp delegate] deliverMenuItem] setAction: NULL];
}


//
//
//
- (void) windowDidLoad
{ 
  // We add our observer for our two notifications
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_loadAccounts)
    name: AccountsHaveChanged
    object: nil];

  // We add our window from our list of opened windows
  [GNUMail addEditWindow: [self window]];
}


//
//
//
- (Message *) message
{
  return message;
}


//
//
//
- (void) setMessage: (Message *) theMessage
{  
  if ( theMessage )
    {
      RETAIN(theMessage);
      RELEASE(message);
      message = theMessage;

      [self _updateViewWithMessage: message
      	    appendSignature: YES];
      [self _updateSizeLabel];
    }
  else
    {
      DESTROY(message);
    }
}


//
//
//
- (Message *) unmodifiedMessage
{
  return unmodifiedMessage;
}


//
//
//
- (void) setUnmodifiedMessage: (Message *) theUnmodifiedMessage
{
  if ( theUnmodifiedMessage )
    {
      RETAIN(theUnmodifiedMessage);
      RELEASE(unmodifiedMessage);
      unmodifiedMessage = theUnmodifiedMessage;
    }
  else
    {
      DESTROY(unmodifiedMessage);
    }
}


//
//
//
- (void) setMessageFromDraftsFolder: (Message *) theMessage
{
  if (theMessage)
    {
      RETAIN(theMessage);
      RELEASE(message);
      message = theMessage;

      [self _updateViewWithMessage: message
	    appendSignature: NO];
    }
  else
    {
      DESTROY(message);
    }
}


//
//
//
- (BOOL) showCc
{
  return showCc;
}


//
//
//
- (void) setShowCc: (BOOL) theBOOL
{
  showCc = theBOOL;
  
  if (showCc)
    {
#ifdef MACOSX
      [ccText setEnabled: YES];
      [addCc setLabel: _(@"Remove Cc")];
      [addCc setImage: [NSImage imageNamed: @"remove_cc_32.tiff"]];
#else
      [[[self window] contentView] addSubview: ccLabel];
      [[[self window] contentView] addSubview: ccText];
      [addCc setTitle: _(@"Remove Cc")];
      [addCc setImage: [NSImage imageNamed: @"remove_cc_48.tiff"]];
#endif
    }
  else
    {
#ifdef MACOSX
      [ccText setEnabled: NO];
      [addCc setLabel: _(@"Add Cc")];
      [addCc setImage: [NSImage imageNamed: @"add_cc_32.tiff"]];
#else
      [ccLabel removeFromSuperviewWithoutNeedingDisplay];
      [ccText removeFromSuperviewWithoutNeedingDisplay];
      [addCc setTitle: _(@"Add Cc")];
      [addCc setImage: [NSImage imageNamed: @"add_cc_48.tiff"]];
#endif
    }
  
#ifndef MACOSX
  [self _adjustWidgetsPosition];
#endif

  [self _adjustNextKeyViews];
}


//
//
//
- (BOOL) showBcc
{
  return showBcc;
}


//
//
//
- (void) setShowBcc: (BOOL) theBOOL
{
  showBcc = theBOOL;
 
  if (showBcc)
    {
#ifdef MACOSX
      [bccText setEnabled: YES];
      [addBcc setLabel: _(@"Remove Bcc")];
      [addBcc setImage: [NSImage imageNamed: @"remove_bcc_32.tiff"]];
#else
      [[[self window] contentView] addSubview: bccLabel];
      [[[self window] contentView] addSubview: bccText];
      [addBcc setTitle: _(@"Remove Bcc")];
      [addBcc setImage: [NSImage imageNamed: @"remove_bcc_48.tiff"]];
#endif
    }
  else
    {
#ifdef MACOSX
      [bccText setEnabled: NO];
      [addBcc setLabel: _(@"Add Bcc")];
      [addBcc setImage: [NSImage imageNamed: @"add_bcc_32.tiff"]];
#else
      [bccLabel removeFromSuperviewWithoutNeedingDisplay];
      [bccText removeFromSuperviewWithoutNeedingDisplay];
      [addBcc setTitle: _(@"Add Bcc")];
      [addBcc setImage: [NSImage imageNamed: @"add_bcc_48.tiff"]];
#endif
    }

#ifndef MACOSX  
  [self _adjustWidgetsPosition];
#endif

  [self _adjustNextKeyViews];
}


//
//
//
- (int) signaturePosition
{
  return signaturePosition;
}


//
//
//
- (void) setSignaturePosition: (int) thePosition
{
  signaturePosition = thePosition;
}


//
//
//
- (BOOL) editingDraftMessage
{
  return (editingDraftMessage);
}


//
//
//
- (void) setEditingDraftMessage: (BOOL) flag
{
  editingDraftMessage = flag;
}


//
//
//
- (NSPopUpButton *) accountPopUpButton
{
  return accountPopUpButton;
}


//
// Changes and select the right account in the popup button.
//
- (void) setAccountName: (NSString *) theAccountName
{
  if ( theAccountName )
    {
      [Utilities loadAccountsInPopUpButton: accountPopUpButton
		 select: theAccountName];
      [self accountSelectionHasChanged: nil];
    }		
}


//
//
//
- (NSString *) charset
{
  return charset;
}


//
//
//
- (void) setCharset: (NSString *) theCharset
{
  if ( theCharset )
    {
      RETAIN(theCharset);
      RELEASE(charset);
      charset = theCharset;
    }
  else
    {
      DESTROY(charset);
    }
}


//
// Other methods
//
- (BOOL) updateMessageContentFromTextView
{
  NSTextStorage *textStorage;
  NSArray *theArray;

  NSDictionary *allAdditionalHeaders, *allValues;
 
  InternetAddress *anInternetAddress;
  BOOL hasFoundXMailerHeader;
  NSString *aString;
  int i;
  
  // We initialize our boolean value to false
  hasFoundXMailerHeader = NO;
  
  // We get the current selected account when sending this email
  [accountPopUpButton synchronizeTitleAndSelectedItem];
  
  // Then, we get our account from our user defaults
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]]
		objectForKey: @"PERSONAL"];
  
  // We set the From header value
  anInternetAddress = [[InternetAddress alloc] initWithPersonal: [allValues objectForKey: @"NAME"]
					       andAddress: [allValues objectForKey: @"EMAILADDR"]];
  [message setFrom: anInternetAddress];
  RELEASE(anInternetAddress);
  
  // We set the Reply-To address (if we need too)
  aString = [allValues objectForKey: @"REPLYTOADDR"];
  
  if (aString && [[aString stringByTrimmingWhiteSpaces] length] > 0)
    {
      anInternetAddress = [[InternetAddress alloc] initWithString: aString];
      
      if (anInternetAddress)
	{
	  [message setReplyTo: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
    }
  else
    {
      [message setReplyTo: nil];
    }
  
  // We set the Organization header value (if we need to)
  aString = [allValues objectForKey: @"ORGANIZATION"];
  
  if (aString && [[aString stringByTrimmingWhiteSpaces] length] > 0)
    {
      [message setOrganization: aString];
    }

  // For now, if there are recipients, we remove them in case a user add or changed them manually
  if ([message recipientsCount] > 0) 
    {
      [message removeAllRecipients];
    }

  // We decode our recipients, let's begin with the To field.
  if ( [[[toText stringValue] stringByTrimmingWhiteSpaces] length] > 0 )
    {
      theArray = [self _recipientsFromString: [toText stringValue] ];
      
      for (i = 0; i < [theArray count]; i++)
	{
	  anInternetAddress = [[InternetAddress alloc] initWithString: [theArray objectAtIndex:i]];
	  
	  if ( !anInternetAddress )
	    {
	      [self _showAlertPanelWithString: [theArray objectAtIndex: i] ];
	      return NO;
	    }
	  
	  [anInternetAddress setType: TO];
	  [message addToRecipients: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
    }

  // We decode our Cc field, if we need to
  if ( showCc )
    {
      theArray = [self _recipientsFromString: [ccText stringValue]];
      
      for (i = 0; i < [theArray count]; i++)
	{
	  anInternetAddress = [[InternetAddress alloc] initWithString: [theArray objectAtIndex:i]];

	  if ( !anInternetAddress )
	    {
	      [self _showAlertPanelWithString: [theArray objectAtIndex: i] ];
	      return NO;
	    }

	  [anInternetAddress setType: CC];
	  [message addToRecipients: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
      
    }

  // We decode our Bcc field, if we need to
  if ( showBcc )
    {
      theArray = [self _recipientsFromString: [bccText stringValue]];
      
      for (i = 0; i < [theArray count]; i++)
	{
	  anInternetAddress = [[InternetAddress alloc] initWithString: [theArray objectAtIndex:i]];
	  
	  if ( !anInternetAddress )
	    {
	      [self _showAlertPanelWithString: [theArray objectAtIndex: i] ];
	      return NO;
	    }

	  [anInternetAddress setType: BCC];
	  [message addToRecipients: anInternetAddress];
	  RELEASE(anInternetAddress);
	}
    }

  
  // We decode our Subject field
  [message setSubject: [subjectText stringValue]];
  
  // We finally add all our addition headers from the user defaults
  allAdditionalHeaders = [[NSUserDefaults standardUserDefaults] 
			   objectForKey: @"ADDITIONALOUTGOINGHEADERS"];
  
  if ( allAdditionalHeaders )
    {
      NSEnumerator *anEnumerator;
      NSString *aKey, *aValue;

      anEnumerator = [allAdditionalHeaders keyEnumerator];
      
      while ( (aKey = [anEnumerator nextObject]) )
	{
	  // We get our value
	  aValue = [allAdditionalHeaders objectForKey: aKey];
	  
	  // We add our X- if we don't have it
	  if ( ![aKey hasPrefix: @"X-"] )
	    {
	      aKey = [NSString stringWithFormat: @"X-%@", aKey];
	    }

	  if ( [aKey compare: @"X-Mailer"  options: NSCaseInsensitiveSearch] == NSOrderedSame )
	    {
	      hasFoundXMailerHeader = YES;
	    }
	  
	  [message addHeader: aKey
		   withValue: aValue];
	}
    }

  // We add our X-Mailer: GNUMail.app (Version ...) header if the user hasn't defined one
  if ( !hasFoundXMailerHeader )
    {
      [message addHeader: @"X-Mailer"
	       withValue: [NSString stringWithFormat: @"GNUMail.app (Version %@)", GNUMailVersion()]];
    }

  // We get our text storage
  textStorage = [textView textStorage];
  
  // We now build our message body - considering all the attachments
  if ( [textStorage length] == 1 && [textStorage containsAttachments] ) 
    {
      NSTextAttachment *aTextAttachment;
      
      aTextAttachment = [textStorage attribute: NSAttachmentAttributeName		      
				     atIndex: 0
				     effectiveRange: NULL];
      
      [self _updatePart: message
	    usingTextAttachment: aTextAttachment];
    }
  else if ( ![textStorage containsAttachments] )
    {
      [self _setPlainTextContentFromString: [textView string]
	    inPart: message];
    }
  else
    {         
      MimeMultipart *aMimeMultipart;
      Part *aPart;      
      
      NSAutoreleasePool *pool;
      NSString *aString;

      // We create our local autorelease pool
      pool = [[NSAutoreleasePool alloc] init];

      aMimeMultipart = [[MimeMultipart alloc] init];
      
      // We first decode our text/plain body
      aPart = [[Part alloc] init];

      [self _setPlainTextContentFromString: [self _plainTextContentFromTextView]
	    inPart: aPart];

      // We add our new part
      [aMimeMultipart addBodyPart: aPart];
      RELEASE(aPart);

      // We get our text store string representation
      aString = [textStorage string];
      
      // We finally add all our attachments
      for (i = 0; i < [textStorage length]; i++)
	{
	  NSTextAttachment *aTextAttachment;
	  
	  if ( [aString characterAtIndex: i] != NSAttachmentCharacter )
	    {
	      continue;
	    }

	  aTextAttachment = [textStorage attribute: NSAttachmentAttributeName		      
					 atIndex: i 
					 effectiveRange: NULL];
	  if ( aTextAttachment ) 
	    {
	      Part *aPart;
	      
	      if ( [[aTextAttachment attachmentCell] respondsToSelector: @selector(part)] )
		{
		  aPart = [(ExtendedAttachmentCell *)[aTextAttachment attachmentCell] part];
		}
	      else
		{
		  aPart = nil;
		}

	      if ( aPart )
		{
		  [aMimeMultipart addBodyPart: (Part *)aPart];
		}
	      else
		{
		  aPart = [[Part alloc] init];
		  
		  [self _updatePart: aPart
			usingTextAttachment: aTextAttachment];
		  
		  [aMimeMultipart addBodyPart: aPart];
		  RELEASE(aPart);		 
		}
	      
	    } // if ( aTextAttachment ) 
	  
	} // for (...)
      
      [message setContentTransferEncoding: NONE];
      [message setContentType: @"multipart/mixed"];     
      [message setContent: aMimeMultipart];
      
      // We generate a new boundary for our message
      [message setBoundary: [MimeUtility generateBoundary]];
      
      RELEASE(aMimeMultipart);
      RELEASE(pool);
    }

  //
  // We inform our bundles that our message WAS encoded
  //
  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle respondsToSelector: @selector(messageWasEncoded:)] )
	{
	  // If AT LEAST ONE bundle failed to encode the message, we
	  // stop everything for now.
	  //
	  // FIXME: should use the returned copy successively instead of
	  //        modifying directly "message"
	  //
	  if ( ![aBundle messageWasEncoded: message] )
	    {
	      return NO;
	    }
	}
    }

  return YES;
}


//
//
//
- (void) insertFileWithName: (NSString *) theFileName
{
  if ( theFileName )
    {
      [textView insertFile: theFileName];
    }
}

@end


//
// private methods
//
@implementation EditWindowController (Private)

- (void) _adjustNextKeyViews
{
  if (showCc && showBcc)
    {
      [toText setNextKeyView: ccText];
      [ccText setNextKeyView: bccText];
      [bccText setNextKeyView: subjectText];
    }	 
  else if (showCc && !showBcc)
    {
      [toText setNextKeyView: ccText];
      [ccText setNextKeyView: subjectText];
    }
  else if (!showCc && showBcc)
    {
      [toText setNextKeyView: bccText];
      [bccText setNextKeyView: subjectText];
    }
  else
    {
      [toText setNextKeyView: subjectText];
    }
}


//
//
//
- (void) _adjustWidgetsPosition
{  
  NSRect rectOfToText;
  float widthOfScrollView;

  rectOfToText = [toText frame];
  widthOfScrollView = [scrollView frame].size.width;
  
  if (showCc && showBcc)
    {
      NSDebugLog(@"showCc is YES showBcc is YES");
      
      // To - 25
      [ccLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-25,50,21)];
      [ccText setFrame: NSMakeRect(65,rectOfToText.origin.y-25,widthOfScrollView-75,21)];
      
      // To - 50
      [bccLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-50,50,21)];
      [bccText setFrame: NSMakeRect(65,rectOfToText.origin.y-50,widthOfScrollView-75,21)];
      
      // To - 75
      [subjectLabel setFrame: NSMakeRect(0,rectOfToText.origin.y-75,55,21)];
      [subjectText setFrame: NSMakeRect(65,rectOfToText.origin.y-75,widthOfScrollView-75,21)];
      
      // To - 100
      [sizeLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-100,200,21)];
      
      // Space left...
      [scrollView setFrame: NSMakeRect(5,5,widthOfScrollView,rectOfToText.origin.y-105)];
    }	 
  else if (showCc && !showBcc)
    {
      NSDebugLog(@"showCc is YES showBcc is NO");

      // To - 25
      [ccLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-25,50,21)];
      [ccText setFrame: NSMakeRect(65,rectOfToText.origin.y-25,widthOfScrollView-75,21)];
      
      // To - 50
      [subjectLabel setFrame: NSMakeRect(0,rectOfToText.origin.y-50,55,21)];
      [subjectText setFrame: NSMakeRect(65,rectOfToText.origin.y-50,widthOfScrollView-75,21)];
      
      // To - 75
      [sizeLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-75,200,21)];

      // Space left...
      [scrollView setFrame: NSMakeRect(5,5,widthOfScrollView,rectOfToText.origin.y-80)];
    }
  else if (!showCc && showBcc)
    {
      NSDebugLog(@"showCc is NO showBcc is YES");

      // To - 25
      [bccLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-25,50,21)];
      [bccText setFrame: NSMakeRect(65,rectOfToText.origin.y-25,widthOfScrollView-75,21)];
      
      // To - 50
      [subjectLabel setFrame: NSMakeRect(0,rectOfToText.origin.y-50,55,21)];
      [subjectText setFrame: NSMakeRect(65,rectOfToText.origin.y-50,widthOfScrollView-75,21)];
      
      // To - 75
      [sizeLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-75,200,21)];

      // Space left...
      [scrollView setFrame: NSMakeRect(5,5,widthOfScrollView,rectOfToText.origin.y-80)];
    }
  else
    {
      NSDebugLog(@"showCc is NO showBcc is NO");
     
      // To - 25
      [subjectLabel setFrame: NSMakeRect(0,rectOfToText.origin.y-25,55,21)];
      [subjectText setFrame: NSMakeRect(65,rectOfToText.origin.y-25,widthOfScrollView-75,21)];
      
      // To - 50
      [sizeLabel setFrame: NSMakeRect(5,rectOfToText.origin.y-50,200,21)];
     
      // Space left...
      [scrollView setFrame: NSMakeRect(5,5,widthOfScrollView,rectOfToText.origin.y-55)];
    }
}


//
//
//
- (void) _loadAccessoryViews
{
  int i;

  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasComposeViewAccessory] )
	{
#ifdef MACOSX
          NSToolbarItem *aToolbarItem;
          NSToolbar *aToolbar;
          id aView;
          
          aToolbarItem = [[NSToolbarItem alloc] initWithItemIdentifier: [aBundle name]];
          [allowedToolbarItemIdentifiers addObject: [aBundle name]];
          
          [additionalToolbarItems setObject: aToolbarItem
                                  forKey: [aBundle name]];
                                                      
          aView = [aBundle composeViewAccessory];
          [aToolbarItem setView: aView];
          [aToolbarItem setLabel: [aBundle name]];               // name
          [aToolbarItem setPaletteLabel: [aBundle description]]; // description
          [aToolbarItem setMinSize: [aView frame].size];
          [aToolbarItem setMaxSize: [aView frame].size];
          RELEASE(aToolbarItem);
          
          aToolbar = [[self window] toolbar];
          [aToolbar insertItemWithItemIdentifier: [aBundle name]
                    atIndex: [[aToolbar visibleItems] count]];
	  
#else    
          NSRect aRect;
	  id aView;
	  
	  aView = [aBundle composeViewAccessory];
	  aRect = [aView frame];      
	  aRect.origin.x = ([addresses frame].origin.x + 72);
	  aRect.origin.y = [addresses frame].origin.y;

	  [aView setFrame: aRect];
	  [aView setAutoresizingMask: NSViewMinYMargin];

	  [[[self window] contentView] addSubview: aView];
#endif
	}

      // We also set the current superview
      [aBundle setCurrentSuperview: [[self window] contentView]];
    }
}


//
//
//
- (void) _loadCharset
{
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"DEFAULT_CHARSET"] )
    {
      NSString *aString;
      
      aString = [[MimeUtility allCharsets] objectForKey: [[[NSUserDefaults standardUserDefaults] 
							    objectForKey: @"DEFAULT_CHARSET"] lowercaseString]];
      
      if ( aString )
	{
	  [self setCharset: aString];
	}
      else
	{
	  [self setCharset: nil];
	}
    }
  else
    {
      [self setCharset: nil];
    }
}


//
//
//
- (void) _loadAccounts
{
  [Utilities loadAccountsInPopUpButton: accountPopUpButton 
	     select: nil];

  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];
}


//
// 
//
- (NSString *) _loadSignature
{
  NSDictionary *allValues;
  NSString *aSignature;

  [accountPopUpButton synchronizeTitleAndSelectedItem];

  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [Utilities accountNameForItemTitle: [accountPopUpButton titleOfSelectedItem]]]
		objectForKey: @"PERSONAL"];
  
  aSignature = nil;
  
  if ( [allValues objectForKey: @"SIGNATURE_SOURCE"] &&
       [[allValues objectForKey: @"SIGNATURE_SOURCE"] intValue] == 0 )
    {
      aSignature = [NSString stringWithContentsOfFile:
			       [allValues objectForKey: @"SIGNATURE"]];
    }
  else if ( [allValues objectForKey: @"SIGNATURE_SOURCE"] &&
	    [[allValues objectForKey: @"SIGNATURE_SOURCE"] intValue] == 1 )
    {
      NSFileHandle *aFileHandle;
      NSString *aString;
      NSTask *aTask;
      NSPipe *aPipe;
      NSData *aData;
      NSRange aRange;

      
      // We get our program's name (and arguments, if any)
      aString = [allValues objectForKey: @"SIGNATURE"];

      // If a signature hasn't been set, let's return.
      if ( !aString )
	{
	  return nil;
	}
      
      aPipe = [NSPipe pipe];
      aFileHandle = [aPipe fileHandleForReading];
      
      aTask = [[NSTask alloc] init];
      [aTask setStandardOutput: aPipe];
      
      // We trim our string from any whitespaces
      aString = [aString stringByTrimmingWhiteSpaces];

      // We verify if our program to lauch has any arguments
      aRange = [aString rangeOfString: @" "];
      
      if ( aRange.length )
	{
	  [aTask setLaunchPath: [aString substringToIndex: aRange.location]];
	  [aTask setArguments: [NSArray arrayWithObjects: [aString substringFromIndex: (aRange.location + 1)],
					nil]];
	}
      else
	{
	  [aTask setLaunchPath: aString];
	}
      
      // We verify if our launch path points to an executable file
      if ( ![[NSFileManager defaultManager] isExecutableFileAtPath: [aTask launchPath]] )
	{
	  NSDebugLog(@"The signature's path doesn't point to an executable! Ignored.");
	  RELEASE(aTask);
	  return nil;
	}
      
      // We launch our task
      [aTask launch];
      
      while ( [aTask isRunning] )
	{
	  [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
				      beforeDate: [NSDate distantFuture]];
	}
      
      aData = [aFileHandle readDataToEndOfFile];
      
      aSignature = [[NSString alloc] initWithData: aData
				     encoding: NSUTF8StringEncoding];
      AUTORELEASE(aSignature);
      
      RELEASE(aTask);
    }
  
  if ( aSignature )
    {
      return [NSString stringWithFormat: @"\n\n-- \n%@", aSignature];
    }

  return nil;
}


//
//
//
- (void) _openPanelDidEnd: (NSOpenPanel *) theOpenPanel
               returnCode: (int) theReturnCode
              contextInfo: (void *) theContextInfo
{
  if (theReturnCode == NSOKButton)
    {
      NSEnumerator *filesToOpenEnumerator;
      NSFileManager *aFileManager;
      NSString *theFilename;
      
      filesToOpenEnumerator = [[theOpenPanel filenames] objectEnumerator];
      aFileManager = [NSFileManager defaultManager];

      while ( (theFilename = [filesToOpenEnumerator nextObject]) )
	{
	  if ( ![aFileManager isReadableFileAtPath: theFilename] )
	    {
	      NSRunAlertPanel(_(@"Error!"),
			      _(@"The file %@ is not readable and has not been attached to this E-Mail."),
			      _(@"OK"),
			      NULL,
			      NULL,
			      theFilename);
	    }
	  else
	    {
	      [self insertFileWithName: theFilename];
	    }
	}
      
      [[self window] makeFirstResponder: textView ];
    }
}


//
//
//
- (void) _replaceSignature
{
  NSString *aSignature;
  
  // If we don't want a signature...
  if ( [self signaturePosition] == SIGNATURE_HIDDEN )
    {
      return;
    }
  
  if ( previousSignatureValue )
    {
      NSRange aRange;
      
      if ( [self signaturePosition] == SIGNATURE_BEGINNING )
	{
	  aRange = [[[textView textStorage] string] rangeOfString: previousSignatureValue];
	}
      else if ( [self signaturePosition] == SIGNATURE_END )
	{
	  aRange = [[[textView textStorage] string] rangeOfString: previousSignatureValue
						    options: NSBackwardsSearch];
	}
      if ( aRange.length )
	{
	  [[textView textStorage] deleteCharactersInRange: aRange];
	}
    }

  // We load the new signature and insert it at the proper position
  aSignature = [self _loadSignature];
  ASSIGN(previousSignatureValue, aSignature);

  if ( aSignature )
    {
      if ( [self signaturePosition] == SIGNATURE_BEGINNING )
  	{
	  NSMutableAttributedString *theMutableAttributedString;
	  
	  if ( [textView font] )
	    {
	      theMutableAttributedString = [[NSMutableAttributedString alloc] initWithString: aSignature
									      attributes: [NSDictionary dictionaryWithObject: [textView font]
													forKey: NSFontAttributeName]];
	    }
	  else
	    {
	      theMutableAttributedString = [[NSMutableAttributedString alloc] initWithString: aSignature];
	    }
	  
	  [theMutableAttributedString appendAttributedString: [textView textStorage]];
	  [[textView textStorage] setAttributedString: theMutableAttributedString];
	  RELEASE(theMutableAttributedString);
	}
      else if ( [self signaturePosition] == SIGNATURE_END )
	{
	  NSAttributedString *theAttributedString;
	  
	  if ( [textView font] )
	    {
	      theAttributedString = [[NSAttributedString alloc] initWithString: aSignature
								attributes: [NSDictionary dictionaryWithObject: [textView font]
											  forKey: NSFontAttributeName]];
	    }
	  else
	    {
	      theAttributedString = [[NSAttributedString alloc] initWithString: aSignature];
	    }

	  [[textView textStorage] appendAttributedString: theAttributedString];
	  RELEASE(theAttributedString);
	}

      [textView setSelectedRange: NSMakeRange(0,0)];
    }

  // We update our size label
  [self _updateSizeLabel];
}


//
//
//
- (NSString *) _plainTextContentFromTextView
{
  NSMutableString *aMutableString;
  NSTextStorage *textStorage;

  NSAutoreleasePool *pool;
  int i, len;
  
  textStorage = [textView textStorage];

  aMutableString = [[NSMutableString alloc] initWithString: [textStorage string]];
 
  pool = [[NSAutoreleasePool alloc] init];
  len = [aMutableString length];

  for (i = (len - 1); i >= 0; i--)
    { 
      NSTextAttachment *aTextAttachment;     
      id cell;
 
      if ( [aMutableString characterAtIndex: i] != NSAttachmentCharacter )
	{
	  continue;
	}

      aTextAttachment = [textStorage attribute: NSAttachmentAttributeName
				     atIndex: i
				     effectiveRange: NULL];
      
      cell = (ExtendedAttachmentCell *)[aTextAttachment attachmentCell];
      
      if ( ![cell respondsToSelector: @selector(part)] )
	{
	  cell = nil;
	}

      if ( cell && [cell part] && [[cell part] filename] )
	{
	  [aMutableString replaceCharactersInRange: NSMakeRange(i, 1)
			  withString: [NSString stringWithFormat: @"<%@>", [[cell part] filename]] ];
	}
      else if ( [[[aTextAttachment fileWrapper] filename] lastPathComponent] )
	{
	  [aMutableString replaceCharactersInRange: NSMakeRange(i, 1)
			  withString: [NSString stringWithFormat: @"<%@>", [[[aTextAttachment fileWrapper] filename]
									     lastPathComponent]] ];
	}
      else
	{
	  [aMutableString replaceCharactersInRange: NSMakeRange(i, 1)
			  withString: @"<unknown>"];
	}
    }
  
  RELEASE(pool);
  
  return AUTORELEASE(aMutableString);
}


//
//
//
- (NSArray *) _recipientsFromString: (NSString *) theString
{
  NSMutableArray *pairsStack, *returnArray;
  NSMutableString *aString;
  unichar c;
  int i;
  
  returnArray = [NSMutableArray array];
  pairsStack = [NSMutableArray array];
  aString = (NSMutableString *)[NSMutableString string];
  
  for (i = 0; i < [theString length]; i++)
    {
      c = [theString characterAtIndex: i];
      switch (c)
	{
	case ',':
	  if ([pairsStack count] == 0 && [aString length])
	    {
	      [returnArray addObject: [NSString stringWithString: aString]];
	      [aString replaceCharactersInRange: NSMakeRange(0, [aString length])
		       withString: @""];
	      continue;
	    }
	  break;
	case ' ':
	  if (![aString length])
	    {
	      continue;
	    }
	  break;
	case '"':
	  if ([pairsStack count] == 0 || [(NSNumber*)[pairsStack lastObject] intValue] != '"')
	    {
	      [pairsStack addObject: [NSNumber numberWithChar: c]];
	    }
	  else
	    {
	      [pairsStack removeLastObject];
	    }
	  break;
	case '<':
	  [pairsStack addObject: [NSNumber numberWithChar: c]];
	  break;
	case '>':
	  if ([pairsStack count] && [(NSNumber*)[pairsStack lastObject] intValue] == '<')
	    {
	      [pairsStack removeLastObject];
	    }
	  break;
	case '(':
	  [pairsStack addObject: [NSNumber numberWithChar: c]];
	  break;
	case ')':
	  if ([pairsStack count] && [(NSNumber*)[pairsStack lastObject] intValue] == '(')
	    {
	      [pairsStack removeLastObject];
	    }
	  break;
	}
      [aString appendFormat: @"%c", c];
    }
  
  if ([pairsStack count] == 0 && [aString length])
    {
      [returnArray addObject: [NSString stringWithString: aString]];
    }

  return returnArray;
}


//
//
//
- (void) _setPlainTextContentFromString: (NSString *) theString
                                 inPart: (Part *) thePart
{
  // We must first verify if our string contains any non-ascii character
  if ( [MimeUtility isASCIIString: theString] )
    {
      int wrapLimit;
      
      wrapLimit = [[NSUserDefaults standardUserDefaults] integerForKey: @"LINE_WRAP_LIMIT"];
      
      if ( !wrapLimit )
	{
	  wrapLimit = 72;
	}
      
      [thePart setContentType: @"text/plain"];                // Content-Type
      [thePart setContentTransferEncoding: NONE];             // encoding -> none
      [thePart setCharset: @"us-ascii"];
      [thePart setFormat: FORMAT_FLOWED];
      [thePart setLineLength: wrapLimit];
      [thePart setContent: theString];
    }
  else
    {
      NSString *aCharset;
     
      // We verify if we are using automatic charset detection (when it's nil)
      if ( ![self charset] )
	{
	  aCharset = [MimeUtility charsetForString: theString];
	}
      else
	{
	  NSArray *allKeys;

	  allKeys = [[MimeUtility allCharsets] allKeysForObject: [self charset]];
	  
	  if ( [allKeys count] )
	    {
	      aCharset = [allKeys objectAtIndex: 0];
	    }
	  else
	    {
	      aCharset = [MimeUtility charsetForString: theString];
	    }
	}
      
      [thePart setContentType: @"text/plain"];               // Content-Type

      // Next, we verify if we can use an encoding other than QP for a specific
      // Content-Type. For now, we only verify for ISO-2022-JP charsets.
      if ( [[aCharset lowercaseString] isEqualToString: @"iso-2022-jp"] )
	{
	  [thePart setContentTransferEncoding: NONE]; // encoding -> none
	}
      else
	{
	  [thePart setContentTransferEncoding: QUOTEDPRINTABLE]; // encoding -> quoted-printable
       }
      
      [thePart setFormat: FORMAT_UNKNOWN];
      [thePart setCharset: aCharset];
      [thePart setContent: theString];
    }
}

//
// Only called under MacOS X
//
- (void) _sheetDidEnd: (NSWindow *) sheet
	   returnCode: (int) returnCode
	  contextInfo: (void *) contextInfo
{
  if ( returnCode == NSAlertAlternateReturn )
    {
      // We append the message to the Drafts folder.
      [[MailboxManagerController singleInstance] saveMessageInDraftsFolderForController: self];
    }
}

//
// Only called under MacOS X
//
- (void) _sheetDidDismiss: (NSWindow *) sheet
	       returnCode: (int) returnCode
	      contextInfo: (void *) contextInfo
{
    // We cancel the closing operation
    if ( returnCode == NSAlertDefaultReturn )
    {
        return;
    }

    [[self window] close];
}

//
//
//
- (void) _showAlertPanelWithString: (NSString *) theString
{
  NSRunInformationalAlertPanel(_(@"Error!"),
			       _(@"An error occured while decoding %@. Please fix this address."),
			       _(@"OK"),
			       NULL,
			       NULL,
			       theString);
}


//
//
//
- (void) _updateViewWithMessage: (Message *) theMessage
		appendSignature: (BOOL) aBOOL
{
  NSMutableString *ccMutableString, *bccMutableString;
  NSEnumerator *recipientsEnumerator;
  InternetAddress *theRecipient;
  
  // We get all recipients and we set our To/Cc/Bcc fields.
  ccMutableString = [[NSMutableString alloc] init];
  bccMutableString = [[NSMutableString alloc] init];
  recipientsEnumerator = [[theMessage recipients] objectEnumerator];
  
  while ( (theRecipient = [recipientsEnumerator nextObject]) )
    {  
      if ( [theRecipient type] == TO )
	{
	  [toText setStringValue: [theRecipient unicodeStringValue]];
	}
      else if ( [theRecipient type] == CC ) 
	{
	  [ccMutableString appendString: [NSString stringWithFormat: @"%@, ", [theRecipient unicodeStringValue]]];
	}
      else if ( [theRecipient type] == BCC ) 
	{
	  [bccMutableString appendString: [NSString stringWithFormat: @"%@, ", [theRecipient unicodeStringValue]]];
	}
    }

  if ( [[toText stringValue] length] == 0 &&
       [[ccText stringValue] length] == 0 &&
       [[bccText stringValue] length] == 0)
    {
      [send setEnabled: NO];
    }
  
  // We set the value to our fields, if we need to.
  if ( [ccMutableString length] > 0 ) 
    {
      [ccText setStringValue: [ccMutableString substringToIndex: ([ccMutableString length]-2)]];
    }

  if ( [bccMutableString length] > 0 ) 
    {
      [bccText setStringValue: [bccMutableString substringToIndex: ([bccMutableString length]-2)]];
    }


  // We are done with our mutable strings - we release them.
  RELEASE(ccMutableString);
  RELEASE(bccMutableString);

  // We set our subject
  if ( [theMessage subject] )
    {
      [subjectText setStringValue: [theMessage subject]];
    }
  else
    {
      [subjectText setStringValue: @""];
    }

  // We now set the content of the message (considering all attachments)
  // in case of a forward / reply. We quote it properly.
  if ( [theMessage content] )
    {
      [[textView textStorage] setAttributedString: 
				[Utilities _quoteMessageContentFromAttributedString: 
					     [Utilities formattedAttributedStringFromAttributedString:
							  [Utilities attributedStringFromContentForPart: theMessage]]] ];
    }

  // We finally set the signature, if we need to
  if ( aBOOL )
    {
      [self _replaceSignature];
    }
}


//
//
//
- (void) _updateSizeLabel
{
  NSTextStorage *aTextStorage;
  NSAutoreleasePool *pool;
  NSString *aString;
  float size;

  pool = [[NSAutoreleasePool alloc] init];

  size = [[textView string] length];
  size = (float)(size / (float)1024);

  aTextStorage = [textView textStorage];
  
  // FIXME
  // This is _very_ slow 
  if ( [aTextStorage containsAttachments] )
    {
      NSTextAttachment *aTextAttachment;
      int i, len;

      len = [aTextStorage length];

      for (i = 0; i < len; i++)
	{  
	  aTextAttachment = [aTextStorage attribute: NSAttachmentAttributeName		      
					  atIndex: i 
					  effectiveRange: NULL];
	  if ( aTextAttachment ) 
	    {
	      Part *aPart;

	      if ( [[aTextAttachment attachmentCell] respondsToSelector: @selector(part)] )
		{
		  aPart = [(ExtendedAttachmentCell *)[aTextAttachment attachmentCell] part];
		}
	      else
		{
		  aPart = nil;
		}

	      if ( aPart )
		{
		  size += (float)((float)[aPart size] / (float)1024);
		}
	      else
		{
		  NSFileWrapper *aFileWrapper;
		  
		  aFileWrapper = [aTextAttachment fileWrapper];

		  size += (float)((float)[[aFileWrapper regularFileContents] length] / (float)1024);
		}
	    }
	}
    }

  // We finally update the string value of our label
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"USE_FIXED_FONT_FOR_TEXT_PLAIN_MESSAGES"] == NSOnState )
    {
      aString = [NSString stringWithFormat: _(@"%0.1fKB (%d characters) - %d characters per line are shown"), size, [aTextStorage length],
			  ((int)floor((float)[textView frame].size.width / (float)[[textView font] maximumAdvancement].width) - 1)];
    }
  else
    {
      aString = [NSString stringWithFormat: _(@"%0.1fKB (%d characters)"), size, [aTextStorage length]];
    }

  [sizeLabel setStringValue: aString];
  [sizeLabel setNeedsDisplay: YES];

  RELEASE(pool);
}

//
//
//
- (void) _updatePart: (Part *) thePart
 usingTextAttachment: (NSTextAttachment *) theTextAttachment
{
  NSFileWrapper *aFileWrapper;
  MimeType *aMimeType;
  NSData *aData;
  
  aFileWrapper = [theTextAttachment fileWrapper];
  [thePart setFilename: [[aFileWrapper filename] lastPathComponent]];
  
  // We search for the content-type to use 
  aMimeType = [[MimeTypeManager singleInstance] bestMimeTypeForFileExtension:
						      [[[aFileWrapper filename] lastPathComponent]
							pathExtension]];
  if ( aMimeType )
    {
      [thePart setContentType: [aMimeType mimeType]];
    }
  else
    {
      [thePart setContentType: @"application/octet-stream"];
    }
  
  [thePart setContentTransferEncoding: BASE64];   // always base64 encoding for now
  [thePart setContentDisposition: @"attachment"]; // always attachment for now
      
  aData = [aFileWrapper regularFileContents];
  [thePart setContent: aData];
}

@end
