#include <config.h>
#include <samplerate.h>
#include "filter.h"
#include "sample.h"
#include "snd.h"

struct filter_format {
    snd *sr;
    track_map_t source_tracks_map;
    track_map_t target_tracks_map;
};

struct filter *
filter_snd_new(struct filter *producer,
               size_t sz,
               snd *sr,
               track_map_t source_track_map,
               track_map_t target_track_map) {
    struct filter *f = filter_new(producer, sz);

    if(!f) 
        return NULL;

    f->read = filter_snd_read;
    f->write = filter_snd_write;
    f->seek = filter_snd_seek;
    f->destroy = filter_snd_destroy;
    f->query = filter_snd_query;

    ((struct filter_snd *)f)->sr = sr;
    ((struct filter_snd *)f)->source_tracks_map = source_tracks_map;
    ((struct filter_snd *)f)->target_tracks_map = target_tracks_map;

    return f;
}

void
filter_snd_destroy(struct filter *f) {
}

long long
filter_snd_read(struct filter *f, 
                void *buf[], 
                long long *count) {
    long long r;

    r = snd_getn(((struct filter_snd *)f)->sr,
                 buf,
                 ((struct filter_snd *)f)->source_tracks_map,
                 f->producer_pos,
                 *count);

    if(r <= 0)
        return r;

    f->producer_pos += r;

    *count = r;

    return r;
}

long long
filter_snd_write(struct filter *f, 
                 void *buf[], 
                 int channels,
                 long long *count) {
    long long r;

    r = snd_putn(((struct filter_snd *)f)->sr,
                 buf,
                 channels,
                 ((struct filter_snd *)f)->target_tracks_map,
                 f->producer_pos,
                 *count);
    
    if(r <= 0)
        return r;

    f->producer_pos += r;

    *count = r;

    return r;
}

long long
filter_snd_seek(struct filter *f, 
                long long *count,
                int whence) {
    switch(whence) {
    case SEEK_SET:
        f->producer_pos = *count;
        break;

    case SEEK_CUR:
        f->producer_pos += *count;
        break;

    case SEEK_END:
        f->producer_pos = 
            snd_frame_count(((struct filter_snd *)f)->sr, MAP_ALL) + *count;
        break;

    default:
        return -EINVAL;
    }

    return f->producer_pos;
}

int
filter_snd_query(struct filter *f,
                 int query,
                 void *ptr) {
    int *ip = (int *)ptr;

    switch(query) {
    case FILTER_QUERY_CHANNELS:
        *ip = ((struct filter *)f)->sr->channels;
        break;

    case FILTER_QUERY_FORMAT:
        *ip = ((struct filter *)f)->sr->sample_type;
        break;

    default:
        return -EINVAL;
    }

    return 0;
}
