# plugs/todo.py
#
#

""" manage todo lists .. by user or by channel .. a time/data string can 
 be provided to set time on a todo item.
"""

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import strtotime, striptime, getwho, today, lockdec
from gozerbot.commands import cmnds
from gozerbot.examples import examples
from gozerbot.users import users
from gozerbot.datadir import datadir
from gozerbot.persist import Persist
from gozerbot.plughelp import plughelp
from gozerbot.aliases import aliases
from gozerbot.config import config
from gozerplugs.plugs.alarm import alarms
import time, thread, os

plughelp.add('todo', 'todo lists')

todolock = thread.allocate_lock()
locked = lockdec(todolock)

class Todoitem:

    """ a todo item """

    def __init__(self, name, descr, ttime=None, duration=None, warnsec=None, \
priority=None, num=0):
        self.name = name
        self.time = ttime
        self.duration = duration
        self.warnsec = warnsec
        self.descr = descr
        self.priority = priority
        self.num = num

    def __str__(self):
        return "name: %s num: %d time: %s duration: %s warnsec: %s \
description: %s priority: %s" % (self.name, self.num, \
time.ctime(self.time), self.duration, self.warnsec, self.descr, self.priority)

class Todolist:

    """ a dict faking list of todo items .. index is number """

    def __init__(self):
        self.max = 0
        self.data = {}

    def __len__(self):
        return len(self.data)

    def __getitem__(self, num):
        return self.data[num]

    def __delitem__(self, num):
        del self.data[num]

    def __iter__(self):
        tmplist = self.data.values()
        tmplist.sort(lambda x, y: cmp(x.priority, y.priority), reverse=True)
        return tmplist.__iter__()

    def append(self, item):
        """ add todo item """
        self.max += 1
        item.num = self.max
        self.data[self.max] = item

    def __str__(self):
        return str(self.data)


class Todo(Persist):

    """ Todoos """

    @locked
    def __init__(self, filename):
        Persist.__init__(self, filename)
        if not self.data:
            return
        for key in self.data.keys():
            todoos = self.data[key]
            for (k, v) in todoos.data.items():
                v.num = k
            newd = Todolist()
            for i in todoos:
                newd.append(i)
            self.data[key] = newd

    def size(self):
        """ return number of todo entries """
        return len(self.data)

    def get(self, name):
        """ get todoos of <name> """
        if self.data.has_key(name):
            return self.data[name]

    @locked
    def add(self, name, txt, ttime, warnsec=0):
        """ add a todo """
        name = name.lower()
        if not self.data.has_key(name):
            self.data[name] = Todolist()
        self.data[name].append(Todoitem(name, txt.strip(), ttime, \
warnsec=0-warnsec))
        self.save()
        return len(self.data[name])

    @locked
    def addnosave(self, name, txt, ttime):
        """ add but don't save """
        name = name.lower()
        if not self.data.has_key(name):
            self.data[name] = Todolist()
        self.data[name].append(Todoitem(name, txt, ttime))


    @locked
    def reset(self, name):
        name = name.lower()
        if self.data.has_key(name):
           self.data[name] = Todolist()
        self.save()

    @locked
    def delete(self, name, nr):
        """ delete todo item """
        if not self.data.has_key(name):
            return 0
        todoos = self.data[name]
        try:
            if todoos[nr].warnsec:
                alarmnr = 0 - todoos[nr].warnsec
                if alarmnr > 0:
                    alarms.delete(alarmnr)
            del todoos[nr]
        except KeyError:
            return 0
        self.save()
        return 1

    def toolate(self, name):
        """ show if there are any time related todoos that are too late """
        now = time.time()
        teller = 0
        for i in self.data[name]:
            if i.time < now:
                teller += 1
        return teller

    def timetodo(self, name):
        """ show todoos with time field set """
        result = []
        if not self.data.has_key(name):
            return result
        for i in self.data[name]:
            if i.time:
                result.append(i)
        return result

    def withintime(self, name, time1, time2):
        """ show todoos within time frame """
        result = []
        if not self.data.has_key(name):
            return result
        for i in self.data[name]:
            if i.time >= time1 and i.time < time2:
                result.append(i)
        return result

    @locked
    def setprio(self, who, itemnr, prio):
        """ set priority of todo item """
        try:
            todoitems = self.get(who)
            todoitems[itemnr].priority = prio
            self.save()
            return 1
        except (KeyError, TypeError):
            pass

    def settime(self, who, itemnr, ttime):
        """ set time of todo item """
        try:
            todoitems = self.get(who)
            todoitems[itemnr].time = ttime
            self.save()
            return 1
        except (KeyError, TypeError):
            pass

class TodoDb(object):

    """ database todo interface """

    def reset(self, name): 
        result = db.execute(""" DELETE FROM todo WHERE name == %s """, name)
        return result

    def size(self):
        """ return number of todo's """
        result = db.execute(""" SELECT COUNT(*) FROM todo """)
        return result[0][0]
        
    def get(self, name):
        """ get todo list of <name> """
        name = name.lower()
        result = db.execute(""" SELECT * FROM todo WHERE name = %s ORDER BY \
priority DESC, indx ASC """, name)
        res = []
        if result:
            for i in result:
                args = [i[1],i[5], i[2], i[3], i[4], i[6], i[0]]
                res.append(Todoitem(*args))
        return res

    def getid(self, idnr):
        """ get todo data of <idnr> """
        result = db.execute(""" SELECT * FROM todo WHERE indx = %s """, idnr)
        res = []
        if result:
            for i in result:
                args = [i[1],i[5], i[2], i[3], i[4], i[6], i[0]]
                res.append(Todoitem(*args))
        return res
        return result

    def setprio(self, who, todonr, prio):
        """ set priority of todonr """
        result = db.execute(""" UPDATE todo SET priority = %s WHERE indx \
= %s """, (prio, todonr))
        return result

    def getprio(self, todonr):
        """ get priority of todonr """
        result = db.execute(""" SELECT name, priority FROM todo WHERE \
indx = %s """, todonr)
        return result

    def getwarnsec(self, todonr):
        """ get priority of todonr """
        result = db.execute(""" SELECT warnsec FROM todo WHERE \
indx = %s """, todonr)
        return result

    def settime(self, who, todonr, ttime):
        """ set time of todonr """
        result = db.execute(""" UPDATE todo SET time = %s WHERE indx \
= %s """, (ttime, todonr))
        return result

    def add(self, name, txt, ttime, alarmnr=None):
        """ add a todo """
        name = name.lower()
        txt = txt.strip()
        if ttime:
            if not alarmnr:
                result = db.execute(""" INSERT INTO todo(name, time, \
descr) VALUES (%s, %s, %s) """, (name, ttime, txt))
            else:
                result = db.execute(""" INSERT INTO todo(name, time, \
descr, warnsec) VALUES (%s, %s, %s, %s) """, (name, ttime, txt, 0-alarmnr))

        else:
            result = db.execute(""" INSERT INTO todo(name, descr) \
VALUES (%s, %s) """, (name, txt))
        return result

    def delete(self, name, indexnr):
        """ delete todo item """
        name = name.lower()
        try:
            warnsec = self.getwarnsec(indexnr)[0][0]
            if warnsec:
                alarmnr = 0 - warnsec
                if alarmnr > 0:
                    alarms.delete(alarmnr)
        except (IndexError, TypeError):
            pass
        result = db.execute(""" DELETE FROM todo WHERE name = %s AND \
indx = %s """, (name, indexnr))
        return result

    def toolate(self, name):
        """ show if there are any time related todoos that are too late """
        name = name.lower()
        now = time.time()
        result = db.execute(""" SELECT * FROM todo WHERE name = %s AND \
time < %s """, (name, now))
        res = []
        if result:
            for i in result:
                args = [i[1],i[5], i[2], i[3], i[4], i[6], i[0]]
                res.append(Todoitem(*args))
        return res

    def withintime(self, name, time1, time2):
        """ show todo list within time frame """
        name = name.lower()
        result = db.execute(""" SELECT * FROM todo WHERE name = %s AND \
time > %s AND time < %s """, (name, time1, time2))
        res = []
        if result:
            for i in result:
                args = [i[1],i[5], i[2], i[3], i[4], i[6], i[0]]
                res.append(Todoitem(*args))
        return res

    def timetodo(self, name):
        name = name.lower()
        now = time.time()
        result = db.execute(""" SELECT * FROM todo WHERE time AND name = %s \
""", name)
        res = []
        if result:
            for i in result:
                args = [i[1],i[5], i[2], i[3], i[4], i[6], i[0]]
                res.append(Todoitem(*args))
        return res

    def reset(self, name):
        """ reset todo items of user with <name> """
        name = name.lower()
        result = db.execute(""" DELETE FROM todo WHERE name = %s """, (name, ))
        return result

if config['dbenable']:
    from gozerbot.db import db
    todo = TodoDb()
else:
    todo = Todo(datadir + os.sep + 'todo')
    if not todo.data:
        todo.data = {}

def size():
    """ return number of todo entries """
    return todo.size()

def handle_todo(bot, ievent):
    """ todo [<item>] .. show todo's or set todo item .. a time/date can be \
given"""
    if len(ievent.args) > 0:
        handle_todo2(bot, ievent)
        return
    name = users.getname(ievent.userhost)
    try:
        todoos = todo.get(name)
    except KeyError:
        ievent.reply('i dont have todo info for %s' % user.name)
        return
    saytodo(bot, ievent, todoos)

def handle_todo2(bot, ievent):
    """ set todo item """
    if not ievent.rest:
        ievent.missing("<what>")
        return
    else:
        what = ievent.rest
    name = users.getname(ievent.userhost)
    ttime = strtotime(what)
    nr = 0
    if not ttime  == None:
        ievent.reply('time detected ' + time.ctime(ttime))
        what = striptime(what)
        alarmnr = alarms.add(bot.name, ievent.nick, ttime, what)
        nr = todo.add(name, what, ttime, alarmnr)
    else:
        nr = todo.add(name, what, None)
    ievent.reply('todo item %s added' % nr)

cmnds.add('todo', handle_todo, 'USER')
examples.add('todo', 'todo [<item>] .. show todo items or add a todo item', \
'1) todo 2) todo program the bot 3) todo 22:00 sleep')

def handle_tododone(bot, ievent):
    """ todo-done <listofnrs> .. remove todo items """
    if len(ievent.args) == 0:
        ievent.missing('<list of nrs>')
        return
    try:
        nrs = []
        for i in ievent.args:
            nrs.append(int(i))
    except ValueError:
        ievent.reply('%s is not an integer' % i)
        return
    name = users.getname(ievent.userhost)
    nrdone = 0
    for i in nrs:
        nrdone += todo.delete(name, i)
    if nrdone == 1:
        ievent.reply('%s item deleted' % nrdone)
    elif nrdone == 0:
        ievent.reply('no items deleted')
    else:
        ievent.reply('%s items deleted' % nrdone)

cmnds.add('todo-done', handle_tododone, 'USER')
examples.add('todo-done', 'todo-done <listofnrs> .. remove items from \
todo list', '1) todo-done 1 2) todo-done 3 5 8')
aliases.data['done'] = 'todo-done'

def handle_chantodo(bot, ievent):
    """ todo-chan [<item>] .. show channel todo's or set todo item for \
channel"""
    if ievent.rest:
        handle_chantodo2(bot, ievent)
        return
    todoos = todo.get(ievent.channel)
    saytodo(bot, ievent, todoos)

def handle_chantodo2(bot, ievent):
    """ set todo item for channel"""
    what = ievent.rest
    ttime = strtotime(what)
    nr = 0
    if not ttime  == None:
        ievent.reply('time detected ' + time.ctime(ttime))
        result = '(%s) ' % ievent.nick + striptime(what)
        alarmnr = alarms.add(bot.name, ievent.channel, ttime, result)
        nr = todo.add(ievent.channel, result, ttime, alarmnr)
    else:
        result = '(%s) ' % ievent.nick + what
        nr = todo.add(ievent.channel, result, None)
    ievent.reply('todo item %s added' % nr)

cmnds.add('todo-chan', handle_chantodo, 'USER')
examples.add('todo-chan', 'todo-chan [<item>] .. add channel todo', \
'todo-chan fix bla')
aliases.data['chantodo'] = 'todo-chan'

def handle_todochandone(bot, ievent):
    """ todo-chandone <listofnrs> .. remove channel todo item """
    if not ievent.rest:
        ievent.missing('<list of nrs>')
        return
    data = ievent.rest.split()
    try:
        nrs = []
        for i in data:
            nrs.append(int(i))
    except ValueError:
        ievent.reply('%s is not an integer' % i)
        return
    nrdone = 0
    for i in nrs:
        nrdone += todo.delete(ievent.channel, i)
    if nrdone == 1:
        ievent.reply('%s item deleted' % nrdone)
    elif nrdone == 0:
        ievent.reply('no items deleted')
    else:
        ievent.reply('%s items deleted' % nrdone)

cmnds.add('todo-chandone', handle_todochandone, 'USER')
examples.add('todo-chandone', 'todo-chandone <listofnrs> .. remove item \
from channel todo list', 'todo-chandone 2')
aliases.data['chandone'] = 'todo-chandone'

def handle_settodo(bot, ievent):
    """ todo-set <name> <txt> .. add a todo to another user's todo list"""
    try:
        who = ievent.args[0]
        what = ' '.join(ievent.args[1:])
    except IndexError:
        ievent.missing('<nick> <what>')
        return
    if not what:
        ievent.missing('<nick> <what>')
        return
    userhost = getwho(bot, who)
    if not userhost:
        ievent.reply("can't find userhost for %s" % who)
        return
    whouser = users.getname(userhost)
    if not whouser:
        ievent.reply("can't find user for %s" % userhost)
        return
    name = users.getname(ievent.userhost)
    if not users.permitted(userhost, name, 'todo'):
        ievent.reply("%s doesn't permit todo sharing for %s " % \
(who, name))
        return
    what = "%s: %s" % (ievent.nick, what)
    ttime = strtotime(what)
    nr = 0
    if not ttime  == None:
        ievent.reply('time detected ' + time.ctime(ttime))
        what = striptime(what)
        alarmnr = alarms.add(bot.name, who, ttime, what)
        nr = todo.add(whouser, what, ttime, alarmnr)
    else:
        nr = todo.add(whouser, what, None)
    ievent.reply('todo item %s added' % nr)

cmnds.add('todo-set', handle_settodo, 'USER')
examples.add('todo-set', 'todo-set <nick> <txt> .. set todo item of \
<nick>', 'todo-set dunker bot proggen')

def handle_gettodo(bot, ievent):
    """ todo-get <nick> .. get todo of another user """
    try:
        who = ievent.args[0]
    except IndexError:
        ievent.missing('<nick>')
        return
    userhost = getwho(bot, who)
    if not userhost:
        ievent.reply("can't find userhost for %s" % who)
        return
    whouser = users.getname(userhost)
    if not whouser:
        ievent.reply("can't find user for %s" % userhost)
        return
    name = users.getname(ievent.userhost)
    if not users.permitted(userhost, name, 'todo'):
        ievent.reply("%s doesn't permit todo sharing for %s " % (who, name))
        return
    todoos = todo.get(whouser)
    saytodo(bot, ievent, todoos)

cmnds.add('todo-get', handle_gettodo, ['USER', 'WEB'])
examples.add('todo-get', 'todo-get <nick> .. get the todo list of \
<nick>', 'todo-get dunker')

def handle_todotime(bot, ievent):
    """ todo-time .. show time related todoos """
    name = users.getname(ievent.userhost)
    todoos = todo.timetodo(name)
    saytodo(bot, ievent, todoos)

cmnds.add('todo-time', handle_todotime, 'USER')
examples.add('todo-time', 'todo-time .. show todo items with time fields', \
'todo-time')
aliases.data['tt'] = 'todo-time'

def handle_todoweek(bot, ievent):
    """ todo-week .. show time related todo items for this week """
    name = users.getname(ievent.userhost)
    todoos = todo.withintime(name, today(), today()+7*24*60*60)
    saytodo(bot, ievent, todoos)

cmnds.add('todo-week', handle_todoweek, 'USER')
examples.add('todo-week', 'todo-week .. todo items for this week', 'todo-week')

def handle_today(bot, ievent):
    """ todo-today .. show time related todo items for today """
    name = users.getname(ievent.userhost)
    todoos = todo.withintime(name, today(), today()+24*60*60)
    saytodo(bot, ievent, todoos)

cmnds.add('todo-today', handle_today, 'USER')
examples.add('todo-today', 'todo-today .. todo items for today', 'todo-today')
aliases.data['today'] = 'todo-today'

def handle_tomorrow(bot, ievent):
    """ todo-tomorrow .. show time related todo items for tomorrow """
    username = users.getname(ievent.userhost)
    if ievent.rest:
        what = ievent.rest
        ttime = strtotime(what)
        if ttime != None:
            if ttime < today() or ttime > today() + 24*60*60:
                ievent.reply("%s is not tomorrow" % \
time.ctime(ttime + 24*60*60))
                return
            ttime += 24*60*60
            ievent.reply('time detected ' + time.ctime(ttime))
            what = striptime(what)
        else:
            ttime = today() + 42*60*60
        todo.add(username, what, ttime)   
        ievent.reply('todo added')    
        return
    todoos = todo.withintime(username, today()+24*60*60, today()+2*24*60*60)
    saytodo(bot, ievent, todoos)

cmnds.add('todo-tomorrow', handle_tomorrow, 'USER')
examples.add('todo-tomorrow', 'todo-tomorrow .. todo items for tomorrow', \
'todo-tomorrow')
aliases.data['tomorrow'] = 'todo-tomorrow'

def handle_setpriority(bot, ievent):
    """ todo-setprio [<channel|name>] <itemnr> <prio> .. show priority \
        on todo item """
    try:
        (who, itemnr, prio) = ievent.args
    except ValueError:
        try:
            (itemnr, prio) = ievent.args
            who = users.getname(ievent.userhost)
        except ValueError:
            ievent.missing('[<channe|namel>] <itemnr> <priority>')
            return
    try:
        itemnr = int(itemnr)
        prio = int(prio)
    except ValueError:
        ievent.missing('[<channel|name>] <itemnr> <priority>')
        return
    who = who.lower()
    if not todo.setprio(who, itemnr, prio):
        ievent.reply('no todo %s found for %s' % (itemnr, who))
        return
    ievent.reply('priority set')

cmnds.add('todo-setprio', handle_setpriority, 'USER')
examples.add('todo-setprio', 'todo-setprio [<channel|name>] <itemnr> <prio> \
.. set todo priority', '1) todo-setprio #dunkbots 2 5 2) todo-setprio owner \
3 10 3) todo-setprio 2 10')
aliases.data['setprio'] = 'todo-setprio'

def handle_todosettime(bot, ievent):
    """ todo-settime [<channel|name>] <itemnr> <timestring> .. set time \
        on todo item """
    ttime = strtotime(ievent.txt)
    if ttime == None:
        ievent.reply("can't detect time")
        return   
    txt = striptime(ievent.txt)
    try:
        (who, itemnr) = txt.split()
    except ValueError:
        try:
            (itemnr, ) = txt.split()
            who = users.getname(ievent.userhost)
        except ValueError:
            ievent.missing('[<channe|namel>] <itemnr> <timestring>')
            return
    try:
        itemnr = int(itemnr)
    except ValueError:
        ievent.missing('[<channel|name>] <itemnr> <timestring>')
        return
    who = who.lower()
    if not todo.settime(who, itemnr, ttime):
        ievent.reply('no todo %s found for %s' % (itemnr, who))
        return
    ievent.reply('time of todo %s set to %s' % (itemnr, time.ctime(ttime)))

cmnds.add('todo-settime', handle_todosettime, 'USER')
examples.add('todo-settime', 'todo-settime [<channel|name>] <itemnr> \
<timestring> .. set todo time', '1) todo-settime #dunkbots 2 13:00 2) \
todo-settime owner 3 2-2-2010 3) todo-settime 2 22:00')

def handle_getpriority(bot, ievent):
    """ todo-getprio <[channel|name]> <itemnr> .. get priority of todo \
        item """
    try:
        (who, itemnr) = ievent.args
    except ValueError:
        try:
            itemnr = ievent.args[0]
            who = users.getname(ievent.userhost)
        except IndexError:
            ievent.missing('[<channel|name>] <itemnr>')
            return
    try:
        itemnr = int(itemnr)
    except ValueError:
        ievent.missing('[<channel|name>] <itemnr>')
        return
    who = who.lower()
    todoitems = todo.get(who)
    if not todoitems:
        ievent.reply('no todoitems known for %s' % who)
        return
    try:
        prio = todoitems[itemnr].priority
    except KeyError:
        ievent.reply('no todo item %s known for %s' % (itemnr, who))
        return
    ievent.reply('priority is %s' % prio)

cmnds.add('todo-getprio', handle_getpriority, 'USER')
examples.add('todo-getprio', 'todo-getprio [<channel|name>] <itemnr> .. get \
todo priority', '1) todo-getprio #dunkbots 5 2) todo-getprio 3')
aliases.data['prio'] = 'todo-getprio'

def saytodo(bot, ievent, todoos):
    """ output todo items of <name> """
    result = []
    now = time.time()
    if not todoos:
        ievent.reply('nothing todo ;]')
        return
    for i in todoos:
        res = ""
        res += "%s) " % i.num
        if i.time:
            if i.time < now:
                res += 'TOO LATE: '
            res += "%s %s " % (time.ctime(i.time), i.descr)
        else:
            res += "%s " % i.descr
        if i.priority:
            res += "[%+d] " % i.priority
        result.append(res.strip())
    if result:
        ievent.reply("todolist of %s: " % ievent.nick, result, nritems=True)
