/* 
   Copyright (C) 1999 Kyle R. Burton, All Rights Reserved
   mortis@voicenet.com
   http://www.bgw.org
   http://www.voicenet.com/~mortis

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <config.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <errno.h>
#include <locale.h>

#include <gperiodic.h>
#include <gparser.h>

#include "table_data.c"

#define cvs_version   "$Header: /home/mortis/cvs/gperiodic/src/gperiodic.c,v 1.3 2001/08/28 02:12:50 mortis Exp $";
#define cvs_revision  "$Revision: 1.3 $";
#define cvs_date      "$Date: 2001/08/28 02:12:50 $";
#define cvs_author    "$Author: mortis $";


char *optstr            = "hsd:cDPivC:";   /* cmdline option string... */
char *bin;                                 /* get name from argv... */
gint gp_sound_enabled   = 0;
char *sound_basedir     = "/usr/share/gperiodic";
char *dsp_device        = "/dev/dsp";
char *lang;
int  enable_colorscale  = 0;
int  debug = 0;

/*
 * exit hanlder for the main window -- exits the application
 *
 */
void main_window_exit( GtkWidget *w, gpointer d ) 
{
  gtk_main_quit();
}

void dialog_destroy( GtkWidget *w, gpointer d )
{
  gtk_grab_remove(GTK_WIDGET(w));
}

void play_pronounciation( GtkWidget *w, gpointer d )
{
  FILE *sound = NULL, *dsp = NULL;
  char *sound_file;
  int error = 0;

  struct table_entry *entry = (struct table_entry *)d;
  sound_file = g_strdup_printf("%s/%s/%s.wav", sound_basedir, lang, entry->info[NAME]);

  g_message("%s(%d) trying to play sound for: %s : %s",__FILE__,__LINE__,
    entry->info[NAME], sound_file);

  sound = fopen(sound_file, "rb");
  if( NULL == sound ) ++error;

  if( !error ) {
    dsp   = fopen(dsp_device,"w");
  }

  if( NULL == dsp ) ++error;

  if(!error) {
    char ch;
    while(!feof(sound)) {
      fread(&ch,1,1,sound);
      fwrite(&ch,1,1,dsp);
    }
  }

  if( NULL != sound )     fclose(sound);
  if( NULL != dsp )       fclose(dsp);
  if( NULL != sound_file) free(sound_file);
}

void dialog_close( GtkWidget *w, gpointer d )
{
  gtk_widget_destroy(GTK_WIDGET(d));
}

void display_element_dialog( GtkWidget *w, gpointer d )
{
  struct table_entry *entry = (struct table_entry *)d;
  GtkWidget *dialog, *vbox, *tbl, *ok_button, *hbox_space, *hbox_ok,
    *sound_button, *hbox_sound;
  int col, x1, x2, y1, y2;
  char *buff;

  _DBG(g_message("%s(%d) display_table_element() w=0x%08x d=0x%08x",
     __FILE__, __LINE__, w, d ) );

  /* display this in a dialog box... */
  dialog = gtk_dialog_new();
  gtk_signal_connect(GTK_OBJECT(dialog),"destroy",
         GTK_SIGNAL_FUNC(dialog_destroy), dialog );
   
  gtk_window_set_title(GTK_WINDOW(dialog), _(entry->info[NAME]));

  gtk_container_set_border_width( GTK_CONTAINER(dialog), 5 );

  vbox = gtk_vbox_new(FALSE,0);

  gtk_box_pack_start(GTK_BOX( GTK_DIALOG(dialog)->action_area),
         vbox, TRUE, TRUE, 0 );
  gtk_widget_show(vbox);

  tbl = gtk_table_new(MAX_INFO_NR, 2,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox), tbl, FALSE, FALSE, 0 );
  gtk_widget_show(tbl);

  /* add the labels to the dialog box verticaly... */
  for (col = 0; col < 2; col++) {
    int row;

    for (row = 0; row < MAX_INFO_NR; row++) {
      GtkWidget *label;

      if (col == 0)
        label = gtk_label_new(_(header.info[row]));
      else if (row == 0)
        label = gtk_label_new(_(entry->info[row]));
      else
        label = gtk_label_new(entry->info[row]);

      gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
      gtk_table_attach(GTK_TABLE(tbl), label,
           col, col + 1, row, row + 1,
           GTK_FILL, GTK_FILL, 0, 0);
      gtk_widget_show(label);
    }
  }


  ok_button = gtk_button_new_with_label(_("Ok"));
  gtk_signal_connect( GTK_OBJECT(ok_button), "clicked",
          GTK_SIGNAL_FUNC(dialog_close), dialog );

  GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);

  hbox_space = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(hbox_space),5);
  gtk_box_pack_start( GTK_BOX(vbox), hbox_space, TRUE, TRUE, 0 );
  gtk_widget_show(hbox_space);

  hbox_ok = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox_ok), ok_button, TRUE, TRUE, 0 );
  gtk_box_pack_end(GTK_BOX(vbox), hbox_ok, TRUE, TRUE, 0 );
  gtk_widget_show(hbox_ok);

  gtk_widget_grab_default(ok_button);

  gtk_widget_show(ok_button);

  /* this is some of the experimental sound stuff... */
  if( gp_sound_enabled ) {
    sound_button = gtk_button_new_with_label(_("Play Pronounciation"));
  
    gtk_signal_connect( GTK_OBJECT(sound_button), "clicked",
      GTK_SIGNAL_FUNC(play_pronounciation), entry );
  
    hbox_sound = gtk_hbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(hbox_sound),5);
    gtk_box_pack_start(GTK_BOX(hbox_sound), sound_button, TRUE, TRUE, 0 );
    gtk_box_pack_start(GTK_BOX(vbox), hbox_sound, TRUE, TRUE, 0 );
  
    gtk_widget_show(hbox_sound);

    gtk_widget_show(sound_button);
  }


  gtk_grab_add(dialog);
  gtk_widget_show(dialog);
}


void menu_file_exit(gpointer callback_data, guint callback_action, GtkWidget *w )
{
  main_window_exit(w,NULL);
}

void menu_help_about( gpointer callback_data, guint callback_action, GtkWidget *widget )
{
  GtkWidget *ok_button    = NULL;
  GtkWidget *about_dialog = NULL;
  GtkWidget *label        = NULL;
  gchar *buff;
  char *authors = "mortis@voicenet.com,\n"
    "a.w.peters@ieee.org,\n"
    "khazuada@univ-littoral.fr,\n"
    "mrivera@cable.net.co,\n"
    "chris@koeln.ccc.de,\n"
    "lalo@linuxcenter.com.br\n"
    "koen_kooi@hotmail.com\n"
    "antigone34@libero.it";
  char *info = _("GPeriodic\n\n"
     "Version: " VERSION "\n\n"
     "Authors: %s\n\n"
     "This program shows the periodic table of elements.");


  about_dialog = gtk_dialog_new();

  gtk_signal_connect(GTK_OBJECT(about_dialog), "destroy", 
    GTK_SIGNAL_FUNC(dialog_destroy),
    about_dialog );

  gtk_window_set_title(GTK_WINDOW(about_dialog), _("GPeriodic - About"));
  gtk_container_set_border_width( GTK_CONTAINER(about_dialog), 5 );

  buff = g_strdup_printf(_(info), authors);
  label = gtk_label_new(buff);
  free(buff);

  gtk_misc_set_padding( GTK_MISC(label), 10, 10 );

  gtk_box_pack_start( GTK_BOX(GTK_DIALOG(about_dialog)->vbox),
    label, TRUE, TRUE, 0 );

  gtk_widget_show(label);


  ok_button = gtk_button_new_with_label("Ok");

  gtk_signal_connect( GTK_OBJECT(ok_button), "clicked", 
    GTK_SIGNAL_FUNC(dialog_close), about_dialog );

  GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);

  gtk_box_pack_start( GTK_BOX( GTK_DIALOG(about_dialog)->action_area ),
    ok_button, TRUE, TRUE, 0 );

  gtk_widget_grab_default(ok_button);

  gtk_widget_show(ok_button);

  gtk_widget_show(about_dialog);

}

void show_usage( void )
{
  printf("%s [opts]\n"
  "Options are:\n"
  "    -h             show this help screen.\n"
  "    -c             enable colorscale\n"
  "    -s             enable element name pronounciation.\n"
  "    -d path        specify the path to the pronounciation files.\n"
  "    -D             dump the periodic table [cvs format].\n"
  "    -i             enter interactive mode.\n"
  "    -v             be verbose\n"
  "    -C commands    execute comands as if entered in interactive mode\n"
  "                   and exit\n"
  "\n",
    bin
  );
}

int criterion2scale(double r, int maxcolor)
{
  int result;
  if(r < 0) return 0;
  result = 1 + (maxcolor-1) * r;
  if (result >= maxcolor) result = maxcolor - 1;
  return result;
}

void colorize_element(
  int i,
  struct colorscale * cs1, criterion c1,
  struct colorscale * cs2, criterion c2,
  struct colorscale * cs3, criterion c3)
{
  int n, c;
  double r;
  GdkColor col, col_;

  c = cs1->maxcolor;
  r = c1(table+i);
  n = criterion2scale(r,c);
  printf("<E9>l<E9>ment %s fluide = %f, %d ",table[i].info[SYMBOL],r,n);
  col = cs1->scale[n];

  if(cs2) {
    c = cs2->maxcolor;
    r = c2(table+i);
    n = criterion2scale(r,c);
    g_message("gaz = %f, %d",r,n);
    col_ = cs2->scale[n];
    col.red += col_.red;
    col.green += col_.green;
    col.blue += col_.blue;
    if(cs3) {
      c = cs3->maxcolor;
      r = c3(table+i);
      n = criterion2scale(r,c);
      col_ = cs3->scale[n];
      col.red += col_.red;
      col.green += col_.green;
      col.blue += col_.blue;
    }
  }

  table[i].style->bg[0].red = col.red;
  table[i].style->bg[0].green = col.green;
  table[i].style->bg[0].blue = col.blue;

  gtk_widget_set_style( table[i].button, table[i].style );
}


void main_prog(void)
{
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *periodic_table;
  GtkAccelGroup *accel_group;
  GtkItemFactory *item_factory;
  int menu_num_items;
  GdkColor tooltip_color_bg, tooltip_color_fg;
  int red, gree, blue;
  char *buff;
  int i;

  /* create the menu using a menufactory... */
  GtkItemFactoryEntry menu[] = {
    {_("/_File"), NULL, 0, 0, "<Branch>"},
    {_("/_File/_Quit"), NULL,
     (GtkItemFactoryCallback)menu_file_exit, 0, "<Item>"},
    {_("/_Help"), NULL, 0, 0, "<Branch>"},
    {_("/_Help/_About ..."), NULL,
     (GtkItemFactoryCallback)menu_help_about, 0, "<Item>"},
  };

  /* create the main window */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window),"GPeriodic");
  gtk_container_border_width(GTK_CONTAINER(window),0);

  /* connect the close event... */
  gtk_signal_connect(GTK_OBJECT(window), "delete-event",
         GTK_SIGNAL_FUNC(main_window_exit), &window );

  /* use a vbox for the menubar and the table of elements... */
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_border_width(GTK_CONTAINER(vbox),2);
  gtk_container_add( GTK_CONTAINER(window), vbox );
  gtk_widget_show(vbox);

  menu_num_items = sizeof(menu) / sizeof(menu[0]);

  /* accelerator group for the menu items/acitons */
  accel_group = gtk_accel_group_new();
  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<blah>",
              accel_group);
  gtk_item_factory_create_items( item_factory, menu_num_items, menu, NULL );
  gtk_accel_group_attach(accel_group, GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(vbox),
    gtk_item_factory_get_widget(item_factory, "<blah>"),
    FALSE, FALSE, 0);

  /* create the table widget to hold the periodic table */
  periodic_table = gtk_table_new(1,1,TRUE);
  gtk_box_pack_end( GTK_BOX(vbox), periodic_table, FALSE, FALSE, 0 );
  gtk_widget_show(periodic_table);

  /* now for each element in the table of elements, create a display */
  /* item for it, and add it to the table... */
  for(i = 0; i < sizeof(table); ++i ) {
    if( !table[i].info[NAME] ) {
      /* stop if no data... */
      _DBG(g_message("%s(%d) %s : %d",__FILE__,__LINE__,
         _("No symbol for index"), i));
      break;
    }

    /* create the button */
    table[i].button = gtk_button_new_with_label(_(table[i].info[SYMBOL]));
    if( !table[i].button ) {
      g_warning("%s(%d) %s", __FILE__, __LINE__, _("Error button was null?"));
      break;
    }

/******************************************************************************/
    {
      int j;
      GtkStyle *button_style = gtk_widget_get_style(window);

      table[i].style = gtk_style_copy(button_style);

      _DBG( g_message("%s(%d) Setting color:  %d %d %d",__FILE__,__LINE__,
        table[i].color.red,
        table[i].color.green,
        table[i].color.blue ) );

      
      if( enable_colorscale ) {
        colorize_element(i,
          &solid_fluid_cs, solid_fluid_crid,
          &condensed_gas_cs, condensed_gas_crit,
          0, 0
        );
      }
      else {
        /* set for 0, 1, and 2 */
        table[i].style->bg[0].red   = table[i].color.red;
        table[i].style->bg[0].green = table[i].color.green;
        table[i].style->bg[0].blue  = table[i].color.blue;
      }

      table[i].style->bg[1].red   = table[i].color.red;
      table[i].style->bg[1].green = table[i].color.green;
      table[i].style->bg[1].blue  = table[i].color.blue;

      table[i].style->bg[2].red   = table[i].color.red;
      table[i].style->bg[2].green = table[i].color.green;
      table[i].style->bg[2].blue  = table[i].color.blue;

      gtk_widget_set_style( table[i].button, table[i].style );
    }
/******************************************************************************/

    /* connect the destroy method to it */
    gtk_signal_connect(GTK_OBJECT(table[i].button), "clicked",
           GTK_SIGNAL_FUNC(display_element_dialog), &table[i] );

    /* set up a string for the tooltips */
    buff = g_strdup_printf(_("%s  n:%s w:%s"), _(table[i].info[NAME]),
       table[i].info[NUMBER], table[i].info[WEIGHT]);

    /* create a new tooltips object... */
    table[i].tooltip = gtk_tooltips_new();
    gtk_tooltips_set_delay(table[i].tooltip,100);
    gtk_tooltips_set_tip(table[i].tooltip,table[i].button,buff,NULL);
    gtk_tooltips_set_colors( table[i].tooltip, &tooltip_color_bg, &tooltip_color_fg );

    /* attach the button to the table */
    gtk_table_attach(GTK_TABLE(periodic_table), table[i].button,
         table[i].x - 1, table[i].x,
         table[i].y - 1, table[i].y,
         GTK_FILL, GTK_FILL,              /* opts */
         2, 2);                           /* ypad */

    /* display it */
    gtk_widget_show(table[i].button);
  }

  /* show the main winodw */
  gtk_widget_show(window);

  /* show all the widgets (table/menubar/buttons/etc...) in case */
  /* some haven't gotten shown already */
  gtk_widget_show_all(window);

  /* give up controll till they exit the application */
  gtk_main();

  free(buff);
}

/* 
 * this is where it all starts...
 *
 */
int main(int argc, char** argv)
{
  char ch;

  textdomain (PACKAGE);
  bindtextdomain(PACKAGE, GPERIODIC_LOCALE_DIR);

  lang = getenv("LANG");
  if( NULL == lang ) {
    lang = "en";
  }

  gtk_init(&argc,&argv);

  bin = *argv;

  /* parse the command line with getopt(3) */
  while( -1 != (ch=getopt(argc,argv,optstr)) ) {
    switch(ch) {
      case 'c':
        enable_colorscale = 1;
        break;
      case 'd':
        sound_basedir = strdup(optarg);
        break;
      case 's':
        gp_sound_enabled = 1;
        break;
      case 'h':
        show_usage();
        exit(0);
        break;
      case 'D':
        dump_table();
        exit(0);
        break;
      case 'P':
        parse_table();
        exit(0);
        break;
      case 'i':
        interactive_shell();
        exit(0);
        break;
      case 'v':
        debug = 1;
        break;
      case 'C':
        gpparser_parse_line(optarg,0);
        return 0;
        break;
      default:
        break;
    }
  }

  if(optind < argc) {
    while(optind < argc) {
      int atomicNumber = atoi(argv[optind]);
      if(!atomicNumber) { 
        atomicNumber = gpdata_get_atomic_number_for_symbol(argv[optind]);
      }
      gpparse_print_element_data_for_num(atomicNumber);
      ++optind;
    }
    return 0;
  }

  main_prog();

  /* exit from the application... */
  return 0;
}

