/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvastext.c - text item.
 */

/**
 * SECTION:goocanvastext
 * @Title: GooCanvasText
 * @Short_Description: a text item.
 *
 * GooCanvasText represents a text item.
 *
 * It is a subclass of #GooCanvasItemSimple and so inherits all of the style
 * properties such as "stroke-color", "fill-color" and "line-width".
 *
 * It also implements the #GooCanvasItem interface, so you can use the
 * #GooCanvasItem functions such as goo_canvas_item_raise() and
 * goo_canvas_item_rotate().
 *
 * To create a #GooCanvasText use goo_canvas_text_new().
 *
 * To get or set the properties of an existing #GooCanvasText, use
 * g_object_get() and g_object_set().
 *
 * To respond to events such as mouse clicks on the text you must connect
 * to the signal handlers of the corresponding #GooCanvasTextView objects.
 * (See goo_canvas_view_get_item_view() and #GooCanvasView::item-view-created.)
 */
#include <config.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include "goocanvastext.h"
#include "goocanvastextview.h"


enum {
  PROP_0,

  PROP_X,
  PROP_Y,
  PROP_WIDTH,
  PROP_TEXT,
  PROP_USE_MARKUP,
  PROP_FONT,
  PROP_FONT_DESC,
  PROP_ANCHOR,
  PROP_ALIGN
};

static void goo_canvas_text_finalize (GObject *object);
static void item_interface_init (GooCanvasItemIface *iface);
static void goo_canvas_text_get_property (GObject            *object,
					  guint               param_id,
					  GValue             *value,
					  GParamSpec         *pspec);
static void goo_canvas_text_set_property (GObject            *object,
					  guint               param_id,
					  const GValue       *value,
					  GParamSpec         *pspec);

G_DEFINE_TYPE_WITH_CODE (GooCanvasText, goo_canvas_text,
			 GOO_TYPE_CANVAS_ITEM_SIMPLE,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_ITEM,
						item_interface_init))


static void
goo_canvas_text_class_init (GooCanvasTextClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass*) klass;

  gobject_class->finalize = goo_canvas_text_finalize;

  gobject_class->get_property = goo_canvas_text_get_property;
  gobject_class->set_property = goo_canvas_text_set_property;

  /* Text */
  g_object_class_install_property (gobject_class, PROP_TEXT,
				   g_param_spec_string ("text",
							_("Text"),
							_("The text to display"),
							NULL,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_USE_MARKUP,
				   g_param_spec_boolean ("use-markup",
							 _("Use Markup"),
							 _("Whether to parse PangoMarkup in the text, to support different styles"),
							 FALSE,
							 G_PARAM_READWRITE));

  /* Position */
  g_object_class_install_property (gobject_class, PROP_X,
				   g_param_spec_double ("x",
							"X",
							_("The x coordinate of the text"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_Y,
				   g_param_spec_double ("y",
							"Y",
							_("The y coordinate of the text"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_WIDTH,
				   g_param_spec_double ("width",
							_("Width"),
							_("The width to use to layout the text"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, -1.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_ANCHOR,
				   g_param_spec_enum ("anchor",
						      _("Anchor"),
						      _("How to position the text relative to the given x and y coordinates"),
						      GTK_TYPE_ANCHOR_TYPE,
						      GTK_ANCHOR_NW,
						      G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_ALIGN,
				   g_param_spec_enum ("alignment",
						      _("Alignment"),
						      _("How to align the text"),
						      PANGO_TYPE_ALIGNMENT,
						      PANGO_ALIGN_LEFT,
						      G_PARAM_READWRITE));

  /* Font */
  g_object_class_install_property (gobject_class, PROP_FONT,
				   g_param_spec_string ("font",
							_("Font"),
							_("The base font to use for the text"),
							NULL,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_FONT_DESC,
				   g_param_spec_boxed ("font-desc",
						       _("Font Description"),
						       _("The attributes specifying which font to use"),
						       PANGO_TYPE_FONT_DESCRIPTION,
						       G_PARAM_READWRITE));
}


static void
goo_canvas_text_init (GooCanvasText *text)
{
  text->width = -1.0;
  text->anchor = GTK_ANCHOR_NW;
}


/**
 * goo_canvas_text_new:
 * @parent: the parent item, or %NULL. If a parent is specified, it will assume
 *  ownership of the item, and the item will automatically be freed when it is
 *  removed from the parent. Otherwise call g_object_unref() to free it.
 * @string: the text to display.
 * @x: the x coordinate of the text.
 * @y: the y coordinate of the text.
 * @width: the width of the text item, or -1 for unlimited width.
 * @anchor: the position of the text relative to the given @x and @y
 *  coordinates. For example an anchor of %GDK_ANCHOR_NW will result in the
 *  top-left of the text being placed at the given @x and @y coordinates.
 *  An anchor of %GDK_ANCHOR_CENTER will result in the center of the text being
 *  placed at the @x and @y coordinates.
 * @first_property: the name of the first property to set, or %NULL.
 * @...: the remaining property names and values to set, terminated with a
 *  %NULL.
 * 
 * Creates a new text item.
 * 
 * <!--PARAMETERS-->
 *
 * Here's an example showing how to create a text item with the bottom right
 * of the text box placed at (500,500):
 *
 * <informalexample><programlisting>
 *  GooCanvasItem *text = goo_canvas_text_new (mygroup, "Hello World", 500.0, 500.0, 200.0, GTK_ANCHOR_SE,
 *                                             "fill-color", "blue",
 *                                             NULL);
 * </programlisting></informalexample>
 * 
 * Returns: a new text item.
 **/
GooCanvasItem*
goo_canvas_text_new (GooCanvasItem *parent,
		     const char    *string,
		     gdouble        x,
		     gdouble        y,
		     gdouble        width,
		     GtkAnchorType  anchor,
		     const gchar   *first_property,
		     ...)
{
  GooCanvasItem *item;
  GooCanvasText *text;
  va_list args;

  item = g_object_new (GOO_TYPE_CANVAS_TEXT, NULL);
  text = GOO_CANVAS_TEXT (item);

  text->text = g_strdup (string);
  text->x = x;
  text->y = y;
  text->width = width;
  text->anchor = anchor;

  va_start (args, first_property);
  g_object_set_valist (G_OBJECT (item), first_property, args);
  va_end (args);

  if (parent)
    {
      goo_canvas_item_add_child (parent, item, -1);
      g_object_unref (item);
    }

  return item;
}


static void
goo_canvas_text_finalize (GObject *object)
{
  GooCanvasText *item = (GooCanvasText*) object;

  g_free (item->text);

  G_OBJECT_CLASS (goo_canvas_text_parent_class)->finalize (object);
}


static GooCanvasItemView*
goo_canvas_text_create_view (GooCanvasItem     *item,
			     GooCanvasView     *canvas_view,
			     GooCanvasItemView *parent_view)
{
  return goo_canvas_text_view_new (canvas_view, parent_view,
				   (GooCanvasText*) item);
}


static void
item_interface_init (GooCanvasItemIface *iface)
{
  iface->create_view = goo_canvas_text_create_view;
}


static void
goo_canvas_text_get_property (GObject              *object,
			      guint                 prop_id,
			      GValue               *value,
			      GParamSpec           *pspec)
{
  GooCanvasText *text = (GooCanvasText*) object;

  switch (prop_id)
    {
    case PROP_X:
      g_value_set_double (value, text->x);
      break;
    case PROP_Y:
      g_value_set_double (value, text->y);
      break;
    case PROP_WIDTH:
      g_value_set_double (value, text->width);
      break;
    case PROP_TEXT:
      g_value_set_string (value, text->text);
      break;
    case PROP_USE_MARKUP:
      g_value_set_boolean (value, text->use_markup);
      break;
    case PROP_FONT:
      if (text->font_desc)
	{
	  char *font = pango_font_description_to_string (text->font_desc);
	  g_value_set_string (value, font);
	  g_free (font);
	}
      else
	{
	  g_value_set_string (value, NULL);
	}
      break;
    case PROP_FONT_DESC:
      g_value_set_boxed (value, text->font_desc);
      break;
    case PROP_ANCHOR:
      g_value_set_enum (value, text->anchor);
      break;
    case PROP_ALIGN:
      g_value_set_enum (value, text->alignment);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static void
goo_canvas_text_set_property (GObject              *object,
			      guint                 prop_id,
			      const GValue         *value,
			      GParamSpec           *pspec)
{
  GooCanvasText *text = (GooCanvasText*) object;
  const char *font_name;
  PangoFontDescription *font_desc;

  switch (prop_id)
    {
    case PROP_X:
      text->x = g_value_get_double (value);
      break;
    case PROP_Y:
      text->y = g_value_get_double (value);
      break;
    case PROP_WIDTH:
      text->width = g_value_get_double (value);
      break;
    case PROP_TEXT:
      g_free (text->text);
      text->text = g_value_dup_string (value);
      break;
    case PROP_USE_MARKUP:
      text->use_markup = g_value_get_boolean (value);
      break;
    case PROP_FONT:
      if (text->font_desc)
	pango_font_description_free (text->font_desc);
      font_name = g_value_get_string (value);
      if (font_name)
	text->font_desc = pango_font_description_from_string (font_name);
      else
	text->font_desc = NULL;
      break;
    case PROP_FONT_DESC:
      if (text->font_desc)
	pango_font_description_free (text->font_desc);
      font_desc = g_value_peek_pointer (value);
      if (font_desc)
	text->font_desc = pango_font_description_copy (font_desc);
      else
	text->font_desc = NULL;
      break;
    case PROP_ANCHOR:
      text->anchor = g_value_get_enum (value);
      break;
    case PROP_ALIGN:
      text->alignment = g_value_get_enum (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_signal_emit_by_name (text, "changed", TRUE);
}

