#include "config.h"

#include <postgresql/libpq/libpq-fs.h>

#include "connection.h"
#include "statement.h"
#include "result-set.h"
#include "database-metadata.h"
#include "object.h"

#include "gql++/exception.h"

namespace GQL
{

namespace PG
{

#if defined(GQL_THREADS)
SigC::Threads::Mutex PGConnection::global_mutex_;
#endif

PGConnection::PGConnection(const string& host,
                           const string& port,
                           const map<string, string>& info,
                           const string& db,
                           const string& url,
                           PGDriver *driver)
{
  if (info.find("user") == info.end())
    throw SQLException("The user property is missing. It is manadatory");
  if (info.find("password") == info.end())
    throw SQLException("The password property is missing. It is manadatory");
  
  auto_commit_ = true;
  driver_ = driver;
  url_ = url;
  database_ = db;
  port_ = port;
  host_ = host;

  global_lock();
  conn_ = PQsetdb(host.c_str(), port.c_str(), NULL, NULL, db.c_str());
  global_unlock();

  driver_->reference();
  
  if (PQstatus(conn_) != CONNECTION_OK)
    throw SQLException(string("Connection failed: ") + PQerrorMessage(conn_));

  metadata_ = new PGDatabaseMetaData(this);
}

PGConnection::~PGConnection()
{
  commit();
  close();
  
  driver_->unreference();
}

void PGConnection::close()
{
  lock();
  PQfinish(conn_);
  conn_ = 0;
  unlock();
}

Statement *PGConnection::create_statement()
{
  return(new PGStatement(this));
}

PreparedStatement *PGConnection::prepare_statement(const string& url)
{
  return(0);
}

CallableStatement * PGConnection::prepare_call(const string& url)
{
  return(0);
}

void PGConnection::set_auto_commit(bool auto_commit)
{
  if (auto_commit_ == auto_commit)
    return;

  if (auto_commit == false)
  {
    // PostgreSQL is always in autocommit mode, one has to "BEGIN" a
    // transaction explicitly
    exec_sql("BEGIN TRANSACTION");
  }
  
  lock();
  auto_commit_ = auto_commit;
  unlock();
}

void PGConnection::commit()
{
  exec_sql("COMMIT");
  if (auto_commit_ == false)
    exec_sql("BEGIN TRANSACTION");
}

void PGConnection::rollback()
{
  exec_sql("ROLLBACK");
  if (auto_commit_ == false)
    exec_sql("BEGIN TRANSACTION");
}

string PGConnection::get_catalog() const
{
  string empty;
  
  return(empty);
}

string PGConnection::native_sql(const string& sql) const
{
  string empty;

  return(empty);
}

DatabaseMetaData *PGConnection::get_meta_data()
{
  return(new PGDatabaseMetaData(this));
}

SQLObject *PGConnection::create_object()
{
  return new PGSQLObject(this, conn_);
}

SQLObject *PGConnection::create_blob()
{
  lock();
  Oid  oid = lo_creat(conn_, INV_READ | INV_WRITE);
  unlock();
  
  PGSQLObject *result = new PGSQLObject(this, conn_);
  result->from_int(oid);
  
  return result;
}

void PGConnection::destroy_blob(SQLObject *blob)
{
  int status;
  
  Oid oid = blob->to_int();
  lock();
  status = lo_unlink(conn_, oid);
  unlock();
  
  if (status == -1)
    throw SQLException(PQerrorMessage(conn_));
}

ResultSet *PGConnection::exec_sql(const string& sql)
{
  PGresult *res;
  PGResultSet *result;
  int status;
  
  if (sql.length() > 8192)
    throw SQLException("SQL Statement too long: " + sql);
  
  lock();
  res = PQexec(conn_, sql.c_str());
  unlock();
  
  if (!res)
    throw SQLException(PQerrorMessage(conn_));
  
  
  status = PQresultStatus(res);
  if (status != PGRES_COMMAND_OK && status != PGRES_TUPLES_OK)
  {
    throw SQLException(PQerrorMessage(conn_));
  }
  
  result = new PGResultSet(this, res);
  
  lock();
  destroy.connect(result->destroy.slot());
  unlock();
  
  return(result);
}

}
}
