/*   grep-dctrl - grep Debian control files
     Copyright (C) 1999, 2000, 2001  Antti-Juhani Kaijanaho
  
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
  
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details. 
  
     You should have received a copy of the GNU General Public License
     along with this program; see the file COPYING.  If not, write to
     the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
  
     The author can be reached via mail at (ISO 8859-1 charset for the city)
        Antti-Juhani Kaijanaho
        Helokantie 1 A 16
        FIN-40640 JYVSKYL
        FINLAND
        EUROPE
     and via electronic mail from
        gaia@iki.fi
     If you have a choice, use the email address; it is more likely to
     stay current.

*/

#include <errno.h>
#include <limits.h>
#include <locale.h>
#include <getopt.h>
#include <publib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "i18n.h"
#include "matcher.h"
#include "msg.h"
#include "rc.h"
#include "strutil.h"

/* If you change these, update the manual page, too */
#define FIELDS_MAXNUM 256
#define FIELD_MAXLEN 512
#define SHOWFIELDS_MAXLEN 512
#define SHOWFIELD_MAXNUM 256

#define NO_REGEX -1
#define EXTENDED_REGEX 1
#define STD_REGEX 0

enum { OPT_RCFILE = UCHAR_MAX + 1, OPT_COUNT };

static void
print_version (const char * progname)
{
  printf ("grep-dctrl %s\n", VERSION);
  puts   ( _("This program is free software and comes WITHOUT ANY KIND "
             "OF WARRANTY."));
  printf (_("See the file %s for more information, or do a\n"), COPYING);
  printf (_("\"%s --copying\", piping the output to your favourite pager.\n"),
          progname);
}

static void
print_usage (const char * progname)
{
  printf (_("Usage: %s [ options ] PATTERN [ FILE ... ]\n"
            "       %s --version | --copying | --help | -V | -C | -h\n"),
          progname, progname);
  puts   (_("Possible options:"));
  puts   (_("       -l LEVEL, --errorlevel=LEVEL"));
  puts   (_("             Set debugging level to LEVEL (0..3,"));
  puts   (_("             biggest is most verbose)."));
  puts   (_("       -F FIELD,FIELD,..., --field=FIELD,FIELD,..."));
  puts   (_("             Restrict pattern matching to the FIELDs."));
  puts   (_("       -s FIELD,FIELD,...; --show-field=FIELD,FIELD,..."));
  puts   (_("             Show only the body of these fields"));
  puts   (_("             from the matching paragraphs."));
  puts   (_("       -n, --no-field-names"));
  puts   (_("             Print only the bodies of the fields to be shown"));
  puts   (_("       -e, --eregex"));
  puts   (_("             The pattern is an extended POSIX "
            "regular expression"));
  puts   (_("       -r, --regex"));
  puts   (_("             The pattern is a POSIX regular expression"));
  printf (_("       -i, --ignore-case\n"));
  puts   (_("             Ignore case when looking for a match."));
  puts   (_("       -v, --invert-match"));
  puts   (_("             Show those paragraphs that don't match."));
  puts   (_("       --config-file=FNAME"));
  puts   (_("             Use FNAME as the config file."));
  printf (_("             (Defaults: %s/grep-dctrl.rc, ~/.grep-dctrlrc)\n"),
          SYSCONF);
  puts   (_("       --count"));
  puts   (_("             Instead of showing the paragraphs that match (or, with -v, "));
  puts   (_("             that don't match), show the count of those paragraphs."));
  puts   (_("       -X, --exact-match"));
  puts   (_("             Require an exact match (not substring match)."));
  puts   (_("       -P    Shorthand for '-FPackage'."));
/*  puts   (_("       -d    Show the short description in matched paragraphs."));*/
  puts   (_("       -V, --version"));
  puts   (_("             Print out version information."));
  puts   (_("       -C, --copying"));
  puts   (_("             Print out the copyright license. (long output)"));
  puts   (_("       -h, --help"));
  puts   (_("             Print out this help screen."));
  puts   ("");
  puts   (_("Please, report bugs to <gaia@iki.fi>."));
}

/* Copy the file called fname to standard outuput stream.  Return zero
   iff problems were encountered. */
static int
to_stdout (const char * fname)
{
  FILE * f;
  int c;
  int rv = 1;
  
  f = fopen (fname, "r");
  if (f == 0)
    {
      message (L_FATAL, strerror (errno), COPYING);
      return 0;
    }

  while ( ( c = getc (f)) != EOF)
    putchar (c);

  if (ferror (f))
    {
      message (L_FATAL, strerror (errno), COPYING);
      rv = 0;
    }
  
  fclose (f);

  return rv;
}

static void
set_match_field (struct matcher_t * matcher, const char * field)
{
  static size_t match_fields_num = 0;
  static char const * match_fields [FIELDS_MAXNUM];
  static size_t match_field_used = 0;
  static char match_field [FIELD_MAXLEN] = "";
  size_t i;

  if (strlen (field) + match_field_used + 1>= FIELD_MAXLEN)
    {
      message (L_FATAL, _("cumulative field name exceeds maximum length"), 0);
      exit (EXIT_FAILURE);
    }

  strcpy (match_field + match_field_used, field);
  
  /* Tokenize */
  i = match_fields_num;
  match_fields [i] = strtok (match_field + match_field_used, ",");
  while (match_fields [i] != 0 && i <  FIELDS_MAXNUM)
          match_fields [++i] = strtok (0, ",");

  if (match_fields [i] != 0)
  {
          message (L_FATAL, _("too many fields to search in"), 0);
          exit (EXIT_FAILURE);
  }
  match_fields_num = i;
  match_field_used += strlen(field) + 1;

  matcher->fields = match_fields;
  matcher->numfields = match_fields_num;
}

int
main (int argc, char * argv [])
{
  int i;
  int print_count_p = 0;
  int regex_class = NO_REGEX;
  int rv = EXIT_SUCCESS;
  struct matcher_t matcher = { MATCH_FIXED, 1, 1 };
  char show_field_string [SHOWFIELDS_MAXLEN] = "";
  char * show_fields [SHOWFIELD_MAXNUM] = { 0 }; /* show these fields only */
  const char * rcname = 0;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  msg_set_progname (argv[0]);

  while (1)
    {
      int c;
      static struct option long_options [] = {
        { "version",        no_argument,       0, 'V' },
        { "help",           no_argument,       0, 'h' },
        { "errorlevel",     required_argument, 0, 'l' },
        { "field",          required_argument, 0, 'F' },
        { "show-field",     required_argument, 0, 's' },
        { "eregex",         no_argument,       0, 'e' },
        { "regex",          no_argument,       0, 'r' },
        { "copying",        no_argument,       0, 'C' },
        { "ignore-case",    no_argument,       0, 'i' },
        { "invert-match",   no_argument,       0, 'v' },
        { "exact-match",    no_argument,       0, 'X' },
        { "config-file",    required_argument, 0, OPT_RCFILE },
        { "count",          no_argument,       0, OPT_COUNT },
        { "no-field-names", no_argument,       0, 'n' },
        { 0, 0, 0, 0 } };
      static const char * short_options = "cCeF:hil:nPrs:vVX";

      c = getopt_long (argc, argv, short_options, long_options, 0);

      if (c == EOF)
        break;

      switch (c)
        {
        case 'V':
          print_version (argv [0]);
          exit (EXIT_SUCCESS);

        case 'h':
          print_usage (argv [0]);
          exit (EXIT_SUCCESS);

        case 'C':
          if (!to_stdout (COPYING))
            exit (EXIT_FAILURE);

          exit (EXIT_SUCCESS);

        case 'n':
          matcher.show_fieldnames = 0;
          break;

        case 'c':
          fprintf(stderr, _("%s: warning: the meaning of -c "
                            "will change in the future\n"), argv[0]);
          fprintf(stderr, _("%s: -c is disabled; "
                            "use --config-file instead\n"), argv[0]);
          exit(EXIT_FAILURE);
          
        case OPT_RCFILE:
          rcname = xstrdup (optarg);
          break;

        case OPT_COUNT:
          matcher.suppress = 1;
          print_count_p = 1;
          break;

        case 'l':
          {
            int ll = str2loglevel (optarg);
            if (ll < 0)
              {
                message (L_FATAL, _("no such log level"), optarg);
                exit (EXIT_FAILURE);
              }
            set_loglevel (ll);
          }
          break;

        case 'P':
          set_match_field (&matcher, "Package");
          break;

        case 'F':
          set_match_field (&matcher, optarg);
          break;

        case 'X':
          if (matcher.type == MATCH_REGEX)
            {
              message (L_FATAL,
                       _("don't know how to do an exact regex match"), 0);
              exit (EXIT_FAILURE);
            }

          matcher.type = MATCH_EXACT;
          debug_message (_("using exact matches"), 0);
          break;

        case 's':
          if (*show_field_string != 0)
            {
              message (L_FATAL, _("you can only use -s once"), 0);
              exit (EXIT_FAILURE);
            }
          if (strlen (optarg) >= SHOWFIELDS_MAXLEN)
            {
              message (L_FATAL, _("output field spec exceeds maximum length"), 0);
              exit (EXIT_FAILURE);
            }
          strcpy (show_field_string, optarg);

          /* Tokenize the field spec into field names. */
          {  
            int i = 0;
            show_fields [i] = strtok (show_field_string, ",");
            while (show_fields [i] != 0 && i < SHOWFIELD_MAXNUM)
              show_fields [++i] = strtok (0, ",");
            if (show_fields [i] != 0)
              {
                message (L_FATAL, _("too many output fields"), 0);
                exit (EXIT_FAILURE);
              }
          }
          break;
                
        case 'e':
          if (regex_class != NO_REGEX)
            {
              message (L_FATAL, _("do not use both -e and -r options"), 0);
              exit (EXIT_FAILURE);
            }
          if (matcher.type == MATCH_EXACT)
            {
              message (L_FATAL,
                       _("don't know how to do an exact regex match"), 0);
              exit (EXIT_FAILURE);
            }
          debug_message (_("using extended regular expressions"), 0);
          matcher.type = MATCH_REGEX;
          regex_class = EXTENDED_REGEX;
          break;

        case 'r':
          if (regex_class != NO_REGEX)
            {
              message (L_FATAL, _("do not use both -e and -r options"), 0);
              exit (EXIT_FAILURE);
            }
          if (matcher.type == MATCH_EXACT)
            {
              message (L_FATAL,
                       _("don't know how to do an exact regex match"), 0);
              exit (EXIT_FAILURE);
            }
          debug_message ("using conventional regular expressions", 0);
          matcher.type = MATCH_REGEX;
          regex_class = STD_REGEX;
          break;

        case 'i':
          debug_message (_("ignoring case"), 0);
          matcher.case_sensitive = 0;
          break;
 
        case 'v':
          debug_message(_("inverted match"), 0);
          matcher.inverse = 1;
          break;

        case '?': case ':':
          exit (EXIT_FAILURE);

        default:
          message (L_FATAL,
                   _("I'm broken - please report this to <gaia@iki.fi>"),
                   "main.c");
          abort ();
        }
          
    }

  if (show_fields [0] == 0 && !matcher.show_fieldnames)
    {
      message (L_FATAL,
               _("cannot suppress field names when showing whole paragraphs"),
               0);
      exit (EXIT_FAILURE);
    }

  if (argc == optind)
    {
      message (L_FATAL, _("a pattern is mandatory"), 0);
      exit (EXIT_FAILURE);
    }

  switch (matcher.type)
    {
      int rerr;
      static regex_t regex;

    case MATCH_FIXED: case MATCH_EXACT:
      matcher.pattern.fixed = argv [optind];
      break;

    case MATCH_REGEX:
      matcher.pattern.regex = &regex;
      rerr = regcomp (matcher.pattern.regex, argv [optind],
                      ( (regex_class == EXTENDED_REGEX ? REG_EXTENDED : 0)
                        | REG_NOSUB
                        | (matcher.case_sensitive ? 0 : REG_ICASE)));
      if (rerr != 0)
        {
          char * s;

          s = get_regerror (rerr, matcher.pattern.regex);
          if (s == 0)
            fatal_enomem (0);

          message (L_IMPORTANT, s, 0);
          free (s);
          exit (EXIT_FAILURE);
        }
      break;

    default:
      message (L_FATAL, _("I'm broken - please report this to <gaia@iki.fi>"),
               "main.c/regex");
      abort ();

    }

  for (i = ++optind; i < argc || (argc == optind && optind == i); i++)
    {
      FILE * f;
      const char * fname;

      if (optind == argc)
        fname = find_ifile_by_exename (fnbase (argv [0]), rcname);
      else
        fname = argv [i];

      if (fname == 0)
        {
          message (L_FATAL, _("need a file name to grep"), 0);
          exit (EXIT_FAILURE);
        }

      if (strcmp (fname, "-") == 0)
        {
          f = stdin;
          fname = "stdin";
        }
      else
        {
          f = fopen (fname, "r");
          if (f == NULL)
            {
              perror (argv[0]);
              break;
            }
        }
      rv = (grep_control (&matcher, f, show_fields, fname))
        ? rv : EXIT_FAILURE;
      if (f != stdin)
        fclose (f);
    }      
  if (print_count_p)
    printf("%llu\n", matcher.match_count);
  return rv;
}
