/* $Id: Communication.cpp,v 1.4 2002/01/04 22:57:23 kordless Exp $ */

/**************************************************************************
 *                                                                        *
 *   Copyright (C) 2000 Grub, Inc.                                        *
 *                                                                        *
 *   This program is free software; you can redistribute it and/or modify *
 *   it under the terms of the GNU General Public License as published by *
 *   the Free Software Foundation; either version 1, or (at your option)  *
 *   any later version.                                                   *
 *                                                                        *
 *   This program is distributed in the hope that it will be useful,      *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *   GNU General Public License for more details.                         *
 *                                                                        *
 *   You should have received a copy of the GNU General Public License    *
 *   along with this program; if not, write to the Free Software          *
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *                                                                        *
 *   Author:  Igor Stojanovski - ozra   (email: ozra@grub.org)            *
 *                                                                        *
 **************************************************************************/

#include "Communication.h"

using std::string;

#ifdef COMM_COMPRESSION

const int Communication::write_zbuf_len = COMPRESSION_WRITE_BUFSIZE;
const int Communication::read_zbuf_len = COMPRESSION_READ_BUFSIZE;

#if 0  /* used by XXX_compressed() */
struct compressed_t {
	Communication *comm;
	int fd;               /* file descriptor */
	int total_read;       /* callback func. adds total bytes read */
};
#endif

/* callback functions used by zread() and zwrite() */

int write_compressed( int fd, const char *buf, int buflen, void *arg )
{
	int ret;
	/* may return zero */
	ret = sock_write( fd, buf, buflen );
	if ( ret > 0 )
		((Communication *)arg)->total_compress_sent += ret;

	return ret;
}

int read_compressed( int fd, char *buf, int buflen, void *arg )
{
	int ret;

	/* never returns zero */
	ret = sock_read( fd, buf, buflen );
	if ( ret > 0 )
		((Communication *)arg)->total_compress_recv += ret;

	return ret;
}
#endif  // COMM_COMPRESSION

/* write_abstract()
 * Used by buffered reads/writes.  This datatype (see buf.h) will call this
 * function any time it needs to write data.  The return codes of this
 * function are then passed back to the caller as return value of buf_write()
 * or buf_flush().
 * Parameters:
 *   fd -- the file descriptor to write to
 *   buf -- the buffer of data to write from
 *   len -- the length of this buffer
 *   arg -- (struct *sock_arg) type; used with select()
 * Returns:
 *   Number of bytes written; or
 *   COMM_TIMEOUT -- if waiting for available write timed out
 *   COMM_SELECTFAILED -- if select() failed; check errno for more info
 *   SOCK_WOULDBLOCK -- if non-blocking I/O used when write() would block
 *   SOCK_ERROR -- if writing to the socked failed
 */
int write_abstract( int fd, char *buf, int len, void *arg ) {
	sock_arg *sock = (sock_arg *)arg;
	int ret;

	if ( arg != NULL && sock->timeout.tv_sec != 0 ) {
		struct timeval timeout = {
			sock->timeout.tv_sec,
			sock->timeout.tv_usec
		};

#ifdef __PTHREAD_TESTCANCEL
		pthread_testcancel();
#endif

		FD_SET( fd, &sock->set );
		ret = select( fd + 1, NULL, &sock->set, NULL, &timeout );

#ifdef __PTHREAD_TESTCANCEL
		pthread_testcancel();
#endif

		if ( ret == 0 ) {

			/* write file desc. didn't become available
			 * after the specified time elapsed
			 */
			return COMM_TIMEOUT;
		}
		else if ( ret == -1 ) {

			/* select() failed; check errno for reason */
			return COMM_SELECTFAILED;
		}
	}

#ifdef COMM_COMPRESSION
	if ( sock && sock->comm->use_comp ) {
		int fret;
		int func_ret;

		/* writes (hopefully) all requested bytes */
		fret = zwrite( sock->comm->write_zstream,
			sock->comm->write_zbuf, sock->comm->write_zbuf_len,
			fd, buf, len,
			(void *)sock->comm, write_compressed, &func_ret );
		if ( fret == ZS_CALLBACK ) {

			/* sock_write() returned a (negative) error code */

			/* returns the value from sock_write() */
			return func_ret;
		}
		else if ( fret != ZS_OK ) {

			/* compression error occured */

/* TODO: we need a better return code here instead of SOCK_ERROR */
			fprintf( stderr, "zwrite(): compression failed. "
				"Returned = %d\n", fret );
exit(1);
			return SOCK_ERROR;
		}

		sock->comm->total_sent += len;

		/* At this point ALL requested bytes were written */
		return len;
	}
#endif  // COMM_COMPRESSION

	ret = (int)sock_write( fd, buf, len );

	if ( ret > 0 ) {

#ifdef COMM_COMPRESSION
		sock->comm->total_sent += ret;
		sock->comm->total_compress_sent += ret;
#endif
	}

	return ret;
}


/* read_abstract()
 * Similar to write_abstract(), ths function is called any time a read is
 * needed.
 * Parameters:
 *   fd -- the file descriptor to read from
 *   buf -- the buffer of data to read to
 *   len -- the length of this buffer
 *   arg -- (struct *sock_arg) type; used with select()
 * Returns:
 *   Number of bytes read, but never zero bytes; or
 *   COMM_TIMEOUT -- if waiting for read timed out
 *   COMM_SELECTFAILED -- if select() failed; check errno for more info
 *   SOCK_WOULDBLOCK -- if non-blocking I/O used when read() would block
 *   SOCK_ERROR -- if reading to the socked failed
 *   SOCK_CLOSED -- if reading from a closed connection
 */
int read_abstract( int fd, char *buf, int len, void *arg ) {
	sock_arg *sock = (sock_arg *)arg;
	int ret;

	if ( arg != NULL && sock->timeout.tv_sec != 0 ) {
		struct timeval timeout = {
			sock->timeout.tv_sec,
			sock->timeout.tv_usec
		};

#ifdef __PTHREAD_TESTCANCEL
		pthread_testcancel();
#endif

		FD_SET( fd, &sock->set );
		ret = select( fd + 1, &sock->set, NULL, NULL, &timeout );

#ifdef __PTHREAD_TESTCANCEL
		pthread_testcancel();
#endif

		if ( ret == 0 ) {

			/* write file desc. didn't become available
			 * after the specified time elapsed
			 */
			return COMM_TIMEOUT;
		}
		else if ( ret == -1 ) {

			/* select() failed; check errno for reason */
			return COMM_SELECTFAILED;
		}
	}

#ifdef COMM_COMPRESSION
	if ( sock && sock->comm->use_comp ) {
		int fret;
		int func_ret;

		fret = zread( sock->comm->read_zstream,
			sock->comm->read_zbuf, sock->comm->read_zbuf_len,
			fd, buf, len,
			(void *)sock->comm, read_compressed, &func_ret );
		if ( fret == ZS_CALLBACK ) {

			/* sock_read() returned a (non-positive) error code */
			/* it should never return zero, though */

			/* returns the value from sock_read() */
			return func_ret;
		}
		else if ( fret < 0 ) {

			/* compression error occured */

/* TODO: we need a better return code here instead of SOCK_ERROR */
			fprintf( stderr, "zread(): compression failed. "
				"Returned = %d\n", fret );
exit(1);
			return SOCK_ERROR;
		}

		sock->comm->total_recv += fret;

		return fret;
	}
#endif  // COMM_COMPRESSION

	ret = (int)sock_read( fd, buf, len );

	if ( ret > 0 ) {

#ifdef COMM_COMPRESSION
		sock->comm->total_recv += ret;
		sock->comm->total_compress_recv += ret;
#endif
	}

	return ret;
}

int write_abstract_custom( int fd, char *buf, int len, void *arg ) {
	Communication *comm = (Communication *)arg;
	int ret = 0;

	if ( comm->write_fn ) {
		if ( comm->timeout.tv_sec > 0 ) {
			struct timeval timeout = {
				comm->timeout.tv_sec,
				comm->timeout.tv_usec
			};

#ifdef __PTHREAD_TESTCANCEL
			pthread_testcancel();
#endif

			FD_SET( fd, &comm->write_custom_fd_set );
			ret = select( fd + 1, NULL,
				&comm->write_custom_fd_set, NULL,
				&timeout );

#ifdef __PTHREAD_TESTCANCEL
			pthread_testcancel();
#endif

			if ( ret == 0 ) {

				/* write file desc. didn't become available
				 * after the specified time elapsed
				 */
				return COMM_TIMEOUT;
			}
			else if ( ret == -1 ) {

				/* select() failed; check errno for reason */
				return COMM_SELECTFAILED;
			}
		}

		ret = comm->write_fn( fd, buf, len, comm, comm->w_fn_arg );
	}

	return ret;
}

int read_abstract_custom( int fd, char *buf, int len, void *arg ) {
	Communication *comm = (Communication *)arg;
	int ret = 0;

	if ( comm->read_fn ) {
		if ( comm->timeout.tv_sec > 0 ) {
			// must copy the struct because Linux modifies it
			struct timeval timeout = {
				comm->timeout.tv_sec,
				comm->timeout.tv_usec
			};

#ifdef __PTHREAD_TESTCANCEL
			pthread_testcancel();
#endif

			FD_SET( fd, &comm->read_custom_fd_set );
			ret = select( fd + 1, &comm->read_custom_fd_set,
				NULL, NULL,
				&timeout );

#ifdef __PTHREAD_TESTCANCEL
			pthread_testcancel();
#endif

			if ( ret == 0 ) {

				/* write file desc. didn't become available
				 * after the specified time elapsed
				 */
				return COMM_TIMEOUT;
			}
			else if ( ret == -1 ) {

				/* select() failed; check errno for reason */
				return COMM_SELECTFAILED;
			}
		}

		ret = comm->read_fn( fd, buf, len, comm, comm->r_fn_arg );
	}

	return ret;
}

// SOCK_ERROR
int Communication::write( const char *buf, int len ) {
	int ret = 0, total_w = 0;

	if ( w_comm ) {

		ret = w_comm->write( buf, len );
		if ( ret < 0 ) {

			copy_err( w_comm );
			return -1;
		}
	}
	else {
		// initialize the time struct, as Linux modifies it every time
		// select() gets called
		write_arg.timeout.tv_sec  = timeout.tv_sec;
		write_arg.timeout.tv_usec = timeout.tv_usec;

#ifdef COMM_COMPRESSION
		write_arg.comm = this;
#endif

		// ret = bytes written, SOCK_WOULDBLOCK, SOCK_ERROR
		if ( use_custom_io )
			ret = buf_write( &buff_w, (char *)buf, len, (void *)this );
		else
			ret = buf_write( &buff_w, (char *)buf, len, (void *)&write_arg );
		if ( ret < 0 ) {
			if ( ret == SOCK_ERROR && sock_err ) {

				Comm_ERROR( SOCK_ERROR, "Write Socket: got EPIPE" );
				return -1;
			}
			else if ( ret == COMM_TIMEOUT ) {

				Comm_ERROR( COMM_TIMEOUT,
					"Write Socket: connection timed out" );
				return -1;
			}
			else if ( ret == COMM_SELECTFAILED ) {

				Comm_ERROR( COMM_SELECTFAILED, strerror( errno ) );
				return -1;
			}
			else if ( ret == SOCK_WOULDBLOCK && sock_err ) {

				Comm_TRACE( SOCK_WOULDBLOCK,
					"Write Socket: WOULD BLOCK" );
				return -1;
			}
			else if ( use_custom_io ) {

				Comm_ERROR( ret, "Custom I/O error" );
				return -1;
			}
			else {

				Comm_ERROR( COMM_IO_ERROR, "I/O error occured" );
				return -1;
			}
		}
	}

	return 0;
}


int Communication::flush() {
	int ret;

	if ( w_comm ) {

		ret = w_comm->flush();
		if ( ret < 0 ) {

			copy_err( w_comm );
			return -1;
		}
	}
	else {
		// initialize the time struct, as Linux modifies it every time
		// select() gets called
		write_arg.timeout.tv_sec  = timeout.tv_sec;
		write_arg.timeout.tv_usec = timeout.tv_usec;

#ifdef COMM_COMPRESSION
		write_arg.comm = this;
#endif

		if ( use_custom_io )
			ret = buf_flush( &buff_w, (void *)this );
		else
			ret = buf_flush( &buff_w, (void *)&write_arg );
		if ( ret < 0 ) {
			if ( ret == SOCK_ERROR && sock_err ) {

				Comm_ERROR( SOCK_ERROR, "Write Socket: got EPIPE" );
				return -1;
			}
			else if ( ret == COMM_TIMEOUT ) {

				Comm_ERROR( COMM_TIMEOUT,
					"Write Socket: connection timed out" );
				return -1;
			}
			else if ( ret == COMM_SELECTFAILED ) {

				Comm_ERROR( COMM_SELECTFAILED, strerror( errno ) );
				return -1;
			}
			else if ( ret == SOCK_WOULDBLOCK && sock_err ) {

				Comm_TRACE( SOCK_WOULDBLOCK,
					"Write Socket: WOULD BLOCK" );
				return -1;
			}
			else if ( use_custom_io ) {

				Comm_ERROR( ret, "Custom I/O error" );
				return -1;
			}
			else {

				Comm_ERROR( COMM_IO_ERROR, "I/O error occured" );
				return -1;
			}
		}
	}

	return 0;
}


// SOCK_ERROR
int Communication::send( const char *buf, int len ) {
	int ret;
	char nul = END_CHAR;

	ret = write( buf, len );
	if ( ret != -1 )
		ret = write( &nul, 1 );

	if ( ret != -1 )
		ret = flush();

	return ret;
}


// SOCK_ERROR
int Communication::finish() {
	int ret;
	char nul = END_CHAR;

	ret = write( &nul, 1 );

	if ( ret != -1 )
		ret = flush();

	return ret;
}

// len cannot be larger than the buffer size
// returns bytes read
// SOCK_ERROR, SOCK_CLOSED
int Communication::read( char *buf, int len ) {
	int ret = 0;

	if ( len < 1 ) return 0;

	if ( r_comm ) {

		ret = r_comm->read( buf, len );
		if ( ret < 0 ) {

			copy_err( r_comm );
			return -1;
		}

#ifdef _DEBUG
	fprintf(stdout, "Communication::read() buf '%s'\n", buf );
#endif

	}
	else {
		// initialize the time struct, as Linux modifies it every time
		// select() gets called
		read_arg.timeout.tv_sec  = timeout.tv_sec;
		read_arg.timeout.tv_usec = timeout.tv_usec;

#ifdef COMM_COMPRESSION
		read_arg.comm = this;
#endif

		if ( use_custom_io )
			ret = buf_read( &buff_r, buf, len, (void *)this );
		else
			// ret = bytes read, SOCK_WOULDBLOCK, SOCK_ERROR
			ret = buf_read( &buff_r, buf, len, (void *)&read_arg );
		if ( ret < 0 ) {
			if ( ret == SOCK_ERROR && sock_err ) {

				Comm_ERROR( SOCK_ERROR, "Read Socket: got EPIPE" );
				return -1;
			}
			else if ( ret == SOCK_CLOSED && sock_err ) {

				Comm_ERROR( SOCK_CLOSED,
					"Read Socket: FIN received on socket" );
				return -1;
			}
			else if ( ret == COMM_TIMEOUT ) {

				Comm_ERROR( COMM_TIMEOUT,
					"Read Socket: connection timed out" );
				return -1;
			}
			else if ( ret == COMM_SELECTFAILED ) {

				Comm_ERROR( COMM_SELECTFAILED, strerror( errno ) );
				return -1;
			}
			else if ( ret == SOCK_WOULDBLOCK && sock_err ) {

				Comm_TRACE( SOCK_WOULDBLOCK,
					"Read Socket: WOULD BLOCK" );
				return -1;
			}
			else if ( use_custom_io ) {

				Comm_ERROR( ret, "Custom I/O error" );
				return -1;
			}
			else {

				Comm_ERROR( COMM_IO_ERROR, "I/O error occured" );
				return -1;
			}
		}
	}

	return ret;
}


// no problems if part of the data consists of control or null characters
int Communication::readline( char *buf, int len, char end_char ) {
	int ret = 0;
	char ch;
	int bytes_ret = 0;

	if ( len < 2 ) return 0;   // sanity check

	if ( r_comm ) {

		ret = r_comm->readline( buf, len, end_char );
		if ( ret < 0 ) {

			copy_err( r_comm );
			return -1;
		}
	}
	else {
		do {
			// initialize the time struct, as Linux modifies it every time
			// select() gets called
			read_arg.timeout.tv_sec  = timeout.tv_sec;
			read_arg.timeout.tv_usec = timeout.tv_usec;

#ifdef COMM_COMPRESSION
			read_arg.comm = this;
#endif

			if ( use_custom_io )
				ret = buf_read( &buff_r, &ch, 1, (void *)this );
			else
				// ret = bytes read, SOCK_WOULDBLOCK, SOCK_ERROR
				ret = buf_read( &buff_r, &ch, 1, (void *)&read_arg );
			if ( ret < 0 ) {
				if ( ret == SOCK_ERROR && sock_err ) {

					Comm_ERROR( SOCK_ERROR,
						"Read Socket: got EPIPE" );
					return -1;
				}
				else if ( ret == SOCK_CLOSED && sock_err ) {

					Comm_ERROR( SOCK_CLOSED,
						"Read Socket: FIN received on socket" );
					return -1;
				}
				else if ( ret == COMM_TIMEOUT ) {

					Comm_ERROR( COMM_TIMEOUT,
						"Read Socket: connection timed out" );
					return -1;
				}
				else if ( ret == COMM_SELECTFAILED ) {

					Comm_ERROR( COMM_SELECTFAILED,
						strerror( errno ) );
					return -1;
				}
				else if ( ret == SOCK_WOULDBLOCK && sock_err ) {

					Comm_TRACE( SOCK_WOULDBLOCK,
						"Read Socket: WOULD BLOCK" );
					return -1;
				}
				else if ( use_custom_io ) {

					Comm_ERROR( ret, "Custom I/O error" );
					return -1;
				}
				else {

					Comm_ERROR( COMM_IO_ERROR, "I/O error occured" );
					return -1;
				}
			}
			else {
				// everything OK, so append the byte to buf
				buf[bytes_ret++] = ch;
			}
		} while ( ch != end_char && bytes_ret < len - 1 );

		buf[bytes_ret++] = '\0';

		return bytes_ret;
	}

	return ret;
}


int Communication::read_spacesplit(
		string arg_list[],
		int arg_list_len,
		int max_len,
		int end_char )
{
	int ret, count;
	char *arg_start, *arg_end;

	if ( max_len > bufsize ) {

		Comm_ERROR( COMM_ARG2LONG,
			"max_len argument in read_spacesplit() is "
			"longer than specified buffer size"
		);
		return -1;
	}

	ret = readline( read_buf, max_len, end_char );
	if ( ret == -1 || ret == 0 ) {

		// ret should never equal 0 anyway
		return -1;
	}
	else if ( ret == 1 ) {

		// this condition should not happen as well
		return 0;
	}

	// checking if the line received was ended properly
	if ( read_buf[ret - 2] != end_char ) {

		Comm_ERROR( COMM_BADCOMMAND, "Message did not terminate with a "
			"proper end character after maximum length reached"
		);
		return -1;
	}

	// terminate the string
	read_buf[ret - 2] = '\0';

	// split the input line in separate strings, delimited by
	// one or more spaces
	count = 0;
	arg_start = next_no_space( read_buf );
	while ( arg_start ) {
		arg_end = strchr( arg_start, ' ' );
		if ( arg_end ) {

			*arg_end = '\0';
			arg_list[count++] = arg_start;			
			if ( count == arg_list_len )
				break;
			arg_start = next_no_space( arg_end + 1 );
		}
		else
			break;
	}
	if ( arg_start && count < arg_list_len )
		arg_list[count++] = arg_start;			

	return count;
}

#ifdef COMM_COMPRESSION
int Communication::init_compression() {

	if ( use_comp )
		goto done;

	write_zstream = zstream_deflate_init( ZS_BEST_COMPRESSION );
	if ( !write_zstream ) goto done_error;
	read_zstream  = zstream_inflate_init();
	if ( !read_zstream )  goto done_error;

	set_errcode(0);
	set_errmsg("");

done:
	use_comp = true;
	return 0;

done_error:
	if ( write_zstream )
		zstream_deflate_end( write_zstream );

	if ( read_zstream )
		zstream_inflate_end( read_zstream );

	read_zstream = 0;
	write_zstream = 0;

	return -1;
}

int Communication::end_compression() {
	int ret;

	if ( ! use_comp )
		return 0;

	if ( write_zstream )
		ret += zstream_deflate_end( write_zstream );

	if ( read_zstream )
		ret += zstream_inflate_end( read_zstream );

	read_zstream = 0;
	write_zstream = 0;

	use_comp = false;

	if ( ret ) {

		set_errcode(0);
		set_errmsg("");
		set_compress_errcode(COMM_Z_ERROR);
		set_compress_errmsg("error closing the ZLIB streams");
	}

	return ret ? -1 : 0;
}
#endif

/* --------------------------------------------------------------------- */

void Communication::init( int timeout_sec ) {

	wbuffer  = new char[bufsize+1];
	rbuffer  = new char[bufsize+1];
	read_buf = new char[bufsize+1];

	// initialize the sets used by select()
	FD_ZERO( &write_arg.set );
	FD_ZERO( &read_arg.set );

	// initialize the time structure used by select()
	timeout.tv_sec = timeout_sec;
	timeout.tv_usec = 0;

	FD_ZERO( &write_custom_fd_set );
	FD_ZERO( &read_custom_fd_set );

	sock_err = false;

	r_comm = w_comm = 0;

	total_sent = 0L;
	total_compress_sent = 0L;
	total_recv = 0L;
	total_compress_recv = 0L;

#ifdef COMM_COMPRESSION
	use_comp = false;
	write_zstream = 0;
	read_zstream = 0;
#endif
}

/*
 * Returns pointer to next char after what ptr points to which is space.
 * If space is not found until '\0' is encountered, NULL is returned.
 */
char *Communication::next_no_space( char *ptr ) {

	while( *ptr == ' ' && *ptr != '\0' )
		ptr++;

	if ( *ptr == '\0' )
		return NULL;
	else
		return ptr;
}

void Communication::copy_err( Communication *comm ) {

	errstr = comm->errstr;
	errnum = comm->errnum;
}

