/* sms.c --- code for management of script files: parts stolen from smssend  */

/* Copyright (c) E. Lassauge, 2000-2001.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * mailto:lassauge@mail.dotcom.fr
 * http://lassauge.free.fr/
 *
 * REVISION HISTORY:
 *
 * 2001-01-30 Eric Lassauge <lassauge@mail.dotcom.fr>
 * added stdout/stderr callbacks to get smssend messages
 *
 * 2001-01-04 Eric Lassauge <lassauge@mail.dotcom.fr>
 * get rid of THREAD_SMSSEND: I do not want to maintain this part of the code.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <pthread.h>
#include <signal.h>
#include <gnome.h>

#include "sms.h"
#include "interface.h"
#include "support.h"

/* global preferences values */
gboolean	UseProxy 		= FALSE;
gchar		ProxyHost[P_LENGTH];
gint		ProxyPort 		= 8080;
gchar		ProxyUser[P_LENGTH];
gchar		ProxyPass[P_LENGTH];
gchar		*HeaderFile 		= (gchar *)NULL;
gint		DebugLevel 		= 0;
gint		TimeOut    		= 0;
gboolean	AutoCheck 		= FALSE;
gboolean	KeepPass  		= FALSE;
gboolean	DelaySend   		= FALSE;
gint		Retries     		= 1;

/* args variables */
gchar		*ScriptPath 		= (gchar *)NULL;
gchar		*ScriptFilename 	= (gchar *)NULL;
gint		MyScripts		= 0;

/* global variables */
GHashTable	*AliasHash	 	= (GHashTable *)NULL;
GSList		*DelayList	 	= (GSList*)NULL;
gint 		NumProvider	 	= 0;
GHashTable	*ShownProviderHash	= (GHashTable *)NULL; /* visible in GUI */
GHashTable	*SavedProviderHash	= (GHashTable *)NULL; /* saved in user prefs */
GHashTable	*ProviderHash		= (GHashTable *)NULL; /* all providers */

/* executable name for smssend */
#define SMSEXECNAME	"smssend"

/* outputs from smssend */
#define MESSAGE1	"You already have the latest version of SmsSend"
#define MESSAGE2	"SmsSend Error "
#define MESSAGE3	"Result :"

/* parameters for DoSms() thread */
#define MAXLINE 1024
#define HALF_SECOND 500000
#define DEF_TIMEOUT 60
#define NOT_DONE    0xdeadbeef

static void *DoSms(void *arg);
const char arg_help[]   = "-help";
const char arg_update[] = "-update";
const char arg_minus[]  = "--";

/* -----------------------------------------------------------------------------------*/
void DeleteProvider(PProvider_t Pv)
{
    /* Free a provider struct: based on smssend code */
    int i;

    if (Pv->Params != NULL)
    {
	for (i = 0; i < Pv->NbParams; i++)
	{
	    if (Pv->Params[i].Name != NULL)
		g_free(Pv->Params[i].Name);
	    if (Pv->Params[i].Value != NULL)
		g_free(Pv->Params[i].Value);
	}
	free(Pv->Params);
    }
    if (Pv->Label != NULL)
	g_free(Pv->Label);
    if (Pv->Filename != NULL)
	g_free(Pv->Filename);
    g_free(Pv);
}

/* -----------------------------------------------------------------------------------*/
PProvider_t CreateProviderFromFile(const char FileName[])
{
    /* Load a provider struct from script file: based on smssend code */
    FILE *fp;
    char Name[MAXLINE], Value[MAXLINE], Saf[MAXLINE], *Str;
    PProvider_t Pv;
    int NbParams;

    fp = fopen(FileName, "r");
    if (fp == NULL)
    {
	g_error(_("Error in provider loader : can't open file !\n"));
	return NULL;
    }

    Pv = g_new0(Provider_t, 1);
    NbParams = 0;

    while (SU_ParseConfig(fp, Name, sizeof(Name), Value, sizeof(Value)))
      {
	  if (g_strncasecmp(Name, "NbParams",9) == 0)
	    {
		Pv->NbParams = atoi(Value);
		Pv->Params = g_new0(Param_t, Pv->NbParams);
	    }
	  else if (Name[0] == '%')
	    {
		if (NbParams >= Pv->NbParams)
		  {
		      g_error(_("Error in provider loader : More than NbParams has been found\n"));
		      DeleteProvider(Pv);
		      return NULL;
		  }
		Pv->Params[NbParams].Name = g_strdup(Name + 1);
		if (Value[0] != 0)
		  {
		      if (Value[0] == ':')
			{
			    Pv->Params[NbParams].Help = g_strdup(SU_TrimLeft(Value + 1));
			}
		      strcpy(Saf, Value);
		      Str = strtok(Value, " ");
		      while (Str != NULL)
			{
			    if (g_strcasecmp(Str, "Hidden") == 0)
				Pv->Params[NbParams].Hidden = 1;
			    else if (g_strcasecmp(Str, "Convert") == 0)
				Pv->Params[NbParams].Convert = 1;
			    else if (g_strncasecmp(Str, "Size", 4) == 0)
				Pv->Params[NbParams].Size = atoi(Str + 5);
			    else if (Str[0] == ':')
			      {
				  Str = strchr(Saf, ':');
				  Pv->Params[NbParams].Help = g_strdup(SU_TrimLeft(Str + 1));
				  break;
			      }
			    else
				g_printerr(_("Unknown option in Param values for %s: %s\n"), FileName, Str);
			    Str = strtok(NULL, " ");
			}
		  }
		NbParams++;
	    }
      }

    fclose(fp);

    return Pv;
}

/* -----------------------------------------------------------------------------------*/
/*
 * Read a line from a descriptor.  Read the line one byte at a time,
 * looking for the newline. Works fine in nonblocking mode..here
 * we return when no more data can be read. 
 * We overwrite the newline with a null.
 * We return the number of characters up to, but not including,
 * the null (the same as strlen(3)).
 */
gint read_line(gint fd, gchar *ptr, gint maxlen) 
{
    gint n, rc;
    gchar c;
    gchar *str;

	str = ptr;
 
        for (n = 1; n < maxlen; n++) {
                if ( (rc = read(fd, &c, 1)) == 1) {
                        *ptr++ = c;
                        if (c == '\n') {
                                break;
                        }

                } else if (rc == 0) {
			/* EOF */
                        if (n == 1)
                                return(0);      /* EOF, no data read */
                        else
                                break;          /* EOF, some data was read */
                } else if (rc == -2) {
			/* timeout while reading string */
			return(-2);
		} else {
			/* nonblocking mode an nothing to read? */
			if (rc == -1 && errno == EAGAIN) {
				if (n == 1) 
					return(-1);
				else
					break;
			}	
                        return(-1);     /* error */
		}
        }

	/* terminate the string */
	*ptr = 0;
 
        /* strip of some trailing chars - yes..we need both levels */
        ptr--;
        if ((*ptr == '\n') || (*ptr == '\r') ) {
                *ptr = 0;
        }
        ptr--;
        if ((*ptr == '\n') || (*ptr == '\r') ) {
                *ptr = 0;
        }
 
	if (strlen(str) == 0) {
		/* if we read an empty string, but are NOT on EOF return 1 */
		return 1;
	} else {
        	return(strlen(str));
	}
}

/* -----------------------------------------------------------------------------------*/
/* get output of smssend (stderr) */
void smssend_stderr(gpointer data, gint source, GdkInputCondition cond)
{
    gint n;
    gint ret=0;
    gchar line[MAXLINE];
    PcmdParam_t sms_param = (PcmdParam_t) data;

    n = read_line(source, line, MAXLINE);

    /* finished? */
    if (n <= 0) 
    {
#ifdef DEBUG
fprintf(stderr,"smssend_stderr: end\n");
#endif
	gtk_input_remove(sms_param->Callback);
	gtk_input_remove(sms_param->Callback2);

	/* pick up return status of child */
	waitpid(sms_param->SmsId, &ret, WNOHANG|WUNTRACED);
    	/* attach a 'ignore' signal handler again to SIGCHLD to avoid zombies */
    	signal(SIGCHLD,SIG_IGN);

	sms_param->Done = WEXITSTATUS(ret);
#ifdef DEBUG
fprintf(stderr,"smssend_stderr: finished (%d)\n",sms_param->Done);
#endif

        gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.0);
	return;
     }
     gnome_appbar_set_status(GNOME_APPBAR(Appbar), line);
}

/* -----------------------------------------------------------------------------------*/
/* get output of smssend (stdout) */
void smssend_stdout(gpointer data, gint source, GdkInputCondition cond)
{
    gint n;
    gint ret=0;
    gchar line[MAXLINE];
    PcmdParam_t sms_param = (PcmdParam_t) data;
    static gint progress = 0;
    static gint dir = +1;

    n = read_line(source, line, MAXLINE);

    /* finished ? */
    if ((n <= 0) ||
	!(g_strncasecmp(line,MESSAGE1,strlen(MESSAGE1))) ||
	!(g_strncasecmp(line,MESSAGE2,strlen(MESSAGE2))) ||
	!(g_strncasecmp(line,MESSAGE3,strlen(MESSAGE3))) )
    {
#ifdef DEBUG
fprintf(stderr,"smssend_stdout: end\n");
#endif
	gtk_input_remove(sms_param->Callback);
	gtk_input_remove(sms_param->Callback2);

	/* pick up return status of child */
	waitpid(sms_param->SmsId, &ret, WNOHANG|WUNTRACED);
    	/* attach a 'ignore' signal handler again to SIGCHLD to avoid zombies */
    	signal(SIGCHLD,SIG_IGN);

	sms_param->Done = WEXITSTATUS(ret);
#ifdef DEBUG
fprintf(stderr,"smssend_stdout: finished (%d)\n",sms_param->Done);
#endif

	if (n > 0)
     	   gnome_appbar_set_status(GNOME_APPBAR(Appbar), line);
        gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 1.0);
	return;
     }
     
     if (progress + 5 > 100)
     {
	dir = -1;
        progress = 100;
     }
     if (progress <= 0)
     {
	dir = +1;
        progress = 0;
     }
     progress = progress + (5 * dir);
     gnome_appbar_set_status(GNOME_APPBAR(Appbar), line);
     gnome_appbar_set_progress(GNOME_APPBAR(Appbar), ((gfloat)progress)/100.0);
}

/* -----------------------------------------------------------------------------------*/

void CreateArgv( PcmdParam_t sms_param )
{
    int argc, i;
    char *value;

    argc = sms_param->Pv->NbParams;
    /* update structure */
    sms_param->argc = argc;
    sms_param->argv = g_new(gchar *, argc);;
    /* get all fields */
    for (i = 0; i < argc; i++)
    {
    	value = (char *)NULL;
	if (g_strcasecmp(sms_param->Pv->Params[i].Name, "Message") == 0)
	{
	    gchar **valuearray;
	    gchar *valuetmp;
	    /* change newline to space (use glib string arrays) */
	    GtkEditable *edit = GTK_EDITABLE(sms_param->Pv->Params[i].Widget);
	    valuetmp = gtk_editable_get_chars(edit, 0, -1);
#ifdef ERASE
	    gtk_editable_delete_text(edit,0,-1);
#endif
	    valuearray = g_strsplit(valuetmp,"\n",sms_param->Pv->Params[i].Size);
	    value = g_strjoinv(" ",valuearray);
	    sms_param->argv[i] = g_strdup_printf("%s", value);
	    g_strfreev(valuearray);
	    g_free(valuetmp);
	    g_free(value);
	}
	else
	{
	    value = gtk_entry_get_text(GTK_ENTRY(sms_param->Pv->Params[i].Widget));
#ifdef ERASE
	    gtk_editable_delete_text(GTK_EDITABLE(sms_param->Pv->Params[i].Widget),0,-1);
#endif
	    if (KeepPass && g_strncasecmp(sms_param->Pv->Params[i].Name, "Password",10) == 0)
	    {
		/* save password in preference file */
		gchar *config_file;
		gchar *path;
		gchar *temp;

		config_file=gnome_util_prepend_user_home(".smssend/smssendrc");
		path = g_strdup_printf("=%s=/Passwords/%s",config_file,sms_param->Pv->Label);
		if (value && strlen(value))
		{
		    temp = g_strdup(value);
		    crypt_password(temp);
		    gnome_config_set_string(path,temp);
		    g_free(temp);
		}
		g_free(path);
		g_free(config_file);
		gnome_config_sync();
	    }
	    sms_param->argv[i] = g_strdup(value);
	    g_free(value);
	}
    }
}

/* -----------------------------------------------------------------------------------*/
/* temporary defines to eliminate appbar updating */
#define Gnome_appbar_set_status(a,b) 
/*
	GDK_THREADS_ENTER(); \
	gnome_appbar_set_status(a,b); \
	GDK_THREADS_LEAVE(); 
*/
#define Gnome_appbar_set_progress(a,b) 
/*
	GDK_THREADS_ENTER(); \
	gnome_appbar_set_progress(a,b); \
	GDK_THREADS_LEAVE();
*/

static void *DoSms(void *arg)
{
    /* thread to call smssend in a fork->exec */
    PcmdParam_t sms_param = (PcmdParam_t) arg;
    gchar  proxy[P_LENGTH+8] = "";
    gchar  pass[P_LENGTH*2+3] = "";
    gchar  debug[10] = "";
    gchar  timeout[5] = "";
    gchar  *headerfile = (gchar *)NULL;
    gchar  *args[16];
    static int   retval=0;
    gint fd1[2], fd2[2], fd3[2];
    int i,ret;

    sms_param->SmsId = -1;
    pthread_detach (pthread_self ());

    /* attach a 'ignore' signal handler to SIGCHLD to avoid zombies */
    signal(SIGCHLD,SIG_IGN);

    /* check smssend presence */
    args[0] = gnome_is_program_in_path (SMSEXECNAME);
    if (!args[0])
    {
	  g_printerr(_("SmsSend not found!\n"));
	  Gnome_appbar_set_status(GNOME_APPBAR(Appbar), _("SmsSend not found!"));
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.0);
    	  pthread_exit((void *)(-1));
          /* Not reached  */
          return ((void *)(-1));
    }

    /* do the pipe-stuff */
    if (pipe(fd1) <0 || pipe(fd2) <0 || pipe(fd3) <0) 
    {
    	  g_free(args[0]);
	  g_printerr(_("SmsSend: pipe error!\n"));
	  Gnome_appbar_set_status(GNOME_APPBAR(Appbar), _("SmsSend: pipe error!"));
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.0);
    	  pthread_exit((void *)(-1));
          /* Not reached  */
          return ((void *)(-1));
    }

    if (UseProxy)
    {
	  /* update proxy args */
	  g_snprintf(proxy, P_LENGTH+8, "-p%s:%d", ProxyHost, ProxyPort);
	  if (strlen(ProxyUser))
	  {
		g_snprintf(pass, P_LENGTH*2+3, "-u%s:%s", ProxyUser, ProxyPass);
	  }
    }

    /* fork a smssend process */
    for (ret=Retries;ret > 0; ret--) { /* RETRIES loop */
    sms_param->SmsId = fork();
    if (sms_param->SmsId == 0)
    {
	  /* child process */
	  int argc=0, lastarg;

	  close(fd1[1]);
	  close(fd2[0]);
          close(fd3[0]);

          /* reroute stdin from child */
          if (fd1[0] != STDIN_FILENO) {
              if (dup2(fd1[0], STDIN_FILENO) != STDIN_FILENO) {
                  g_error(_("SmsSend: dup2 error on stdin\n"));
              }
              close(fd1[0]);
          }
          /* reroute stdout from child */
          if (fd2[1] != STDOUT_FILENO) {
              if (dup2(fd2[1], STDOUT_FILENO) != STDOUT_FILENO) {
                  g_error(_("SmsSend: dup2 error on stdout\n"));
              }
              close(fd2[1]);
          }
          /* reroute stderr from child */
          if (fd3[1] != STDERR_FILENO) {
              if (dup2(fd3[1], STDERR_FILENO) != STDERR_FILENO) {
                  g_error(_("SmsSend: dup2 error on stderr\n"));
              }
              close(fd3[1]);
          }
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.25);

	  /* create args array */
	  if (sms_param->Pv)
	  {
              args[1] = sms_param->Pv->Filename;
	      if (sms_param->argc <= 0)
              	CreateArgv(sms_param);
              argc = sms_param->argc;
    	      for (i=0;i<argc;i++)
		  args[2 + i] = sms_param->argv[i];

	  }
	  else if (sms_param->Help)
	  {
	      args[1] = (char *)arg_help;
	  }
	  else if (sms_param->Update)
	  {
	      args[1] = (char *)arg_update;
	  }

          lastarg = argc + 2;

	  if ((UseProxy)||(DebugLevel)||(TimeOut)||(HeaderFile))
		/* don't forget the '--' ! */
		args[lastarg++] = (char *)arg_minus;
	  if (UseProxy)
	  {
		args[lastarg++] = proxy;
		args[lastarg++] = pass;
	  }
	  if (DebugLevel)
	  {
	  	g_snprintf(debug, 10, "-d%d", DebugLevel);
		args[lastarg++] = debug;
	  }
	  if (TimeOut)
	  {
	  	g_snprintf(timeout, 5, "-t%d", TimeOut);
		args[lastarg++] = timeout;
	  }
	  if (HeaderFile)
	  {
		headerfile=g_strdup_printf("-h%s",HeaderFile);
		args[lastarg++] = headerfile;
	  }
	  args[lastarg] = NULL;
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.50);
	  /* call smssend */
	  execvp(SMSEXECNAME, args);
          /* if exec() returns, there is something wrong */
	  g_printerr(_("SmsSend exec failed!\n"));
	  Gnome_appbar_set_status(GNOME_APPBAR(Appbar), _("SmsSend exec failed!"));
	  /* free all allocated strings */
	  if (headerfile)
	      g_free(headerfile);
	  for (i = 0; i < argc; i++)
		g_free(args[2 + i]);
	  g_free(sms_param->argv);
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.0);
	  retval = -2;
          /* exit child. note the use of _exit() instead of exit() */
	  _exit(-2);
    	  /* Not reached  */
    }

    /* parent: wait for child process to end before exiting thread */
    if (sms_param->SmsId > 0)
    {
	  int maxsleep;
          /* set output to nonblocking - otherwise our callback would block */
          fcntl(fd2[0], F_SETFL, O_NONBLOCK);
          fcntl(fd3[0], F_SETFL, O_NONBLOCK);

          /* catch output of child */
	  sms_param->Callback = gdk_input_add(fd2[0], GDK_INPUT_READ,
                                  (GdkInputFunction) smssend_stdout,
                                  (gpointer)arg);
	  sms_param->Callback2 = gdk_input_add(fd3[0], GDK_INPUT_READ,
                                  (GdkInputFunction) smssend_stderr, 
                                  (gpointer)arg);
	  /* wait, doing nothing with a timeout */
	  i = 0;
          if (TimeOut) maxsleep = TimeOut * 2;
	  else         maxsleep = DEF_TIMEOUT * 2;
          while ((sms_param->Done == NOT_DONE) && (i++ < maxsleep))
	  {
		usleep(HALF_SECOND);
#ifdef DEBUG
fprintf(stderr,".");
#endif
	  }
	  if (sms_param->Done == NOT_DONE)
	  {
		/* 'twas a time out ! */
		gtk_input_remove(sms_param->Callback);
		gtk_input_remove(sms_param->Callback2);
		kill(sms_param->SmsId,SIGTERM);
#ifdef DEBUG
fprintf(stderr,"-");
#endif
	  }
	  close(fd2[0]);
	  close(fd3[0]);
	  close(fd1[1]);
          retval = sms_param->Done;
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 1.0);
    }
    else
    {
	  g_printerr(_("SmsSend fork failed!\n"));
	  Gnome_appbar_set_status(GNOME_APPBAR(Appbar), _("SmsSend fork failed!"));
	  Gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.0);
	  retval = -3;
    }
    if (retval == 0)
	    break;
    else
	  Gnome_appbar_set_status(GNOME_APPBAR(Appbar), _("SmsSend fork retried!"));
    /* End RETRIES loop */
    }

    /* free allocated memories */
    g_free(args[0]);
    if (headerfile)
        g_free(headerfile);
    if (sms_param->argc)
    {
    	for (i = 0; i < sms_param->argc; i++)
	    g_free(sms_param->argv[i]);
    	g_free(sms_param->argv);
    }
    g_free(sms_param);
#ifdef DEBUG
fprintf(stderr,"Exit thread\n");
#endif
    pthread_exit((void *)retval);
    /* Not reached  */
    return ((void *)retval);
}
#undef Gnome_appbar_set_status
#undef Gnome_appbar_set_progress

/* -----------------------------------------------------------------------------------*/
void DelayedSms(gpointer data, gpointer user_data)
{
    /* thread stuff */
    pthread_t      thread_sms;
    int            thread_status;
    PcmdParam_t sms_param = (PcmdParam_t) data;

    thread_status = pthread_create(&thread_sms, NULL, DoSms, (void *)sms_param);
    if (thread_status < 0)
       g_printerr(_("Error creating delayed SmsSend thread!\n"));

}

/* -----------------------------------------------------------------------------------*/
void LaunchSms(PProvider_t Pv, gboolean Help, gboolean Update)
{
    /* allocate parameters struct */
    PcmdParam_t sms_param = g_new(cmdParam_t, 1);
    /* thread stuff */
    pthread_t      thread_sms;
    int            thread_status;

    sms_param->Pv        = Pv;
    sms_param->Help      = Help;
    sms_param->Update    = Update;
    sms_param->Callback  = 0;
    sms_param->Callback2 = 0;
    sms_param->Done      = NOT_DONE;
    sms_param->argc      = 0;
    sms_param->argv      = NULL;

    gnome_appbar_set_status(GNOME_APPBAR(Appbar), "SmsSend ...");
    gnome_appbar_set_progress(GNOME_APPBAR(Appbar), 0.0);

    thread_status = pthread_create(&thread_sms, NULL, DoSms, (void *)sms_param);
    if (thread_status < 0)
       g_printerr(_("Error creating SmsSend thread!\n"));
}
