/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgnome/libgnome.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/xmlmemory.h>

#include <stdio.h>
#include <sys/stat.h>   
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>

#include "typedefs.h"
#include "comments.h"
#include "file-utils.h"

#define COMMENT_TAG  "Comment"
#define PLACE_TAG    "Place"
#define TIME_TAG     "Time"
#define NOTE_TAG     "Note"
#define KEYWORDS_TAG "Keywords"
#define FORMAT_TAG   "format"
#define FORMAT_VER   "1.0"


CommentData *
comment_data_new ()
{
	CommentData *data;

	data = g_new (CommentData, 1);

	data->place = NULL;
	data->time = 0;
	data->comment = NULL;
	data->keywords_n = 0;
	data->keywords = NULL;

	return data;
}


void
comment_data_free_keywords (CommentData *data)
{
	if (data->keywords != NULL) {
                int i;
                for (i = 0; i < data->keywords_n; i++)
                        g_free (data->keywords[i]);
                g_free (data->keywords);
		data->keywords = NULL;
		data->keywords_n = 0;
        }
}


void
comment_data_free (CommentData *data)
{
        if (data == NULL)
                return;

        if (data->place != NULL) 
                g_free (data->place);

        if (data->comment != NULL)
                g_free (data->comment);

	comment_data_free_keywords (data);

        g_free (data);
}


gchar *
comments_get_comment_filename (const gchar *source) 
{
	gchar *path;
	gchar *directory;
	const gchar *filename;

	if (!source) return NULL;

	directory = remove_level_from_path (source);
	filename = file_name_from_path (source);

	path = g_strconcat (g_get_home_dir(), 
			    "/", 
			    RC_COMMENTS_DIR, 
			    directory,
			    "/",
			    filename, 
			    COMMENT_EXT, 
			    NULL);

	g_free (directory);

	return path;
}


void
comment_copy (const gchar *src,
	      const gchar *dest)
{
	gchar *comment_src;
	gchar *comment_dest;

	comment_src = comments_get_comment_filename (src);
	comment_dest = comments_get_comment_filename (dest);

	file_copy (comment_src, comment_dest);

	g_free (comment_src);
	g_free (comment_dest);
}


void
comment_move (const gchar *src,
	      const gchar *dest)
{
	gchar *comment_src;
	gchar *comment_dest;

	comment_src = comments_get_comment_filename (src);
	comment_dest = comments_get_comment_filename (dest);

	file_move (comment_src, comment_dest);

	g_free (comment_src);
	g_free (comment_dest);
}


void
comment_delete (const gchar *filename)
{
	gchar *comment_name;

	comment_name = comments_get_comment_filename (filename);
	unlink (comment_name);
	g_free (comment_name);
}


/* This checks all files in ~/.gqview/comments/DIR and
 * if CLEAR_ALL is TRUE removes them all otherwise removes only those who 
 * have no source counterpart.
 */
void
comments_remove_old_comments (const gchar *dir,
			      gboolean recursive,
			      gboolean clear_all)
{
	visit_rc_directory (RC_COMMENTS_DIR,
			    COMMENT_EXT,
			    dir,
			    recursive,
			    clear_all);
}


/* ----- comments_remove_old_comments_async implememtation. ------ */

static void 
check_comment_file (gchar *real_file, 
		    gchar *rc_file, 
		    gpointer data)
{
	gboolean clear_all = GPOINTER_TO_INT (data);

	if (clear_all || ! path_is_file (real_file)) 
		if ((unlink (rc_file) < 0)) 
			g_warning ("Cannot delete %s\n", rc_file);
}


void
remove_comments_done (const GList *dir_list, 
		      gpointer data)
{
	gboolean clear_all = GPOINTER_TO_INT (data);
	const GList *scan;
	
	if (!clear_all)
		return;

	for (scan = dir_list; scan; scan = scan->next) {
		gchar *dir = scan->data;
		rmdir (dir);
	}
}


void
comments_remove_old_comments_async (const gchar *dir,
				    gboolean recursive,
				    gboolean clear_all)
{
	visit_rc_directory_async (RC_COMMENTS_DIR,
				  COMMENT_EXT,
				  dir,
				  recursive,
				  check_comment_file,
				  remove_comments_done,
				  GINT_TO_POINTER (clear_all));
}


static void
get_keywords (xmlChar *value, 
              CommentData *data)
{
        int n, i;
        xmlChar *c, *sub_str;
        gboolean done;

        if ((value == NULL) || (*value == 0)) 
                return;

        n = 1;
        for (c = value; *c != 0; c++)
                if (*c == ',')
                        n++;
        data->keywords_n = n;
        data->keywords = g_new0 (char*, n + 1);

        i = 0;
        sub_str = c = value;
        do {
                done = (*c == 0); 

                if (*c == ',')
                        *c = 0;
                if (*c == 0) {
                        data->keywords[i++] = g_strdup (sub_str);
                        sub_str = c + 1;
                }
                c++;
        } while (! done);
}


CommentData *
comments_load_comment (const gchar *filename)
{
	CommentData *data;
	gchar *comment_file;
	xmlDocPtr doc;
        xmlNodePtr root, node;
        xmlChar *value;

	comment_file = comments_get_comment_filename (filename);
	if (! path_is_file (comment_file)) {
		g_free (comment_file);
		return NULL;
	}

        doc = xmlParseFile (comment_file);
	if (doc == NULL) {
		g_free (comment_file);
		return NULL;
	}

	data = g_new (CommentData, 1);
        data->place = NULL;
        data->time = 0;
        data->comment = NULL;
        data->keywords = NULL;
        data->keywords_n = 0;

        root = xmlDocGetRootElement (doc);
        node = root->xmlChildrenNode;

	for (; node; node = node->next) {
                value = xmlNodeListGetString (doc, node->xmlChildrenNode, 1);

                if (strcmp (node->name, PLACE_TAG) == 0) {
                        data->place = value;
                        value = NULL;

                } else if (strcmp (node->name, NOTE_TAG) == 0) {
                        data->comment = value;
                        value = NULL;

                } else if (strcmp (node->name, KEYWORDS_TAG) == 0) {
                        get_keywords (value, data);

                } else if (strcmp (node->name, TIME_TAG) == 0) {
                        if (value != NULL)
				data->time = atol (value);
		
                } 
		
                if (value)
                        xmlFree (value);
        }

        xmlFreeDoc (doc);
	g_free (comment_file);

	return data;
}


void
comments_save_comment (const gchar *filename,
		       CommentData *data)
{
	xmlDocPtr doc;
        xmlNodePtr tree, subtree;
	gchar *comment_file;
	gchar *time_str;
	gchar *keywords_str;
	gchar *dest_dir;

	/* convert data to strings. */
	time_str = g_strdup_printf ("%ld", data->time);
	if (data->keywords_n > 0) {
		if (data->keywords_n == 1)
			keywords_str = g_strdup (data->keywords[0]);
		else
			keywords_str = g_strjoinv (",", data->keywords);
	} else
		keywords_str = g_strdup ("");

	/* create the xml tree. */
        doc = xmlNewDoc ("1.0");

        doc->xmlRootNode = xmlNewDocNode (doc, NULL, COMMENT_TAG, NULL); 
        xmlSetProp (doc->xmlRootNode, FORMAT_TAG, FORMAT_VER);

        tree = doc->xmlRootNode;
        subtree = xmlNewChild (tree, NULL, PLACE_TAG, data->place);
        subtree = xmlNewChild (tree, NULL, TIME_TAG, time_str);
        subtree = xmlNewChild (tree, NULL, NOTE_TAG, data->comment);
	subtree = xmlNewChild (tree, NULL, KEYWORDS_TAG, keywords_str);

	g_free (time_str);
	g_free (keywords_str);

        /* write to disk. */
	comment_file = comments_get_comment_filename (filename);
	dest_dir = remove_level_from_path (comment_file);
	if (ensure_dir_exists (dest_dir)) {
		xmlSetDocCompressMode (doc, 3);
		xmlSaveFile (comment_file, doc);
	}
	g_free (dest_dir);
	g_free (comment_file);

        xmlFreeDoc (doc);
}


gchar *
comments_get_comment_as_string (CommentData *data)
{
	gchar *s = NULL;
	gchar time_txt[50];
	gchar *comment_txt;
	gchar *place_txt;
	struct tm *tm;

	if (data == NULL) return NULL;
 
	if (data->time != 0) {
		tm = localtime (& data->time);
		strftime (time_txt, 50, "(%d %b %Y)", tm);
	} else
		time_txt[0] = 0;

	if (data->comment)
		comment_txt = g_strdup (data->comment);
	else
		comment_txt = g_strdup ("");

	if (data->place)
		place_txt = g_strdup (data->place);
	else
		place_txt = g_strdup ("");

	if ((data->comment == NULL) && (data->place == NULL) 
	    && (data->time == 0)) {
		if (data->keywords_n > 0) 
			s = NULL;
		else
			s = g_strdup (_("(No Comment)"));
	} else
		s = g_strconcat (comment_txt,
				 (data->comment && data->place) ? ", " : "",
				 place_txt, 
				 (data->comment || data->place) ? " " : "",
				 time_txt, NULL);

	g_free (comment_txt);
	g_free (place_txt);

	return s;
}

