/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <gtk/gtk.h>
#include "gthumb-init.h"
#include "bookmark-list.h"
#include "typedefs.h"
#include "pixbuf-utils.h"
#include "preferences.h"
#include "file-utils.h"

#include "icons/dir.xpm"
#include "icons/catalog.xpm"
#include "icons/catalog_search.xpm"


static gint
sort_by_name (GtkCList      *clist,
	      gconstpointer  ptr1,
	      gconstpointer  ptr2)
{
        const GtkCListRow *row1 = ptr1, *row2 = ptr2;
	const gchar *name1, *name2;

	name1 = file_name_from_path ((gchar*) row1->data);
	name2 = file_name_from_path ((gchar*) row2->data);

	if ((name1 == NULL) || (name2 == NULL))
		return 0;

	return strcasecmp (name1, name2);
}


BookmarkList *
bookmark_list_new ()
{
	BookmarkList * book_list;
	GtkWidget *clist;
	GtkWidget *scrolled;
	gint height;

	book_list = g_new (BookmarkList, 1);

	book_list->list = NULL;

	/* Create the widgets. */
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_ALWAYS);

	clist = gtk_clist_new (1);  
	gtk_clist_column_titles_passive (GTK_CLIST (clist)); 
	gtk_widget_set_usize (clist, CLIST_WIDTH_DEFAULT, CLIST_HEIGHT_DEFAULT);
	height = (GTK_WIDGET (clist)->style->font->ascent +
		  GTK_WIDGET (clist)->style->font->descent + CLIST_ROW_PAD); 
	gtk_clist_set_row_height (GTK_CLIST (clist), height);	
	gtk_clist_set_compare_func (GTK_CLIST (clist), sort_by_name);

	gtk_container_add (GTK_CONTAINER (scrolled), clist);
	book_list->clist = clist;
	
	book_list->root_widget = scrolled;

	return book_list;
}


void
bookmark_list_free (BookmarkList *book_list)
{
	g_return_if_fail (book_list != NULL);

	if (book_list->list != NULL)
		path_list_free (book_list->list);
	g_free (book_list);
}


const gchar *
get_menu_name (const gchar *label)
{
	gint offset = 0;

	if (pref_util_location_is_catalog (label)
	    || pref_util_location_is_search (label)) {
		gchar *rc_dir_prefix;

		rc_dir_prefix = g_strconcat (g_get_home_dir (),
					     "/",
					     RC_CATALOG_DIR,
					     "/",
					     NULL);

		offset = strlen (rc_dir_prefix);

		g_free (rc_dir_prefix);
	}

	return pref_util_remove_prefix (label) + offset;
}


void 
bookmark_list_set (BookmarkList *book_list,
		   GList *list)
{
#define PADDING "          "
	GList *scan;
	GtkWidget *clist;
	GdkPixmap *dir_pixmap, *search_pixmap, *catalog_pixmap;
	GdkBitmap *dir_bitmap, *search_bitmap, *catalog_bitmap;

	g_return_if_fail (book_list != NULL);

	clist = book_list->clist;
	gtk_clist_freeze (GTK_CLIST (clist));
	gtk_clist_clear (GTK_CLIST (clist));

	pixmap_from_xpm ((const char **) dir_xpm, &dir_pixmap, &dir_bitmap);
	pixmap_from_xpm ((const char **) catalog_xpm, &catalog_pixmap, &catalog_bitmap);
	pixmap_from_xpm ((const char **) catalog_search_xpm, &search_pixmap, &search_bitmap);

	if (book_list->list)
		path_list_free (book_list->list);

	/* copy the list of paths. */
	book_list->list = NULL;

	for (scan = list; scan; scan = scan->next)
		book_list->list = g_list_prepend (
			book_list->list, g_strdup ((gchar*) scan->data));

	book_list->list = g_list_reverse (book_list->list);

	/* insert the bookmarks in the clist. */
	for (scan = book_list->list; scan; scan = scan->next) {
		gchar *buf[2];  
		gint row;
		GdkPixmap *book_pixmap;
		GdkBitmap *book_bitmap;

		buf[0] = scan->data;
		buf[1] = NULL; 

		row = gtk_clist_append (GTK_CLIST (clist), buf);

		if (pref_util_location_is_catalog (scan->data)) {
			book_pixmap = catalog_pixmap;
			book_bitmap = catalog_bitmap;
		} else if (pref_util_location_is_search (scan->data)) {
			book_pixmap = search_pixmap;
			book_bitmap = search_bitmap;
		} else {
			book_pixmap = dir_pixmap;
			book_bitmap = dir_bitmap;
		}

		gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
				       bookmarks_get_menu_tip (preferences.bookmarks, scan->data),
				       10, 
				       book_pixmap, book_bitmap);

		gtk_clist_set_row_data (GTK_CLIST (clist), row, scan->data);
	}

	gtk_clist_sort (GTK_CLIST (clist));

	gdk_pixmap_unref (dir_pixmap);
	gdk_bitmap_unref (dir_bitmap);
	gdk_pixmap_unref (catalog_pixmap);
	gdk_bitmap_unref (catalog_bitmap);
	gdk_pixmap_unref (search_pixmap);
	gdk_bitmap_unref (search_bitmap);

	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 0, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 1, TRUE);
	gtk_clist_thaw (GTK_CLIST (clist));
#undef PADDING
}


gchar *
bookmark_list_path_from_row (BookmarkList *book_list,
			     gint row)
{
	gchar *name;

	name = gtk_clist_get_row_data (GTK_CLIST (book_list->clist), row);
	return g_strdup (name);
}
