/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeui/gnome-winhints.h>
#include "fullscreen.h"
#include "menu-callbacks.h"

#define HIDE_DELAY 2000

gint
image_key_press_cb (GtkWidget *widget, 
		    GdkEventKey *event,
		    gpointer data)
{
	FullScreen *fullscreen = data;
	GThumbWindow *window = fullscreen->related_win;
	ImageViewer *viewer = IMAGE_VIEWER (window->viewer);

	if (event->state & GDK_SHIFT_MASK) {
		switch (event->keyval) {
		case GDK_Down:
		case GDK_KP_Down:
			image_viewer_scroll_step_y (viewer, TRUE);
			return TRUE;

		case GDK_Up:
		case GDK_KP_Up:
			image_viewer_scroll_step_y (viewer, FALSE);
			return TRUE;

		case GDK_Right:
		case GDK_KP_Right:
			image_viewer_scroll_step_x (viewer, TRUE);
			return TRUE;

		case GDK_Left:
		case GDK_KP_Left:
			image_viewer_scroll_step_x (viewer, FALSE);
			return TRUE;
		}
	}

	if (event->state & GDK_CONTROL_MASK) {
		switch (event->keyval) {
		case GDK_Down:
		case GDK_KP_Down:
			image_viewer_scroll_page_y (viewer, TRUE);
			return TRUE;

		case GDK_Up:
		case GDK_KP_Up:
			image_viewer_scroll_page_y (viewer, FALSE);
			return TRUE;

		case GDK_Right:
		case GDK_KP_Right:
			image_viewer_scroll_page_x (viewer, TRUE);
			return TRUE;

		case GDK_Left:
		case GDK_KP_Left:
			image_viewer_scroll_page_x (viewer, FALSE);
			return TRUE;
		}
	}

	switch (event->keyval) {
		
		/* Exit fullscreen mode. */
	case GDK_Escape:
	case GDK_q:
	case GDK_v:
	case GDK_F11:
		fullscreen_stop (fullscreen);
		break;

		/* Next image. */
	case GDK_space:
	case GDK_KP_Right:
	case GDK_KP_Down:
	case GDK_KP_Page_Down:
	case GDK_Right:
	case GDK_Down:
	case GDK_Page_Down: 
		window_show_next_image (window);
		break;

		/* Previous image. */
	case GDK_b:
	case GDK_BackSpace:
	case GDK_Up:
	case GDK_Page_Up:
	case GDK_Left: 
	case GDK_KP_Up:
	case GDK_KP_Page_Up:
	case GDK_KP_Left:
		window_show_prev_image (window);
		break;

		/* Show first image. */
	case GDK_Home: 
	case GDK_KP_Home:
		window_show_first_image (window);
		break;
		
		/* Show last image. */
	case GDK_End: 
	case GDK_KP_End:
		window_show_last_image (window);
		break;

		/* Zoom in. */
	case GDK_plus:
	case GDK_equal:
	case GDK_KP_Add:
		image_viewer_zoom_in (viewer);
		break;

		/* Zoom out. */
	case GDK_minus:
	case GDK_KP_Subtract:
		image_viewer_zoom_out (viewer);
		break;

		/* Actual size. */
	case GDK_KP_Divide:
	case GDK_1:
	case GDK_z:
		image_viewer_set_zoom (viewer, 1.0);
		break;

		/* Zoom to fit. */
	case GDK_x:
		image_viewer_zoom_to_fit (viewer);
		break;

		/* Start/Stop Slideshow. */
	case GDK_s:
		slideshow_cb (widget, window);
		break;

		/* Toggle animation. */
	case GDK_g:
		toggle_animation_cb (widget, window);
		break;

		/* Step animation. */
	case GDK_j:
		step_animation_cb (widget, window);
		break;

#if 0
		/* Delete selection. */
	case GDK_Delete: 
	case GDK_KP_Delete:
		/* show the mouse pointer. */
		if (fullscreen->mouse_hide_id != 0)
			gtk_timeout_remove (fullscreen->mouse_hide_id);
		image_viewer_show_cursor (IMAGE_VIEWER (fullscreen->viewer));

		/* delete. */
		if (window->sidebar_content == DIR_LIST)
			delete_cb (NULL, window);
		else if (window->sidebar_content == CATALOG_LIST)
			remove_from_catalog_cb (NULL, window);
		break;
#endif

		/* Flip image. */
	case GDK_f:
	case GDK_F:
		flip_cb (widget, window);
		break;

		/* Rotate. */
	case GDK_r:
	case GDK_R:
	case GDK_bracketright:
		image_viewer_alter (viewer, ALTER_ROTATE_90);
		break;

	case GDK_bracketleft:
		image_viewer_alter (viewer, ALTER_ROTATE_90_CC);
		break;

		/* Mirror. */
	case GDK_m:
	case GDK_M:
		mirror_cb (widget, window);
		break;

	default:
		return TRUE;
	}

	return TRUE;
}


static gint 
hide_mouse_pointer_cb (gpointer data)
{
        FullScreen *fullscreen = data;

	image_viewer_hide_cursor (IMAGE_VIEWER (fullscreen->viewer));
	fullscreen->mouse_hide_id = 0;

        return FALSE;
}


static gint
fs_motion_notify_cb (GtkWidget *widget, 
		     GdkEventButton *bevent, 
		     gpointer data)
{
        FullScreen *fullscreen = data;

	gdk_window_get_pointer (widget->window, NULL, NULL, NULL);
	image_viewer_show_cursor (IMAGE_VIEWER (fullscreen->viewer));

	if (fullscreen->mouse_hide_id != 0)
		gtk_timeout_remove (fullscreen->mouse_hide_id);
	fullscreen->mouse_hide_id = gtk_timeout_add (HIDE_DELAY,
						     hide_mouse_pointer_cb,
						     fullscreen);

	return FALSE;
}


FullScreen *
fullscreen_new ()
{
	FullScreen *fullscreen;

	fullscreen = g_new (FullScreen, 1);
	fullscreen->motion_id = 0;
	fullscreen->mouse_hide_id = 0;
	fullscreen->window = gtk_window_new (GTK_WINDOW_POPUP);
	gtk_window_set_wmclass (GTK_WINDOW (fullscreen->window), "",
				"gthumb_fullscreen");

	gtk_widget_set_uposition (fullscreen->window, 0, 0);
  	gtk_widget_set_usize (fullscreen->window, 
			      gdk_screen_width (), 
			      gdk_screen_height ());

	fullscreen->related_win = NULL;
	fullscreen->viewer = NULL;

	gtk_signal_connect (GTK_OBJECT (fullscreen->window), 
			    "key_press_event",
			    (GtkSignalFunc) image_key_press_cb, 
			    fullscreen);

	gtk_signal_connect (GTK_OBJECT (fullscreen->window),
			    "motion_notify_event",
			    (GtkSignalFunc) fs_motion_notify_cb,
			    fullscreen);

	return fullscreen;
}


void
fullscreen_close (FullScreen *fullscreen)
{
	g_return_if_fail (fullscreen != NULL);

	gtk_widget_destroy (fullscreen->window);
	g_free (fullscreen);
}


void
fullscreen_start (FullScreen *fullscreen,
		  GThumbWindow *window)
{
	g_return_if_fail (fullscreen != NULL);

	if (fullscreen->related_win != NULL)
		return;

	/* ensure fullscreen menu item is active. */
	gtk_signal_handler_block_by_data (GTK_OBJECT (window->popmitem_fullscreen), window);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (window->popmitem_fullscreen), TRUE);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (window->popmitem_fullscreen), window);

	window->fullscreen = TRUE;
	fullscreen->related_win = window;
	fullscreen->viewer = window->viewer;
	gtk_widget_reparent (window->viewer, fullscreen->window);

	image_viewer_set_black_background (IMAGE_VIEWER (fullscreen->viewer),
					   TRUE);

	gtk_widget_show_all (fullscreen->window);

	/* capture keyboard events. */
	gdk_keyboard_grab (fullscreen->window->window, TRUE, GDK_CURRENT_TIME);
        gtk_grab_add (fullscreen->window);
        gtk_widget_grab_focus (fullscreen->window);

	/* auto hide mouse pointer staff. */
	fullscreen->mouse_hide_id = gtk_timeout_add (HIDE_DELAY,
						     hide_mouse_pointer_cb,
						     fullscreen);
}


void
fullscreen_stop (FullScreen *fullscreen)
{
	GThumbWindow *window;

	g_return_if_fail (fullscreen != NULL);

	if (fullscreen->related_win == NULL)
		return;

	window = fullscreen->related_win;

	/* auto hide mouse pointer staff. */
	image_viewer_show_cursor (IMAGE_VIEWER (fullscreen->viewer));
	if (fullscreen->mouse_hide_id)
		gtk_timeout_remove (fullscreen->mouse_hide_id);

	/* ensure fullscreen menu item is not active. */
	gtk_signal_handler_block_by_data (GTK_OBJECT (window->popmitem_fullscreen), window);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (window->popmitem_fullscreen), FALSE);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (window->popmitem_fullscreen), window);

	image_viewer_set_black_background (IMAGE_VIEWER (fullscreen->viewer),
					   FALSE);
	gtk_widget_reparent (fullscreen->viewer, window->viewer_container);
	window->fullscreen = FALSE;
	fullscreen->related_win = NULL;
	fullscreen->viewer = NULL;

	/* release keyboard focus. */ 
	gdk_keyboard_ungrab (GDK_CURRENT_TIME);
	gtk_grab_remove (fullscreen->window);

	gtk_widget_hide (fullscreen->window);
}
