/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>

#include "main.h"
#include "catalog.h"
#include "dlg-catalog.h"
#include "file-data.h"
#include "file-utils.h"
#include "image-list.h"
#include "window.h"


static GList *
get_selection_list (ImageList *ilist)
{
	gint row;
	FileData *fd;
	GList *scan, *file_list;

	file_list = NULL;
	for (scan = ilist->selection; scan; scan = scan->next) {
		row = GPOINTER_TO_INT (scan->data);
		fd = image_list_get_image_data (ilist, row);
		file_list = g_list_prepend (file_list, g_strdup (fd->path));
	}
	file_list = g_list_reverse (file_list);

	return file_list;
}


void
add_to_catalog_cb (GtkWidget *widget,
		   void *data)
{
	GThumbWindow *window = data;
	GList *list;
	ImageList *ilist;

	window = data;
	ilist = IMAGE_LIST (window->file_list->ilist);

	list = get_selection_list (ilist);
	dlg_add_to_catalog (window, list);

	/* the list is deallocated when the dialog is closed. */
}


void
move_to_catalog_dir_cb (GtkWidget *widget,
			void *data)
{
	GThumbWindow *window = data;
	gchar *catalog_path;
	GtkCList *clist;
	gint row;

	if (GTK_CLIST (window->catalog_list->clist)->selection == NULL)
		return;

	window = data;
	clist = GTK_CLIST (window->catalog_list->clist);
	row = GPOINTER_TO_INT (clist->selection->data);
	catalog_path = g_strdup (gtk_clist_get_row_data (clist, row));

	dlg_move_to_catalog_directory (window, catalog_path);

	/* the catalog_path is deallocated when the dialog is closed. */
}


static void
reply_cb (gint reply,
	  gpointer data)
{
	GThumbWindow *window = data;
	ImageList *ilist;
	GList *list, *scan;
	Catalog *catalog;

	if (reply != 0)
		return;
	
	ilist = IMAGE_LIST (window->file_list->ilist);
	list = get_selection_list (ilist);
	
	catalog = catalog_new ();
	catalog_load_from_disk (catalog, window->catalog_path);

	for (scan = list; scan; scan = scan->next) 
		catalog_remove_item (catalog, (gchar*) scan->data);

	catalog_write_to_disk (catalog);

	all_windows_notify_cat_files_deleted (window->catalog_path, list);
	path_list_free (list);
}


void
remove_from_catalog_cb (GtkWidget *widget,
			void *data)
{
	GThumbWindow *window = data;
	GtkWidget *dialog;

	if (! preferences.confirm_file_del) {
		reply_cb (0, window);
		return;
	}

	dialog = gnome_question_dialog_parented (_("The selected files will be deleted\n from the catalog, are you sure ?"),
						 reply_cb,
						 window,
						 GTK_WINDOW (window->app));
	gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
}


void
go_to_container_cb (GtkWidget *widget, 
		    void *data)
{
	GThumbWindow *window = data;
	ImageList *ilist;
	GList *list;
	gchar *path;

	ilist = IMAGE_LIST (window->file_list->ilist);
	list = get_selection_list (ilist);

	g_assert (list != NULL);

	path = remove_level_from_path (list->data);
	window_go_to_directory (window, path);
	g_free (path);
	path_list_free (list);
}


static void
string_cb (gchar *string, 
	   gpointer data)
{
	char *old_name, *new_name, *path;

	if (string == NULL)
		return;

	old_name = (gchar*) data;
	path = remove_level_from_path (old_name);
	new_name = g_strconcat (path,
				"/",
				string,
				CATALOG_EXT,
				NULL);

	rename (old_name, new_name);

	g_free (path);
	g_free (new_name);

	all_windows_update_catalog_list ();
}


void
rename_catalog_cb (GtkWidget *widget,
		   void *data)
{
	GtkWidget *dialog;
	GThumbWindow *window;
	GtkCList *clist;
	gint row;
	gchar *catalog_name, *name_only;

	window = data;
	clist = GTK_CLIST (window->catalog_list->clist);
	row = GPOINTER_TO_INT (clist->selection->data);
	catalog_name = g_strdup (gtk_clist_get_row_data (clist, row));
	name_only = remove_extension_from_path (file_name_from_path (catalog_name));

	dialog = gnome_request_dialog (FALSE,
				       _("Enter new catalog name : "),
				       name_only,
				       1024,
				       string_cb,
				       catalog_name,
				       GTK_WINDOW (window->app));
	gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);

	gnome_dialog_run (GNOME_DIALOG (dialog));

	g_free (catalog_name);
	g_free (name_only);
}


static void
catalog_reply_cb (gint reply,
		  gpointer data)
{
	GThumbWindow *window = data;
	gint row;
	gchar *path;

	if (reply != 0)
		return;

	row = GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data);
	path = catalog_list_path_from_row (window->catalog_list, row);

	if (path_is_dir (path)) {
		if (! dir_is_empty (path)) {
			GtkWidget *dialog;
			
			dialog = gnome_error_dialog_parented (
				_("Cannot delete non empty catalog directory"),
				GTK_WINDOW (window->app));
			gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		} else 
			delete_catalog_dir (path, FALSE);
	} else 
		delete_catalog (path);

	g_free (path);
	all_windows_update_catalog_list ();
}


void
delete_catalog_cb (GtkWidget *widget,
		   void *data)
{
	GThumbWindow *window = data;
	GtkWidget *dialog;
	gchar *message;
	gchar *path;
	gint row;

	if (! preferences.confirm_catalog_del) {
		catalog_reply_cb (0, window);
		return;
	}

	row = GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data);
	path = catalog_list_path_from_row (window->catalog_list, row);

	if (path_is_dir (path)) 
		message = g_strdup (_("The selected catalog directory will be deleted, are you sure ?"));
	else
		message = g_strdup (_("The selected catalog will be deleted, are you sure ?"));

	dialog = gnome_question_dialog_parented (message,
						 catalog_reply_cb,
						 window,
						 GTK_WINDOW (window->app));
	gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	g_free (path);
	g_free (message);
}
