/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#ifdef HAVE_PRINT

#include <gnome.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <libgnomeprint/gnome-print-master.h>
#include <libgnomeprint/gnome-print-master-preview.h>
#include <libgnomeprint/gnome-print-dialog.h>
#include <libgnomeprint/gnome-printer-dialog.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <glade/glade.h>

#include "window.h"
#include "image-viewer.h"
#include "pixbuf-utils.h"

#define GLADE_PRINT_FILE "gthumb_print.glade"

typedef struct {
	int ref_count;

	GtkWidget *canvas;
	GnomeCanvasItem *ci_image;

	/* Paper Informations. */
	const GnomePaper *paper;
	gdouble paper_width;
	gdouble paper_height;
	gdouble paper_lmargin;
	gdouble paper_rmargin;
	gdouble paper_tmargin;
	gdouble paper_bmargin;

	gboolean portrait;

	/* Image Informations. */
	GdkPixbuf *pixbuf;
	
	gchar *filename;
	
	double image_w, image_h;
	
	double scale_x, scale_y;
	double trans_x, trans_y;
	double zoom;

	double min_x, min_y;
	double max_x, max_y;

	gboolean use_colors;
} PrintInfo;


typedef struct {
	GThumbWindow *window;

	GladeXML *gui;
	GtkWidget *dialog;
	
	GtkWidget *btn_close;
	GtkWidget *btn_center;
	GtkWidget *btn_print;
	GtkWidget *hscale1;
	GtkWidget *combo_paper_size;
	GtkWidget *opt_orient;

	GtkAdjustment *adj;

	PrintInfo *pi;
} DialogData;


void 
print_info_unref (PrintInfo *pi)
{
	g_return_if_fail (pi != NULL);
	g_return_if_fail (pi->ref_count > 0);
	
	pi->ref_count--;

	if (pi->ref_count == 0) {
		if (pi->pixbuf)
			gdk_pixbuf_unref (pi->pixbuf);
		if (pi->filename)
			g_free (pi->filename);
		g_free (pi);
	}
}


static void
print_image (GnomePrintContext *pc,
	     PrintInfo *pi, 
	     gboolean border) 
{
	gdouble w, h;
	gdouble lmargin, rmargin, tmargin, bmargin;
	gint pw, ph, rs;
        guchar *p;

	w = pi->paper_width;
	h = pi->paper_height;
	lmargin = pi->paper_lmargin;
	rmargin = pi->paper_rmargin;
	bmargin = pi->paper_bmargin;
	tmargin = pi->paper_tmargin;

	gnome_print_beginpage (pc, "1");

	if (border) {
		gnome_print_gsave (pc);
		gnome_print_moveto (pc, lmargin, bmargin);
		gnome_print_lineto (pc, lmargin, h - tmargin);
		gnome_print_lineto (pc, w - rmargin, h - tmargin);
		gnome_print_lineto (pc, w - rmargin, bmargin);
		gnome_print_lineto (pc, lmargin, bmargin);
		gnome_print_stroke (pc);
	}

	if (! pi->portrait) { /* Landscape. */
		GdkPixbuf *new_pixbuf;
		gdouble temp;

		new_pixbuf = pixbuf_copy_rotate_90 (pi->pixbuf, TRUE);
		gdk_pixbuf_unref (pi->pixbuf);
		pi->pixbuf = new_pixbuf;

		pi->trans_y = (pi->paper_height - pi->scale_y) / pi->scale_y - pi->trans_y;

		temp = pi->trans_x;
		pi->trans_x = pi->trans_y;
		pi->trans_y = temp;

		temp = pi->scale_x;
		pi->scale_x = pi->scale_y;
		pi->scale_y = temp;
	}

	p = gdk_pixbuf_get_pixels (pi->pixbuf);
        pw = gdk_pixbuf_get_width (pi->pixbuf);
        ph = gdk_pixbuf_get_height (pi->pixbuf);
        rs = gdk_pixbuf_get_rowstride (pi->pixbuf);

	gnome_print_gsave (pc);
	gnome_print_scale (pc, pi->scale_x, pi->scale_y);
	gnome_print_translate (pc, pi->trans_x, pi->trans_y);
	if (pi->use_colors) {
		if (gdk_pixbuf_get_has_alpha (pi->pixbuf)) 
			gnome_print_rgbaimage (pc, p, pw, ph, rs);
		else 
			gnome_print_rgbimage  (pc, p, pw, ph, rs);
	} else
		gnome_print_grayimage (pc, p, pw, ph, rs);
        gnome_print_grestore (pc);

	gnome_print_showpage (pc);
}


/* called when the preview dialog is closed. */
static void
preview_destroy_cb (GtkWidget *widget, 
		    PrintInfo *pi)
{
	print_info_unref (pi);
}


void
print_dialog (PrintInfo *pi, 
	      gboolean border)
{
	GnomePrintDialog *gpd;
	GnomePrintMaster *gpm = NULL;
	gboolean do_preview;
	gint copies, collate;

	gpd = GNOME_PRINT_DIALOG (gnome_print_dialog_new (_("Print Image"), GNOME_PRINT_DIALOG_COPIES));

	do_preview = FALSE;
	switch (gnome_dialog_run (GNOME_DIALOG (gpd))) {
	case GNOME_PRINT_PRINT:
		break;

	case GNOME_PRINT_PREVIEW:
		do_preview = TRUE;
		break;

	case GNOME_PRINT_CANCEL:
		gnome_dialog_close (GNOME_DIALOG (gpd));
		return;
	}

	gpm = gnome_print_master_new ();
	gnome_print_dialog_get_copies (gpd, &copies, &collate);
	gnome_print_master_set_copies (gpm, copies, collate); 
	gnome_print_master_set_paper (gpm, pi->paper);
	gnome_print_master_set_printer (gpm, gnome_print_dialog_get_printer (gpd));
	gnome_dialog_close (GNOME_DIALOG (gpd));
	
	print_image (gnome_print_master_get_context (gpm), pi, border);
	gnome_print_master_close (gpm);

	if (do_preview) {
		GnomePrintMasterPreview *pmp;
		gnome_print_master_set_printer (gpm, NULL);
		pmp = gnome_print_master_preview_new (gpm, _("Print Preview"));
		gtk_signal_connect (GTK_OBJECT (pmp), "destroy",
				    (GtkSignalFunc) preview_destroy_cb,
				    pi);
		gtk_widget_show (GTK_WIDGET (pmp));

	} else 
		gnome_print_master_print (gpm);	
}



static void
add_simulated_page (GnomeCanvas *canvas)
{
	GnomeCanvasItem *i;
	double x1, y1, x2, y2, width, height;

	gnome_canvas_get_scroll_region (canvas, &x1, &y1, &x2, &y2);
	width = x2 - x1;
	height = y2 - y1;

	i = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (gnome_canvas_root (canvas)),
		gnome_canvas_rect_get_type (),
		"x1",   	 0.0,
		"y1",   	 0.0,
		"x2",   	 width,
		"y2",   	 height,
		"fill_color",    "white",
		"outline_color", "black",
		"width_pixels",  1,
		NULL);
	gnome_canvas_item_lower_to_bottom (i);
	i = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (gnome_canvas_root (canvas)),
		gnome_canvas_rect_get_type (),
		"x1",   	 3.0,
		"y1",   	 3.0,
		"x2",   	 width + 3.0,
		"y2",   	 height + 3.0,
		"fill_color",    "black",
		NULL);
	gnome_canvas_item_lower_to_bottom (i);
}


static GnomeCanvasItem * 
add_image (GnomeCanvas *canvas, 
	   const char *data, 
	   double x,
	   double y,
	   double scale_x,
	   double scale_y,
	   int width, 
	   int height, 
	   int rowstride, 
	   int bytes_per_pixel)
{
	GnomeCanvasGroup * group;
	GnomeCanvasItem *item;
	ArtPixBuf *pixbuf;
	int size, bpp;
	void *dup;
	
	/*
	 * We do convert gray scale images to RGB
	 */

	if (bytes_per_pixel == 1)
		bpp = 3;
	else
		bpp = bytes_per_pixel;
	
	size = width * height * bpp;
	dup = art_alloc (size);
	if (!dup)
		return NULL;

	if (bytes_per_pixel == 3){
		memcpy (dup, data, size);
		pixbuf = art_pixbuf_new_rgb (dup, width, height, rowstride);
	} else if (bytes_per_pixel == 4){
		memcpy (dup, data, size);
		pixbuf = art_pixbuf_new_rgba (dup, width, height, rowstride);
	} else if (bytes_per_pixel == 1){
		const char *source;
		char *target;
		int  ix, iy;

		source = data;
		target = dup;

		for (iy = 0; iy < height; iy++){
			for (ix = 0; ix < width; ix++){
				*target++ = *source;
				*target++ = *source;
				*target++ = *source;
				source++;
			}
		}
		pixbuf = art_pixbuf_new_rgb (dup, width, height, rowstride * 3);
	} else
		return NULL;
	
	group = GNOME_CANVAS_GROUP (gnome_canvas_root (canvas));
	
	item = gnome_canvas_item_new (group,
				      gnome_canvas_image_get_type (),
				      "pixbuf", pixbuf,
				      "x",      x,
				      "y",      y,
				      "width",  (gdouble) scale_x,
				      "height", (gdouble) scale_y,
				      "anchor", GTK_ANCHOR_NW,
				      NULL);
	
	return item;
}


static gint
item_event (GnomeCanvasItem *item, 
	    GdkEvent *event, 
	    PrintInfo *pi)
{
	static double start_x, start_y;
	static double img_start_x, img_start_y;
	static int dragging;
	double x, y;
	GdkCursor *fleur;

	x = event->button.x;
	y = event->button.y;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		switch (event->button.button) {
		case 1:
			start_x = x;
			start_y = y;
			gtk_object_get (GTK_OBJECT (item),
					"x", &img_start_x,
					"y", &img_start_y,
					NULL);

			fleur = gdk_cursor_new (GDK_FLEUR);
			gnome_canvas_item_grab (item,
						(GDK_POINTER_MOTION_MASK 
						 | GDK_BUTTON_RELEASE_MASK),
						fleur,
						event->button.time);
			gdk_cursor_destroy (fleur);
			dragging = TRUE;
			break;

		default:
			break;
		}
		break;

	case GDK_MOTION_NOTIFY:
		if (dragging && (event->motion.state & GDK_BUTTON1_MASK)) {
			double x1, y1;

			x1 = img_start_x + (x - start_x);
			y1 = img_start_y + (y - start_y);

			x1 = MAX (x1, pi->min_x);
			x1 = MIN (x1, pi->max_x - pi->image_w * pi->zoom);
			y1 = MAX (y1, pi->min_y);
			y1 = MIN (y1, pi->max_y - pi->image_h * pi->zoom);

			gnome_canvas_item_set (item,
					       "x", x1,
					       "y", y1,
					       NULL);
		}
		break;

	case GDK_BUTTON_RELEASE:
		gnome_canvas_item_ungrab (item, event->button.time);
		dragging = FALSE;
		break;

	default:
		break;
	}

	return FALSE;
}


static void
add_image_preview (PrintInfo *pi, 
		   gboolean border)
{
	gdouble w, h;
	gdouble lmargin, rmargin, tmargin, bmargin;
	gdouble iw, ih;
	gdouble max_w, max_h;
	gint pw, ph, rs, bpp;
        guchar *p;

	w = pi->paper_width;
	h = pi->paper_height;
	lmargin = pi->paper_lmargin;
	rmargin = pi->paper_rmargin;
	bmargin = pi->paper_bmargin;
	tmargin = pi->paper_tmargin;
	iw = (double) gdk_pixbuf_get_width (pi->pixbuf); 
	ih = (double) gdk_pixbuf_get_height (pi->pixbuf);

	/* scale if image does not fit on the window. */
	max_w = w - lmargin - rmargin;
	max_h = h - bmargin - tmargin;
	
	if (max_w / iw < max_h / ih) {
		ih = ih * (max_w / iw);
		iw = max_w;
	} else {
		iw = iw * (max_h / ih);
		ih = max_h;
	}

	pi->image_w = iw;
	pi->image_h = ih;

	/* center & check bounds. */
	pi->trans_x = MAX ((w - iw) / 2, lmargin);
	pi->trans_y = MAX ((h - ih) / 2, bmargin);
	
	pi->zoom = 1.0;

	pi->min_x = lmargin;
	pi->min_y = tmargin;
	pi->max_x = lmargin + max_w;
	pi->max_y = tmargin + max_h;

	if (border) 
		gnome_canvas_item_new (
			gnome_canvas_root ( GNOME_CANVAS (pi->canvas)),
			gnome_canvas_rect_get_type (),
			"x1",   	 lmargin,
			"y1",   	 tmargin,
			"x2",   	 lmargin + max_w,
			"y2",   	 tmargin + max_h,
			"outline_color", "gray",
			"width_pixels",  1,
			NULL);
	

	p = gdk_pixbuf_get_pixels (pi->pixbuf);
        pw = gdk_pixbuf_get_width (pi->pixbuf);
        ph = gdk_pixbuf_get_height (pi->pixbuf);
        rs = gdk_pixbuf_get_rowstride (pi->pixbuf);

	if (pi->use_colors) {
		if (gdk_pixbuf_get_has_alpha (pi->pixbuf)) 
			bpp = 4;
		else 
			bpp = 3;
	} else
		bpp = 1;

	pi->ci_image = add_image (GNOME_CANVAS (pi->canvas), 
				  p, 
				  pi->trans_x, pi->trans_y, 
				  iw, ih,
				  pw, ph, rs, bpp);
	if (pi->ci_image == NULL)
		g_error ("Cannot create image preview\n");

	gtk_signal_connect (GTK_OBJECT (pi->ci_image), "event",
			    (GtkSignalFunc) item_event,
			    pi);
}


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget *widget, 
	    DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	print_info_unref (data->pi);
	g_free (data);
}


/* called when the center button is clicked. */
static void
center_cb (GtkWidget *widget, 
	   DialogData *data)
{
	PrintInfo *pi = data->pi;
	gdouble x1, y1, w, h;

	w = pi->max_x - pi->min_x;
	h = pi->max_y - pi->min_y;
	x1 = pi->min_x + (w - pi->image_w * pi->zoom) / 2;
	y1 = pi->min_y + (h - pi->image_h * pi->zoom) / 2;

	gnome_canvas_item_set (pi->ci_image,
			       "x", x1,
			       "y", y1,
			       NULL);
}


static void 
zoom_changed (GtkAdjustment *adj,
	      DialogData *data)
{
	PrintInfo *pi = data->pi;
	gdouble w, h;
	gdouble x1, y1;

	pi->zoom = adj->value / 100.0;
	w = pi->image_w * pi->zoom;
	h = pi->image_h * pi->zoom;

	gnome_canvas_item_set (pi->ci_image,
			       "width", w,
			       "height", h,
			       NULL);

	/* keep image within page borders. */
	gtk_object_get (GTK_OBJECT (pi->ci_image),
			"x", &x1,
			"y", &y1,
			NULL);
	x1 = MAX (x1, pi->min_x);
	x1 = MIN (x1, pi->max_x - pi->image_w * pi->zoom);
	y1 = MAX (y1, pi->min_y);
	y1 = MIN (y1, pi->max_y - pi->image_h * pi->zoom);

	gnome_canvas_item_set (pi->ci_image,
			       "x", x1,
			       "y", y1,
			       NULL);
}


/* called when the print button is clicked. */
static void 
print_cb (GtkWidget *widget,
	  DialogData *data)
{
	PrintInfo *pi = data->pi;
	gdouble x, y;

	gtk_object_get (GTK_OBJECT (pi->ci_image),
			"x", &x,
			"y", &y,
			NULL);
	pi->scale_x = pi->image_w * pi->zoom;
	pi->scale_y = pi->image_h * pi->zoom;
	pi->trans_x = x / pi->scale_x;
	pi->trans_y = (pi->paper_height - pi->scale_y - y) / pi->scale_y;

	/* pi is used by print_dialog, so we must add a reference. */
	pi->ref_count++;
	gtk_widget_destroy (data->dialog); 
	print_dialog (pi, FALSE);
}


static void
clear_canvas (GnomeCanvasGroup *group)
{
	GList *list = group->item_list;

	while (list) {
		gtk_object_destroy (GTK_OBJECT (list->data));
		list = group->item_list;
	}
}


static void
page_update (DialogData *data)
{
	PrintInfo *pi = data->pi;

	/* Clear canvas. */
	clear_canvas (GNOME_CANVAS_GROUP (GNOME_CANVAS (pi->canvas)->root));

	if (pi->portrait) {
		pi->paper_width = gnome_paper_pswidth (pi->paper);
		pi->paper_height = gnome_paper_psheight (pi->paper);
		pi->paper_lmargin = gnome_paper_lmargin (pi->paper);
		pi->paper_tmargin = gnome_paper_tmargin (pi->paper);
		pi->paper_rmargin = gnome_paper_rmargin (pi->paper);
		pi->paper_bmargin = gnome_paper_bmargin (pi->paper);
	} else {
		pi->paper_width = gnome_paper_psheight (pi->paper);
		pi->paper_height = gnome_paper_pswidth (pi->paper);
		pi->paper_lmargin = gnome_paper_tmargin (pi->paper);
		pi->paper_tmargin = gnome_paper_rmargin (pi->paper);
		pi->paper_rmargin = gnome_paper_bmargin (pi->paper);
		pi->paper_bmargin = gnome_paper_lmargin (pi->paper);
	}
	gnome_canvas_set_scroll_region (GNOME_CANVAS (pi->canvas), 
					0, 0,
					pi->paper_width, pi->paper_height);

	add_simulated_page (GNOME_CANVAS (pi->canvas));
	add_image_preview (pi, TRUE);

	/* reset zoom to 100%. */
	gtk_signal_handler_block_by_data (GTK_OBJECT (data->adj), data);
	gtk_adjustment_set_value (data->adj, 100.0);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (data->adj), data);
}


static void
paper_size_changed (GtkEntry *entry, 
		    DialogData *data)
{
	char *text;

	text = gtk_entry_get_text (entry);
	data->pi->paper = gnome_paper_with_name (text);
	page_update (data);
}


static void
orient_port_cb (GtkWidget *widget, 
		DialogData *data)
{
	PrintInfo *pi = data->pi;

	if (pi->portrait) return;

	pi->portrait = TRUE;
	page_update (data);
}


static void
orient_land_cb (GtkWidget *widget, 
		DialogData *data)
{
	PrintInfo *pi = data->pi;

	if (! pi->portrait) return;

	pi->portrait = FALSE;
	page_update (data);
}


void
print_image_cb (GtkWidget *widget, 
		void *wdata)
{
	GThumbWindow *window = wdata;
	PrintInfo *pi;
	DialogData *data;
	GtkWidget *menu, *mitem;
	gint i;
	const static GtkSignalFunc orient_fn[] = {
		GTK_SIGNAL_FUNC (orient_port_cb),
		GTK_SIGNAL_FUNC (orient_land_cb)};

	if (image_viewer_is_void (IMAGE_VIEWER (window->viewer))) return;

	pi = g_new0 (PrintInfo, 1);
	pi->ref_count = 1;
	pi->zoom = 1.0;
	pi->filename = image_viewer_get_image_filename (IMAGE_VIEWER (window->viewer));
	if (pi->filename == NULL) {
		g_warning ("Cannot get image filename");
		print_info_unref (pi);
		return;
	}

	pi->paper = gnome_paper_with_name (gnome_paper_name_default ());
	if (pi->paper == NULL) {
		g_warning ("Cannot get paper %s", gnome_paper_name_default ());
		print_info_unref (pi);
		return;
	}

	pi->pixbuf = gdk_pixbuf_new_from_file (pi->filename);
	if (pi->pixbuf == NULL) {
		g_warning ("Cannot load image %s", pi->filename);
		print_info_unref (pi);
		return;
	}

	pi->image_w = (gdouble) gdk_pixbuf_get_width (pi->pixbuf);
	pi->image_h = (gdouble) gdk_pixbuf_get_height (pi->pixbuf);
	pi->portrait = TRUE;
	pi->use_colors = TRUE;

	data = g_new (DialogData, 1);
	data->window = window;
	data->pi = pi;
	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_PRINT_FILE, NULL);
        if (! data->gui) {
                g_warning ("Cannot find " GLADE_PRINT_FILE "\n");
		print_info_unref (pi);
		g_free (data);
                return;
        }

	/* Get the widgets. */
	data->dialog = glade_xml_get_widget (data->gui, "page_setup_dialog");
	data->btn_close = glade_xml_get_widget (data->gui, "btn_close");
	data->hscale1 = glade_xml_get_widget (data->gui, "hscale1");
	data->btn_center = glade_xml_get_widget (data->gui, "btn_center");
	data->btn_print = glade_xml_get_widget (data->gui, "btn_print");
	data->combo_paper_size = glade_xml_get_widget (data->gui, "combo_paper_size");
	data->opt_orient = glade_xml_get_widget (data->gui, "opt_orientation");
	pi->canvas = glade_xml_get_widget (data->gui, "canvas");

	/* Set widgets data. */
	gtk_combo_set_popdown_strings (GTK_COMBO (data->combo_paper_size), 
				       gnome_paper_name_list ());
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (data->combo_paper_size)->entry),
			    gnome_paper_name (pi->paper));

	menu = gtk_menu_new ();
	for (i = 0; i < 2; i++) {
		mitem = gtk_menu_item_new_with_label ((i == 0) ? _("Portrait") : _("Landscape"));
		gtk_menu_append (GTK_MENU (menu), mitem);
		gtk_signal_connect (GTK_OBJECT (mitem), "activate",
				    GTK_SIGNAL_FUNC (orient_fn[i]),
				    data);
		gtk_widget_show (mitem);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (data->opt_orient), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_orient), 0);

	/* Set the signals handlers. */
	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) destroy_cb,
			    data);
	gtk_signal_connect_object (GTK_OBJECT (data->btn_close), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));
	gtk_signal_connect (GTK_OBJECT (data->btn_center), "clicked",
			    GTK_SIGNAL_FUNC (center_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->btn_print), "clicked",
			    GTK_SIGNAL_FUNC (print_cb),
			    data);

	data->adj = gtk_range_get_adjustment (GTK_RANGE (data->hscale1));
	gtk_signal_connect (GTK_OBJECT (data->adj), "value_changed",
			    (GtkSignalFunc) zoom_changed,
			    data);

	gtk_signal_connect (GTK_OBJECT (GTK_COMBO (data->combo_paper_size)->entry), "changed",
			    paper_size_changed, 
			    data);

	/**/
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), 
				      GTK_WINDOW (window->app));

	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (pi->canvas), 0.25);
	page_update (data);

	gtk_widget_show_all (data->dialog);
}

#endif /* HAVE_PRINT */
