/* lighthouseblue_theme_draw.c -- implementation of LighthouseBlue style
 
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Author: Jan Rosczak <jan.rosczak@web.de>  
*/

#include <gmodule.h>
#include <gtk/gtk.h>
#include <string.h>

#include "lighthouseblue.h"


#define DETAIL(xx)   ((detail) && (!strcmp(xx, detail)))


static GtkStyleClass *parent_class = NULL;


static gboolean
sanitize_size (GdkWindow * window, gint * width, gint * height)
{
  gboolean set_bg = FALSE;

  if ((*width == -1) && (*height == -1))
    {
      /*set_bg = GDK_IS_WINDOW (window); */
      gdk_window_get_size (window, width, height);
    }
  else if (*width == -1)
    gdk_window_get_size (window, width, NULL);
  else if (*height == -1)
    gdk_window_get_size (window, NULL, height);

  return set_bg;
}

static void
theme_draw_rectangle (GdkDrawable * drawable,
		      GdkGC * gc,
		      gint filled, gint x, gint y, gint width, gint height)
{
  gdk_draw_line (drawable, gc, x + 1, y, x + width - 1, y);
  gdk_draw_line (drawable, gc, x + 1, y + height, x + width - 1, y + height);

  gdk_draw_line (drawable, gc, x, y + 1, x, y + height - 1);
  gdk_draw_line (drawable, gc, x + width, y + 1, x + width, y + height - 1);
}


static void
draw_hline (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget, gchar * detail, gint x1, gint x2, gint y)
{
  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (area)
    {
      gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
    }

  if (DETAIL ("label"))
    {
      gdk_draw_line (window, style->fg_gc[state_type], x1, y, x2, y);
    }
  else
    {
      gdk_draw_line (window, style->dark_gc[state_type], x1, y, x2, y);
      if (DETAIL ("menuitem"))
	{
	  gdk_draw_line (window, style->light_gc[state_type], x1, y + 1, x2,
			 y + 1);
	}
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
    }
}

static void
draw_vline (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget, gchar * detail, gint y1, gint y2, gint x)
{
  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (area)
    {
      gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
    }

  gdk_draw_line (window, style->dark_gc[state_type], x, y1, x, y2);

  if (area)
    {
      gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
    }
}

static void
draw_shadow (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *outer_gc;
  GdkGC *bg_gc;

  gint xthickness = 1;
  gint ythickness = 1;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  sanitize_size (window, &width, &height);

  if (DETAIL ("button") || DETAIL ("optionmenu") || DETAIL ("hscrollbar")
      || DETAIL ("vscrollbar"))
    {
      outer_gc = engine_data->shade_gc[6];
    }
  else
    {
      outer_gc = engine_data->shade_gc[5];
    }

  if (GTK_IS_WIDGET (widget) && widget->parent != NULL)
    {
      bg_gc = widget->parent->style->bg_gc[GTK_STATE_NORMAL];
    }
  else
    {
      bg_gc = style->bg_gc[GTK_STATE_NORMAL];
    }

  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      return;
    case GTK_SHADOW_IN:
    case GTK_SHADOW_ETCHED_IN:
      gc1 = style->light_gc[state_type];	/* white_gc ? */
      gc2 = engine_data->shade_gc[2];
      break;
    case GTK_SHADOW_OUT:
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = engine_data->shade_gc[2];
      gc2 = style->light_gc[state_type];	/* white_gc ? */
      break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
      gdk_gc_set_clip_rectangle (bg_gc, area);

      if (shadow_type == GTK_SHADOW_IN || shadow_type == GTK_SHADOW_OUT)
	{
	  gdk_gc_set_clip_rectangle (style->black_gc, area);
	  gdk_gc_set_clip_rectangle (outer_gc, area);
	}
    }

  if (DETAIL ("buttondefault"))
    {
      theme_draw_rectangle (window, outer_gc, FALSE, x, y, width - 1,
			    height - 1);
      gdk_draw_point (window, bg_gc, x, y);
      gdk_draw_point (window, bg_gc, x + width - 1, y);
      gdk_draw_point (window, bg_gc, x, y + height - 1);
      gdk_draw_point (window, bg_gc, x + width - 1, y + height - 1);
      return;
    }

  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      return;

    case GTK_SHADOW_IN:
      theme_draw_rectangle (window, outer_gc, FALSE, x, y, width - 1,
			    height - 1);
      gdk_draw_point (window, bg_gc, x, y);
      gdk_draw_point (window, bg_gc, x + width - 1, y);
      gdk_draw_point (window, bg_gc, x, y + height - 1);
      gdk_draw_point (window, bg_gc, x + width - 1, y + height - 1);

      /* Light around right and bottom edge */
      if (ythickness > 0)
	{
	  gdk_draw_line (window, gc1, x + 1, y + height - 2, x + width - 2,
			 y + height - 2);
	}

      if (xthickness > 0)
	{
	  gdk_draw_line (window, gc1, x + width - 2, y + 1, x + width - 2,
			 y + height - 2);
	}

      /* Dark around left and top */
      if (ythickness > 0)
	{
	  gdk_draw_line (window, gc2, x + 1, y + 1, x + width - 3, y + 1);
	}

      if (xthickness > 0)
	{
	  gdk_draw_line (window, gc2, x + 1, y + 1, x + 1, y + height - 3);
	}
      break;

    case GTK_SHADOW_OUT:
      theme_draw_rectangle (window, outer_gc, FALSE, x, y, width - 1,
			    height - 1);
      gdk_draw_point (window, bg_gc, x, y);
      gdk_draw_point (window, bg_gc, x + width - 1, y);
      gdk_draw_point (window, bg_gc, x, y + height - 1);
      gdk_draw_point (window, bg_gc, x + width - 1, y + height - 1);

      /* Dark around right and bottom edge */
      if (ythickness > 0)
	{
	  gdk_draw_line (window, gc1, x + 2, y + height - 2, x + width - 2,
			 y + height - 2);
	}

      if (xthickness > 0)
	{
	  gdk_draw_line (window, gc1, x + width - 2, y + 2, x + width - 2,
			 y + height - 2);
	}

      /* Light around top and left */
      if (ythickness > 0)
	{
	  gdk_draw_line (window, gc2, x + 1, y + 1, x + width - 2, y + 1);
	}

      if (xthickness > 0)
	{
	  gdk_draw_line (window, gc2, x + 1, y + 1, x + 1, y + height - 2);
	}
      break;

    case GTK_SHADOW_ETCHED_IN:
    case GTK_SHADOW_ETCHED_OUT:
      gdk_draw_line (window, gc1, x + 1, y + height - 1, x + width - 3,
		     y + height - 1);
      gdk_draw_line (window, gc1, x + width - 1, y + 1, x + width - 1,
		     y + height - 3);

      gdk_draw_line (window, gc2, x + 1, y, x + width - 3, y);
      gdk_draw_line (window, gc2, x, y + 1, x, y + height - 3);

      gdk_draw_line (window, gc1, x + 1, y + 1, x + width - 2, y + 1);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 2);

      gdk_draw_line (window, gc2, x + 1, y + height - 2, x + width - 3,
		     y + height - 2);
      gdk_draw_line (window, gc2, x + width - 2, y + 1, x + width - 2,
		     y + height - 3);
      break;
    }

  if (DETAIL ("button") && GTK_WIDGET_HAS_DEFAULT (widget))	/* && !GTK_WIDGET_HAS_FOCUS (widget)) */
    {
      GdkGC *gc = style->base_gc[GTK_STATE_SELECTED];	/*(LIGHTHOUSEBLUE_STYLE (style))->shade_gc[6]; */
      theme_draw_rectangle (window, gc, FALSE, x + 3, y + 3, width - 7,
			    height - 7);
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, NULL);
      gdk_gc_set_clip_rectangle (gc2, NULL);
      gdk_gc_set_clip_rectangle (bg_gc, NULL);

      if (shadow_type == GTK_SHADOW_IN || shadow_type == GTK_SHADOW_OUT)
	{
	  gdk_gc_set_clip_rectangle (style->black_gc, NULL);
	  gdk_gc_set_clip_rectangle (outer_gc, NULL);
	}
    }
}

static void
draw_varrow (GdkWindow * window,
	     GdkGC * gc,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkArrowType arrow_type, gint x, gint y, gint width, gint height)
{
  gint steps, extra;
  gint y_start, y_increment;
  gint i;

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc, area);
    }

  width = width + width % 2 - 1;	/* Force odd */
  steps = 1 + width / 2;
  extra = height - steps;

  if (arrow_type == GTK_ARROW_DOWN)
    {
      y_start = y;
      y_increment = 1;
    }
  else
    {
      y_start = y + height - 1;
      y_increment = -1;
    }

  for (i = 0; i < extra; i++)
    {
      gdk_draw_line (window, gc, x, y_start + i * y_increment, x + width - 1,
		     y_start + i * y_increment);
    }
  for (; i < height; i++)
    {
      gdk_draw_line (window, gc, x + (i - extra), y_start + i * y_increment,
		     x + width - (i - extra) - 1, y_start + i * y_increment);
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc, NULL);
    }
}

static void
draw_harrow (GdkWindow * window,
	     GdkGC * gc,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkArrowType arrow_type, gint x, gint y, gint width, gint height)
{
  gint steps, extra;
  gint x_start, x_increment;
  gint i;

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc, area);
    }

  height = height + height % 2 - 1;	/* Force odd */
  steps = 1 + height / 2;
  extra = width - steps;

  if (arrow_type == GTK_ARROW_RIGHT)
    {
      x_start = x;
      x_increment = 1;
    }
  else
    {
      x_start = x + width - 1;
      x_increment = -1;
    }

  for (i = 0; i < extra; i++)
    {
      gdk_draw_line (window, gc, x_start + i * x_increment, y,
		     x_start + i * x_increment, y + height - 1);
    }
  for (; i < width; i++)
    {
      gdk_draw_line (window, gc, x_start + i * x_increment, y + (i - extra),
		     x_start + i * x_increment, y + height - (i - extra) - 1);
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc, NULL);
    }
}

static void
draw_arrow (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    GtkArrowType arrow_type,
	    gboolean fill, gint x, gint y, gint width, gint height)
{
  gint i, j;
  gint orig_width, orig_x;
  GdkGC *gc;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (DETAIL ("menuitem") && state_type == GTK_STATE_INSENSITIVE)
    {
      return;
    }

  sanitize_size (window, &width, &height);

  switch (state_type)
    {
    case GTK_STATE_INSENSITIVE:
      gc = style->white_gc;
      break;
    case GTK_STATE_NORMAL:
      gc = engine_data->shade_gc[7];
      break;
    default:
      gc = style->fg_gc[state_type];
    }

  if (DETAIL ("spinbutton"))
    {
      x += (width - 7) / 2;

      if (arrow_type == GTK_ARROW_UP)
	{
	  y += (height - 4) / 2;
	}
      else
	{
	  y += (1 + height - 4) / 2;
	}

      draw_varrow (window, gc, shadow_type, area, arrow_type, x, y, 7, 4);
    }
  else if (DETAIL ("vscrollbar"))
    {
      parent_class->draw_box (style, window, state_type, shadow_type, area,
			      widget, detail, x, y, width, height);

      x += (width - 7) / 2;
      y += (height - 5) / 2;

      draw_varrow (window, gc, shadow_type, area, arrow_type, x, y, 7, 5);

    }
  else if (DETAIL ("hscrollbar"))
    {
      parent_class->draw_box (style, window, state_type, shadow_type, area,
			      widget, detail, x, y, width, height);

      y += (height - 7) / 2;
      x += (width - 5) / 2;

      draw_harrow (window, gc, shadow_type, area, arrow_type, x, y, 5, 7);
    }
  else
    {
      if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN)
	{
	  x += (width - 7) / 2;
	  y += (height - 5) / 2;

	  draw_varrow (window, gc, shadow_type, area, arrow_type, x, y, 7, 5);
	}
      else
	{
	  x += (width - 5) / 2;
	  y += (height - 7) / 2;

	  draw_harrow (window, gc, shadow_type, area, arrow_type, x, y, 5, 7);
	}
    }
}

static void
draw_slider (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y, gint width, gint height, GtkOrientation orientation)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *gc3;
  GdkGC *gc4;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  sanitize_size (window, &width, &height);

  gc1 = style->light_gc[state_type];
  gc2 = style->dark_gc[state_type];

  gc3 = style->bg_gc[state_type];
  gc4 = engine_data->shade_gc[6];

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
      gdk_gc_set_clip_rectangle (gc3, area);
      gdk_gc_set_clip_rectangle (gc4, area);
    }

  if (GTK_IS_SCROLLBAR (widget) || GTK_IS_SCALE (widget))
    {
      gdk_draw_rectangle (window, gc3, TRUE, x + 1, y + 1, width - 3,
			  height - 3);

      gdk_draw_line (window, gc2, x + 2, y + height - 2, x + width - 2,
		     y + height - 2);
      gdk_draw_line (window, gc2, x + width - 2, y + 2, x + width - 2,
		     y + height - 2);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + width - 2, y + 1);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 2);

      if (GTK_IS_VSCROLLBAR (widget) || GTK_IS_VSCALE (widget))
	{
	  if (GTK_IS_VSCROLLBAR (widget))
	    {
	      gdk_draw_rectangle (window, gc4, FALSE, x, y, width - 1,
				  height - 1);
	    }
	  else
	    {
	      theme_draw_rectangle (window, gc4, FALSE, x, y, width - 1,
				    height - 1);
	    }

	  if (height >= 14 && width >= 12)
	    {
	      gint gy = y + height / 2 - 8;
	      gint glength = width - 10;

	      gdk_draw_line (window, gc1, x + 5, gy + 5, x + 5 + glength,
			     gy + 5);
	      gdk_draw_line (window, gc2, x + 4, gy + 4, x + 4 + glength,
			     gy + 4);

	      gdk_draw_line (window, gc1, x + 5, gy + 8, x + 5 + glength,
			     gy + 8);
	      gdk_draw_line (window, gc2, x + 4, gy + 7, x + 4 + glength,
			     gy + 7);

	      gdk_draw_line (window, gc1, x + 5, gy + 11, x + 5 + glength,
			     gy + 11);
	      gdk_draw_line (window, gc2, x + 4, gy + 10, x + 4 + glength,
			     gy + 10);
	    }
	}
      else
	{
	  if (GTK_IS_HSCROLLBAR (widget))
	    {
	      gdk_draw_rectangle (window, gc4, FALSE, x, y, width - 1,
				  height - 1);
	    }
	  else
	    {
	      theme_draw_rectangle (window, gc4, FALSE, x, y, width - 1,
				    height - 1);
	    }

	  if (width > 14 && height >= 12)
	    {
	      gint gx = x + width / 2 - 8;
	      gint glength = height - 10;

	      gdk_draw_line (window, gc1, gx + 5, y + 5, gx + 5,
			     y + 5 + glength);
	      gdk_draw_line (window, gc2, gx + 4, y + 4, gx + 4,
			     y + 4 + glength);

	      gdk_draw_line (window, gc1, gx + 8, y + 5, gx + 8,
			     y + 5 + glength);
	      gdk_draw_line (window, gc2, gx + 7, y + 4, gx + 7,
			     y + 4 + glength);

	      gdk_draw_line (window, gc1, gx + 11, y + 5, gx + 11,
			     y + 5 + glength);
	      gdk_draw_line (window, gc2, gx + 10, y + 4, gx + 10,
			     y + 4 + glength);
	    }
	}
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
      gdk_gc_set_clip_rectangle (gc3, area);
      gdk_gc_set_clip_rectangle (gc4, area);
    }
}

static void
draw_box (GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail, gint x, gint y, gint width, gint height)
{
  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  sanitize_size (window, &width, &height);

  if (widget
      && (DETAIL ("menubar") || DETAIL ("toolbar") || DETAIL ("dockitem_bin"))
      && shadow_type != GTK_SHADOW_NONE)
    {
      GdkGC *gc;

      gc = engine_data->shade_gc[2];

      if (area)
	{
	  gdk_gc_set_clip_rectangle (gc, area);
	}

      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x, y, width,
					  height);


      if (!DETAIL ("menubar"))	/* Only toolbars have the top line */
	{
	  gdk_draw_line (window, engine_data->shade_gc[0], x, y, x + width,
			 y);
	}

      gdk_draw_line (window, gc, x, y + height - 1, x + width,
		     y + height - 1);

      /*gdk_draw_line (window, gc, x, y, x + width, y);
         gdk_draw_line (window, gc, x, y + height - 1, x + width, y + height - 1); */


      if (area)
	{
	  gdk_gc_set_clip_rectangle (gc, NULL);
	}
    }
  else if (DETAIL ("menuitem"))
    {
      if (area)
	{
	  gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
	}

      gdk_draw_rectangle (window, style->bg_gc[state_type], TRUE, x + 1,
			  y + 1, width - 2, height - 2);
      theme_draw_rectangle (window, style->bg_gc[state_type], FALSE, x, y,
			    width - 1, height - 1);

      if (area)
	{
	  gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
	}
    }
  else if (widget && DETAIL ("trough"))	/* Scrollbars */
    {
      if (area)
	{
	  gdk_gc_set_clip_rectangle (engine_data->shade_gc[4], area);
	  gdk_gc_set_clip_rectangle (engine_data->shade_gc[6], area);
	}

      if GTK_IS_HSCALE
	(widget)
	{
	  y += (height - SCALE_SIZE) / 2;
	  height = SCALE_SIZE;
	}

      if GTK_IS_VSCALE
	(widget)
	{
	  x += (width - SCALE_SIZE) / 2;
	  width = SCALE_SIZE;
	}

      gdk_draw_rectangle (window, engine_data->shade_gc[4], TRUE, x, y, width,
			  height);
      gdk_draw_rectangle (window, engine_data->shade_gc[6], FALSE, x, y,
			  width - 1, height - 1);

      if (GTK_IS_HSCALE (widget) || GTK_IS_VSCALE (widget))
	{
	  gdk_draw_line (window, engine_data->shade_gc[4], x + 1, y + 1,
			 x + width - 2, y + 1);
	  gdk_draw_line (window, engine_data->shade_gc[4], x + 1, y + 1,
			 x + 1, y + height - 2);
	}

      if (area)
	{
	  gdk_gc_set_clip_rectangle (engine_data->shade_gc[4], NULL);
	  gdk_gc_set_clip_rectangle (engine_data->shade_gc[5], NULL);
	}
    }
  else if (DETAIL ("slider"))
    {
      draw_slider (style, window, state_type, shadow_type, area, widget,
		   detail, x, y, width, height, GTK_ORIENTATION_VERTICAL);
    }
  else if (DETAIL ("spinbutton_up") || DETAIL ("spinbutton_down"))
    {
      /* Make the top button overlap the first line of the bottom button */
      if (strcmp (detail, "spinbutton_up") == 0)
	{
	  height += 1;
	}

      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x, y, width,
					  height);
      gtk_paint_shadow (style, window, state_type, shadow_type, area, widget,
			detail, x, y, width, height);
    }
  else if (widget && DETAIL ("bar") && GTK_IS_PROGRESS_BAR (widget))
    {
      GtkProgressBarOrientation orientation;

      if (area)
	{
	  gdk_gc_set_clip_rectangle (style->base_gc[GTK_STATE_SELECTED],
				     area);
	}

      orientation = GTK_PROGRESS_BAR (widget)->orientation;

      if (orientation == GTK_PROGRESS_LEFT_TO_RIGHT
	  || GTK_PROGRESS_RIGHT_TO_LEFT)
	{
	  if (width > 2)
	    {
	      width -= 1;
	    }
	}
      else
	{
	  if (height > 2)
	    {
	      height -= 1;
	    }
	}

      gdk_draw_rectangle (window, style->base_gc[GTK_STATE_SELECTED], TRUE,
			  x, y, width - 1, height);

      if (area)
	{
	  gdk_gc_set_clip_rectangle (style->base_gc[GTK_STATE_SELECTED],
				     NULL);
	}
    }
  else
    {
      parent_class->draw_box (style, window, state_type, shadow_type, area,
			      widget, detail, x, y, width, height);
    }
}

static void
draw_flat_box (GtkStyle * style,
	       GdkWindow * window,
	       GtkStateType state_type,
	       GtkShadowType shadow_type,
	       GdkRectangle * area,
	       GtkWidget * widget,
	       gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1;

  g_return_if_fail (window != NULL);

  sanitize_size (window, &width, &height);

  if (detail)
    {
      if (state_type == GTK_STATE_SELECTED)
	{
	  if (!strcmp ("text", detail))
	    {
	      gc1 = style->bg_gc[GTK_STATE_SELECTED];
	    }
	  else if (!strncmp ("cell_even", detail, strlen ("cell_even")) ||
		   !strncmp ("cell_odd", detail, strlen ("cell_odd")))
	    {
	      /* This has to be really broken; alex made me do it. -jrb */
	      if (GTK_WIDGET_HAS_FOCUS (widget))
		{
		  gc1 = style->base_gc[state_type];
		}
	      else
		{
		  gc1 = style->base_gc[GTK_STATE_ACTIVE];
		}
	    }
	  else
	    {
	      gc1 = style->bg_gc[state_type];
	    }
	}
      else
	{
	  if (!strcmp ("viewportbin", detail))
	    {
	      gc1 = style->bg_gc[GTK_STATE_NORMAL];
	    }
	  else if (!strcmp ("entry_bg", detail))
	    {
	      gc1 = style->base_gc[state_type];
	    }

	  /* For trees: even rows are base color, odd rows are a shade of
	   * the base color, the sort column is a shade of the original color
	   * for that row.
	   */

	  /* FIXME when we have style properties, clean this up.
	   */

	  else if (!strcmp ("cell_even", detail) ||
		   !strcmp ("cell_odd", detail) ||
		   !strcmp ("cell_even_ruled", detail))
	    {
	      gc1 = style->base_gc[state_type];
	    }
	  else
	    {
	      gc1 = style->bg_gc[state_type];
	    }
	}
    }
  else
    {
      gc1 = style->bg_gc[state_type];
    }

  if (!style->bg_pixmap[state_type] || gc1 != style->bg_gc[state_type] ||
      gdk_window_get_type (window) == GDK_WINDOW_PIXMAP)
    {
      if (area)
	{
	  gdk_gc_set_clip_rectangle (gc1, area);
	}
      if (state_type == GTK_STATE_SELECTED)
	{
	  gdk_draw_rectangle (window, gc1, TRUE, x + 3, y, width - 6,
			      height - 1);
	  theme_draw_rectangle (window, gc1, FALSE, x + 2, y, width - 5,
				height - 1);
	}
      else
	{
	  gdk_draw_rectangle (window, gc1, TRUE, x, y, width, height);
	}

      if (detail && !strcmp ("tooltip", detail))
	{
	  gdk_draw_rectangle (window, style->black_gc, FALSE, x, y, width - 1,
			      height - 1);
	}

      if (area)
	{
	  gdk_gc_set_clip_rectangle (gc1, NULL);
	}
    }
  else
    {
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x, y, width,
					  height);
    }
}

static void
draw_check (GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *gc3;
  GdkGC *gc4;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (DETAIL ("check"))		/* menuitem */
    {
      if (shadow_type == GTK_SHADOW_IN)
	{
	  gdk_draw_line (window, style->fg_gc[state_type], x + 3, y + 5,
			 x + 3, y + 9);
	  gdk_draw_line (window, style->fg_gc[state_type], x + 4, y + 5,
			 x + 4, y + 9);

	  gdk_draw_line (window, style->fg_gc[state_type], x + 5, y + 8,
			 x + 9, y + 4);
	  gdk_draw_line (window, style->fg_gc[state_type], x + 5, y + 7,
			 x + 9, y + 3);
	}

      return;
    }

  gc1 = style->light_gc[state_type];
  gc2 = style->dark_gc[state_type];
  gc3 = style->bg_gc[state_type];
  gc4 = style->fg_gc[GTK_STATE_SELECTED];

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
      gdk_gc_set_clip_rectangle (gc3, area);
      gdk_gc_set_clip_rectangle (gc4, area);
    }

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
      gdk_draw_rectangle (window, gc3, TRUE, x + 1, y + 1, width - 1,
			  height - 1);
      gdk_draw_line (window, gc2, x + 1, y, x + width - 1, y);
      gdk_draw_line (window, gc2, x, y + 1, x, y + height - 1);

      gdk_draw_line (window, gc1, x + 1, y + height, x + width - 1,
		     y + height);
      gdk_draw_line (window, gc1, x + width, y + 1, x + width,
		     y + height - 1);
      gdk_draw_rectangle (window, gc4, TRUE, x + 3, y + 3, width - 5,
			  height - 5);
    }
  else
    {
      gdk_draw_rectangle (window, gc3, TRUE, x + 1, y + 1, width - 1,
			  height - 1);
      gdk_draw_line (window, gc1, x + 1, y, x + width - 1, y);
      gdk_draw_line (window, gc1, x, y + 1, x, y + height - 1);

      gdk_draw_line (window, gc2, x + 1, y + height, x + width - 1,
		     y + height);
      gdk_draw_line (window, gc2, x + width, y + 1, x + width,
		     y + height - 1);
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, NULL);
      gdk_gc_set_clip_rectangle (gc2, NULL);
      gdk_gc_set_clip_rectangle (gc3, NULL);
      gdk_gc_set_clip_rectangle (gc4, NULL);
    }
}

static void
draw_option (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *gc3;
  GdkGC *gc4;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (DETAIL ("option"))	/* menuitem */
    {
      if (shadow_type == GTK_SHADOW_IN)
	{
	  gdk_draw_arc (window, style->fg_gc[state_type], TRUE, x, y,
			width - 1, height - 1, 0, 360 * 64);
	}

      return;
    }

  gc1 = style->light_gc[state_type];
  gc2 = style->dark_gc[state_type];
  gc3 = style->bg_gc[state_type];
  gc4 = style->fg_gc[GTK_STATE_SELECTED];

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
      gdk_gc_set_clip_rectangle (gc3, area);
      gdk_gc_set_clip_rectangle (gc4, area);
    }

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
      gdk_draw_arc (window, gc3, TRUE, x, y, width, height, 0, 360 * 64);
      gdk_draw_arc (window, gc2, FALSE, x, y, width, height, 45 * 64,
		    225 * 64);
      gdk_draw_arc (window, gc1, FALSE, x, y, width, height, 225 * 64,
		    180 * 64);
      gdk_draw_arc (window, gc4, TRUE, x + 2, y + 2, width - 4, height - 4, 0,
		    360 * 64);
    }
  else
    {
      gdk_draw_arc (window, gc3, TRUE, x, y, width, height, 0, 360 * 64);
      gdk_draw_arc (window, gc1, FALSE, x, y, width, height, 45 * 64,
		    225 * 64);
      gdk_draw_arc (window, gc2, FALSE, x, y, width, height, 225 * 64,
		    180 * 64);
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, NULL);
      gdk_gc_set_clip_rectangle (gc2, NULL);
      gdk_gc_set_clip_rectangle (gc3, NULL);
      gdk_gc_set_clip_rectangle (gc4, NULL);
    }
}

static void
draw_shadow_gap (GtkStyle * style,
		 GdkWindow * window,
		 GtkStateType state_type,
		 GtkShadowType shadow_type,
		 GdkRectangle * area,
		 GtkWidget * widget,
		 gchar * detail,
		 gint x,
		 gint y,
		 gint width,
		 gint height,
		 GtkPositionType gap_side, gint gap_x, gint gap_width)
{
  GdkGC *gc;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (DETAIL ("frame"))
    {
      sanitize_size (window, &width, &height);
      gc = engine_data->shade_gc[5];

      if (area)
	{
	  gdk_gc_set_clip_rectangle (gc, area);
	}

      gdk_draw_line (window, gc, x, y + 1, x, y + height - 2);
      gdk_draw_line (window, gc, x + 1, y + height - 1, x + width - 2,
		     y + height - 1);
      gdk_draw_line (window, gc, x + width - 1, y + 1, x + width - 1,
		     y + height - 2);

      if (gap_x > 0)
	{
	  gdk_draw_line (window, gc, x + 1, y, x + gap_x - 1, y);
	}
      if ((width - (gap_x + gap_width)) > 0)
	{
	  gdk_draw_line (window, gc, x + gap_x + gap_width, y, x + width - 2,
			 y);
	}

      if (area)
	{
	  gdk_gc_set_clip_rectangle (gc, NULL);
	}

    }
  else
    {
      parent_class->draw_shadow_gap (style, window, state_type, shadow_type,
				     area, widget, detail, x, y, width,
				     height, gap_side, gap_x, gap_width);

    }
}

static void
draw_box_gap (GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail,
	      gint x,
	      gint y,
	      gint width,
	      gint height,
	      GtkPositionType gap_side, gint gap_x, gint gap_width)
{
  GdkGC *gc1 = NULL;
  GdkGC *gc2 = NULL;
  GdkGC *outer_gc = NULL;
  gint offset = 0;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  switch (gap_side)
    {
    case GTK_POS_LEFT:
    case GTK_POS_RIGHT:
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x, y + 1, width,
					  height - 2);
      break;

    case GTK_POS_TOP:
    case GTK_POS_BOTTOM:
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x + 1, y,
					  width - 2, height);
      break;
    }

  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      return;

    case GTK_SHADOW_IN:
    case GTK_SHADOW_ETCHED_IN:
      gc1 = engine_data->shade_gc[2];
      gc2 = style->light_gc[state_type];	/*style->white_gc; */
      break;

    case GTK_SHADOW_OUT:
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = style->light_gc[state_type];	/*style->white_gc; */
      gc2 = engine_data->shade_gc[2];
      break;
    }

  outer_gc = engine_data->shade_gc[6];

  if (area)
    {
      gdk_gc_set_clip_rectangle (outer_gc, area);
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
    }

  switch (gap_side)
    {
    case GTK_POS_TOP:
      gdk_draw_line (window, outer_gc, x, y + offset, x, y + height - 2);

      gdk_draw_line (window, gc1, x + 1, y, x + 1, y + height - 2);

      gdk_draw_line (window, gc2, x + 1, y + height - 2, x + width - 2,
		     y + height - 2);
      gdk_draw_line (window, gc2, x + width - 2, y, x + width - 2,
		     y + height - 2);

      gdk_draw_line (window, outer_gc, x + 1, y + height - 1, x + width - 2,
		     y + height - 1);
      gdk_draw_line (window, outer_gc, x + width - 1, y + offset,
		     x + width - 1, y + height - 2);

      if (gap_x > 0)
	{
	  gdk_draw_line (window, outer_gc, x + offset, y, x + gap_x - 1, y);
	  gdk_draw_line (window, gc1, x + 1, y + 1, x + gap_x - 1, y + 1);
	}
      if ((width - (gap_x + gap_width)) > 0)
	{
	  gdk_draw_line (window, outer_gc, x + gap_x + gap_width, y,
			 x + width - 2, y);
	  gdk_draw_line (window, gc1, x + gap_x + gap_width, y + 1,
			 x + width - 2, y + 1);
	}
      break;

    case GTK_POS_BOTTOM:
      gdk_draw_line (window, outer_gc, x + 1, y, x + width - 2, y);
      gdk_draw_line (window, outer_gc, x, y + 1, x, y + height - 1 - offset);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + width - 2, y + 1);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 1);

      gdk_draw_line (window, gc2, x + width - 2, y + 1, x + width - 2,
		     y + height - 1);
      gdk_draw_line (window, outer_gc, x + width - 1, y + 1, x + width - 1,
		     y + height - 1 - offset);

      if (gap_x > 0)
	{
	  gdk_draw_line (window, outer_gc, x + offset, y + height - 1,
			 x + gap_x - 1, y + height - 1);
	  gdk_draw_line (window, gc2, x + 1, y + height - 2, x + gap_x - 1,
			 y + height - 2);
	}
      if ((width - (gap_x + gap_width)) > 0)
	{
	  gdk_draw_line (window, outer_gc, x + gap_x + gap_width,
			 y + height - 1, x + width - 2, y + height - 1);
	  gdk_draw_line (window, gc2, x + gap_x + gap_width, y + height - 2,
			 x + width - 2, y + height - 2);
	}
      break;

    case GTK_POS_LEFT:
      gdk_draw_line (window, outer_gc, x + offset, y, x + width - 2, y);
      gdk_draw_line (window, gc1, x, y + 1, x + width - 2, y + 1);

      gdk_draw_line (window, gc2, x, y + height - 2, x + width - 2,
		     y + height - 2);
      gdk_draw_line (window, gc2, x + width - 2, y + 1, x + width - 2,
		     y + height - 2);
      gdk_draw_line (window, outer_gc, x + offset, y + height - 1,
		     x + width - 2, y + height - 1);
      gdk_draw_line (window, outer_gc, x + width - 1, y + 1, x + width - 1,
		     y + height - 2);

      if (gap_x > 0)
	{
	  gdk_draw_line (window, outer_gc, x, y + offset, x, y + gap_x - 1);
	  gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + gap_x - 1);
	}
      if ((width - (gap_x + gap_width)) > 0)
	{
	  gdk_draw_line (window, outer_gc, x, y + gap_x + gap_width, x,
			 y + height - 2);
	  gdk_draw_line (window, gc1, x + 1, y + gap_x + gap_width, x + 1,
			 y + height - 2);
	}
      break;

    case GTK_POS_RIGHT:
      gdk_draw_line (window, outer_gc, x + 1, y, x + width - 1 - offset, y);
      gdk_draw_line (window, outer_gc, x, y + 1, x, y + height - 2);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + width - 1, y + 1);
      gdk_draw_line (window, gc1, x + 1, y + 1, x + 1, y + height - 2);

      gdk_draw_line (window, gc2, x + 1, y + height - 2, x + width - 1,
		     y + height - 2);
      gdk_draw_line (window, outer_gc, x + 1, y + height - 1,
		     x + width - 1 - offset, y + height - 1);

      if (gap_x > 0)
	{
	  gdk_draw_line (window, outer_gc, x + width - 1, y + offset,
			 x + width - 1, y + gap_x - 1);
	  gdk_draw_line (window, gc2, x + width - 2, y + 1, x + width - 2,
			 y + gap_x - 1);
	}
      if ((width - (gap_x + gap_width)) > 0)
	{
	  gdk_draw_line (window, outer_gc, x + width - 1,
			 y + gap_x + gap_width, x + width - 1,
			 y + height - 2);
	  gdk_draw_line (window, gc2, x + width - 2, y + gap_x + gap_width,
			 x + width - 2, y + height - 2);
	}
      break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (outer_gc, NULL);
      gdk_gc_set_clip_rectangle (gc1, NULL);
      gdk_gc_set_clip_rectangle (gc2, NULL);
    }
}

static void
draw_extension (GtkStyle * style,
		GdkWindow * window,
		GtkStateType state_type,
		GtkShadowType shadow_type,
		GdkRectangle * area,
		GtkWidget * widget,
		gchar * detail,
		gint x,
		gint y, gint width, gint height, GtkPositionType gap_side)
{
  GdkGC *gc1;
  GdkGC *gc2;
  GdkGC *outer_gc;
  GdkGC *bg_gc;
  gint xthickness = 2;
  gint ythickness = 2;
  gint d = 0;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  sanitize_size (window, &width, &height);

  outer_gc = engine_data->shade_gc[6];
  bg_gc = widget->parent->style->bg_gc[GTK_STATE_NORMAL];

  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      return;

    case GTK_SHADOW_IN:
    case GTK_SHADOW_ETCHED_IN:
      gc1 = engine_data->shade_gc[2];
      gc2 = style->light_gc[state_type];	/*style->white_gc; */
      break;

    case GTK_SHADOW_OUT:
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = style->light_gc[state_type];	/*style->white_gc; */
      gc2 = engine_data->shade_gc[2];
      break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (gc1, area);
      gdk_gc_set_clip_rectangle (gc2, area);
      gdk_gc_set_clip_rectangle (outer_gc, area);
      gdk_gc_set_clip_rectangle (bg_gc, area);
    }

  //Unselected tab should be drawn 2 pixels lower than selected tab
  if (state_type == GTK_STATE_ACTIVE)
    {
      d = 2;
    }

  switch (gap_side)
    {
    case GTK_POS_TOP:
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x + xthickness, y,
					  width - (2 * xthickness),
					  height - (ythickness) - d);

      gdk_draw_line (window, outer_gc, x, y, x, y + height - 2 - d);

      gdk_draw_line (window, gc1, x + 1, y, x + 1, y + height - 2 - d);

      gdk_draw_line (window, gc2, x + 2, y + height - 2 - d, x + width - 2,
		     y + height - 2 - d);
      gdk_draw_line (window, gc2, x + width - 2, y, x + width - 2,
		     y + height - 2 - d);

      gdk_draw_line (window, outer_gc, x + 1, y + height - 1 - d,
		     x + width - 2, y + height - 1 - d);
      gdk_draw_line (window, outer_gc, x + width - 1, y, x + width - 1,
		     y + height - 2 - d);

      gdk_draw_point (window, bg_gc, x, y + height - 1 - d);
      gdk_draw_point (window, bg_gc, x + width - 1, y + height - 1 - d);

      if (state_type == GTK_STATE_ACTIVE)
	{
	  /* Clear vestige of selected tab */
	  gdk_draw_line (window, bg_gc, x, y + height - 2, x + width - 1,
			 y + height - 2);
	  gdk_draw_line (window, bg_gc, x, y + height - 1, x + width - 1,
			 y + height - 1);
	}

      break;

    case GTK_POS_BOTTOM:
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x + xthickness,
					  y + d + ythickness,
					  width - (2 * xthickness),
					  height - (ythickness) - d);

      gdk_draw_line (window, outer_gc, x + 1, y + d, x + width - 2, y + d);
      gdk_draw_line (window, outer_gc, x, y + d + 1, x, y + height - 1);

      gdk_draw_line (window, gc1, x + 1, y + d + 1, x + width - 2, y + d + 1);
      gdk_draw_line (window, gc1, x + 1, y + d + 1, x + 1, y + height - 1);

      gdk_draw_line (window, gc2, x + width - 2, y + d + 1, x + width - 2,
		     y + height - 1);

      gdk_draw_line (window, outer_gc, x + width - 1, y + d + 1,
		     x + width - 1, y + height - 1);

      gdk_draw_point (window, bg_gc, x, y + d);
      gdk_draw_point (window, bg_gc, x + width - 1, y + d);

      if (state_type == GTK_STATE_ACTIVE)
	{
	  /* Clear vestige of selected tab */
	  gdk_draw_line (window, bg_gc, x, y, x + width - 1, y);
	  gdk_draw_line (window, bg_gc, x, y + 1, x + width - 1, y + 1);
	}

      break;

    case GTK_POS_LEFT:
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area, x, y + ythickness,
					  width - (xthickness) - d,
					  height - (2 * ythickness));

      gdk_draw_line (window, outer_gc, x, y, x + width - 2 - d, y);
      gdk_draw_line (window, gc1, x, y + 1, x + width - 2 - d, y + 1);

      gdk_draw_line (window, gc2, x, y + height - 2, x + width - 2 - d,
		     y + height - 2);
      gdk_draw_line (window, gc2, x + width - 2 - d, y + 2, x + width - 2 - d,
		     y + height - 2);

      gdk_draw_line (window, outer_gc, x, y + height - 1, x + width - 2 - d,
		     y + height - 1);
      gdk_draw_line (window, outer_gc, x + width - 1 - d, y + 1,
		     x + width - 1 - d, y + height - 2);

      gdk_draw_point (window, bg_gc, x + width - 1 - d, y);
      gdk_draw_point (window, bg_gc, x + width - 1 - d, y + height - 1);

      if (state_type == GTK_STATE_ACTIVE)
	{
	  /* Clear vestige of selected tab */
	  gdk_draw_line (window, bg_gc, x + width - 2, y, x + width - 2,
			 y + height - 1);
	  gdk_draw_line (window, bg_gc, x + width - 1, y, x + width - 1,
			 y + height - 1);
	}

      break;

    case GTK_POS_RIGHT:
      gtk_style_apply_default_background (style, window, widget
					  && !GTK_WIDGET_NO_WINDOW (widget),
					  state_type, area,
					  x + xthickness + d, y + ythickness,
					  width - (xthickness) - d,
					  height - (2 * ythickness));

      gdk_draw_line (window, outer_gc, x + 1 + d, y, x + width - 1, y);
      gdk_draw_line (window, outer_gc, x + d, y + 1, x + d, y + height - 2);

      gdk_draw_line (window, gc1, x + 1 + d, y + 1, x + width - 1, y + 1);
      gdk_draw_line (window, gc1, x + 1 + d, y + 1, x + 1 + d,
		     y + height - 1);

      gdk_draw_line (window, gc2, x + 1 + d, y + height - 2, x + width - 1,
		     y + height - 2);

      gdk_draw_line (window, outer_gc, x + 1 + d, y + height - 1,
		     x + width - 1, y + height - 1);

      gdk_draw_point (window, bg_gc, x + d, y);
      gdk_draw_point (window, bg_gc, x + d, y + height - 1);

      if (state_type == GTK_STATE_ACTIVE)
	{
	  /* Clear vestige of selected tab */
	  gdk_draw_line (window, bg_gc, x + 1, y, x + 1, y + height - 1);
	  gdk_draw_line (window, bg_gc, x, y, x, y + height - 1);
	}

      break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (bg_gc, NULL);
      gdk_gc_set_clip_rectangle (outer_gc, NULL);
      gdk_gc_set_clip_rectangle (gc1, NULL);
      gdk_gc_set_clip_rectangle (gc2, NULL);
    }
}

static void
draw_focus (GtkStyle * style,
	    GdkWindow * window,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail, gint x, gint y, gint width, gint height)
{
  GdkPoint points[5];
  GdkGC *gc;
  gboolean free_dash_list = FALSE;
  gint line_width = 1;
  gchar *dash_list = "\1\1";
  gint dash_len;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  if (DETAIL ("treeview"))	/* focus is represented by color only (see base color definitions) */
    {
      return;
    }

  if (DETAIL ("button") && GTK_WIDGET_HAS_DEFAULT (widget))
    {
      return;			/* focus is represented by blue rectangle painted in draw_shadow */
    }

  gc = engine_data->shade_gc[6];

  /*if (widget)
     {
     gtk_widget_style_get (widget,
     "focus-line-width", &line_width,
     "focus-line-pattern", (gchar *)&dash_list,
     NULL);

     free_dash_list = TRUE;
     } */

  sanitize_size (window, &width, &height);

  if (area)
    gdk_gc_set_clip_rectangle (gc, area);

  gdk_gc_set_line_attributes (gc, line_width,
			      dash_list[0] ? GDK_LINE_ON_OFF_DASH :
			      GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);


  if (detail && !strcmp (detail, "add-mode"))
    {
      if (free_dash_list)
	g_free (dash_list);

      dash_list = "\4\4";
      free_dash_list = FALSE;
    }

  points[0].x = x + line_width / 2;
  points[0].y = y + line_width / 2;
  points[1].x = x + width - line_width + line_width / 2;
  points[1].y = y + line_width / 2;
  points[2].x = x + width - line_width + line_width / 2;
  points[2].y = y + height - line_width + line_width / 2;
  points[3].x = x + line_width / 2;
  points[3].y = y + height - line_width + line_width / 2;
  points[4] = points[0];

  if (!dash_list[0])
    {
      gdk_draw_lines (window, gc, points, 5);
    }
  else
    {
      /* We go through all the pain below because the X rasterization
       * rules don't really work right for dashed lines if you
       * want continuity in segments that go between top/right
       * and left/bottom. For instance, a top left corner
       * with a 1-1 dash is drawn as:
       *
       *  X X X 
       *  X
       *
       *  X
       *
       * This is because pixels on the top and left boundaries
       * of polygons are drawn, but not on the bottom and right.
       * So, if you have a line going up that turns the corner
       * and goes right, there is a one pixel shift in the pattern.
       *
       * So, to fix this, we drawn the top and right in one call,
       * then the left and bottom in another call, fixing up
       * the dash offset for the second call ourselves to get
       * continuity at the upper left.
       *
       * It's not perfect since we really should have a join at
       * the upper left and lower right instead of two intersecting
       * lines but that's only really apparent for no-dashes,
       * which (for this reason) are done as one polygon and
       * don't to through this code path.
       */

      dash_len = strlen (dash_list);

      if (dash_list[0])
	gdk_gc_set_dashes (gc, 0, dash_list, dash_len);

      gdk_draw_lines (window, gc, points, 3);

      /* We draw this line one farther over than it is "supposed" to
       * because of another rasterization problem ... if two 1 pixel
       * unjoined lines meet at the lower right, there will be a missing
       * pixel.
       */
      points[2].x += 1;

      if (dash_list[0])
	{
	  gint dash_pixels = 0;
	  gint i;

	  /* Adjust the dash offset for the bottom and left so we
	   * match up at the upper left.
	   */
	  for (i = 0; i < dash_len; i++)
	    dash_pixels += dash_list[i];

	  if (dash_len % 2 == 1)
	    dash_pixels *= 2;

	  gdk_gc_set_dashes (gc,
			     dash_pixels - (width + height -
					    2 * line_width) % dash_pixels,
			     dash_list, dash_len);
	}

      gdk_draw_lines (window, gc, points + 2, 3);
    }

  gdk_gc_set_line_attributes (gc, 0, GDK_LINE_SOLID, GDK_CAP_BUTT,
			      GDK_JOIN_MITER);

  if (area)
    gdk_gc_set_clip_rectangle (gc, NULL);

  if (free_dash_list)
    g_free (dash_list);
}


static void
draw_handle (GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y, gint width, gint height, GtkOrientation orientation)
{
  GdkGC *light_gc;
  GdkGC *dark_gc;

  LighthouseBlueData *engine_data = style->engine_data;

  g_return_if_fail (window != NULL);

  sanitize_size (window, &width, &height);

  light_gc = style->light_gc[state_type];
  dark_gc = engine_data->shade_gc[5];

  if (area)
    {
      gdk_gc_set_clip_rectangle (light_gc, area);
      gdk_gc_set_clip_rectangle (dark_gc, area);
    }

  gtk_style_apply_default_background (style, window, widget
				      && !GTK_WIDGET_NO_WINDOW (widget),
				      state_type, area, x, y, width, height);

  if ((DETAIL ("handlebox") && widget && GTK_IS_HANDLE_BOX (widget))
      || DETAIL ("dockitem"))
    {
      /* toolbar stuff */
      gdk_draw_line (window, engine_data->shade_gc[0], x, y, x + width, y);
      gdk_draw_line (window, engine_data->shade_gc[2], x, y + height - 1,
		     x + width, y + height - 1);

      gdk_draw_line (window, engine_data->shade_gc[2], x + width - 1, y + 2,
		     x + width - 1, y + height - 3);
      gdk_draw_line (window, engine_data->shade_gc[0], x + width, y + 2,
		     x + width, y + height - 3);
    }
  else if (DETAIL ("paned"))
    {
      gint px;
      gint py;

      if (orientation == GTK_ORIENTATION_HORIZONTAL)
	{
	  py = y + height / 2 - 1;
	  for (px = x + width / 2 - 15; px <= x + width / 2 + 15; px += 5)
	    {
	      gdk_draw_point (window, light_gc, px, py);
	      gdk_draw_point (window, light_gc, px + 1, py);
	      gdk_draw_point (window, light_gc, px, py + 1);
	      gdk_draw_point (window, dark_gc, px + 1, py + 2);
	      gdk_draw_point (window, dark_gc, px + 2, py + 1);
	      gdk_draw_point (window, dark_gc, px + 2, py + 2);
	    }
	}
      else
	{
	  px = x + width / 2 - 1;
	  for (py = y + height / 2 - 15; py <= y + height / 2 + 15; py += 5)
	    {
	      gdk_draw_point (window, light_gc, px, py);
	      gdk_draw_point (window, light_gc, px + 1, py);
	      gdk_draw_point (window, light_gc, px, py + 1);
	      gdk_draw_point (window, dark_gc, px + 1, py + 2);
	      gdk_draw_point (window, dark_gc, px + 2, py + 1);
	      gdk_draw_point (window, dark_gc, px + 2, py + 2);
	    }
	}

      return;
    }
  else
    {
      draw_box (style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height);
    }

  if (orientation == GTK_ORIENTATION_VERTICAL)
    {
      if (height >= 14 && width >= 8)
	{
	  gint gy = y + height / 2 - 8;
	  gint glength = width - 6;

	  gdk_draw_line (window, light_gc, x + 3, gy + 5, x + 3 + glength,
			 gy + 5);
	  gdk_draw_line (window, dark_gc, x + 2, gy + 4, x + 2 + glength,
			 gy + 4);

	  gdk_draw_line (window, light_gc, x + 3, gy + 8, x + 3 + glength,
			 gy + 8);
	  gdk_draw_line (window, dark_gc, x + 2, gy + 7, x + 2 + glength,
			 gy + 7);

	  gdk_draw_line (window, light_gc, x + 3, gy + 11, x + 3 + glength,
			 gy + 11);
	  gdk_draw_line (window, dark_gc, x + 2, gy + 10, x + 2 + glength,
			 gy + 10);
	}
    }
  else
    {
      if (width > 14 && height >= 8)
	{
	  gint gx = x + width / 2 - 8;
	  gint glength = height - 6;

	  gdk_draw_line (window, light_gc, gx + 5, y + 3, gx + 5,
			 y + 3 + glength);
	  gdk_draw_line (window, dark_gc, gx + 4, y + 2, gx + 4,
			 y + 2 + glength);

	  gdk_draw_line (window, light_gc, gx + 8, y + 3, gx + 8,
			 y + 3 + glength);
	  gdk_draw_line (window, dark_gc, gx + 7, y + 2, gx + 7,
			 y + 2 + glength);

	  gdk_draw_line (window, light_gc, gx + 11, y + 3, gx + 11,
			 y + 3 + glength);
	  gdk_draw_line (window, dark_gc, gx + 10, y + 2, gx + 10,
			 y + 2 + glength);
	}
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle (light_gc, NULL);
      gdk_gc_set_clip_rectangle (dark_gc, NULL);
    }
}

void
lighthouseblue_initialize_style (GtkStyleClass * klass,
				 GtkStyleClass * parent)
{
  parent_class = parent;
  *klass = *parent;

  klass->draw_hline = draw_hline;
  klass->draw_vline = draw_vline;
  klass->draw_shadow = draw_shadow;

  klass->draw_box = draw_box;
  klass->draw_flat_box = draw_flat_box;
  klass->draw_check = draw_check;
  klass->draw_option = draw_option;

  klass->draw_shadow_gap = draw_shadow_gap;
  klass->draw_box_gap = draw_box_gap;
  klass->draw_extension = draw_extension;
  klass->draw_slider = draw_slider;
  klass->draw_handle = draw_handle;

  klass->draw_arrow = draw_arrow;
  klass->draw_focus = draw_focus;
}
