/* GNOME Transcript
 * Copyright (C) 1999-2000 the Free Software Foundation
 * Authors: Matias Mutchinick, Jos Miguel Roquillo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gnome.h>
#include <gtk/gtksignal.h>
#include <gtrans_ifase/gtrans_ifase.h>

#include "gtrans_conn_dialog.h"
#include "gtrans_button.h"


enum {
	CONNECT,
	LAST_SIGNAL
};


static guint gtrans_conn_dialog_signals[LAST_SIGNAL] = { 0 };

static GtkWindow *parent_class = NULL;




/*
 * gtrans_conn_dialog_destroy:
 * @object : The GTransConnDialog to destroy.
 *
 * Destroy the connection dialog, and close unused gmodules.
 */
static void 
gtrans_conn_dialog_destroy(GtkObject *object)
{
	GTransConnDialog  *dlg;
	GTransIFasePlugin *plugin;
	GtkCList          *clist;
	gint               i , rows;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GTRANS_IS_CONN_DIALOG(object));
	
	dlg = GTRANS_CONN_DIALOG(object);
	clist = GTK_CLIST(dlg->dbsel);
	rows = clist->rows;
	
	for ( i = 0 ; i < rows ; i++){
		
		plugin = (GTransIFasePlugin *)
			gtk_clist_get_row_data(clist,i);
		
		/* Leave open the selected gmodule */
		if (i != clist->focus_row)
			if (plugin->plugin)
				g_module_close(plugin->plugin);
		
		g_free(plugin);
	}
	
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}




/**
 * gtrans_conn_dialog_class_init
 */
static void
gtrans_conn_dialog_class_init(GTransConnDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	
	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;
	
	gtrans_conn_dialog_signals[CONNECT] =
		gtk_signal_new ("connect",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransConnDialogClass, connect),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals(object_class, 
				     gtrans_conn_dialog_signals,
				     LAST_SIGNAL);
	
	parent_class = gtk_type_class(gtk_window_get_type());

	/* Destroy */
	object_class->destroy = gtrans_conn_dialog_destroy;
}




/**
 * gtrans_conn_dialog_init
 */
static void
gtrans_conn_dialog_init(GTransConnDialog *dialog)
{
	dialog->plugin = NULL;	
	GTK_WINDOW(dialog)->type = GTK_WINDOW_DIALOG;
}




/**
 * gtrans_conn_dialog_get_type
 */
guint 
gtrans_conn_dialog_get_type()
{
	static guint gtrans_conn_dialog_type = 0;
	
	if(!gtrans_conn_dialog_type) {
		GtkTypeInfo gtrans_conn_dialog_info = {
			"GTransConnDialog",
			sizeof(GTransConnDialog),
			sizeof(GTransConnDialogClass),
			(GtkClassInitFunc) gtrans_conn_dialog_class_init,
			(GtkObjectInitFunc) gtrans_conn_dialog_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		gtrans_conn_dialog_type = 
			gtk_type_unique(gtk_window_get_type(),
					&gtrans_conn_dialog_info);
	}
	return gtrans_conn_dialog_type;
}





/**
 * gtrans_conn_dialog_select_row_cb
 * @clist   : The GtkClist who emited the "select_row" signal
 * @row     : The row selected
 * @dlg     : The GTransConnDialog
 *
 * Point @dlg->plugin to the selected plugin.
 */
static void
gtrans_conn_dialog_select_row_cb(GtkCList  *clist, 
	      gint       row, 
	      gint       column, 
	      GdkEvent  *event, 
	      GtkWidget *dlg)
{
	GTransIFasePlugin *plugin;
	
	plugin = (GTransIFasePlugin *)
		gtk_clist_get_row_data(GTK_CLIST(clist), row);
	
	GTRANS_CONN_DIALOG(dlg)->plugin = plugin;
}





/**
 * gtrans_conn_dialog_unselect_row_cb
 * @dlg : The Connection dialaog.
 *
 * Set @dlg->plugin to NULL cause the plugin has just been unselected.
 */
static void
gtrans_conn_dialog_unselect_row_cb(GtkCList  *clist, 
		gint       row, 
		gint       column, 
		GdkEvent  *event, 
		GtkWidget *dlg)
{
	GTRANS_CONN_DIALOG(dlg)->plugin = NULL;
}



/*
 * gtrans_conn_dialog_plugin_clist_new
 * @dlg     : The connection dialog to create the clist for
 * @plugins : The GList of found plugins (GTransIFasePlugin)
 *
 * Create the GtkClist to select a pluign from.
 */
static GtkWidget *
gtrans_conn_dialog_plugin_clist_new(GTransConnDialog *dlg,
				    GList            *plugins)
{
	GtkWidget          *clist;
	GList              *tmp;
	gchar              *text[2];
	gchar              *titles[2]={N_("Plugin"), N_("File name")};
	GTransIFasePlugin  *module;
	gint                row;
	
	clist = gtk_clist_new_with_titles (2, titles);		
	gtk_clist_set_column_auto_resize (GTK_CLIST(clist), 0, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(clist), 1, TRUE);

	gtk_signal_connect (GTK_OBJECT (clist), "select_row", 
			    GTK_SIGNAL_FUNC (gtrans_conn_dialog_select_row_cb), 
			    (gpointer) dlg);
	
	gtk_signal_connect (GTK_OBJECT (clist), "unselect_row", 
			    GTK_SIGNAL_FUNC (gtrans_conn_dialog_unselect_row_cb),
			    (gpointer)dlg);
	
	for (tmp = plugins ; tmp != NULL ; tmp = tmp->next) { 
		
		module = (GTransIFasePlugin *) tmp->data;
		
		text[0] = module->name;
		text[1] = module->filename;
		
		row = gtk_clist_append (GTK_CLIST(clist), text);
		gtk_clist_set_row_data (GTK_CLIST(clist), row, module);
	}
	
	g_list_free(plugins);
	
	return clist;
}




/*
 * gtrans_conn_dialog_emit_connect
 * @widget : GTransConnDialog widget 
 *
 * Emit the "connect" signal. 
 **/
static void
gtrans_conn_dialog_emit_connect(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_conn_dialog_signals[CONNECT]);	
}




/**
 * gtrans_conn_dialog_new:
 * @plugins : A GList of found (GTransIFasePlugins) plugins.
 *
 * Create the connection dialog that allows the user to select a plugin
 * to establish a connection to a database server.
 */
GtkWidget *
gtrans_conn_dialog_new(GList *plugins)
{
	GTransConnDialog   *dialog;
	GtkWidget          *vbox;
	GtkWidget          *sep1;
	GtkWidget          *table;
	GtkWidget          *label;
	GtkWidget          *hbox;
	GtkWidget          *conn_button;
	GtkWidget          *cancel_button;
	GtkWidget          *sep2;
	GtkWidget          *scrolled_win;
	
	dialog = gtk_type_new(gtrans_conn_dialog_get_type());
	gtk_window_set_title(GTK_WINDOW(dialog),N_("Connection Dialog"));
	gtk_window_set_policy (GTK_WINDOW (dialog), TRUE, FALSE, TRUE);
	gtk_container_set_border_width(GTK_CONTAINER(dialog),10);
	gtk_signal_connect(GTK_OBJECT(dialog),"delete-event",
			   GTK_SIGNAL_FUNC(gtk_widget_destroy),
			   (gpointer) NULL);
	
	vbox = gtk_vbox_new (FALSE,10);
	gtk_container_add(GTK_CONTAINER(dialog),vbox);
	
	dialog->dbsel = gtrans_conn_dialog_plugin_clist_new(dialog,plugins);	
	
	scrolled_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_widget_set_usize(scrolled_win,200,120);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW (scrolled_win), 
					      dialog->dbsel);
	gtk_box_pack_start(GTK_BOX(vbox),scrolled_win,TRUE,TRUE,0);
	
	sep1 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep1,FALSE,TRUE,0);


	table = gtk_table_new(5,2,FALSE);
	gtk_box_pack_start(GTK_BOX(vbox),table,TRUE,TRUE,0);
	
	/* Db */
	label = gtk_label_new(_("Database: "));
	gtk_misc_set_alignment(GTK_MISC(label),0.0,0.0);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
		
	dialog->db = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), dialog->db, 1, 2, 0, 1);
	
	/* Host */
	label = gtk_label_new(_("Host: "));
	gtk_misc_set_alignment(GTK_MISC(label),0.0,0.0);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	
	dialog->host = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), dialog->host, 1, 2, 1, 2);
	
	/* Port */
	label = gtk_label_new(_("Port: "));
	gtk_misc_set_alignment(GTK_MISC(label),0.0,0.0);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	
	dialog->port = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), dialog->port, 1, 2, 2, 3);
	
	/* Username */
	label = gtk_label_new(_("Username: "));
	gtk_misc_set_alignment(GTK_MISC(label),0.0,0.0);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
	
	dialog->login = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), dialog->login, 1, 2, 3, 4);
			 	
	/* Pwd */
	label = gtk_label_new(_("Password: "));
	gtk_misc_set_alignment(GTK_MISC(label),0.0,0.0);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
	
	dialog->pwd = gtk_entry_new();
	gtk_entry_set_visibility(GTK_ENTRY(dialog->pwd),FALSE);
	gtk_table_attach_defaults(GTK_TABLE(table), dialog->pwd, 1, 2, 4, 5);
	
	
	sep2 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep2,FALSE,FALSE,0);
	
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox),GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox),0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
	conn_button = gtrans_button_with_stock_n_label(GNOME_STOCK_BUTTON_OK,
						       _("Connect"));
	GTK_WIDGET_SET_FLAGS(conn_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbox),conn_button,TRUE,TRUE,0);
	gtk_widget_grab_default(conn_button);
	gtk_signal_connect_object(GTK_OBJECT(conn_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_conn_dialog_emit_connect),
				  GTK_OBJECT(dialog));
	
	cancel_button = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
	GTK_WIDGET_SET_FLAGS(cancel_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbox),cancel_button,TRUE,TRUE,0);
	gtk_signal_connect_object(GTK_OBJECT (cancel_button),"clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  GTK_OBJECT (dialog));
	
	gtk_widget_show_all(vbox);
	
	return GTK_WIDGET(dialog); 
}
