/* GNOME Transcript
 * Copyright (C) 1999-2000 the Free Software Foundation 
 * Authors : Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
 * The table designer widget implementation. 
 * It uses the plugins to define itself for 
 * each specific database server.
 */


#include <gnome.h>
#include <gtrans_ifase/gtrans_ifase.h>

#include "gtkmatrix.h"
#include "gtrans_table_designer.h"
#include "gtrans_button.h"


enum {
	OK,
	CLOSE,
	LAST_SIGNAL
};



static guint gtrans_table_designer_signals[LAST_SIGNAL] = { 0 };

static GtkWindow *parent_class = NULL;



/**
 * gtrans_table_designer_destroy
 * @object : Tje GTransTableDesigner
 *
 * Destroy the GTransTableDesigner.
 */
static void
gtrans_table_designer_destroy(GtkObject *object)
{
	GTransTableDesigner  *gtd;
	GList                *tmp, *aux;
	gint                  i;
	gchar               **fattrs;
	
	g_return_if_fail(object != NULL);
	g_return_if_fail(GTRANS_IS_TABLE_DESIGNER(object));
	
	gtd = GTRANS_TABLE_DESIGNER(object);
	
	
	/* 
	 * The list of attributes names 
	 */
	if (gtd->attr_names){

		for (tmp = gtd->attr_names ; tmp != NULL ; tmp = tmp->next)
			if (tmp->data)
				g_free(tmp->data);
		g_list_free(gtd->attr_names);
	}

	
	/* 
	 * The list of field types 
	 */
	if (gtd->field_types){
		
		for (tmp = gtd->field_types ; tmp != NULL ; tmp = tmp->next)
			if (tmp->data)
				g_free(tmp->data);
		g_list_free(gtd->field_types);
	}

	
	/* 
	 * The list of filed masks 
	 */
	if (gtd->type_masks)
		g_list_free(gtd->type_masks);
	
	
	/* 
	 * The list of attributes values 
	 */
	if (gtd->attr_vals){
		
		for (tmp = gtd->attr_vals ; tmp != NULL ; tmp = tmp->next){
			
			if (tmp->data == NULL)
				continue;
			
			for (aux = tmp->data ; aux != NULL ; aux = aux->next)
				if (aux->data)
					g_free(aux->data);
			g_list_free(tmp->data);
			
		}
		g_list_free(gtd->attr_vals);
	}

	
	/* 
	 * The list of attributes entrys list, I dont free th entrys
	 * those are freed when the frame is destroyed 
	 */
	if (gtd->attr_entrys)
		g_list_free(gtd->attr_entrys);
	
	
	/*
	 * The attributes data sets
	 */
	if (gtd->attr_data){
		
		for (tmp = gtd->attr_data ; tmp != NULL ; tmp = tmp->next){
			
			if ( (fattrs = (gchar**)tmp->data) == NULL)
				continue;
			
			for (i = 0 ; i < gtd->nattrs ; i++)
				if (fattrs[i])
					g_free(fattrs[i]);
			
			g_free(fattrs);
		}
		g_list_free(gtd->attr_data);
	}
	
	
	/* 
	 * The entry and combo 
	 */
	gtk_widget_destroy(gtd->entry);
	gtk_widget_destroy(gtd->combo);

	
	/* 
	 * The Designer, and everything contained in it 
	 */
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}





/**
 * gtrans_table_designer_class_init
 *
 * Widget stuff.
 */
static void
gtrans_table_designer_class_init(GTransTableDesignerClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	
	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;
	
	gtrans_table_designer_signals[OK] =
		gtk_signal_new ("ok",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransTableDesignerClass, ok),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtrans_table_designer_signals[CLOSE] =
		gtk_signal_new ("close",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransTableDesignerClass, close),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	
	gtk_object_class_add_signals(object_class, 
				     gtrans_table_designer_signals,
				     LAST_SIGNAL);
	
	parent_class = gtk_type_class(gtk_window_get_type());

	/* Destroy */
	object_class->destroy = gtrans_table_designer_destroy;
}




/**
 * gtrans_table_designer_init
 *
 * Initialize some values for the GTransTableDesigner.
 */
static void
gtrans_table_designer_init(GTransTableDesigner *gtd)
{
	gtd->active_col = 0;
	gtd->active_row = 0;
	gtd->attr_entrys = NULL;
	gtd->attr_data   = NULL;
}




/**
 * gtrans_table_designer_get_type
 *
 * Widget stuff
 */
guint 
gtrans_table_designer_get_type()
{
	static guint gtrans_table_designer_type = 0;
	
	if(!gtrans_table_designer_type) {
		GtkTypeInfo gtrans_table_designer_info = {
			"GTransTableDesigner",
			sizeof(GTransTableDesigner),
			sizeof(GTransTableDesignerClass),
			(GtkClassInitFunc) gtrans_table_designer_class_init,
			(GtkObjectInitFunc) gtrans_table_designer_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		gtrans_table_designer_type = 
			gtk_type_unique(gtk_window_get_type(),
					&gtrans_table_designer_info);
	}
	return gtrans_table_designer_type;
}



/**
 * gtrans_table_designer_emit__ok
 *
 * Emits the "ok" signal, called when the ok button is clicked.
 */
static void
gtrans_table_designer_emit__ok(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_table_designer_signals[OK]);
}



/**
 * gtrans_table_designer_emit__close
 *
 * Emits the "close" signal, called when the ok button is clicked.
 */
static void
gtrans_table_designer_emit__close(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_table_designer_signals[CLOSE]);
}








/**
 * gtrans_table_designer_alloc_attr_data
 * @gtd : The GTransTableDesigner
 *
 * Allocates room for one more attribute data set, this is,
 * appends to @gtd->attr_data an arry of NULL gchar pointers.
 * The arry must be size @gtd->nattrs (nomber of attribute entries).
 */
static void
gtrans_table_designer_alloc_attr_data(GTransTableDesigner *gtd)
{
	gchar    **data;
		
	g_return_if_fail(gtd != NULL);
		
	data = g_new0(gchar*,gtd->nattrs);
	
	gtd->attr_data = g_list_append(gtd->attr_data,data);
}





/**
 * gtrans_table_designer_get_attr_entry_text
 * @widget : The widget of an attribute entry.
 *
 * Returns the text written in the widget.
 */
static gchar *
gtrans_table_designer_get_attr_entry_text(GtkWidget *widget)
{
	gchar *text;
	
	g_return_val_if_fail(widget != NULL, NULL);
	
	if (GTK_IS_COMBO(widget))
		text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(widget)->entry));
	
	else if (GTK_IS_ENTRY(widget))
		text = gtk_entry_get_text(GTK_ENTRY(widget));
	
	else 
		return NULL;
	
	if (strlen(text) == 0)
		return NULL;
	
	return text;
}




/**
 * gtrans_table_designer_set_attr_entry_text
 * @widget : The widget of an attribute entry
 * @text   : Some text
 *
 * Sets the text in the widget.
 */
static void
gtrans_table_designer_set_attr_entry_text(GtkWidget *widget,
					  gchar     *text)
{
	g_return_if_fail(widget != NULL);
	
	if (text == NULL) text = "";
	
	if (GTK_IS_COMBO(widget))
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(widget)->entry),text);
	
	else if (GTK_IS_ENTRY(widget))
		gtk_entry_set_text(GTK_ENTRY(widget),text);
}




/**
 * gtrans_table_designer_is_attr_editable
 * @mask  : An integer that reperesents a bit mask for a 
 *          field type attributes
 * @pos   : A position of a bit
 *
 * Return 1 if the bit as @pos is set, otherwise return 0.
 */
static gint
gtrans_table_designer_is_attr_editable(gint mask,gint pos)
{
	return ( ( mask >> pos ) & ~( ~0 << 1 ) );
}





/**
 * gtrans_table_designer_get_field_type_mask
 * @gtd  : A GTransTableDesigner
 * @type : A field type name
 *
 * Returns a integer number (the bit mask) of the field type
 * attributes.
 */
static gint
gtrans_table_designer_get_field_type_mask(GTransTableDesigner *gtd,
					  gchar               *type)
{
	GList   *tmp;
	gint     i = 0;
	
	if (type == NULL)
		return 0;
	
	for (tmp = gtd->field_types ; tmp != NULL ; tmp = tmp->next){
		
		if (!strcmp(type,(gchar *)tmp->data))
		 	return (gint) g_list_nth_data(gtd->type_masks,i);
		i++;
	}
	return 0;
}




/**
 * gtrans_table_designer_parse_attr_entrys
 * @gtd  : A GTransTableDesigner
 * @row  : The row of the field definition.
 *
 * Parse and set sensitive or insensitive the entrys of the
 * field arrtibutes, acording to their corresponding
 * field-type mask.
 */
static void
gtrans_table_designer_parse_attr_entrys(GTransTableDesigner *gtd,
					gint                 row)
{
	GtkWidget    *widget;
	gchar        *text;
	gint          i = 0, mask, editable;
	
	text = gtk_matrix_cell_get_text(GTK_MATRIX(gtd->matrix),row,1);
	mask = gtrans_table_designer_get_field_type_mask(gtd,text);
	
	for ( i = 0 ; i < gtd->nattrs ; i++ ){
		
		widget = g_list_nth_data(gtd->attr_entrys,i);
		
		editable = gtrans_table_designer_is_attr_editable(mask,i);
		gtk_widget_set_sensitive(widget,editable);
		
		if (!editable)
			gtrans_table_designer_set_attr_entry_text(widget,NULL);
	}	
}





/**
 * gtrans_table_designer_open_attr_entrys_data
 *
 */
static void
gtrans_table_designer_open_attr_entrys_data(GTransTableDesigner *gtd,
					    gint                 row)
{
	GtkWidget   *entry;
	gchar      **data;
	gint         i;

	data = g_list_nth_data(gtd->attr_data,row);
	
	for (i = 0 ; i < gtd->nattrs ; i++){
		entry = g_list_nth_data(gtd->attr_entrys,i);
		gtrans_table_designer_set_attr_entry_text(entry,data[i]);
	}
}




/**
 * gtrans_table_designer_save_attr_entrys_data
 *
 */
static void
gtrans_table_designer_save_attr_entrys_data(GTransTableDesigner *gtd)
{
	GtkWidget   *entry;
	gchar      **data, *text;
	gint         i;

	data = g_list_nth_data(gtd->attr_data,gtd->active_row);
	
	for (i = 0 ; i < gtd->nattrs ; i++){
		
		if (data[i])
			g_free(data[i]);
		
		entry = g_list_nth_data(gtd->attr_entrys,i); 
		text = gtrans_table_designer_get_attr_entry_text(entry);
		data[i] = g_strdup(text);
	}
}




/**
 * gtrans_table_designer_reset_attr_entrys
 * @gtd   : The GTransTableDesigner
 * @row   : A row's number
 *
 * Copies the text in the attribute entries to it's
 * correspondin attribte data set, and
 * sets the attribute data set of the new selected @row
 * in to the attribute entries.
 */
static void
gtrans_table_designer_reset_attr_entrys(GTransTableDesigner *gtd,
					gint                 row)
{
	gtrans_table_designer_save_attr_entrys_data(gtd);
	gtrans_table_designer_open_attr_entrys_data(gtd,row);
}




/**
 * gtrans_table_designer_add_field
 * @gtd : The GTransTableDesigner
 *
 * Adds a row to the matrix and allocs room for one more
 * attribute data set.
 */
static void
gtrans_table_designer_add_field(GTransTableDesigner *gtd)
{
	GtkMatrix      *matrix;
	
	g_return_if_fail(gtd != NULL);
	
	matrix = GTK_MATRIX(gtd->matrix);
	gtk_matrix_add_row(GTK_MATRIX(matrix),1);
	
	gtrans_table_designer_alloc_attr_data(gtd);
}




/**
 * gtrans_table_designer_cell_changed
 * @matrix : A GtkMatrix
 * @row   : the row of the cell changed
 * @col   : the col of the cell changed
 * @gtd   : the GTransTableDesigner passes as user data
 *
 * Takes care of updating the GTransTableDesigner when a cell 
 * has changed it's text. 
 * This function if needed calls *gtrans_table_designer_parse_field_attrs*
 * that takes care of validating the field attributes according
 * to their field type mask.
 *
 * Also if the cell belongs to the last row adds space for a 
 * field definition
 */
static void
gtrans_table_designer_cell_changed(GtkMatrix               *matrix,
				   gint                     row,
				   gint                     col,
				   GTransTableDesigner     *gtd)
{
	if (col == 1)
		gtrans_table_designer_parse_attr_entrys(gtd,row);
	
	if (row == matrix->maxrow)
		gtrans_table_designer_add_field(gtd);		
}



/**
 * gtrans_table_designer_cell_activated
 * @matrix : The GtkMatrix that emited the "activated" signal
 * @row   : The row of the cell that was "activated"
 * @col   : The col of the cell that was "activated"
 * @gtd   : The GTransTableDesigner passed as user data
 *
 * This function is called whenever a cell is "activated".
 * It takes care of changing the matrix_entry when needed 
 * acording to the column the "activated" cell belongs to.
 *
 * It also sets the data corresponding to @row in the
 * attribute entries, and parses the attribute entries.
 */
static void
gtrans_table_designer_cell_activated(GtkMatrix               *matrix,
				     gint                     row,
				     gint                     col,
				     GTransTableDesigner     *gtd)
{
	if (col != gtd->active_col){
		if (col == 0)
			gtk_matrix_change_entry(matrix,gtd->entry);
		else
			gtk_matrix_change_entry(matrix,gtd->combo);
	}
	
	if(gtd->active_row != row){
		gtrans_table_designer_reset_attr_entrys(gtd,row);
		gtrans_table_designer_parse_attr_entrys(gtd,row);
	}
	
	gtd->active_row = row;
	gtd->active_col = col;
}




/**
 * gtrans_table_designer_init_attr_data
 * @gtd : The GTransTableDesigner
 *
 * Initializes @gtd->attr_data, this is, allocates room for
 * 5 set's of attributes data, 1 for each row initialized 
 * in the designer.
 */
static void
gtrans_table_designer_init_attr_data(GTransTableDesigner *gtd)
{
	gint  i;
	
	g_return_if_fail(gtd != NULL);

	for (i = 0 ; i < 5 ; i++)
		gtrans_table_designer_alloc_attr_data(gtd);
}





/**
 * gtrans_table_designer_build_frame
 * @gtd  : The GTransTableDesigner
 *
 * Builds in @gtd->frame the frame for the attribute entries.
 * The attribute entries are the entries (GtkCombo or GtkEntry)
 * where we enter each field attribute values.
 * Each attribute entry has a label (the attribute name), 
 * and are arranged in a table to look be aligned.
 *
 * How do we know the type (combo or entry) of the attribute entry?
 * these are defined by the plugin. The plugin must have a function
 * that returns an attribute entry values list, this is a GList with
 * one link for each attribute entry we want to create, if the link
 * points to NULL the entry is a GtkEntry, if the link points to
 * abother GList of strings then we have a GtkCombo.
 * This attribute entry value list is read when we create the designer
 * and is kept in @gtd->attr_vals.
 */
static void
gtrans_table_designer_build_frame(GTransTableDesigner *gtd)
{
	GtkWidget   *table;
	GtkWidget   *widget, *name_label;
	gint         i;  
	GList       *strings;
	gchar       *name;
	
	g_return_if_fail(gtd != NULL);
	
	gtd->frame = gtk_frame_new("Field Attributes");
	gtk_container_set_border_width(GTK_CONTAINER(gtd->frame),8);
	
	table = gtk_table_new(gtd->nattrs,2,FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table),5);
	gtk_container_add(GTK_CONTAINER(gtd->frame),table);

	
	for (i = 0 ; i < gtd->nattrs ; i++){
		
		strings = g_list_nth_data(gtd->attr_vals,i);
		name    = g_list_nth_data(gtd->attr_names,i);
		
		name_label = gtk_label_new(name);
		gtk_misc_set_alignment(GTK_MISC(name_label), 0.0, 0.0);
		gtk_table_attach(GTK_TABLE(table), name_label, 0, 1, i, i+1,
				 GTK_FILL, GTK_EXPAND, 15, 2);
		
		if (strings == NULL)
			widget = gtk_entry_new();
		else {
			widget = gtk_combo_new();
			gtk_combo_set_popdown_strings(GTK_COMBO(widget),strings);
			gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(widget)->entry),FALSE);
		}

		gtk_table_attach(GTK_TABLE(table), widget, 1, 2, i, i+1,
				 GTK_FILL, GTK_EXPAND, 15, 2);
		
		gtk_widget_set_sensitive(widget,FALSE);
		
		/* Add the entry to the glist of this attr entries */
		gtd->attr_entrys = g_list_append(gtd->attr_entrys,widget);
	}
}





/**
 * gtrans_table_designer_build_matrix
 * @gtd  : The GTransTableDesigner
 * 
 * Builds the GtkMatrix for the fields name and type 
 * definition.
 * This funcion must be called after @gtd->field_types
 * has been retrived correctly from the plugin set.
 */
static void
gtrans_table_designer_build_matrix(GTransTableDesigner *gtd)
{
	g_return_if_fail(gtd != NULL);
	
	gtd->entry = gtk_entry_new();
	
	gtd->combo = gtk_combo_new();
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(gtd->combo)->entry),FALSE);
	gtk_combo_set_popdown_strings(GTK_COMBO(gtd->combo),gtd->field_types);
	
	gtd->matrix = gtk_matrix_new_with_custom_entry(5,2,gtd->entry);
	
	
	gtk_matrix_column_button_add_label(GTK_MATRIX(gtd->matrix),0,"Name");
	gtk_matrix_column_button_add_label(GTK_MATRIX(gtd->matrix),1,"Type");
	GTK_MATRIX_SET_FLAGS(GTK_MATRIX(gtd->matrix),GTK_MATRIX_AUTORESIZE);
	GTK_MATRIX_SET_FLAGS(GTK_MATRIX(gtd->matrix),GTK_MATRIX_ROW_FROZEN);	
	
	gtk_signal_connect(GTK_OBJECT(gtd->matrix),"activate",
			   GTK_SIGNAL_FUNC(gtrans_table_designer_cell_activated),
			   (gpointer) gtd);
	
	gtk_signal_connect(GTK_OBJECT(gtd->matrix),"entry_changed",
			   GTK_SIGNAL_FUNC(gtrans_table_designer_cell_changed),
			   (gpointer) gtd);
}





/**
 * gtrans_table_designer_new
 * @conn  : The GTransIFaseConn to the database we are working on.
 * @owner : The widget who will be the owner of the designer,
 *          should be the GTransDbEditor who poped it.
 *
 * Create a new GTransTableDesigner which will commit the design
 * to @conn and signal to @owner that should update itself.
 * A GTransTableDesigner allows us to create new tables in the database.
 */
GtkWidget *
gtrans_table_designer_new(GTransIFaseConn *conn,
			  GtkWidget       *owner)
{
	GTransTableDesigner   *gtd;
	GtkWidget             *vbox;
	GtkWidget             *hbox;
	
	GtkWidget             *lab_name;
	GtkWidget             *sep1;
	
	GtkWidget             *scrolled;
	
	GtkWidget             *hbutt_box;
	GtkWidget             *ok_button;
	GtkWidget             *cancel_button;
	GtkWidget             *sep2;

	g_return_val_if_fail(conn != NULL, NULL);
	

	gtd = gtk_type_new(gtrans_table_designer_get_type());
	gtd->conn = conn;
	gtd->owner = owner;
	gtk_window_set_title(GTK_WINDOW(gtd),"Table Designer");
	gtk_signal_connect(GTK_OBJECT(gtd),"delete-event",
			   GTK_SIGNAL_FUNC(gtk_widget_destroy),
			   (gpointer) NULL);
	
	
	vbox = gtk_vbox_new(FALSE,10);
	gtk_container_set_border_width(GTK_CONTAINER(vbox),10);
	gtk_container_add(GTK_CONTAINER(gtd),vbox);
	
	hbox = gtk_hbox_new(FALSE,5);
        gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
	lab_name = gtk_label_new("Table Name:");
	gtk_box_pack_start(GTK_BOX(hbox),lab_name,FALSE,FALSE,0);
	
	gtd->name_e = gtk_entry_new();
 	gtk_box_pack_start(GTK_BOX(hbox),gtd->name_e,FALSE,FALSE,0);
	
	sep1 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep1,FALSE,FALSE,0);
	
	scrolled = gtk_scrolled_window_new(NULL,NULL);
	gtk_widget_set_usize(scrolled,400,150);
	gtk_box_pack_start_defaults(GTK_BOX(vbox),scrolled);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	
	
	/* Read the definition from the plugin */
	gtd->attr_names   = gtrans_ifase_table_designer_attr_names(conn);
	gtd->nattrs       = g_list_length(gtd->attr_names);
	gtd->attr_vals    = gtrans_ifase_table_designer_attr_vals(conn);
	gtd->field_types  = gtrans_ifase_table_designer_field_types(conn);
	gtd->type_masks   = gtrans_ifase_table_designer_type_masks(conn);
	

	/* Create the matrix */
	gtrans_table_designer_build_matrix(gtd);
	gtk_container_add(GTK_CONTAINER(scrolled),gtd->matrix);
	gtrans_table_designer_init_attr_data(gtd);

		
	/* Create the attributes frame */
	gtrans_table_designer_build_frame(gtd);
	gtk_box_pack_start(GTK_BOX(vbox),gtd->frame,FALSE,FALSE,0);
	
	
	sep2 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep2,FALSE,FALSE,0);
	
	hbutt_box = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbutt_box),GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbutt_box),0);
        gtk_box_pack_start(GTK_BOX(vbox),hbutt_box,FALSE,FALSE,0);
	
	ok_button = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
	GTK_WIDGET_SET_FLAGS(ok_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbutt_box),ok_button,TRUE,TRUE,0);
	gtk_widget_grab_default(ok_button);
	gtk_signal_connect_object(GTK_OBJECT(ok_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_table_designer_emit__ok),
				  GTK_OBJECT(gtd));
	
	cancel_button = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
	GTK_WIDGET_SET_FLAGS(cancel_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbutt_box),cancel_button,TRUE,TRUE,0);
	gtk_signal_connect_object(GTK_OBJECT(cancel_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_table_designer_emit__close),
				  GTK_OBJECT(gtd));
	
	gtk_signal_connect_object(GTK_OBJECT(gtd),"delete-event",
				  GTK_SIGNAL_FUNC(gtrans_table_designer_emit__close),
				  GTK_OBJECT(gtd));
	
	gtk_widget_show_all(vbox);
	
	return GTK_WIDGET(gtd); 
}




/**
 * gtrans_table_designer_get_table_name
 * @gtd  : The GTransTableDesigner
 *
 * Returns the name of the table defined in @gtd.
 */
gchar *
gtrans_table_designer_get_table_name(GTransTableDesigner *gtd)
{
	g_return_val_if_fail(gtd != NULL, NULL);
	
	return gtk_entry_get_text(GTK_ENTRY(gtd->name_e));
}




/**
 * gtrans_table_designer_get_field_def 
 * @gtd : The GTransTableDesigner.
 * @row : The number of a row (or field definition)
 *
 * Returns the definition of the field defined in the row @row 
 * (The definition of a field is represented by an arry of strings).
 * Each of the strings of the the arry's is the string you entered
 * while designing you table in the table designer, so 
 * arry[0] -> field's name
 * arry[1] -> field's type
 * arry[2] -> field's attribute 0
 * arry[3] -> field's attribute 1
 * and so on...
 *
 * The "name" and "type" are read from the matrix widget,
 * each of the field attributes are read from the @gtd->attr_data
 * list that holds the values entered in the attribute entries for
 * each row in the designer.
 */
static gchar **
gtrans_table_designer_get_field_def(GTransTableDesigner *gtd,
				    gint                 row)
{
	GtkMatrix  *matrix;
	gchar     **fdef, **fattr;
	gint        i;
	
	g_return_val_if_fail(gtd != NULL, NULL);
	g_return_val_if_fail(GTK_MATRIX(gtd->matrix)->maxrow > row, NULL);
	
	matrix = GTK_MATRIX(gtd->matrix);
	
	fattr  = g_list_nth_data(gtd->attr_data,row);
	
	fdef = g_new0(gchar*, gtd->nattrs+2);
	
	fdef[0] = g_strdup(gtk_matrix_cell_get_text(matrix,row,0));
	fdef[1] = g_strdup(gtk_matrix_cell_get_text(matrix,row,1));
	
	if (fdef[0] == NULL && fdef[1] == NULL)
		return NULL;
	
	for ( i = 0 ; i < gtd->nattrs ; i++)
		fdef[i+2] = g_strdup(fattr[i]);
	
	return fdef;		
}




/**
 * gtrans_table_designer_get_table_def
 * @gtd : The GTransTableDesigner.
 *
 * Returns the design of the table.
 * The design of the table is defined by a GList*, where each
 * link points to an arry of strings, each of the arry's defines
 * one field, and all together define the table.
 * Each of the strings of the the arry's is the string you entered
 * while designing you table in the table designer, so 
 * arry[0] -> field's name
 * arry[1] -> field's type
 * arry[2] -> field's attribute 0
 * arry[3] -> field's attribute 1
 * and so on...
 * With this information the table is created by a function in the plugin.
 *
 * Note : This definition must be freed by the plugin after the 
 *        table is created.
 */
GList *
gtrans_table_designer_get_table_def(GTransTableDesigner *gtd)
{
	GList      *list = NULL;
	gchar     **fdef;
	gint        rows , i;
	
	g_return_val_if_fail(gtd != NULL, NULL);

	rows = GTK_MATRIX(gtd->matrix)->maxrow; /* The last row is never filled */
	
	/* MAYBE : Copy the text in the entrys to the attr data set */
	gtrans_table_designer_save_attr_entrys_data(gtd);
	
	for ( i = 0 ; i < rows ; i++ ){
		fdef = gtrans_table_designer_get_field_def(gtd,i);
		list = g_list_append(list,fdef);
	}
	return list;
}




