/*
 * Common environment for most program functions
 *
 * Copyright (C) 2003--2008  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME "missing appname"
#define VERSION "unknown"
#endif

#include "GuessnetEnvironment.h"

#include "GuessnetParser.h"
#include "IfaceParser.h"

#include <wibble/regexp.h>
#include <wibble/commandline/parser.h>
#include <util/output.h>
#include "util/netsender.h"
#include "util/netwatcher.h"
#include "scanner/iwscan.h"
#include <set>

#include <stdio.h>
#include <stdarg.h>

namespace wibble {
namespace commandline {

struct GuessnetOptions : public StandardParserWithManpage
{
public:
	BoolOption* verbose;
	BoolOption* debug;
	BoolOption* syslog;
	BoolOption* ifupdown;
	BoolOption* autofilter;
	StringOption* defprof;
	IntOption* timeout;
	IntOption* inittime;
	IntOption* initdelay;
	ExistingFileOption* configfile;

	GuessnetOptions() 
		: StandardParserWithManpage(APPNAME, VERSION, 8, "enrico@enricozini.org")
	{
		usage = "[options] [iface]";
		description = "Guess the current network location";

		verbose = add<BoolOption>("verbose", 'v', "verbose", "",
						"enable verbose output");
		debug = add<BoolOption>("debug", 0, "debug", "",
						"enable debugging output (including verbose output)");
		syslog = add<BoolOption>("syslog", 0, "syslog", "",
						"send messages to syslog facility DAEMON, in addition to stderr");
		configfile = add<ExistingFileOption>("configfile", 'C', "config-file", "",
						"name of the configuration file to read (default: stdin or"
						"/etc/network/interfaces in ifupdown mode");
		ifupdown = add<BoolOption>("ifupdown", 'i', "ifupdown-mode", "",
						"use /etc/network/interfaces file instead of the usual"
						" guessnet configuration file");
		defprof = add<StringOption>("defprof", 'd', "default", "name",
						"profile name to report if no known networks are found"
						" (defaults to \"none\")");
		timeout = add<IntOption>("timeout", 't', "timeout", "seconds",
						"timeout (in seconds) used to wait for response packets"
						" (defaults to 5 seconds)");
		inittime = add<IntOption>("inittime", 0, "init-time", "seconds",
						"time (in seconds) to wait for the interface to initialize"
						" when not found already up (defaults to 3 seconds)");
		autofilter = add<BoolOption>("autofilter", 0, "autofilter", "",
						"enable autofiltering interfaces based on their name (default: off)");
		initdelay = add<IntOption>("initdelay", 0, "init-delay", "seconds",
		                           "sleep a given number of seconds before starting operations (default: 0)");
	}
};

}
}

using namespace std;

static GuessnetEnvironment* instance;

GuessnetEnvironment& GuessnetEnvironment::get() throw ()
{
	return *instance;
}


class StandaloneEnvironment : public GuessnetEnvironment
{
protected:
	string configFile;

public:
	StandaloneEnvironment(wibble::commandline::GuessnetOptions& opts)
	{
		// Set verbosity
		util::Output::get().verbose(opts.verbose->boolValue());
		util::Output::get().debug(opts.debug->boolValue());
    util::Output::get().syslog(opts.syslog->boolValue());

		// Find out the interface to be tested
		if (opts.hasNext())
			iface(opts.next());
		else
			iface("eth0");

		// Find out the configuration file to use
		if (opts.hasNext())
			configFile = opts.next();
		else if (opts.configfile->isSet())
			configFile = opts.configfile->stringValue();

		// Find out the default profile name
		if (opts.defprof->boolValue())
		{
			defprof(opts.defprof->stringValue());
			::verbose("Default profile set to `%s'\n", defprof().c_str());
		}

		// Find out the test timeout
		if (opts.timeout->boolValue())
			timeout(opts.timeout->intValue());

		// Find out the init timeout
		if (opts.inittime->boolValue())
			initTimeout(opts.inittime->intValue());

		// Set autofiltering
		autofilter(opts.autofilter->boolValue());

		// Set initial delay to avoid race conditions
		initdelay(opts.initdelay->intValue());
	}

	virtual void parseConfigFile();
};

void StandaloneEnvironment::parseConfigFile()
{
	/* Open the specified config file or stdin if not specified */
	FILE* input = stdin;
	if (!configFile.empty())
	{
		input = fopen(configFile.c_str(), "rt");
		if (!input)
			throw wibble::exception::File(configFile, "opening file");
	}

	GuessnetParser::parse(input);

	if (input != stdin)
		fclose(input);
}


class Tokenizer
{
protected:
	std::string str;
	std::string::size_type s;
public:
	Tokenizer(const std::string& str) throw ()
		: str(str), s(0) {}

	std::string next()
	{
		// Skip leading spaces
		while (s < str.size() && isspace(str[s]))
			s++;

		if (s == str.size()) return string();
		
		string::size_type start = s;

		while (s < str.size() && !isspace(str[s]))
			s++;

		return str.substr(start, s - start);
	}
};


class IfupdownEnvironment : public StandaloneEnvironment, public IfaceFilter
{
protected:
	set<string> ifupdownProfiles;
	bool ifupdownProfilesMatchInverted;

public:
	IfupdownEnvironment(wibble::commandline::GuessnetOptions& opt) :
		StandaloneEnvironment(opt), ifupdownProfilesMatchInverted(false)
	{
		// We have a default config file name in ifupdown mode
		if (configFile.empty())
			configFile = "/etc/network/interfaces";

		::debug("program name is guessnet-ifupdown: enabling ifupdown mode\n");

		// Read stuff from stdin
		wibble::ERegexp null_line("^[[:blank:]]*(#.*)?$");
		wibble::ERegexp parm_line("^[[:blank:]]*([A-Za-z_-]+):[[:blank:]]*(.+)$", 3);

		string line;
		int linenum = 1;
		int c;
		while ((c = fgetc(stdin)) != EOF)
		{
			if (c != '\n')
				line += c;
			else
			{
				if (null_line.match(line))
				{
					//fprintf(stderr, "EMPTY\n");
				}
				else if (parm_line.match(line))
				{
					string name = parm_line[1];
					Tokenizer t(parm_line[2]);
					string val = t.next();

					if (name == "default")
					{
						if (!val.empty())
							defprof(val);
					}
					else if (name == "verbose")
					{
						if (!val.empty())
							util::Output::get().verbose(val == "true");
					}
					else if (name == "debug")
					{
						if (!val.empty())
							util::Output::get().debug(val == "true");
					}
					else if (name == "syslog")
					{
						if (!val.empty())
							util::Output::get().syslog(val == "true");
					}
					else if (name == "timeout")
					{
						int v = atoi(val.c_str());
						if (v > 0)
							timeout(v);
					}
					else if (name == "init-time")
					{
						int v = atoi(val.c_str());
						if (v > 0)
							initTimeout(v);
					}
					else if (name == "autofilter")
					{
						autofilter(val == "true");
					}
					else if (name == "init-delay")
					{
						int v = atoi(val.c_str());
						if (v > 0)
							initdelay(v);
					}
				}
				else
				{
					Tokenizer t(line);
					bool first = true;
					for (string w = t.next(); !w.empty(); w = t.next())
					{
						if (first)
						{
							ifupdownProfilesMatchInverted = w[0] == '!';
							first = false;
						}
						if (w[0] == '!')
						{
							if (!ifupdownProfilesMatchInverted)
								throw wibble::exception::Consistency(
										"parsing list of interfaces to use",
										"found negated interface "+w+" after a normal interface");
							ifupdownProfiles.insert(w.substr(1));
							std::string dm = "Added " + w + "\n";
							::debug(dm.c_str());
						} else {
							if (ifupdownProfilesMatchInverted)
								throw wibble::exception::Consistency(
										"parsing list of interfaces to use",
										"found normal interface "+w+" after negated interface");
							ifupdownProfiles.insert(w);
							std::string dm = "Added " + w + "\n";
							::debug(dm.c_str());
						}
					}
				}
				line = string();
				linenum++;
			}
		}
	}

    virtual bool operator()(const std::string& name) const;
	virtual void parseConfigFile();
};

bool IfupdownEnvironment::operator()(const std::string& name) const
{
    bool hasmatch = false;
	if (autofilter()) 
		hasmatch = (name.substr(0, instance->iface().size()+1) == instance->iface()+"-") ;
	else 
		hasmatch = ifupdownProfilesMatchInverted || (ifupdownProfiles.size() == 0);

	if (ifupdownProfilesMatchInverted && ifupdownProfiles.find(name) != ifupdownProfiles.end())
  {
		hasmatch = false;
  }
	else if (!ifupdownProfilesMatchInverted && ifupdownProfiles.find(name) != ifupdownProfiles.end())
  {
		hasmatch = true;
  } 
	return hasmatch;
}

void IfupdownEnvironment::parseConfigFile()
{
	/* Open the specified config file or stdin if not specified */
	FILE* input = fopen(configFile.c_str(), "rt");;
	if (!input)
		throw wibble::exception::File(configFile, "opening file");

	IfaceParser::parse(input, *this);

	fclose(input);
}

void GuessnetEnvironment::init(int argc, const char* argv[])
{
	wibble::commandline::GuessnetOptions opts;

	if (opts.parse(argc, (const char**)argv))
		exit(0);

	// Check user id
	/*
	if (geteuid() != 0)
		fatal_error("You must run this command as root.");
	*/

	// Find out wether we should run in ifupdown mode
	bool ifupdown_mode = opts.ifupdown->boolValue();
	if (!ifupdown_mode)
	{
		const char* pname = strrchr(argv[0], '/');
		pname = pname ? pname + 1 : argv[0];
		if (strcmp(pname, "guessnet-ifupdown") == 0)
			ifupdown_mode = true;
	}

	if (ifupdown_mode)
	{
		instance = new IfupdownEnvironment(opts);
	} else {
		instance = new StandaloneEnvironment(opts);
	}
	Environment::init(instance);

	util::NetSender::configure(instance->iface());
	util::NetWatcher::configure(instance->iface());
	scanner::IWScan::configure(instance->iface());
	
	instance->parseConfigFile();
}


// vim:set ts=4 sw=4:
