/*C* -*-c++-*-
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

/*
 * game.cc
 *
 * This beastie /is/ hatman. The game itself is mostly all in here. The game
 * object manages all game-related variables etc, and it's member functions
 * manage the whole lot.
 *
 * Basically, it is like this: The object is initialised and play() is called,
 * with the filename of the level to be played. play() sets everything up and
 * then calls mainLoop, which is just a big loop. mainLoop() sets quit to
 * false, and exits when it is true.
 *
 * The loop in mainLoop() first checks for keyboard input, and for arrow keys
 * sets the appropriate directions in the player object. The functions
 * moveGhosts(), movePlayer() and moveFruit() are then called, which handle the
 * movement and collisions of the named objects (movePlayer() does not handle
 * any collisions). Finally, mainLoop() keeps track of time, and if there is
 * plenty spare it redraws the screen by calling draw().
 *
 * The Game object is also responsible for making sure no-one bangs into walls,
 * and controls the movement of ghosts and fruit.
 *
 * Game also draws the score and lives indicator (drawScoreBox()).
 *
 * Addition: Game::initGame() must now be called prior to each "game". play()
 * returns true if the player completed the level.
 *
 * Ok, its now completely different in many ways. Never mind. I can't be doing
 * with all this commentiong malarkey.
 */

#include "Background.h"
#include "Game.h"
#include "GameItem.h"
#include "GameItems.h"
#include "Data.h"
#include "Fruit.h"
#include "Player.h"
#include "files.h"
#include "fonts.h"
#include "gameitems.h"
#include "gluetext.h"
#include "level.h"
#include "misc.h"
#include "speedcontrol.h"
#include "../gl/Animation.h"
#include "../gl/Aset.h"
#include "../gl/Console.h"
#include "../gl/Font.h"
#include "../gl/Keyboard.h"
#include "../gl/keycodes.h"
#include "../gl/Sprite.h"
#include "../gl/VgaBlur.h"
#include "../util/Ticker.h"
#include "../util/debug.h"
#include "../util/types.h"
#include "../util/util.h"
#include <sys/stat.h>

Game* game;

//-----------------------------------------------------------------------------
Game::Game()
{
 ghosts = new Collection<Ghost>(8, 8);
 level = new Level;
 player = new Player;
 fruit = new Fruit;
 fruitStarts = new Collection<Pos>(8, 8);
 gtm = new GlueTextManager(casualFont);
 //fontThin->setHAlign(Font::Centre);
 //fontThin->setVAlign(Font::Middle);
 playerSpeed = new SpeedControl;
 playerSlowSpeed = new SpeedControl;
 ghostSpeed = new SpeedControl;
 ghostScaredSpeed = new SpeedControl;
 ghostEyesSpeed = new SpeedControl;
 fruitSpeed = new SpeedControl;

 hatmanAset = ghostAset = fruitAset = NULL;

 gameItems = new GameItems();
}

//-----------------------------------------------------------------------------
Game::~Game()
{
 delete ghosts;
 delete level;
 delete player;
 delete fruit;
 delete fruitStarts;
 delete gtm;
 delete playerSpeed;
 delete playerSlowSpeed;
 delete ghostSpeed;
 delete ghostScaredSpeed;
 delete ghostEyesSpeed;
 delete fruitSpeed;

 if(hatmanAset) delete hatmanAset;
 if(ghostAset) delete ghostAset;
 if(fruitAset) delete fruitAset;
}

//-----------------------------------------------------------------------------
void Game::initGame()
{
 player->reset();
}

//-----------------------------------------------------------------------------
bool Game::playerCanMove(Dir dir)
{
 Pos pos = player->getPos();
 if(pos.x == stageSize.x && (dir == Up || dir == Down)) return false;
 if(pos.y == stageSize.y && (dir == Left || dir == Right)) return false;
 pos += Pos(xInc[dir], yInc[dir]);
 if(pos.x < 0 || pos.x >= lvlW || pos.y < 0 || pos.y >= lvlH) return true;
 int sq = level->getSquare(pos);
 return (sq < 3 || sq > 22)? true : false;
}

//-----------------------------------------------------------------------------
bool Game::ghostCanMove(int i, Dir dir)
{
 Ghost* ghost = ghosts->at(i);
 Pos pos = ghost->getPos();
 if(pos.x == stageSize.x && (dir == Up || dir == Down)) return false;
 if(pos.y == stageSize.y && (dir == Left || dir == Right)) return false;
 pos += Pos(xInc[dir], yInc[dir]);
 if(pos.x < 0 || pos.x >= stageSize.x || pos.y < 0 || pos.y >= stageSize.y) return true;
 int sq = level->getSquare(pos);
 if(levelOptions->ghostCantHide && ghost->state == Ghost::Scared && (sq == 22)) return false;
 return (sq < 3 || sq > 21)? true : false;
}

//-----------------------------------------------------------------------------
bool Game::fruitCanMove(Dir dir)
     /*
      * Returns true if the fruit can move in the given direction.
      */
{
 Pos pos = fruit->getPos() + Pos(xInc[dir], yInc[dir]);
 if(pos.x < 0 || pos.x >= stageSize.x || pos.y < 0 || pos.y >= stageSize.y) return true;
 int sq = level->getSquare(pos);
 return (sq < 3 || sq > 22)? true : false;
}

//-----------------------------------------------------------------------------
void Game::draw()
     /*
      * This is called once a frame to update the screen. The first drawn goes
      * lowest, etc...
      */
{
 if(fruitActive) screen->blur2(fruitAset->at(0)->next()->draw(screen, fruit->screenPos()));
 player->draw();                // player
 every(*ghosts, draw());        // ghosts
 gtm->blur(screen);             // gluetext
 screen->copyBlurs();
}

//-----------------------------------------------------------------------------
void Game::drawScoreBox()
     /*
      * Redraws the score box, including lives.
      */
{
 screen->restore(scoreRect);

 // first redraw the lives indicator
 Point p = scoreRect.topLeft() + (Point(sprW, sprH) - sprExtra->size()) / 2;
 for(int i=0; i+1<player->lives; i++, p.x += sprExtra->size().x)
  sprExtra->draw(screen, p);

 // then draw the score
 casualFont->hAlign = Font::HRight;
 casualFont->vAlign = Font::VMiddle;
 casualFont->writef(screen, scoreRect.topRight() + Point(- scoreRect.h / 4, scoreRect.h / 2), colorWhite, "%d", player->score);

 screen->update(scoreRect);
}

//-----------------------------------------------------------------------------
void Game::kill()
     /*
      * Called when the player loses a life.
      */
{
 if(--player->lives == 0) { quit = true; return; }
 power = 0;
 player->setPos(player->startPos);
 player->setDir(Right);
 player->slowTime = 0;
 player->pause = 0;
 every(*ghosts, restart());
 drawScoreBox();
}

//-----------------------------------------------------------------------------
void Game::moveGhosts(int ticks)
     /*
      * This procedure performs once-per-frame operations on every ghost in turn.
      */
{
 int aliveMoves = ghostSpeed->moves();
 int scaredMoves = ghostScaredSpeed->moves();
 int deadMoves = ghostEyesSpeed->moves();
 foreach(*ghosts, i)
  {
   Ghost* ghost = (*ghosts)[i];
   int moves = (ghost->state == Ghost::Alive)? aliveMoves : ((ghost->state == Ghost::Scared)? scaredMoves : deadMoves);
   if(moves)
    {
     if(!ghost->moving || ghost->move())
      {
       // Setup some useful variables
       Pos g = ghost->getPos(), target = (ghost->state == Ghost::Dead)? base : player->getPos();
       int xDif = abs(target.x - g.x), yDif = abs(target.y - g.y);
       Dir oldDir = ghost->getDir();

       Dir dir1, dir2;
       if(rInt(256) < levelOptions->ghostSkill)
	{
	 int r = rInt(8);
	 // decide which direction the target is
	 Dir dirHoriz = (g.x == target.x)? ((r & 0x01)? Left : Right) : ((g.x > target.x)? Left : Right);
	 Dir dirVert = (g.y == target.y)? ((r & 0x02)? Up : Down) : ((g.y > target.y)? Up : Down);
	 if((xDif == yDif && (r & 0x04)) || (xDif > yDif))
	  {
	   dir1 = dirHoriz;
	   dir2 = dirVert;
	  }
	 else
	  {
	   dir1 = dirVert;
	   dir2 = dirHoriz;
	  }
	 
	 // go the opposite direction if we are running away from the target
	 if(ghost->state == Ghost::Scared) { dir1 = behindOf[dir1]; dir2 = behindOf[dir2]; }
	}
       else // choose a random direction
	{
	 int r = rInt(4);
	 dir1 = (Dir) rInt(4);
	 if(dir1 == Left || dir1 == Right) dir2 = (r & 0x01)? Up : Down;
	 else dir2 = (r & 0x02)? Left : Right;
	}

       // decide the best way we can go
       Dir dir3 = behindOf[dir2];
       Dir dir4 = behindOf[dir1];
       Dir badDir = behindOf[oldDir];
       if (ghostCanMove(i, dir1) && dir1 != badDir) ghost->setDir(dir1);
       else if (ghostCanMove(i, dir2) && dir2 != badDir) ghost->setDir(dir2);
       else if (ghostCanMove(i, dir3) && dir3 != badDir) ghost->setDir(dir3);
       else if (ghostCanMove(i, dir4) && dir4 != badDir) ghost->setDir(dir4);
       else ghost->setDir(badDir);
      } // if(ghost->move())

     // check if we have collided with the player
     if(ghost->screenPos().near(player->screenPos(), collisionThreshold))
      {
       if(ghost->state == Ghost::Scared)
	{
	 ghost->state = Ghost::Dead;
	 player->score += ghostScore;
	 gtm->add(new GlueText(ghost->screenPos() + Point(sprW/2, sprH/2), scoreFlashTime, "%d", ghostScore));
	 ghostScore <<= 1;
	 drawScoreBox();
	}
       else if (ghost->state == Ghost::Alive) kill();
       continue;
      }
    } // if(moves)

   // if we are on the base square, give us our life back
   if((ghost->getPos() == base) && (ghost->state == Ghost::Dead)) ghost->state = Ghost::Alive;
  }
}

//-----------------------------------------------------------------------------
void Game::checkInput()
{
 KeyboardEvent* ke = keyboard->getEvent();
 if(ke)
  {
   KeyDownEvent* kde = dynamic_cast<KeyDownEvent*>(ke);
   if(kde)
    {
     int key = kde->key;

     if(IS_KEY_Q(key) || IS_KEY_ESCAPE(key))
      quit = true;

     // handle direction keys

     Dir dir = None;
     if(IS_KEY_UP(key)) dir = Up;
     if(IS_KEY_DOWN(key)) dir = Down;
     if(IS_KEY_LEFT(key)) dir = Left;
     if(IS_KEY_RIGHT(key)) dir = Right;

     if((dir != None) && ((dir != player->getDir()) || !player->moving))
      {
       if(dir == behindOf[player->getDir()])
	{
	 player->turnBack();
	 player->nextDir = None;
	}
       else
	{
	 player->nextDir = dir;
	 if(!player->onSquare() && (player->getDisplacement() <= dirGuessThreshold) && playerCanMove(dir)) player->turnBack();
	}
      }

    }
   delete ke;
  }
}

//-----------------------------------------------------------------------------
void Game::movePlayer(int ticks)
{
 for(; ticks; ticks--)
  {
   if(player->pause) player->pause--;
   else
    {
     // do stuff when player is on a square
     if(player->onSquare())
      {
       Pos playerPos = player->getPos();
       if(playerPos.x >= 0 && playerPos.x < lvlW && playerPos.y >= 0 && playerPos.y < lvlH)
	{
	 // eat dots
	 if(level->getSquare(playerPos) == 1)
	  {
	   player->score += levelOptions->dotScore;
	   drawScoreBox();
	   level->clearAndDrawKBo(playerPos);
	   player->slowTime = sprW;
	   player->pause = levelOptions->playerEatingPause;
	  }
	 // eat pellets
	 if(level->getSquare(playerPos) == 2)
	  {
	   player->score += levelOptions->pelletScore;
	   drawScoreBox();
	   power = levelOptions->ghostScaredTime;
	   ghostScore = levelOptions->ghostScore;
	   every(*ghosts, pellet());
	   Ghost::flash = 0;
	   level->clearAndDrawKBo(playerPos);
	   player->slowTime = sprW;
	   player->pause = levelOptions->playerEatingPause;
	  }
	}
       // change direction if requested
       if(player->nextDir != None && playerCanMove(player->nextDir)) player->setDir(player->nextDir);
       else if(!playerCanMove(player->getDir())) player->stop();
      }

     if(player->slowTime? playerSlowSpeed->moves() : playerSpeed->moves())
      {
       if(player->slowTime) player->slowTime--;
       player->move();
       checkInput();
      }
    } // if(player->pause)
   
   // control power pellets
   if(power)
    {
     if(--power == 0)
      {
       every(*ghosts, pelletEnd());
       Ghost::flash = 0;
      }
     else if(power < ghostFlashes * ghostFlashTime)
      Ghost::flash = ((power % (ghostFlashTime << 1)) / ghostFlashTime)? 1 : 0;
    }
  }
}

//-----------------------------------------------------------------------------
void Game::moveFruit(int ticks)
 /* Called once per frame to handle all things fruit-related */
{
 if(fruitActive)
  {
   if(fruit->screenPos().near(player->screenPos(), collisionThreshold))
    {
     fruitActive = false;
     player->score += levelOptions->fruitScore;
     gtm->add(new GlueText(fruit->screenPos() + Point(sprW / 2, sprH / 2), scoreFlashTime, "%d", levelOptions->fruitScore));
     drawScoreBox();
    }
   else if(fruitSpeed->moves() && fruit->move())
    {
     // the fruit has moved 1 square...
     // if we are off the screen disappear
     if(!stageSize.contains(fruit->getPos())) fruitActive = false;
     else
      {
       // otherwse, figure out where to go next
       Dir badDir = behindOf[fruit->getDir()];
       int r = rInt(2);
       Dir dir1 = (rInt(256) < levelOptions->fruitSkill)? fruitDir : (Dir)rInt(4); // 1 in 3 times go randomly
       Dir dir2 = (r & 0x01)? leftOf[dir1] : rightOf[dir1];
       Dir dir3 = behindOf[dir2];
       Dir dir4 = behindOf[dir1];
       if(dir1 != badDir && fruitCanMove(dir1)) fruit->setDir(dir1);
       else if(dir2 != badDir && fruitCanMove(dir2)) fruit->setDir(dir2);
       else if(dir3 != badDir && fruitCanMove(dir3)) fruit->setDir(dir3);
       else if(dir4 != badDir && fruitCanMove(dir4)) fruit->setDir(dir4);
       else fruit->setDir(badDir);
      }
    }
  } // if(fruitActive)
 // if there is no fruit, make some randomly
 else if(!rInt(levelOptions->fruitChance) && fruitStarts->length() > 0)
  {
   fruitActive = true;
   
   // pick a starting position at random
   Pos pos = *fruitStarts->at(rInt(fruitStarts->length()));
   fruit->setPos(pos);

   // figure out what direction we should be going
   if(pos.x == -1) fruitDir = Right;
   else if(pos.y == -1) fruitDir = Down;
   else if(pos.x == lvlW) fruitDir = Left;
   else if(pos.y == lvlH) fruitDir = Up;
   fruit->setDir(fruitDir);

   // set the type. needs work.
   fruit->type = 0;
  }
}

class Timer : public Object
{
private:
 timeval lastTime;

public:
 Timer()
  {
   gettimeofday(&lastTime, NULL);
  }

 int32 operator () ()
  {
   timeval thisTime;
   gettimeofday(&thisTime, NULL);
   // NB: we assume tv_sec has changed by no more than 1
   //     this is not a bug, it is a feature :)
   int32 ret = thisTime.tv_usec - lastTime.tv_usec + ((thisTime.tv_sec == lastTime.tv_sec)? 0 : 1000000);
   lastTime = thisTime;
   return ret;
  }
};

//-----------------------------------------------------------------------------

void Game::mainLoop()
{
 // MAIN GAME LOOP
 int frames = 0, ticks = 0;
 Timer timer;
 uint32 usLeft = 0;
 quit = false;
 while(!quit)
  {
   draw();

   uint32 us = timer() + usLeft;
   uint32 ms = us / 1000;
   usLeft = us % 1000;

   // advance animations

   asetHatman->at(0)->advance(ms);
   asetHatman->at(1)->advance(ms);
   asetHatman->at(2)->advance(ms);
   asetHatman->at(3)->advance(ms);
   fruitAset->at(0)->advance(ms);

   // move stuff around the level and whatever

   movePlayer(ms);
   for(uint32 i=0; i<ms; i++)
    {
     moveGhosts(1);
     moveFruit(1);
     gtm->tick();

     if(level->dots == 0) quit = true;
     if(quit) break;
    }

   frames++;
   ticks += ms;
  }

 DPRINTF("<game> frame rate: %d (max: 1000)\n", frames * 1000 / ticks);
}

//-----------------------------------------------------------------------------
bool Game::play()
{
 // check the level
 if(!level->checkLevel()) { nonFatal("Invalid level"); return false; }

 // load graphics as specified by levelOptions

 if(fruitAset) delete fruitAset; fruitAset = NULL;
 if(NULL == (fruitAset = (Aset*) load(libFileName(levelOptions->fruitAset), (objectCreateFunc) Aset::create))) return false;

 // initialise
 srand(time(0));
 power = 0;
 stagePos = Point(sprW / 2, sprH / 2);

 // check background
 //if(bk->filename != levelOptions->background)
 //bk->load(libFileName(levelOptions->background));

 // play initial sound effect
#ifdef SOUND
 if(options->sound) Effects->Play(sndBegin, 0);
#endif

 // take object positions from the level
 player->startPos = level->find(24);
 base = level->find(22);
 scoreRect = Rect(stage2screen(level->find(19)),
		  stage2screen(level->find(21) + Pos(1, 1)) - stage2screen(level->find(19)));
 level->clear(player->startPos);
 level->getGhosts(ghosts);

 // setup fruit and stuff
 fruitStarts->delAll();
 while(1)
  {
   Pos pos = level->find(23);
   if(pos.x == -1) break;
   if(pos.x == 0) fruitStarts->add(new Pos(-1, pos.y));
   else if(pos.y == 0) fruitStarts->add(new Pos(pos.x, -1));
   else if(pos.x == lvlW-1) fruitStarts->add(new Pos(lvlW, pos.y));
   else if(pos.y == lvlH-1) fruitStarts->add(new Pos(pos.x, lvlH));
   level->clear(pos);
  }
 fruitActive = false;
 fruit->type = levelOptions->fruit;

 // initialise the player
 player->setPos(player->startPos);
 player->setDir(Right);

 // set up speeds
 *playerSpeed = levelOptions->playerSpeed;
 *playerSlowSpeed = levelOptions->playerSlowSpeed;
 *ghostSpeed = levelOptions->ghostSpeed;
 *ghostScaredSpeed = levelOptions->ghostScaredSpeed;
 *ghostEyesSpeed = levelOptions->ghostEyesSpeed;
 *fruitSpeed = levelOptions->fruitSpeed;
 
 // initlialise the screen
 screen->clipping = true;
 screen->clippingRect = screenRect;
 bk->copyTo(screen);
 level->draw();
 screen->keep();
 screen->wipeUpdate(16, 50000);
 drawScoreBox();

 mainLoop();

 // clean up
 ghosts->delAll();

 return (level->dots == 0)? true : false;
}

#if 0
//-----------------------------------------------------------------------------
bool Game::playLevel(const char* filename)
{
 initGame();
 if(!level->load(filename)) return false;
 levelOptions = levelOptionsDefault;
 return play();
}
#endif

#if 0
//-----------------------------------------------------------------------------
bool Game::playLevel(const Level* _level)
{
 initGame();
 *level = *_level;
 levelOptions = levelOptionsDefault;
 return play();
}
#endif

//--------------------------------------------------------------------------------------------------------------------------------

#if 0
bool Game::playGame(const char* filename)
{
 VPRINTF("<game> Game::playGame(%s)\n", filename);
 GameItems gameItems;
 if(!gameItems.loadGame(filename)) return false;
 initGame();
 foreach(gameItems, i)
  {
   levelOptions = ((GameItemLevel*)gameItems[i])->levelOptions;
   if(!level->load(libFileName(levelOptions->filename))) return false;
   if(!play()) return true;
  }
 return true;
}
#endif

//--------------------------------------------------------------------------------------------------------------------------------

bool Game::loadGame(const char* filename)
{
 VPRINTF("<game> loading game \"%s\"\n", filename);
 gameItems->delAll();
 if(!gameItems->loadGame(filename)) return false;
 return true;
}

bool Game::playGame()
{
 VPRINTF("<game> playing\n");
 initGame();
 foreach(*gameItems, i)
  {
   levelOptions = ((GameItemLevel*)gameItems->at(i))->levelOptions;
   if(!level->load(libFileName(levelOptions->filename))) return false;
   if(!play()) break;
  }
 return true;
}

//--------------------------------------------------------------------------------------------------------------------------------

