/*
 * Copyright (C) 2002  Emmanuel VARAGNAT <hddtemp@guzu.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


// Include file generated by ./configure
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Gettext includes
#if ENABLE_NLS
#include <libintl.h>
#define _(String) gettext (String)
#else
#define _(String) (String)
#endif

// Standard includes
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <scsi/scsi.h>
#include <linux/hdreg.h>

// Application specific includes
#include "hddtemp.h"
#include "satacmds.h"
#include "scsicmds.h"

#define swapb(x) \
({ \
        u16 __x = (x); \
        (x) = ((u16)( \
                (((u16)(__x) & (u16)0x00ffU) << 8) | \
                (((u16)(__x) & (u16)0xff00U) >> 8) )); \
})

 
static int sata_probe(int device) {
  int bus_num;
  unsigned char cmd[4] = { WIN_IDENTIFY, 0, 0, 1 };
  unsigned char identify[512];
  char buf[36]; /* should be 36 for unsafe devices (like USB mass storage stuff)
                        otherwise they can lock up! SPC sections 7.4 and 8.6 */
 
  /* SATA disks are difficult to detect as they answer to both ATA and SCSI 
     commands */

  /* First check that the device is accessible through SCSI */
  if(ioctl(device, SCSI_IOCTL_GET_BUS_NUMBER, &bus_num))
    return 0;
  
  /* Get SCSI name and verify it starts with "ATA " */
  if (scsi_inquiry(device, buf))
    return 0;
  else if (strncmp(buf + 8, "ATA ", 4))
     return 0;

  /* Verify that it supports ATA pass thru */
  if (sata_pass_thru(device, cmd, identify) != 0)
    return 0;
  else
    return 1;
}

static const char *sata_model (int device) {
  unsigned char cmd[4] = { WIN_IDENTIFY, 0, 0, 1 };
  unsigned char identify[512];
  
  if(device == -1 || sata_pass_thru(device, cmd, identify))
    return strdup(_("unknown"));
  else
  {
    sata_fixstring(identify + 54, 24);
    return strdup(identify + 54);
  }
}

static unsigned char* sata_search_temperature(const unsigned char* smart_data, int attribute_id) {
  int i, n;

  n = 3;
  i = 0;
  while((debug || *(smart_data + n) != attribute_id) && i < 30) {
    if(debug && *(smart_data + n))
      printf(_("field(%d)\t = %d\n"), *(smart_data + n), *(smart_data + n + 3));

    n += 12;
    i++;
  }

  if(i >= 30)
    return NULL;
  else
    return (unsigned char*)(smart_data + n);
}

static enum e_gettemp sata_get_temperature(struct disk *dsk) {   
  unsigned char    values[512];
  unsigned char *  field;
  int              i;
  u16 *            p;

  if(dsk->db_entry && dsk->db_entry->attribute_id == 0) {
    close(dsk->fd);
    dsk->fd = -1;
    return GETTEMP_NOSENSOR;
  }
  
  /* get SMART values */
  if(sata_enable_smart(dsk->fd) != 0) {
    enum e_gettemp ret;
    if(errno == EIO) {
      snprintf(dsk->errormsg, MAX_ERRORMSG_SIZE, _("S.M.A.R.T. not available"));
      ret = GETTEMP_NOT_APPLICABLE;
    }
    else
    {
      snprintf(dsk->errormsg, MAX_ERRORMSG_SIZE, "%s", strerror(errno));
      ret = GETTEMP_ERROR;
    }
    close(dsk->fd);
    dsk->fd = -1;
    return ret;
  }

  if(sata_get_smart_values(dsk->fd, values)) {
    snprintf(dsk->errormsg, MAX_ERRORMSG_SIZE, "%s", strerror(errno));
    close(dsk->fd);
    dsk->fd = -1;
    return GETTEMP_ERROR;
  }

  p = (u16*)values;
  for(i = 0; i < 256; i++) {
    swapb(*(p+i));
  }

  /* temperature */
  if(dsk->db_entry && dsk->db_entry->attribute_id > 0)
    field = sata_search_temperature(values, dsk->db_entry->attribute_id);
  else
    field = sata_search_temperature(values, DEFAULT_ATTRIBUTE_ID);

  if(field)
    dsk->value = *(field+3);

  if(dsk->db_entry && dsk->value != -1)
    return GETTEMP_KNOWN;
  else {
    if(dsk->value != -1) {
      return GETTEMP_GUESS;
    }
    else {
      return GETTEMP_UNKNOWN;
    }
  }

  /* never reached */
}


/*******************************
 *******************************/

struct bustype sata_bus = {
  sata_probe,
  sata_model,
  sata_get_temperature
};
