/* -*- C++ -*- */

/*

  Heap Layers: An Extensible Memory Allocation Infrastructure
  
  Copyright (C) 2000-2003 by Emery Berger
  http://www.cs.umass.edu/~emery
  emery@cs.umass.edu
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef _STLALLOCATOR_H_
#define _STLALLOCATOR_H_

#include <stdio.h>
#include <new>
#include <stdlib.h>

#include <memory> // STL

using namespace std;

/**
 * @class STLAllocator
 * @brief An allocator adapter for STL.
 * 
 * This mixin lets you use any Heap Layers allocator as the allocator
 * for an STL container.
 *
 * Example:
 * <TT>
 *   typedef STLAllocator<int, MyHeapType> MyAllocator;<BR>
 *   list<int, MyAllocator> l;<BR>
 * </TT>
 */

namespace HL {

template <class T, class SuperHeap>
class STLAllocator {
public:

  typedef T value_type;
  typedef std::size_t size_type;
  typedef std::ptrdiff_t difference_type;
  typedef T * pointer;
  typedef const T * const_pointer;
  typedef T& reference;
  typedef const T& const_reference;

  STLAllocator (void) { }
  STLAllocator (const STLAllocator&) {}
  virtual ~STLAllocator (void) {}

#if defined(_WIN32)
  char * _Charalloc (size_type n) {
    return (char *) allocate (n);
  }
#endif

#if defined(__SUNPRO_CC)
  inline void * allocate (size_type n,
			  const void * = 0) {
    return reinterpret_cast<void *>(getHeap().malloc (sizeof(T) * n));
  }
#else
  inline pointer allocate (size_type n,
			  const void * = 0) {
    return reinterpret_cast<pointer>(getHeap().malloc (sizeof(T) * n));
  }
#endif

  inline void deallocate (void * p, size_type) {
    getHeap().free (p);
  }

  inline void deallocate (pointer p, size_type) {
    getHeap().free (p);
  }
  
  pointer address (reference x) const { return &x; }
  const_pointer address (const_reference x) const { return &x; }

  void construct (pointer p, const T& val) { new (p) T (val); }
  void destroy (pointer p) { p->~T(); }

  /// Make the maximum size be the largest possible object.
  size_type max_size(void) const
  {
    size_type n = (size_type)(-1);
    return (n > 0 ? n : (size_type)(n));
  }

  template <class U> STLAllocator( const STLAllocator<U, SuperHeap> &) {}
  template <class U> struct rebind { typedef STLAllocator<U,SuperHeap> other; };

private:

  /// Maintain exactly one instance of the SuperHeap.
  SuperHeap& getHeap (void) {
    // Initialize the heap inside a buffer -- this means that it won't
    // be deallocated when this allocator is destroyed. Because of
    // ordering issues, *that* can be big trouble.
    static double thBuf[sizeof(SuperHeap) / sizeof(double) + 1];
    static SuperHeap * th = new (thBuf) SuperHeap;
    return *th;
  }

};

}


#endif
