/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: sm1rendr.cpp,v 1.3.2.1 2004/07/09 01:57:37 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#if defined (_AIX)
#include <X11/Xlib.h>
#endif

#include <stdio.h>

#include "sm1rendr.ver"
#include "hxtypes.h"

#include "hxwintyp.h"
#include "chxxtype.h"
#include "hxcom.h"
#include "hxcomm.h"
#include "ihxpckts.h"
#include "hxfiles.h"
#include "hxcore.h"
#include "hxprefs.h"
#include "hxrendr.h"
#include "hxasm.h"
#include "hxplugn.h"
#include "hxengin.h"
#include "hxwin.h"
#include "hxgroup.h"
#include "hxsite2.h"
#include "hxclsnk.h"
#include "hxerror.h"
#include "hxupgrd.h"
#include "hxprefs.h"
#include "hxvsurf.h"
#include "hxcmenu.h"
#include "hxhyper.h"
#include "hxmon.h"	// for IHXRegistry
#include "hxxres.h"
#include "hxxrsmg.h"
#include "hxmeta1.h"   /* renamed for SHAZAM; used to be hxmeta.h */
#include "hxsm2sm.h"

#include "hxxml.h"
#include "hxxmlprs.h"
#include "xmlreslt.h"

#include "hxstack.h"
#include "hxslist.h"
#include "hxstring.h"
#include "hxstrutl.h"
#include "hxmap.h"
#include "chxpckts.h"
#include "hxmap.h"
#include "hxurl.h"
#include "asmrulep.h"
#include "hxver.h"

#include "smlpkt.h"
#include "sm1elem.h"
#include "sm1doc.h"
#include "sm1error.h"
#include "smilres.h"
#include "xmlres.h"
#include "smlrendr.h" /* for SMIL2+ renderer (CSmilRenderer) */
#include "sm1rendr.h"
#include "addupcol.h"

/* We should really define it in a common header file */
#if defined (_WINDOWS ) || defined (WIN32)
#define OS_SEPARATOR_CHAR	'\\'
#define OS_SEPARATOR_STRING	"\\"
#elif defined (_UNIX)
#define OS_SEPARATOR_CHAR	'/'
#define OS_SEPARATOR_STRING	"/"
#elif defined (_MACINTOSH)
#define OS_SEPARATOR_CHAR	':'
#define OS_SEPARATOR_STRING	":"
#endif // defined (_WINDOWS ) || defined (WIN32)

#include "hxheap.h"
#ifdef _DEBUG
#undef HX_THIS_FILE		
static const char HX_THIS_FILE[] = __FILE__;
#endif

#ifdef _WINDOWS
extern HINSTANCE g_hInstance;
#endif

// /Defining the following fixes PR 62481 and seems to work fine with other
// SMIL 2.0 content coming from a SMIL 1.0 File Format plug-in:
#define DO_INIT_PERSISTENT_ON_NEXTGEN_SMILREND    1


static const UINT32 INITIAL_GRANULARITY = 66;

/****************************************************************************
 * 
 *  Method:
 * 
 *	HXCreateInstance()
 * 
 *  Purpose:
 * 
 *	Function implemented by all plugin DLL's to create an instance of 
 *	any of the objects supported by the DLL. This method is similar to 
 *	Window's CoCreateInstance() in its purpose, except that it only 
 *	creates objects from this plugin DLL.
 *
 *	NOTE: Aggregation is never used. Therefore and outer unknown is
 *	not passed to this function, and you do not need to code for this
 *	situation.
 * 
 */
HX_RESULT STDAPICALLTYPE CSmil1Renderer::HXCreateInstance(IUnknown** ppIUnknown)
{
    HX_RESULT retVal = HXR_FAIL;

    if (ppIUnknown)
    {
        CSmil1Renderer* pObj = new CSmil1Renderer();
        if (pObj)
        {
            retVal = pObj->QueryInterface(IID_IUnknown, (void**) ppIUnknown);
            if (FAILED(retVal))
            {
                HX_DELETE(pObj);
            }
        }
    }

    return retVal;
}

HX_RESULT STDAPICALLTYPE CSmil1Renderer::CanUnload2()
{
    return ((CHXBaseCountingObject::ObjectsActive() > 0) ? HXR_FAIL : HXR_OK );
}

const char* const CSmil1Renderer::zm_pName	      = "SMIL 1.0";
const char* const CSmil1Renderer::zm_pDescription    = "RealNetworks RMA Driver Renderer Plugin";
const char* const CSmil1Renderer::zm_pCopyright      = HXVER_COPYRIGHT;
const char* const CSmil1Renderer::zm_pMoreInfoURL    = HXVER_MOREINFO;
const char* const CSmil1Renderer::zm_pStreamMimeTypes[] = { "application/vnd.rn-rmadriver", "application/rma-driver", NULL };


/************************************************************************
 *  Method:
 *    IHXPlugin::InitPlugin
 *  Purpose:
 *    Initializes the plugin for use. This interface must always be
 *    called before any other method is called. This is primarily needed 
 *    so that the plugin can have access to the context for creation of
 *    IHXBuffers and IMalloc.
 */
STDMETHODIMP CSmil1Renderer::InitPlugin(IUnknown* /*IN*/ pContext)
{
#if defined(HELIX_FEATURE_SMIL2)
    HX_ASSERT(!m_bIsHigherVersionSmilStreamFromOldSMIL1FF);
#endif /* defined(HELIX_FEATURE_SMIL2). */

    m_pContext = pContext;
    m_pContext->AddRef();

    m_pContext->QueryInterface(IID_IHXCommonClassFactory,
	    (void**)&m_pCommonClassFactory);

    return HXR_OK;
}

/************************************************************************
 *  Method:
 *    IHXPlugin::GetPluginInfo
 *  Purpose:
 *    Returns the basic information about this plugin. Including:
 *
 *    bLoadMultiple	whether or not this plugin DLL can be loaded
 *			multiple times. All File Formats must set
 *			this value to TRUE.
 *    pDescription	which is used in about UIs (can be NULL)
 *    pCopyright	which is used in about UIs (can be NULL)
 *    pMoreInfoURL	which is used in about UIs (can be NULL)
 */
STDMETHODIMP CSmil1Renderer::GetPluginInfo
(
    REF(BOOL)        /*OUT*/ bLoadMultiple,
    REF(const char*) /*OUT*/ pDescription,
    REF(const char*) /*OUT*/ pCopyright,
    REF(const char*) /*OUT*/ pMoreInfoURL,
    REF(ULONG32)     /*OUT*/ ulVersionNumber
)
{
#if defined(HELIX_FEATURE_SMIL2)
    HX_ASSERT(!m_bIsHigherVersionSmilStreamFromOldSMIL1FF);
#endif /* defined(HELIX_FEATURE_SMIL2). */

    bLoadMultiple = TRUE;   // Must be true for file formats.

    pDescription    = (const char*) zm_pDescription;
    pCopyright	    = (const char*) zm_pCopyright;
    pMoreInfoURL    = (const char*) zm_pMoreInfoURL;
    ulVersionNumber = TARVER_ULONG32_VERSION;

    return HXR_OK;
}

/************************************************************************
 *  Method:
 *    IHXPlugin::GetRendererInfo
 *  Purpose:
 *    If this object is a file format object this method returns
 *    information vital to the instantiation of file format plugins.
 *    If this object is not a file format object, it should return
 *    HXR_UNEXPECTED.
 */
STDMETHODIMP CSmil1Renderer::GetRendererInfo
(
    REF(const char**) /*OUT*/ pStreamMimeTypes,
    REF(UINT32)      /*OUT*/ unInitialGranularity
)
{
#if defined(HELIX_FEATURE_SMIL2)
    HX_ASSERT(!m_bIsHigherVersionSmilStreamFromOldSMIL1FF);
#endif /* defined(HELIX_FEATURE_SMIL2). */

    pStreamMimeTypes = (const char**) zm_pStreamMimeTypes;
    unInitialGranularity = m_ulGranularity;

    return HXR_OK;
}

CSmil1Renderer::CSmil1Renderer()
	: m_lRefCount(0)
	, m_pContext(NULL)
	, m_pStream(NULL)
	, m_pPlayer(NULL)
	, m_ulLastTime(0)
	, m_pCommonClassFactory(0)
	, m_pPacketParser(0)
	, m_pSmilDocRenderer(0)
	, m_pEngine(0)
	, m_pClientContext(0)
	, m_uLayoutRule(0)
	, m_uSourceRule(0)
	, m_ulTotalSMILPackets(0)
	, m_ulGranularity(INITIAL_GRANULARITY)
	, m_pURLFragment(0)
	, m_lastOnPacketResult(HXR_OK)
#if defined(HELIX_FEATURE_SMIL2)
	// /All of the following are needed for handling SMIL 2+ streams:
	, m_pNextGenSmilRenderer(NULL)
	, m_bIsHigherVersionSmilStreamFromOldSMIL1FF(FALSE)
	, m_pHeader(NULL)
	, m_pDefaultNamespace(NULL)
	, m_bCSmil1StartStreamHasBeenCalled(FALSE)
	, m_bCSmil1EndStreamHasBeenCalled(FALSE)
	, m_bCSmil1GetDisplayTypeHasBeenCalled(FALSE)
	, m_bCSmil1InitializeStatisticsHasBeenCalled(FALSE)

	, m_bCSmil1InitPersistenHasBeenCalled(FALSE)

	, m_bCSmil1GetElementPropertiesHasBeenCalled(FALSE)
	, m_bCSmil1RemoveLayoutSiteGroupHasBeenCalled(FALSE)
	, m_ulPersistentComponentID(0)
	, m_uPersistentGroupID(0)
	, m_uPersistentTrackID(0)
	, m_pPersistentParent(NULL)
	, m_ulRegistryID(0)
	, m_uGroupID(0)
	, m_uTrackID(0)
	// /End "...handling SMIL 2+ streams".
#endif /* defined(HELIX_FEATURE_SMIL2). */
	, m_bUseNestedMeta(TRUE)
	, m_pPersistentComponentManager(NULL)    
	, m_ulPersistentVersion(0)
	, m_persistentType(PersistentSMIL)
{
#if defined(_DEBUG) && defined(XXXMEH_CHECK_FOR_LEAKS)
    char szDbgStr[128]; /* Flawfinder: ignore */
    sprintf(szDbgStr, "CON CSmil1Renderer 0x%08x\n", this); /* Flawfinder: ignore */
    OutputDebugString(szDbgStr);
#endif
    m_ulPersistentVersion = HX_ENCODE_PROD_VERSION(1, 0, 0, 0);
};

CSmil1Renderer::~CSmil1Renderer()
{
#if defined(_DEBUG) && defined(XXXMEH_CHECK_FOR_LEAKS)
    char szDbgStr[128]; /* Flawfinder: ignore */
    sprintf(szDbgStr, "DES CSmil1Renderer 0x%08x\n", this); /* Flawfinder: ignore */
    OutputDebugString(szDbgStr);
#endif
    if(m_pSmilDocRenderer)
    {
	m_pSmilDocRenderer->close(this);
	HX_RELEASE(m_pSmilDocRenderer);
    }

#if defined(HELIX_FEATURE_SMIL2)
    HX_RELEASE(m_pDefaultNamespace);
    HX_RELEASE(m_pNextGenSmilRenderer);
    HX_RELEASE(m_pHeader);
    HX_RELEASE(m_pPersistentParent);
#endif /* defined(HELIX_FEATURE_SMIL2). */

    HX_RELEASE(m_pPersistentComponentManager);
    HX_RELEASE(m_pContext);
    HX_RELEASE(m_pClientContext);
    HX_RELEASE(m_pStream);
    HX_RELEASE(m_pEngine);
    HX_RELEASE(m_pCommonClassFactory);

    HX_DELETE(m_pPacketParser);
    HX_RELEASE(m_pPlayer);
    HX_VECTOR_DELETE(m_pURLFragment);
}


// *** IUnknown methods ***

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IUnknown::QueryInterface
//  Purpose:
//	Implement this to export the interfaces supported by your 
//	object.
//
STDMETHODIMP CSmil1Renderer::QueryInterface(REFIID riid, void** ppvObj)
{
    if (IsEqualIID(riid, IID_IUnknown))
    {
	AddRef();
	*ppvObj = this;
	return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXPlugin))
    {
	AddRef();
	*ppvObj = (IHXPlugin*)this;
	return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXRenderer))
    {
	AddRef();
	*ppvObj = (IHXRenderer*)this;
	return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXSiteUser))
    {
	AddRef();
	*ppvObj = (IHXSiteUser*)this;
	return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXStatistics))
    {
	AddRef();
	*ppvObj = (IHXStatistics*)this;
	return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXPersistentRenderer))
    {
	AddRef();
	*ppvObj = (IHXPersistentRenderer*)this;
	return HXR_OK;
    }
    else if (m_pSmilDocRenderer &&
	     HXR_OK == m_pSmilDocRenderer->QueryInterface(riid, ppvObj))
    {
	return HXR_OK;
    }
#if defined(HELIX_FEATURE_SMIL2)
    else if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    return HXR_UNEXPECTED;
	}
	if (HXR_OK ==
		m_pNextGenSmilRenderer->QueryInterface(riid, ppvObj))
	{
	    return HXR_OK;
	}
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */


    *ppvObj = NULL;
    return HXR_NOINTERFACE;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IUnknown::AddRef
//  Purpose:
//	Everyone usually implements this the same... feel free to use
//	this implementation.
//
STDMETHODIMP_(ULONG32) CSmil1Renderer::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IUnknown::Release
//  Purpose:
//	Everyone usually implements this the same... feel free to use
//	this implementation.
//
STDMETHODIMP_(ULONG32) CSmil1Renderer::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
        return m_lRefCount;
    }

    delete this;
    return 0;
}

// *** IHXRenderer methods ***

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::StartStream
//  Purpose:
//	Called by client engine to inform the renderer of the stream it
//	will be rendering. The stream interface can provide access to
//	its source or player. This method also provides access to the 
//	primary client controller interface.
//
STDMETHODIMP CSmil1Renderer::StartStream
(
    IHXStream*	    pStream,
    IHXPlayer*	    pPlayer
)
{
#if defined(HELIX_FEATURE_SMIL2)
    m_bCSmil1StartStreamHasBeenCalled = TRUE;
    HX_ASSERT(!m_bIsHigherVersionSmilStreamFromOldSMIL1FF);
#endif /* defined(HELIX_FEATURE_SMIL2). */

    HX_RESULT rc = HXR_OK;

    // Save for later use!
    m_pStream  = pStream;
    m_pStream->AddRef();
    m_pPlayer  = pPlayer;
    m_pPlayer->AddRef();

    m_pPlayer->GetClientEngine(m_pEngine);
    m_pPlayer->GetClientContext(m_pClientContext);

    IHXBuffer* pBuffer = NULL;
    IHXPreferences* pPreferences = NULL;
    IHXRendererAdviseSink* pRendererAdviseSink = NULL;

    if (HXR_OK == m_pPlayer->QueryInterface(IID_IHXPreferences, (void**)&pPreferences))
    {
	if (HXR_OK == pPreferences->ReadPref("useNestedMeta", pBuffer))
	{
	    m_bUseNestedMeta = atoi((const char*)pBuffer->GetBuffer()) ? TRUE : FALSE;
	}
	HX_RELEASE(pBuffer);
    }
    HX_RELEASE(pPreferences);

    m_pSmilDocRenderer = new CSmil1DocumentRenderer(this, m_pContext);
    m_pSmilDocRenderer->AddRef();

    if (m_bUseNestedMeta)
    {
	IHXPersistentComponent* pPersistentComponent = NULL;

	if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXPersistentComponentManager, 
					       (void**)&m_pPersistentComponentManager))
	{
	    m_pPersistentComponentManager->CreatePersistentComponent(pPersistentComponent);

	    pPersistentComponent->Init((IHXPersistentRenderer*)this);
	    pPersistentComponent->AddRendererAdviseSink((IHXRendererAdviseSink*)m_pSmilDocRenderer);
	    pPersistentComponent->AddGroupSink((IHXGroupSink*)m_pSmilDocRenderer);

	    rc = m_pPersistentComponentManager->AddPersistentComponent(pPersistentComponent);
	
	    generatePreFix();
	}
    }
    else
    {
	IHXPersistenceManager* pMgr = 0;
	if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXPersistenceManager, 
	    (void**)&pMgr))
	{
	    IUnknown* pUnk = 0;
	    if(HXR_OK == pMgr->GetPersistentComponent(pUnk))
	    {
		rc = HXR_FAIL;
		CSmil1SMILSyntaxErrorHandler errHandler(m_pContext);
		errHandler.ReportError(SMILErrorMetaDatatype, NULL, 0);
		pUnk->Release();
	    }
	    else if(HXR_OK == QueryInterface(IID_IUnknown, (void**)&pUnk))
	    {
		pMgr->AddPersistentComponent(pUnk);
		pUnk->Release();
		generatePreFix();
	    }
	    pMgr->Release();
	}

	IHXDriverStreamManager* pDriverStreamMgr = 0;
	if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXDriverStreamManager,
	    (void**)&pDriverStreamMgr))
	{
	    if(HXR_OK == m_pSmilDocRenderer->QueryInterface(IID_IHXRendererAdviseSink, 
							    (void**)&pRendererAdviseSink))
	    {
		pDriverStreamMgr->AddRendererAdviseSink(pRendererAdviseSink);
		pRendererAdviseSink->Release();
	    }
	    pDriverStreamMgr->Release();
	}

	IHXGroupManager* pGrpMgr = 0;
	if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXGroupManager,
	    (void**)&pGrpMgr))
	{
	    IHXGroupSink* pSnk = 0;
	    if(HXR_OK == m_pSmilDocRenderer->QueryInterface(
		IID_IHXGroupSink, (void**)&pSnk))
	    {
		pGrpMgr->AddSink(pSnk);
		pSnk->Release();
	    }
	}
	HX_RELEASE(pGrpMgr);
    }

    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::EndStream
//  Purpose:
//	Called by client engine to inform the renderer that the stream
//	is was rendering is closed.
//
STDMETHODIMP CSmil1Renderer::EndStream()
{
#if defined(HELIX_FEATURE_SMIL2)
    m_bCSmil1EndStreamHasBeenCalled = TRUE;
#endif /* defined(HELIX_FEATURE_SMIL2). */

    if(m_pPlayer)
    {
	if (m_bUseNestedMeta)
	{
	    IHXPersistentComponent* pPersistentComponent = NULL;
	    IHXRendererAdviseSink* pRendererAdviseSink = 0;

	    if (HXR_OK == m_pPersistentComponentManager->GetPersistentComponent(m_pSmilDocRenderer->m_ulPersistentComponentID,
										pPersistentComponent))
	    {
		pPersistentComponent->RemoveRendererAdviseSink((IHXRendererAdviseSink*)m_pSmilDocRenderer);
		pPersistentComponent->RemoveGroupSink((IHXGroupSink*)m_pSmilDocRenderer);
	    }
	    HX_RELEASE(pPersistentComponent);	    
	}
	else
	{
	    IHXGroupManager* pGrpMgr = 0;
	    if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXGroupManager, 
		(void**)&pGrpMgr))
	    {
		IHXGroupSink* pSnk = 0;
		if(m_pSmilDocRenderer &&
		    HXR_OK == m_pSmilDocRenderer->QueryInterface(IID_IHXGroupSink, 
		    (void**)&pSnk))
		{
		    pGrpMgr->RemoveSink(pSnk);
		    pSnk->Release();
		}
		pGrpMgr->Release();
	    }

	    IHXDriverStreamManager* pStrmMgr = 0;
	    if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXDriverStreamManager,
		(void**)&pStrmMgr))
	    {
		IHXRendererAdviseSink* pSnk = 0;
		if(m_pSmilDocRenderer &&
		    HXR_OK == m_pSmilDocRenderer->QueryInterface(
		    IID_IHXRendererAdviseSink, (void**)&pSnk))
		{
		    pStrmMgr->RemoveRendererAdviseSink(pSnk);
		    pSnk->Release();
		}
		pStrmMgr->Release();
	    }
	}
    }
    HX_RELEASE(m_pStream);

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    return HXR_UNEXPECTED;
	}

	return m_pNextGenSmilRenderer->EndStream();
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    if(m_pSmilDocRenderer)
    {
	m_pSmilDocRenderer->endStream();
    }


    return HXR_OK;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnHeader
//  Purpose:
//	Called by client engine when a header for this renderer is 
//	available. The header will arrive before any packets.
//
STDMETHODIMP CSmil1Renderer::OnHeader(IHXValues* pHeader)
{
    // check stream and content versions so an upgrade can
    // be called if necessary...
    HX_RESULT	rc = HXR_OK;

    if (!pHeader)
    {
	return HXR_UNEXPECTED;
    }

#if defined(HELIX_FEATURE_SMIL2)
    // /Hang onto header in case we need to send it up to a higher-version
    // SMIL renderer:
    m_pHeader = pHeader;
    m_pHeader->AddRef();

    HX_ASSERT(!m_bIsHigherVersionSmilStreamFromOldSMIL1FF);
#endif /* defined(HELIX_FEATURE_SMIL2). */

    BOOL bVersionOK = TRUE;
    UINT16 uGroupCount = 0;
    UINT16 uCurrentGroupID = 0;
    UINT16 uSupportedType = 0;	// G2 style by default
    UINT32 ulStreamVersion = 0;
    UINT32 ulContentVersion = 0;
    IHXGroupManager* pGrpMgr = NULL;

    if(HXR_OK == pHeader->GetPropertyULONG32("StreamVersion", 
	ulStreamVersion))
    {
	UINT32 ulMajorVersion = HX_GET_MAJOR_VERSION(ulStreamVersion);
	UINT32 ulMinorVersion = HX_GET_MINOR_VERSION(ulStreamVersion);

	if((ulMajorVersion > STREAM_MAJOR_VERSION) ||
	   (ulMinorVersion > STREAM_MINOR_VERSION &&
	    ulMajorVersion == STREAM_MAJOR_VERSION))
	{
	    bVersionOK = FALSE;
	}
    }

    if(bVersionOK &&
       HXR_OK == pHeader->GetPropertyULONG32("ContentVersion",
           ulContentVersion))
    {
	UINT32 ulMajorVersion = HX_GET_MAJOR_VERSION(ulContentVersion);
	UINT32 ulMinorVersion = HX_GET_MINOR_VERSION(ulContentVersion);

	if((ulMajorVersion > CONTENT_MAJOR_VERSION) ||
	   (ulMinorVersion > CONTENT_MINOR_VERSION &&
	    ulMajorVersion == CONTENT_MAJOR_VERSION))
	{
	    bVersionOK = FALSE;
	}
    }

    if(!bVersionOK)
    {
        AddToAutoUpgradeCollection((const char*) zm_pStreamMimeTypes[0], m_pContext);
	rc = HXR_FAILED;
	goto cleanup;
    }

    m_pPacketParser = new CSmilPacketParser;

    m_pSmilDocRenderer->onHeader(pHeader);     

    if (m_bUseNestedMeta)
    {
	if (!m_pSmilDocRenderer->IsNestedMetaSupported(uSupportedType))
	{
	    rc = HXR_INVALID_METAFILE;
	    goto cleanup;
	}

	switch (uSupportedType)
	{
	case 1:
	    // RP8 SS3 - limited SMIL in RAM support, blow-away all the existing groups
	    // when it is the last SMIL in RAM.
	    if (HXR_OK == m_pPlayer->QueryInterface(IID_IHXGroupManager, (void**)&pGrpMgr))
	    {
		uGroupCount = pGrpMgr->GetGroupCount();
		pGrpMgr->GetCurrentGroup(uCurrentGroupID);
		
		if ((uCurrentGroupID + 1) == uGroupCount)
		{	
		    m_pSmilDocRenderer->m_bLastGroupInRAM20 = TRUE;
		    while(pGrpMgr && pGrpMgr->GetGroupCount() > 0)
		    {
			pGrpMgr->RemoveGroup(0);
		    }
		}
	    }
	    HX_RELEASE(pGrpMgr);
	    break;
	default:
	    break;
	}
    }
    else
    {
	if (HXR_OK == m_pPlayer->QueryInterface(IID_IHXGroupManager, (void**)&pGrpMgr))
	{
    	    while(pGrpMgr && pGrpMgr->GetGroupCount() > 0)
	    {
	        pGrpMgr->RemoveGroup(0);
	    }	    
	}
	HX_RELEASE(pGrpMgr);
    }

cleanup:

    return rc;
}


/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnPacket
//  Purpose:
//	Called by client engine when a packet for this renderer is 
//	due.
//
STDMETHODIMP CSmil1Renderer::OnPacket(IHXPacket* pPacket, 
					 LONG32 lTimeOffset)
{
    HX_RESULT rc = HXR_OK;
    IHXBuffer* pBuffer = NULL;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnPacket(pPacket, lTimeOffset);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    HX_ASSERT(lTimeOffset <= 0);
    m_pSmilDocRenderer->m_ulPersistentComponentDelay = -lTimeOffset;

    //Fix for PR 23352: if we already returned a fail value from
    // a prior OnPacket(), don't process this OnPacket():
    if (HXR_OK != m_lastOnPacketResult)
    {
	rc = m_lastOnPacketResult;
	goto cleanup;
    }
    pBuffer = pPacket->GetBuffer();
    if(pBuffer)
    {
	// /If parse() runs into a missing-quote situation,
	// we should stop and display an error.  Note that
	// CSmilPacketParser::parse(..)  does not keep track of
	// line numbers; TODO: add line-number counting to it.
	CSmilPacketParser::SMILPacketParseResult pktPrsRslt = CSmilPacketParser::SMILUnknown;
	CSmilPacket* pSmilPacket = m_pPacketParser->parse(
	    pBuffer, pktPrsRslt);
	if (CSmilPacketParser::SMILMissingQuote == pktPrsRslt)
	{
	    CSmil1SMILSyntaxErrorHandler errHandler(m_pContext);
	    errHandler.ReportError(SMILErrorBadAttribute,
		    "missing quote", 0);
	}
	else if(pSmilPacket)
	{
	    switch(pSmilPacket->m_type)
	    {
		case CSmilPacket::SMILDocument:
		{
#if defined(HELIX_FEATURE_SMIL2)
		    if (!m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
		    {
			// /findSmilTagAndVersion returns TRUE if it finds
			// the <smil> tag, regardless of the presence of
			// a default namespace:
			if (HXR_OK == findSmilTagAndVersion(pBuffer))
			{
			    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
			    {
				HX_RESULT pSUS2R = setUpNextGenSmilRenderer();
				if (HXR_OK!=pSUS2R  ||  !m_pNextGenSmilRenderer)
				{
				    rc = HXR_UNEXPECTED;
				    goto cleanup;
				}
			    }
			}

		    }

		    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
		    {
			rc = m_pNextGenSmilRenderer->OnPacket(pPacket,
				lTimeOffset);
		    }
		    else
#endif /* defined(HELIX_FEATURE_SMIL2). */
		    {
			rc = handleSMILDocumentPacket(
			    (CSmilDocumentPacket*)pSmilPacket);
		    }
		    if (HXR_OK != rc)
		    {
			m_lastOnPacketResult = rc;
		    }
		}
		break;

		default:
		break;
	    }
	    delete pSmilPacket;
	}
	HX_RELEASE(pBuffer);
    }

cleanup:
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnTimeSync
//  Purpose:
//	Called by client engine to inform the renderer of the current
//	time relative to the streams synchronized time-line. The 
//	renderer should use this time value to update its display or
//	render it's stream data accordingly.
//
STDMETHODIMP CSmil1Renderer::OnTimeSync(ULONG32 ulTime)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnTimeSync(ulTime);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    if(m_pSmilDocRenderer)
    {
	rc = m_pSmilDocRenderer->onTimeSync(ulTime);
    }

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnPreSeek
//  Purpose:
//	Called by client engine to inform the renderer that a seek is
//	about to occur. The render is informed the last time for the 
//	stream's time line before the seek, as well as the first new
//	time for the stream's time line after the seek will be completed.
//
STDMETHODIMP CSmil1Renderer::OnPreSeek(ULONG32 ulOldTime, ULONG32 ulNewTime)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnPreSeek(ulOldTime, ulNewTime);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    if(m_pSmilDocRenderer)
    {
	rc = m_pSmilDocRenderer->onPreSeek(ulOldTime, ulNewTime);
    }

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnPostSeek
//  Purpose:
//	Called by client engine to inform the renderer that a seek has
//	just occured. The render is informed the last time for the 
//	stream's time line before the seek, as well as the first new
//	time for the stream's time line after the seek.
//
STDMETHODIMP CSmil1Renderer::OnPostSeek(ULONG32 ulOldTime,
					   ULONG32 ulNewTime)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnPostSeek(ulOldTime, ulNewTime);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnPause
//  Purpose:
//	Called by client engine to inform the renderer that a pause has
//	just occured. The render is informed the last time for the 
//	stream's time line before the pause.
//
STDMETHODIMP CSmil1Renderer::OnPause(ULONG32 ulTime)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnPause(ulTime);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnBegin
//  Purpose:
//	Called by client engine to inform the renderer that a begin or
//	resume has just occured. The render is informed the first time 
//	for the stream's time line after the resume.
//
STDMETHODIMP CSmil1Renderer::OnBegin(ULONG32 ulTime)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnBegin(ulTime);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::OnBuffering
//  Purpose:
//	Called by client engine to inform the renderer that buffering
//	of data is occuring. The render is informed of the reason for
//	the buffering (start-up of stream, seek has occured, network
//	congestion, etc.), as well as percentage complete of the 
//	buffering process.
//
STDMETHODIMP CSmil1Renderer::OnBuffering(ULONG32 ulFlags,
	UINT16 unPercentComplete)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnBuffering(ulFlags,
		unPercentComplete);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IHXRenderer::GetDisplayType
//  Purpose:
//	Called by client engine to ask the renderer for it's preferred
//	display type. When layout information is not present, the 
//	renderer will be asked for it's prefered display type. Depending
//	on the display type a buffer of additional information may be 
//	needed. This buffer could contain information about preferred
//	window size.
//
STDMETHODIMP CSmil1Renderer::GetDisplayType
(
    REF(HX_DISPLAY_TYPE)   ulFlags,
    REF(IHXBuffer*)	    pBuffer
)
{
    HX_RESULT rc = HXR_OK;

    ulFlags = HX_DISPLAY_NONE;

#if defined(HELIX_FEATURE_SMIL2)
    m_bCSmil1GetDisplayTypeHasBeenCalled = TRUE;
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->GetDisplayType(ulFlags,
		pBuffer);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/************************************************************************
 *	Method:
 *	    IHXRenderer::OnEndofPackets
 *	Purpose:
 *	    Called by client engine to inform the renderer that all the
 *	    packets have been delivered. However, if the user seeks before
 *	    EndStream() is called, renderer may start getting packets again
 *	    and the client engine will eventually call this function again.
 */
STDMETHODIMP CSmil1Renderer::OnEndofPackets(void)
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->OnEndofPackets();
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/************************************************************************
 *	Method:
 *	    IHXStatistics::Init
 *	Purpose:
 *	    Pass registry ID to the caller
 *
 */
STDMETHODIMP
CSmil1Renderer::InitializeStatistics
(
    UINT32	/*IN*/ ulRegistryID
)
{
    HX_RESULT rc = HXR_OK;

    IHXRegistry* pRegistry = NULL;

#if defined(HELIX_FEATURE_SMIL2)
    m_ulRegistryID = ulRegistryID;
    m_bCSmil1InitializeStatisticsHasBeenCalled = TRUE;
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->InitializeStatistics(m_ulRegistryID);
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    if (m_pContext &&
	HXR_OK == m_pContext->QueryInterface(IID_IHXRegistry, (void**)&pRegistry))
    {
	char	    szRegistryEntry[MAX_DISPLAY_NAME] = {0}; /* Flawfinder: ignore */
	IHXBuffer*	    pszRegistryName = NULL;

	// Get the current registry key name
	if (HXR_OK == pRegistry->GetPropName(ulRegistryID, pszRegistryName))
	{
	    IHXBuffer* pValue = new CHXBuffer();
	    pValue->AddRef();

	    SafeSprintf(szRegistryEntry, MAX_DISPLAY_NAME, "%s.name", pszRegistryName->GetBuffer());

	    pValue->Set((const UCHAR*)zm_pName, strlen((const char*) zm_pName) + 1);
	    pRegistry->AddStr(szRegistryEntry, pValue);

	    HX_RELEASE(pValue);
	    HX_RELEASE(pszRegistryName);
	}

	HX_RELEASE(pRegistry);
    }

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

/************************************************************************
 *	Method:
 *	    IHXStatistics::Update
 *	Purpose:
 *	    Notify the client to update its statistics stored in the registry
 *
 */
STDMETHODIMP 
CSmil1Renderer::UpdateStatistics()
{
    HX_RESULT rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->UpdateStatistics();
        goto cleanup;
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

// IHXPersistentRenderer methods
STDMETHODIMP
CSmil1Renderer::InitPersistent(UINT32			ulPersistentComponentID,
			     UINT16			uPersistentGroupID,
			     UINT16			uPersistentTrackID,
			     IHXPersistentRenderer*	pPersistentParent)
{
    HX_RESULT rc = HXR_OK;

    m_ulPersistentComponentID = ulPersistentComponentID;
    m_uPersistentGroupID = uPersistentGroupID;
    m_uPersistentTrackID = uPersistentTrackID;
    m_pPersistentParent = pPersistentParent;
    if (m_pPersistentParent)
    {
	m_pPersistentParent->AddRef();
    }

#if defined(HELIX_FEATURE_SMIL2)
#if defined(DO_INIT_PERSISTENT_ON_NEXTGEN_SMILREND)
    m_bCSmil1InitPersistenHasBeenCalled = TRUE;
    HX_ASSERT (!m_bIsHigherVersionSmilStreamFromOldSMIL1FF);
#endif
#endif /* defined(HELIX_FEATURE_SMIL2). */

    rc = m_pSmilDocRenderer->InitPersistent(ulPersistentComponentID,
					      uPersistentGroupID,
					      uPersistentTrackID,
					      pPersistentParent);

    return rc;
}

STDMETHODIMP
CSmil1Renderer::GetPersistentID(REF(UINT32) ulPersistentComponentID)
{
    HX_RESULT	rc = HXR_OK;

    ulPersistentComponentID = m_pSmilDocRenderer->m_ulPersistentComponentID;

    return rc;
}

STDMETHODIMP
CSmil1Renderer::GetPersistentProperties(REF(IHXValues*) pProperties)
{
    HX_RESULT	rc = HXR_OK;

    pProperties = new CHXHeader();
    pProperties->AddRef();

    pProperties->SetPropertyULONG32("PersistentType", m_persistentType);
    pProperties->SetPropertyULONG32("PersistentVersion", m_ulPersistentVersion);

    return rc;
}

STDMETHODIMP
CSmil1Renderer::GetElementProperties(UINT16		uGroupID,
				    UINT16		uTrackID,
                               	    REF(IHXValues*)	pProperties)
{
    HX_RESULT	rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    m_bCSmil1GetElementPropertiesHasBeenCalled = TRUE;
    m_uGroupID = uGroupID;
    m_uTrackID = uTrackID;
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->GetElementProperties(
		uGroupID, uTrackID, pProperties);
	HX_ASSERT(0  &&  "XXXEH- call on both sm1 and sm2 rends?");
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    rc = m_pSmilDocRenderer->GetElementProperties(uGroupID,
						    uTrackID,
						    pProperties);

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

STDMETHODIMP
CSmil1Renderer::AttachElementLayout(UINT16	    uGroupID,
				    UINT16	    uTrackID,
				    IHXRenderer*   pRenderer,
				    IHXStream*	    pStream,
				    IHXValues*	    pProps)
{
    HX_RESULT	rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->AttachElementLayout(uGroupID,
							 uTrackID,
							 pRenderer,
							 pStream,
							 pProps);
	HX_ASSERT(0  &&  "XXXEH- call on both sm1 and sm2 rends?");
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    rc = m_pSmilDocRenderer->AttachElementLayout(uGroupID, 
						 uTrackID,
						 pRenderer,
						 pStream,
						 pProps);

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

STDMETHODIMP
CSmil1Renderer::DetachElementLayout(IUnknown* pLSG)
{
    HX_RESULT	rc = HXR_OK;

#if defined(HELIX_FEATURE_SMIL2)
    m_bCSmil1RemoveLayoutSiteGroupHasBeenCalled = TRUE;
    if (m_bIsHigherVersionSmilStreamFromOldSMIL1FF)
    {
	if (!m_pNextGenSmilRenderer)
	{
	    rc = HXR_UNEXPECTED;
	    goto cleanup;
	}
	rc = m_pNextGenSmilRenderer->DetachElementLayout(pLSG);
	HX_ASSERT(0  &&  "XXXEH- call on both sm1 and sm2 rends?");
    }
#endif /* defined(HELIX_FEATURE_SMIL2). */

    rc = m_pSmilDocRenderer->DetachElementLayout(pLSG);

#if defined(HELIX_FEATURE_SMIL2)
cleanup:
#endif /* defined(HELIX_FEATURE_SMIL2). */
    return rc;
}

STDMETHODIMP
CSmil1Renderer::GetElementStatus(UINT16		    uGroupID,
				UINT16		    uTrackID,
				UINT32		    ulCurrentTime,
				REF(IHXValues*)    pStatus)
{
    return m_pSmilDocRenderer->GetElementStatus(uGroupID,
						uTrackID,
						ulCurrentTime,
						pStatus);
}

HX_RESULT
CSmil1Renderer::HandleAddLayoutSiteGroup(IUnknown* pLSG)
{
    HX_RESULT		    rc = HXR_OK;
    IHXValues*		    pProps = NULL;
    IHXPersistentRenderer* pPersistentParentRenderer = NULL;

    pPersistentParentRenderer = m_pSmilDocRenderer->m_pPersistentParentRenderer;

    // nested meta, get layout from its parent
    if (pPersistentParentRenderer)
    {
	pProps = new CHXHeader();	
	if (pProps)
	{
	    pProps->AddRef();
	    pProps->SetPropertyULONG32("PersistentType", m_persistentType);
	}

	rc = pPersistentParentRenderer->AttachElementLayout(m_pSmilDocRenderer->m_uPersistentGroupID,
							    m_pSmilDocRenderer->m_uPersistentTrackID,
							    (IHXRenderer*)pLSG,
							    NULL,
							    pProps);
    }
    else if (m_pPlayer)
    {
	IHXLayoutSiteGroupManager* pLSGMgr = 0;
	if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXLayoutSiteGroupManager, (void**)&pLSGMgr))
	{
	    rc = pLSGMgr->AddLayoutSiteGroup(pLSG);
	    pLSGMgr->Release();
	}
    }

    HX_RELEASE(pProps);

    return rc;
}

HX_RESULT
CSmil1Renderer::HandleRemoveLayoutSiteGroup(IUnknown* pLSG)
{
    HX_RESULT		    rc = HXR_OK;
    IHXPersistentRenderer* pPersistentParentRenderer = NULL;

    pPersistentParentRenderer = m_pSmilDocRenderer->m_pPersistentParentRenderer;

    // nested meta, remove layout from its parent
    if (pPersistentParentRenderer)
    {
	rc = pPersistentParentRenderer->DetachElementLayout(pLSG);
    }
    else if (m_pPlayer)
    {
	IHXLayoutSiteGroupManager* pLSGMgr = 0;
	if(HXR_OK == m_pPlayer->QueryInterface(IID_IHXLayoutSiteGroupManager, (void**)&pLSGMgr))
	{
	    rc = pLSGMgr->RemoveLayoutSiteGroup(pLSG);
	    pLSGMgr->Release();
	}
    }
    return rc;
}

HX_RESULT
CSmil1Renderer::HandleAttachElementLayout(IUnknown* pLSG, IHXValues* pProps)
{
    HX_RESULT	rc = HXR_OK;

    if (m_pPersistentComponentManager)
    {
	rc = m_pPersistentComponentManager->AttachPersistentComponentLayout(pLSG, pProps);
    }

    return rc;
}

HX_RESULT
CSmil1Renderer::handleSMILDocumentPacket(CSmilDocumentPacket* pPacket)
{
    HX_RESULT rc = HXR_OK;

    if(pPacket->m_version == RMA_DRIVER_VERSION)
    {
	CHXBuffer* pBuffer = new CHXBuffer;
	pBuffer->AddRef();
	BOOL bLastPacket = FALSE;
	UINT32 ulDocLen = (UINT32)pPacket->m_document.GetLength();
	HX_ASSERT(ulDocLen);
	// /Extra safety check:
	if (0 == ulDocLen)
	{
	    rc = HXR_UNEXPECTED;
	    bLastPacket = FALSE;
	}
	else
	{
	    pBuffer->Set((const BYTE*)(const char*)pPacket->m_document,
		    pPacket->m_document.GetLength());
	    m_ulTotalSMILPackets++;
	    bLastPacket = pPacket->m_ulTotalPackets == m_ulTotalSMILPackets;
	    rc = m_pSmilDocRenderer->onPacket(pBuffer, bLastPacket);
	}
	HX_RELEASE(pBuffer);

	if(HXR_OK != rc)
	{
	    // XML parsing error
	    UINT32 ulLineNumber = 0;
	    UINT32 ulColumnNumber = 0;
	    IHXBuffer* pErrorText = NULL;
	    m_pSmilDocRenderer->getErrorInfo(ulLineNumber,
		ulColumnNumber, pErrorText);

	    const char* pActualErrorText = NULL;
	    if(pErrorText)
	    {
		pActualErrorText = (const char*)pErrorText->GetBuffer();
	    }
	    CSmil1XMLSyntaxErrorHandler errHandler(m_pContext);
	    errHandler.ReportError(rc, pActualErrorText, ulLineNumber);
	    HX_RELEASE(pErrorText);
	}
	else if(bLastPacket)
	{
	    //[SMIL 1.0 compliance] Handle error from setDocument()
	    // otherwise it "plays" (nothing for 20 seconds) when it
	    // should halt under error conditions:
	    rc = m_pSmilDocRenderer->setDocument(m_pURLFragment);
	}
    }
    return rc;
}

void
CSmil1Renderer::generatePreFix()
{
    // get the protocol/server for later...
    IHXStreamSource* pSource = 0;
    m_pStream->GetSource(pSource);
    HX_ASSERT(pSource);
    if (!pSource)
    {
	return;
    }

    const char* pURL = pSource->GetURL();
    HX_ASSERT(pURL);
    
    if (!pURL)
    {
	return;
    }

    CHXURL::GeneratePrefixRootFragment(pURL, m_urlPrefix, m_urlRoot, m_pURLFragment);
    HX_RELEASE(pSource);

    return;
}


#if defined(HELIX_FEATURE_SMIL2)
HX_RESULT 
CSmil1Renderer::findValidDefaultNamespace(IHXBuffer* pStartOfFile)
{
    HX_RESULT pnreslt = HXR_OK;
    const char* pSmilTag = NULL;
    const char* pCloseOfSmilTag = NULL;
    const char* pXmlns = NULL;
    const char* pXmlnsOpenQuotationMark = NULL;
    const char* pBuf = NULL;
    ULONG32 ulBufLen = 0;
    const char* pEqualsSign = NULL;
    const char* pTmp = NULL;
    char* pTmp2 = NULL;
    char* pszStartOfFile = NULL;
    ULONG32 ulCount = 0;
    LONG32 lNumCommentsOpen = 0;

    if (!pStartOfFile)
    {
	pnreslt = HXR_BUFFERTOOSMALL;
	goto cleanup;
    }

    // /Fixes PR 59282: pStartOfFile is not necessarily NULL-terminated,
    // so we have to limit our access of it to the size of the buffer:
    ulBufLen = pStartOfFile->GetSize();

    pBuf = (const char*)pStartOfFile->GetBuffer();

    if (!pBuf  ||  !ulBufLen)
    {
	pnreslt = HXR_BUFFERTOOSMALL;
	goto cleanup;
    }

    pszStartOfFile = new char[ulBufLen+1];
    if (!pszStartOfFile)
    {
	pnreslt = HXR_OUTOFMEMORY;
	goto cleanup;
    }

    // /Now, walk through and copy each character from non-NULL terminated buf:
    pTmp = pBuf; 
    pTmp2 = pszStartOfFile;
    while (*pTmp  &&  ulCount<ulBufLen)
    {
	*pTmp2 = *pTmp;
	pTmp++;
	pTmp2++;
	ulCount++;
    }
    pszStartOfFile[ulCount] = '\0'; // /NULL-terminate it.

    // /Now, let's walk through the start of pszStartOfFile looking for
    // namespace declaration(s) inside the <smil ...> tag, and make sure that
    // the smil tag is NOT inside of a comment:
    pTmp = pszStartOfFile;
    while (*pTmp)
    {
	if (0==strncmp(pTmp, "<!--", 4) )
	{
	    lNumCommentsOpen++;
	    pTmp+=4;
	}
	else if (0==strncmp(pTmp, "-->", 3) )
	{
	    lNumCommentsOpen--;
	    pTmp+=3;
	}
	else if (lNumCommentsOpen<=0  &&  0==strncmp(pTmp, "<smil", 5) )
	{
	    pSmilTag = pTmp; // /We found the smil tag outside of a comment.
	    break;
	}
	else
	{
	    pTmp++;
	}
    }
    if (!pSmilTag  ||  ulBufLen-(pSmilTag-pszStartOfFile) < 6) // /6==min size: "<smil>"
    {
	// /There had better be a <smil...> tag!
	pnreslt = HXR_UNEXPECTED;
	goto cleanup;
    }

    pCloseOfSmilTag = strchr(pSmilTag, '>');
    // /Add 6 so that we don't allow "<smilxmlns ..." (which is invalid):
    pXmlns = strstr(pSmilTag+6, "xmlns");
    if (pXmlns  &&  isspace(*(pXmlns-1)) ) // /"xmlns" must follow a space.
    {
	pEqualsSign = strchr(pXmlns, '=');
	if (pEqualsSign)
	{
	    pXmlnsOpenQuotationMark = strchr(pXmlns, '\"');
	}
    }
    if (pXmlns  &&   pEqualsSign  &&  pXmlnsOpenQuotationMark  &&
	    (!pCloseOfSmilTag  ||  pXmlns<pCloseOfSmilTag))
    {
	m_bIsHigherVersionSmilStreamFromOldSMIL1FF = TRUE;

	// /First, make sure there is nothing but whitespace between
	// the "xmlns" and the '=' as well as between the '=' and the
	// quotation mark:
	char* pTmp = (char*)(pXmlns + strlen("xmlns"));
	while (pTmp<pEqualsSign)
	{
	    if (!isspace(*pTmp))
	    {
		m_bIsHigherVersionSmilStreamFromOldSMIL1FF = FALSE;
		break;
	    }
	    pTmp++;
	}
	pTmp = (char*)(pEqualsSign+1);
	while(pTmp<pXmlnsOpenQuotationMark)
	{
	    if (!isspace(*pTmp)  &&  *pTmp != '\\')
	    {
		m_bIsHigherVersionSmilStreamFromOldSMIL1FF = FALSE;
		break;
	    }
	    pTmp++;
	}

	char* pDefaultNamespace = (char*)pXmlnsOpenQuotationMark+1;
	if (strlen(pDefaultNamespace) > 0)
	{
	    char* pXmlnsCloseQuotationMark = strstr(pDefaultNamespace, "\\\"");
	    if (pXmlnsCloseQuotationMark)
	    {
		*pXmlnsCloseQuotationMark = '\0'; // /NULL-terminate it.
	    }
	    else
	    {
		HX_ASSERT(pXmlnsCloseQuotationMark);
		pnreslt = HXR_FAIL;
		goto cleanup;
	    }
	}
	HX_RELEASE(m_pDefaultNamespace);
	if (m_pCommonClassFactory  &&
		HXR_OK == m_pCommonClassFactory->CreateInstance(CLSID_IHXBuffer,
		(void**)&m_pDefaultNamespace) )
	{
	    LONG32 lLen = strlen(pDefaultNamespace);
	    if (lLen > 0  &&  lLen < 10000) // /sanity check
	    {
		m_pDefaultNamespace->Set((const BYTE*)pDefaultNamespace,
		        lLen + 1);
	    }
	    pnreslt = HXR_OK;
	}

    }

cleanup:
    if (pszStartOfFile)
    {
	HX_VECTOR_DELETE(pszStartOfFile);
    }

    return pnreslt;
}
#endif /* defined(HELIX_FEATURE_SMIL2). */


#if defined(HELIX_FEATURE_SMIL2)
// /The following two methods are used for looking for and handling SMIL 2+
// streams coming from SMIL 1.0 File Format plugins (that didn't have smarts
// to look for default namespace which identifies the SMIL file version):
HX_RESULT
CSmil1Renderer::findSmilTagAndVersion(IHXBuffer* pFileChunk)
{
    HX_RESULT rc = HXR_OK;

    ULONG32 ulLen = 0;
    if (!pFileChunk  ||  (ulLen=pFileChunk->GetSize()) < 6)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    rc = findValidDefaultNamespace(pFileChunk);

cleanup:
    return rc;
}


HX_RESULT
CSmil1Renderer::setUpNextGenSmilRenderer()
{
    HX_RESULT rc = HXR_OK;

    rc = CSmilRenderer::HXCreateInstance((IUnknown**)(&m_pNextGenSmilRenderer));

    if (HXR_OK == rc  &&  m_pNextGenSmilRenderer)
    {
	IHXSmilToSmilRendererCommunicator* pSmilToSmilComm = NULL;
	if (HXR_OK == m_pNextGenSmilRenderer->
		QueryInterface(IID_IHXSmilToSmilRendererCommunicator,
		(void**) &pSmilToSmilComm))
	{
	    if (HXR_OK ==
		    pSmilToSmilComm->InitSmilStreamProxiedByOtherRenderer(
		    m_pDefaultNamespace))
	    {
		rc = m_pNextGenSmilRenderer->InitPlugin(m_pContext);
	    }
	    HX_RELEASE(pSmilToSmilComm);
	}
	if (HXR_OK == rc)
	{
	    // /XXXEH- note that the REF parameters in GetPluginInfo() and
	    // GetRendererInfo are set by the callee, but we can't do
	    // anything with the values passed back as we've already set these
	    // to our own vals and there's no way to re-set them in the core:
	    BOOL bLoadMultiple = FALSE;
	    const char* pszDescription = NULL;
	    const char* pszCopyright = NULL;
	    const char* pszMoreInfoURL = NULL;
	    ULONG32 ulVersionNumber = 0;
	    rc = m_pNextGenSmilRenderer->GetPluginInfo(bLoadMultiple,
		    pszDescription, pszCopyright, pszMoreInfoURL,
		    ulVersionNumber);
	    if (HXR_OK == rc)
	    {
		const char** ppStreamMimeTypes = NULL;
		ULONG32 ulInitialGranularity = m_ulGranularity;
		rc = m_pNextGenSmilRenderer->GetRendererInfo(
			ppStreamMimeTypes, ulInitialGranularity);
	    }
	    if (HXR_OK == rc)
	    {
		HX_ASSERT(m_bCSmil1StartStreamHasBeenCalled);
		rc = m_pNextGenSmilRenderer->StartStream(m_pStream,
			m_pPlayer);
	    }
	    if (HXR_OK == rc  &&  m_bCSmil1EndStreamHasBeenCalled)
	    {
		rc = m_pNextGenSmilRenderer->EndStream();
	    }
	    if (HXR_OK == rc)
	    {
		if (!m_pHeader)
		{
		    HX_ASSERT(m_pHeader);
		    rc = HXR_UNEXPECTED;
		}
		else
		{
		    rc = m_pNextGenSmilRenderer->OnHeader(m_pHeader);
		}
	    }
	    if (HXR_OK == rc  &&  m_bCSmil1GetDisplayTypeHasBeenCalled)
	    {
		HX_DISPLAY_TYPE ulFlags = HX_DISPLAY_NONE;
		IHXBuffer* pBuffer = NULL;
		rc = m_pNextGenSmilRenderer->GetDisplayType(ulFlags, pBuffer);
		if (pBuffer)
		{
		    // /XXXEH- We can't use it here, so just clean it up:
		    HX_DELETE(pBuffer);
		}
	    }
	    if (HXR_OK == rc  &&  m_bCSmil1InitializeStatisticsHasBeenCalled)
	    {
		rc = m_pNextGenSmilRenderer->InitializeStatistics(m_ulRegistryID);
	    }

#if defined(DO_INIT_PERSISTENT_ON_NEXTGEN_SMILREND)
	    if (HXR_OK == rc  &&  m_bCSmil1InitPersistenHasBeenCalled)
	    {
		// /NOTE: it's perfectly OK for m_pPersistentParent to be NULL
		rc = m_pNextGenSmilRenderer->InitPersistent(
			m_ulPersistentComponentID, m_uPersistentGroupID,
			m_uPersistentTrackID, m_pPersistentParent);
	    }
#endif
	    if (HXR_OK == rc  &&  m_bCSmil1GetElementPropertiesHasBeenCalled)
	    {
		IHXValues* pProperties = NULL;
		// /XXXEH- TODO: we need to notify the original caller of this
		// of the next-gen renderer's setting of pProperties.
		rc = m_pNextGenSmilRenderer->GetElementProperties(
			m_uGroupID, m_uTrackID, pProperties);
	    }
	    if (HXR_OK == rc  &&  m_bCSmil1RemoveLayoutSiteGroupHasBeenCalled)
	    {
		// /XXXEH- Need to handle this (unexpected) case by saving the 
		// removed layout site group (which could differ from the
		// add site layout group,  m_pLayoutSiteGroup (or not?):
		HX_ASSERT(!m_bCSmil1RemoveLayoutSiteGroupHasBeenCalled);
	    }
	}
    }

    return rc;
}
#endif /* defined(HELIX_FEATURE_SMIL2). */

