/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Clase base de ficheros.
 * @file    JFile.h
 * @author  Juan Carlos Seijo Prez
 * @date    15/06/2003
 * @version 0.0.1 - 15/06/2003 - Primera versin.
 */

#ifndef _JFILE_INCLUDED
#define _JFILE_INCLUDED

#include <JLib/Util/JTypes.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <JLib/Util/JCompatibility.h>

/** Clase que encapsula un fichero.
 */
class JFile
{
protected:
  FILE *file;               /**< Descriptor de fichero */
  s8 *buff;                 /**< Buffer auxiliar de lectura */
  s8 *name;                 /**< Nombre del fichero asociado */
	bool exists;              /**< Flag de existencia */
  struct stat statInfo;     /**< Informacin del fichero */
  u32 size;                 /**< Tamao del fichero */
  u32 buffSize;             /**< Tamao del buffer */

	/** Determina si el fichero dado es del tipo dado
   * @param  name Nombre del fichero a comprobar.
   * @param  mode Tipo de fichero.
   * @return <b>true</b> en caso de que el fichero sea del tipo dado, <b>false</b>
   * si no lo es.
   */
	static bool IsOfType(const s8 *name, mode_t mode);

public:
  
  /** Crea un objeto fichero. Para comenzar a trabajar con l se debe usar Open().
   * @param  filename Nombre del fichero.
   */
  JFile(const s8 *filename = 0);

  /** Destruye el objeto y libera la memoria asociada.
   */
  ~JFile();

  /** Vuelve a consultar la informacin del fichero, (tamao, permisos, etc.).
   * @return 0 (cero) si todo va bien, -1 en caso de error y -2 si no se dio un nombre.
   */
	s32 Refresh();

  /** Abre el fichero. 
   * @param  filename Nombre del fichero a abrir.
   * @param  mode Modo de apertura en la forma {w|r|a}[+][{b|t}].
   * por defecto el modo es "ab" (adicin, creacin si no existe y binario).
   */
  virtual bool Open(const s8 *filename = 0, const s8 *mode = "ab");
  
  /** Lee del fichero al buffer interno.
   * @param  readSize Tamao en bytes a leer del fichero. Si es cero, lee todo el fichero.
   * @return Nmero de bytes leidos o cero si hubo algn error. 
   */
  virtual u32 Read(u32 readSize = 0);
  
  /** Lee del fichero al buffer dado.
   * @param  buff Buffer a rellenar con los datos ledos.
   * @param  size Tamao en bytes a leer del fichero. Si es cero, lee todo el fichero.
   * @return Nmero de bytes leidos o cero si hubo algn error. 
   */
  virtual u32 Read(void *buff, u32 size);

  /** Lee del fichero al buffer dado y deja el puntero en su posicin original.
   * @param  buff Buffer a rellenar con los datos ledos.
   * @param  size Tamao en bytes a leer del fichero. Si es cero, lee todo el fichero.
   * @return Nmero de bytes leidos o cero si hubo algn error. 
   */
  virtual u32 Peep(void *buff, u32 size);

  /** Carga todo el fichero. Equivalente a Open() + Read(0).
   * @param  filename Nombre del fichero a cargar.
   * @param  mode Modo de apertura en la forma {w|r|a}[+][{b|t}].
   * por defecto el modo es "rb" (lectura, binario).
   * @return <b>true</b> Si se carg el fichero correctamente, <b>false</b> si
   * hubo algn error.
   */
  virtual bool Load(const s8 *filename, const s8 *mode = "rb");
  
  /** Escribe en el fichero.
   * @param  buff Buffer del que tomar los datos a escribir.
   * @param  size Tamao en bytes a escribir en el fichero.
   * @return Nmero de bytes escritos o cero si hubo algn error.
   */
  u32 Write(void *buff, u32 size);

  /** Devuelve el buffer ledo.
   * @return Buffer interno del fichero. No se debe borrar nunca desde fuera.
   */
  s8 *Buffer() {return buff;}
  
  /** Devuelve el tamao del ltimo buffer ledo mediante Read(size).
   * @return Tamao del ltimo buffer leido.
   */
  u32 BufferSize() {return buffSize;}
  
  /** Devuelve el tamao del fichero. Si el fichero pudiera haber cambiado
   * su tamao desde su creacin, es necesario actualizarlo con Refresh().
   * @return Tamao en bytes del fichero.
   */
  inline u32 FileSize() {return statInfo.st_size;}
  
  /** Libera la memoria del buffer interno.
   */
  void FreeBuffer();

  /** Situa el puntero al inicio del fichero.
   */
  void Rewind();

  /** Cierra el fichero.
   * @return <b>true</b> si liber memoria, <b>false</b> si no lo hizo.
   */
  bool Close();

  /** Devuelve el descriptor interno de fichero.
   * @return Descriptor de fichero asociado a este fichero.
   */
  const FILE * File() {return file;}

  /** Devuelve el nombre de fichero.
   * @return Nombre del fichero.
   */
  const s8 * Name() {return name;}

  /** Devuelve la posicin actual dentro del fichero desde el origen.
   * @return Posicin actual en bytes desde el comienzo.
   */
  s32 Pos() {return ftell(file);}

  /** Establece la posicin actual dentro del fichero desde el origen.
   * @param  pos Nueva posicin actual.
   * @return <b>true</b> si la posicin estaba comprendida entre el comienzo y 
   * el fin del fichero, <b>false</b> si no era una posicin vlida.
   */
  bool Pos(s32 pos) {return 0 == fseek(file, pos, SEEK_SET);}

	/** Determina si existe el fichero. Si se pudiera haber creado el fichero desde fuera
   * es necesario llamar a Refresh().
   * @return <b>true</b> si existe, <b>false</b> si no.
   */
	bool Exists() {return exists;}

	/** Devuelve el tamao del fichero.
   * @return Tamao en bytes del fichero.
   */
	u32 Size() {return statInfo.st_size;}

	/** Determina si es un fichero regular (no es directorio, enlace, etc.)
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	bool IsFile() {return 0 != (statInfo.st_mode & S_IFREG);}

	/** Determina si es un directorio
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	bool IsDir() {return 0 != (statInfo.st_mode & S_IFDIR);}

	/** Determina si es un enlace a otro fichero
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	bool IsLink() {return 0 != (statInfo.st_mode & S_IFLNK);}

	/** Determina si es un dispositivo de caracter
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	bool IsCharDev() {return 0 != (statInfo.st_mode & S_IFCHR);}

	/** Determina si es un dispositivo de bloque
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	bool IsBlockDev() {return 0 != (statInfo.st_mode & S_IFBLK);}

	/** Determina si existe el fichero.
   * @param  name Nombre del fichero.
   * @return <b>true</b> si existe, <b>false</b> si no.
   */
	static bool Exists(const s8 *name);

	/** Determina el tamao del fichero.
   * @param  name Nombre del fichero.
   * @return Tamao en bytes del fichero. En caso de que no exista devuelve 0 (cero).
   */
	static u32 Size(const s8 *name);

	/** Determina si es un fichero regular (no es directorio, enlace, etc.)
   * @param  name Nombre del fichero.
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	static bool IsFile(const s8 *name);

	/** Determina si es un directorio.
   * @param  name Nombre del fichero.
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	static bool IsDir(const s8 *name);

	/** Determina si es un enlace a otro fichero/directorio.
   * @param  name Nombre del fichero.
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	static bool IsLink(const s8 *name);

	/** Determina si es un dispositivo de caracteres (puerto de comunicaciones, etc.)
   * @param  name Nombre del fichero.
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	static bool IsCharDev(const s8 *name);

	/** Determina si es un dispositivo de bloque (disco duro, etc.).
   * @param  name Nombre del fichero.
   * @return <b>true</b> si lo es, <b>false</b> si no.
   */
	static bool IsBlockDev(const s8 *name);
};

#endif  // _JFILE_INCLUDED
