/*
 * linux/fs/binfmt_elf.c
 *
 * These are the functions used to load ELF format executables as used
 * on SVr4 machines.  Information on the format may be found in the book
 * "UNIX SYSTEM V RELEASE 4 Programmers Guide: Ansi C and Programming Support
 * Tools".
 *
 * Copyright 1993, 1994: Eric Youngdale (ericy@cais.com).
 */
#include <linux/config.h>

#include <linux/module.h>
#include <linux/version.h>

#include <asm/uaccess.h>

#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/a.out.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/binfmts.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/shm.h>
#include <linux/personality.h>

#include <asm/pgtable.h>

#include <ibcs/ibcs.h>

#define DLINFO_ITEMS 8

#include <linux/elf.h>

#ifdef ELF_TRACE
#include <ibcs/trace.h>
#endif


#undef ELF_PARSE_COMMENTS


static int load_elf_binary(struct linux_binprm * bprm, struct pt_regs * regs);
static int load_elf_library(int fd);


#ifdef ELF_PARSE_COMMENTS
static int parse_comments(struct linux_binprm *exe_bprm, Elf32_Shdr * sect);

/*
 *  The following table gives clues to the "personality" of the executable
 *  which we hope to find in the .comment sections of the binary.
 *  The set here may not be comprehensive even for those systems listed.
 *  Use 'mcs -p' to list the .comments sections of a binary to see what
 *  clues might be there. Or use 'strings' if you don't have mcs.
 */
struct elf_clue {
    short len;		/* negative number uses strstr for lookup */
    char *text;
    unsigned long personality;
};

static struct elf_clue elf_clues[] = {
    /* SCO Unix OpenServer 5 */
    { 18, "@(#)SCO OpenServer",				PER_SCOSVR3 },
    { 22, "@(#) crt1.s.source 20.",			PER_SCOSVR3 },
#if 0
    /* Interactive SVR4 */
    { 33, "@(#)UNIX System V/386 Release 4.0",		PER_SVR4 },
#endif
#if 0
    /* UnixWare 1.x (at _end_ of string) */
    {  -7, " Univel",					PER_SVR4 },
#endif

    /* End of table */
    { 0, 0, 0 }
};
#endif /* ELF_PARSE_COMMENTS */


struct linux_binfmt ibcs_elf_format = {
	NULL, &__this_module, load_elf_binary, load_elf_library, NULL
};


/* We need to explicitly zero any fractional pages
   after the data section (i.e. bss).  This would
   contain the junk from the file that should not
   be in memory */

static void
padzero(unsigned long elf_bss)
{
	unsigned int nbyte;
  
	nbyte = elf_bss & (PAGE_SIZE-1);
	if (nbyte) {
		nbyte = PAGE_SIZE - nbyte;
		clear_user((void *)elf_bss, nbyte);
	}
}

unsigned long * create_elf_tables(char * p,int argc,int envc,struct elfhdr * exec, unsigned int load_addr, unsigned int interp_load_addr, int ibcs)
{
	unsigned long *argv,*envp, *dlinfo;
	unsigned long * sp;

	sp = (unsigned long *) (0xfffffffc & (unsigned long) p);
	if(exec) sp -= DLINFO_ITEMS*2;
	dlinfo = sp;
	sp -= envc+1;
	envp = sp;
	sp -= argc+1;
	argv = sp;
	if (!ibcs) {
		sp--;
		put_user((unsigned long)envp, sp);
		sp--;
		put_user((unsigned long)argv, sp);
	}

	/* The constant numbers (0-9) that we are writing here are
	   described in the header file sys/auxv.h on at least
	   some versions of SVr4 */
	if(exec) { /* Put this here for an ELF program interpreter */
	  struct elf_phdr * eppnt;
	  eppnt = (struct elf_phdr *) exec->e_phoff;
	  put_user(3, dlinfo); dlinfo++;
	  put_user(load_addr + exec->e_phoff, dlinfo); dlinfo++;
	  put_user(4, dlinfo); dlinfo++;
	  put_user(sizeof(struct elf_phdr), dlinfo); dlinfo++;
	  put_user(5, dlinfo); dlinfo++;
	  put_user(exec->e_phnum, dlinfo); dlinfo++;
	  put_user(9, dlinfo); dlinfo++;
	  put_user((unsigned long)exec->e_entry, dlinfo); dlinfo++;
	  put_user(7, dlinfo); dlinfo++;
	  put_user(interp_load_addr, dlinfo); dlinfo++;
	  put_user(8, dlinfo); dlinfo++;
	  put_user(0, dlinfo); dlinfo++;
	  put_user(6, dlinfo); dlinfo++;
	  put_user(PAGE_SIZE, dlinfo); dlinfo++;
	  put_user(0, dlinfo); dlinfo++;
	  put_user(0, dlinfo); dlinfo++;
	}

	sp--;
	put_user((unsigned long)argc, sp);
	current->mm->arg_start = (unsigned long) p;
	while (argc-->0) {
		char *junk;
		put_user((unsigned long)p, argv); argv++;
		p += strlen_user(p);
	}
	put_user(0, argv);
	current->mm->arg_end = current->mm->env_start = (unsigned long) p;
	while (envc-->0) {
		char *junk;
		put_user((unsigned long) p, envp); envp++;
		p += strlen_user(p);
	}
	put_user(0, envp);
	current->mm->env_end = (unsigned long) p;
	return sp;
}


/* This is much more generalized than the library routine read function,
   so we keep this separate.  Technically the library read function
   is only provided so that we can read a.out libraries that have
   an ELF header */

static unsigned int load_elf_interp(struct elfhdr * interp_elf_ex,
			     struct dentry * interpreter_dentry, unsigned int *interp_load_addr)
{
        struct file * file;
	struct elf_phdr *elf_phdata  =  NULL;
	struct elf_phdr *eppnt;
	unsigned int len;
	unsigned int load_addr;
	int load_addr_set = 0;
	int elf_exec_fileno;
	int elf_bss;
	int retval;
	unsigned int last_bss;
	int error;
	int i;
	unsigned int k;
	
	elf_bss = 0;
	last_bss = 0;
	error = load_addr = 0;
	
	/* First of all, some simple consistency checks */
	if((interp_elf_ex->e_type != ET_EXEC && 
	    interp_elf_ex->e_type != ET_DYN) || 
	   (interp_elf_ex->e_machine != EM_386 && interp_elf_ex->e_machine != EM_486) ||
	   (!interpreter_dentry->d_inode->i_op ||
	    !interpreter_dentry->d_inode->i_op->default_file_ops->mmap)){
		return 0xffffffff;
	}
	
	/* Now read in all of the header information */
	
	if(sizeof(struct elf_phdr) * interp_elf_ex->e_phnum > PAGE_SIZE) 
	    return 0xffffffff;
	
	elf_phdata =  (struct elf_phdr *) 
		kmalloc(sizeof(struct elf_phdr) * interp_elf_ex->e_phnum, GFP_KERNEL);
	if(!elf_phdata) return 0xffffffff;
	
	retval = read_exec(interpreter_dentry, interp_elf_ex->e_phoff, (char *) elf_phdata,
			   sizeof(struct elf_phdr) * interp_elf_ex->e_phnum, 1);
	
	elf_exec_fileno = open_dentry(interpreter_dentry, O_RDONLY);
	if (elf_exec_fileno < 0) return 0xffffffff;
	file = fget(elf_exec_fileno);

	eppnt = elf_phdata;
	for(i=0; i<interp_elf_ex->e_phnum; i++, eppnt++)
	  if(eppnt->p_type == PT_LOAD) {
#ifdef PF_R
		int elf_type = MAP_PRIVATE | MAP_DENYWRITE;
		int elf_prot = 0;
		unsigned long vaddr = 0;
	    if (eppnt->p_flags & PF_R) elf_prot = PROT_READ;
	    if (eppnt->p_flags & PF_W) elf_prot |= PROT_WRITE;
	    if (eppnt->p_flags & PF_X) elf_prot |= PROT_EXEC;
		if (interp_elf_ex->e_type == ET_EXEC || load_addr_set) {
			elf_type |= MAP_FIXED;
			vaddr = eppnt->p_vaddr;
	    }
	    error = do_mmap(file, 
			    load_addr + (vaddr & 0xfffff000),
			    eppnt->p_filesz + (vaddr & 0xfff),
				elf_prot,
				elf_type,
			    eppnt->p_offset & 0xfffff000);
#else
	    error = do_mmap(file, 
			    load_addr + (eppnt->p_vaddr & 0xfffff000),
			    eppnt->p_filesz + (eppnt->p_vaddr & 0xfff),
			    PROT_READ | PROT_WRITE | PROT_EXEC,
			    MAP_PRIVATE | MAP_DENYWRITE | MAP_EXECUTABLE | (interp_elf_ex->e_type == ET_EXEC ? MAP_FIXED : 0),
			    eppnt->p_offset & 0xfffff000);
#endif /* PF_R */
	    
	    if (!load_addr_set && interp_elf_ex->e_type == ET_DYN) {
	      load_addr = error;
	      load_addr_set = 1;
	    }
	    k = load_addr + eppnt->p_vaddr + eppnt->p_filesz;
	    if(k > elf_bss) elf_bss = k;
	    if (error < 0 && error > -1024) break;  /* Real error */
	    k = load_addr + eppnt->p_memsz + eppnt->p_vaddr;
	    if(k > last_bss) last_bss = k;
	  }
	

	/* Now use mmap to map the library into memory. */

	SYS(close)(elf_exec_fileno);
	if(error < 0 && error > -1024) {
		fput(file);
	        kfree(elf_phdata);
		return 0xffffffff;
	}

	padzero(elf_bss);
	len = (elf_bss + 0xfff) & 0xfffff000; /* What we have mapped so far */

	/* Map the last of the bss segment */
	if (last_bss > len)
	  do_mmap(NULL, len, last_bss-len,
		  PROT_READ|PROT_WRITE|PROT_EXEC,
		  MAP_FIXED|MAP_PRIVATE, 0);
	kfree(elf_phdata);

	*interp_load_addr = load_addr;
	return ((unsigned int) interp_elf_ex->e_entry) + load_addr;
}

static unsigned int load_aout_interp(struct exec * interp_ex,
			     struct dentry * interpreter_dentry)
{
  int retval;
  unsigned int elf_entry;

  current->mm->brk = interp_ex->a_bss +
    (current->mm->end_data = interp_ex->a_data +
     (current->mm->end_code = interp_ex->a_text));
  elf_entry = interp_ex->a_entry;
  
  
  if (N_MAGIC(*interp_ex) == OMAGIC) {
    do_mmap(NULL, 0, interp_ex->a_text+interp_ex->a_data,
	    PROT_READ|PROT_WRITE|PROT_EXEC,
	    MAP_FIXED|MAP_PRIVATE, 0);
    retval = read_exec(interpreter_dentry, 32, (char *) 0, 
		       interp_ex->a_text+interp_ex->a_data, 0);
  } else if (N_MAGIC(*interp_ex) == ZMAGIC || N_MAGIC(*interp_ex) == QMAGIC) {
    do_mmap(NULL, 0, interp_ex->a_text+interp_ex->a_data,
	    PROT_READ|PROT_WRITE|PROT_EXEC,
	    MAP_FIXED|MAP_PRIVATE, 0);
    retval = read_exec(interpreter_dentry,
		       N_TXTOFF(*interp_ex) ,
		       (char *) N_TXTADDR(*interp_ex),
		       interp_ex->a_text+interp_ex->a_data, 0);
  } else
    retval = -1;
  
  if(retval >= 0)
    do_mmap(NULL, (interp_ex->a_text + interp_ex->a_data + 0xfff) & 
	    0xfffff000, interp_ex->a_bss,
	    PROT_READ|PROT_WRITE|PROT_EXEC,
	    MAP_FIXED|MAP_PRIVATE, 0);
  if(retval < 0) return 0xffffffff;
  return elf_entry;
}

/*
 * These are the functions used to load ELF style executables and shared
 * libraries.  There is no binary dependent code anywhere else.
 */

#define INTERPRETER_NONE 0
#define INTERPRETER_AOUT 1
#define INTERPRETER_ELF 2

static int load_elf_binary(struct linux_binprm * bprm, struct pt_regs * regs)
{
	struct elfhdr elf_ex;
	struct elfhdr interp_elf_ex;
	struct file * file;
  	struct exec interp_ex;
	struct inode *interpreter_inode;
	unsigned int load_addr;
	int load_addr_set = 0;
	unsigned int interpreter_type = INTERPRETER_NONE;
	unsigned char ibcs2_interpreter;
	int i;
	mm_segment_t old_fs;
	int error;
	struct elf_phdr * elf_ppnt, *elf_phdata;
	int elf_exec_fileno;
	unsigned int elf_bss, k, elf_brk;
	int retval;
	char * elf_interpreter;
	unsigned int elf_entry, interp_load_addr = 0;
	int status;
	unsigned int start_code, end_code, end_data;
	unsigned int elf_stack;
	char passed_fileno[6];
	
	MOD_INC_USE_COUNT;

	ibcs2_interpreter = 0;
	status = 0;
	load_addr = 0;
	elf_ex = *((struct elfhdr *) bprm->buf);	  /* exec-header */
	
	if (elf_ex.e_ident[0] != 0x7f ||
	    strncmp(&elf_ex.e_ident[1], "ELF",3) != 0) {
		MOD_DEC_USE_COUNT;
		return  -ENOEXEC;
	}
	
	
	/* First of all, some simple consistency checks */
	if(elf_ex.e_type != ET_EXEC || 
	   (elf_ex.e_machine != EM_386 && elf_ex.e_machine != EM_486) ||
	   (!bprm->dentry->d_inode->i_op || !bprm->dentry->d_inode->i_op->default_file_ops ||
	    !bprm->dentry->d_inode->i_op->default_file_ops->mmap)){
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}
	
	current->personality = PER_LINUX;

#ifdef ELF_PARSE_COMMENTS
	/* First scan any sections for .comments which may have
	 * personality clues.
	 */
	if (elf_ex.e_shoff) {
		Elf32_Shdr * elf_spnt, *elf_shdata;

		elf_shdata = (Elf32_Shdr *)kmalloc(elf_ex.e_shentsize *
						   elf_ex.e_shnum, GFP_KERNEL);
		retval = read_exec(bprm->dentry, elf_ex.e_shoff,
				(char *) elf_shdata,
		 		elf_ex.e_shentsize * elf_ex.e_shnum, 1);
		if (retval < 0) {
	        	kfree(elf_shdata);
			MOD_DEC_USE_COUNT;
			return retval;
		}

		elf_spnt = elf_shdata;
		for(i=0; retval && i < elf_ex.e_shnum; i++){
			if (elf_spnt->sh_type == SHT_PROGBITS
			&& elf_spnt->sh_flags == 0
			&& (status = parse_comments(bprm, elf_spnt)) > 0)
				break;
			elf_spnt++;
			retval -= elf_ex.e_shentsize;
		}

		kfree(elf_shdata);
	}
#endif /* ELF_PARSE_COMMENTS */

#ifdef CONFIG_BINFMT_ELF
	/* If we didn't recognise it as being other than Linux return
	 * ENOEXEC and let the standard Linux ELF loader handle it.
	 * Otherwise we will save a reference to this loader for each
	 * Linux ELF binary which either risks a kernel crash if we
	 * unload iBCS or prevents us unloading iBCS entirely.
	 */
	if (current->personality == PER_LINUX) {
#ifdef ELF_TRACE
		if ((ibcs_trace & TRACE_ELF_LD)) {
			printk(KERN_DEBUG "ELF: Linux or generic SVR4"
					" - hand off to kernel loader.\n");
		}
#endif
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}
#endif

	/* Now read in all of the header information */
	
	elf_phdata = (struct elf_phdr *) kmalloc(elf_ex.e_phentsize * 
						 elf_ex.e_phnum, GFP_KERNEL);
	
	retval = read_exec(bprm->dentry, elf_ex.e_phoff, (char *) elf_phdata,
			   elf_ex.e_phentsize * elf_ex.e_phnum, 1);
	if (retval < 0) {
	        kfree (elf_phdata);
		MOD_DEC_USE_COUNT;
		return retval;
	}
	
	elf_ppnt = elf_phdata;
	
	elf_bss = 0;
	elf_brk = 0;
	
	elf_exec_fileno = open_dentry(bprm->dentry, O_RDONLY);

	if (elf_exec_fileno < 0) {
	        kfree (elf_phdata);
		MOD_DEC_USE_COUNT;
		return elf_exec_fileno;
	}
	
	file = fget(elf_exec_fileno);
	
	elf_stack = 0xffffffff;
	elf_interpreter = NULL;
	start_code = 0xffffffff;
	end_code = 0;
	end_data = 0;
	
	for(i=0;i < elf_ex.e_phnum; i++){
		if(elf_ppnt->p_type == PT_INTERP) {
			/* This is the program interpreter used for shared libraries - 
			   for now assume that this is an a.out format binary */
			
			elf_interpreter = (char *) kmalloc(elf_ppnt->p_filesz, 
							   GFP_KERNEL);
			if (elf_interpreter == NULL) {
				fput(file);
				kfree (elf_phdata);
				kfree(elf_interpreter);
				SYS(close)(elf_exec_fileno);
				MOD_DEC_USE_COUNT;
				return -ENOMEM;
			}
			
			retval = read_exec(bprm->dentry,elf_ppnt->p_offset,elf_interpreter,
					   elf_ppnt->p_filesz, 1);

			/* If the program interpreter is one of these two,
			   then assume an iBCS2 image. Otherwise assume
			   a native linux image. */
			if (strcmp(elf_interpreter,"/usr/lib/libc.so.1") == 0 ||
			    strcmp(elf_interpreter,"/usr/lib/ld.so.1") == 0)
			  ibcs2_interpreter = 1;
#if 0
			printk("Using ELF interpreter %s\n", elf_interpreter);
#endif
			if(retval >= 0) {
				old_fs = get_fs();
				set_fs(get_ds());
				retval = namei(elf_interpreter, &interpreter_inode);
				set_fs(old_fs);
			}
			if(retval >= 0)
				retval = read_exec(interpreter_inode,0,bprm->buf,128, 1);
			
			if(retval >= 0) {
				interp_ex = *((struct exec *) bprm->buf);		/* exec-header */
				interp_elf_ex = *((struct elfhdr *) bprm->buf);	  /* exec-header */
				
			}
			if(retval < 0) {
				fput(file);
				kfree (elf_phdata);
				kfree(elf_interpreter);
				SYS(close)(elf_exec_fileno);
				MOD_DEC_USE_COUNT;
				return retval;
			}
		}
		elf_ppnt++;
	}
	
	/* Some simple consistency checks for the interpreter */
	if(elf_interpreter){
	        interpreter_type = INTERPRETER_ELF | INTERPRETER_AOUT;
		if(retval < 0) {
			fput(file);
			kfree(elf_interpreter);
			kfree(elf_phdata);
			SYS(close)(elf_exec_fileno);
			MOD_DEC_USE_COUNT;
			return -ELIBACC;
		}
		/* Now figure out which format our binary is */
		if((N_MAGIC(interp_ex) != OMAGIC) && 
		   (N_MAGIC(interp_ex) != ZMAGIC) &&
		   (N_MAGIC(interp_ex) != QMAGIC)) 
		  interpreter_type = INTERPRETER_ELF;

		if (interp_elf_ex.e_ident[0] != 0x7f ||
		    strncmp(&interp_elf_ex.e_ident[1], "ELF",3) != 0)
		  interpreter_type &= ~INTERPRETER_ELF;

		if(!interpreter_type)
		  {
		    fput(file);
		    kfree(elf_interpreter);
		    kfree(elf_phdata);
		    SYS(close)(elf_exec_fileno);
		    MOD_DEC_USE_COUNT;
		    return -ELIBBAD;
		  }
	}
	
	/* OK, we are done with that, now set up the arg stuff,
	   and then start this sucker up */
	
	if (!bprm->sh_bang) {
		char * passed_p;
		
		if(interpreter_type == INTERPRETER_AOUT) {
		  sprintf(passed_fileno, "%d", elf_exec_fileno);
		  passed_p = passed_fileno;
		
		  if(elf_interpreter) {
		    bprm->p = copy_strings(1,&passed_p,bprm->page,bprm->p,2);
		    bprm->argc++;
		  }
		}
		if (!bprm->p) {
			fput(file);
		        if(elf_interpreter) {
			      kfree(elf_interpreter);
			}
		        kfree (elf_phdata);
			SYS(close)(elf_exec_fileno);
			MOD_DEC_USE_COUNT;
			return -E2BIG;
		}
	}
	
	/* OK, This is the point of no return */
	flush_old_exec(bprm);

	current->mm->end_data = 0;
	current->mm->end_code = 0;
	current->mm->start_mmap = ELF_START_MMAP;
	current->mm->mmap = NULL;
	elf_entry = (unsigned int) elf_ex.e_entry;
	
	/* Do this so that we can load the interpreter, if need be.  We will
	   change some of these later */
	current->mm->rss = 0;
	bprm->p = setup_arg_pages(bprm->p, bprm);
	current->mm->start_stack = bprm->p;
	
	/* Now we do a little grungy work by mmaping the ELF image into
	   the correct location in memory.  At this point, we assume that
	   the image should be loaded at fixed address, not at a variable
	   address. */
	
	old_fs = get_fs();
	set_fs(get_ds());
	
	elf_ppnt = elf_phdata;
	for(i=0;i < elf_ex.e_phnum; i++){
		
		if(elf_ppnt->p_type == PT_INTERP) {
			/* Set these up so that we are able to load the interpreter */
		  /* Now load the interpreter into user address space */
		  set_fs(old_fs);

		  if(interpreter_type & 1) elf_entry = 
		    load_aout_interp(&interp_ex, interpreter_inode);

		  if(interpreter_type & 2) elf_entry = 
		    load_elf_interp(&interp_elf_ex, interpreter_inode, &interp_load_addr);

		  old_fs = get_fs();
		  set_fs(get_ds());

		  iput(interpreter_inode);
		  kfree(elf_interpreter);
			
		  if(elf_entry == 0xffffffff) { 
		    printk("Unable to load interpreter\n");
		    kfree(elf_phdata);
		    send_sig(SIGSEGV, current, 0);
		    MOD_DEC_USE_COUNT;
		    return 0;
		  }
		}
		
		
		if(elf_ppnt->p_type == PT_LOAD) {
#ifdef PF_R
			int elf_prot = (elf_ppnt->p_flags & PF_R) ? PROT_READ : 0;
			if (elf_ppnt->p_flags & PF_W) elf_prot |= PROT_WRITE;
			if (elf_ppnt->p_flags & PF_X) elf_prot |= PROT_EXEC;
			error = do_mmap(file,
					elf_ppnt->p_vaddr & 0xfffff000,
					elf_ppnt->p_filesz + (elf_ppnt->p_vaddr & 0xfff),
					elf_prot,
					MAP_FIXED | MAP_PRIVATE | MAP_DENYWRITE | MAP_EXECUTABLE,
					elf_ppnt->p_offset & 0xfffff000);
#else
			error = do_mmap(file,
					elf_ppnt->p_vaddr & 0xfffff000,
					elf_ppnt->p_filesz + (elf_ppnt->p_vaddr & 0xfff),
					PROT_READ | PROT_WRITE | PROT_EXEC,
					MAP_FIXED | MAP_PRIVATE | MAP_DENYWRITE | MAP_EXECUTABLE,
					elf_ppnt->p_offset & 0xfffff000);
#endif /* PF_R */

#ifdef LOW_ELF_STACK
			if((elf_ppnt->p_vaddr & 0xfffff000) < elf_stack) 
				elf_stack = elf_ppnt->p_vaddr & 0xfffff000;
#endif
			
			if (!load_addr_set) {
			  load_addr = elf_ppnt->p_vaddr - elf_ppnt->p_offset;
			  load_addr_set = 1;
			}
			k = elf_ppnt->p_vaddr;
			if(k < start_code) start_code = k;
			k = elf_ppnt->p_vaddr + elf_ppnt->p_filesz;
			if(k > elf_bss) elf_bss = k;
#ifdef PF_X
			if((elf_ppnt->p_flags & PF_X) && end_code <  k)
#else
			if((elf_ppnt->p_flags | PROT_WRITE) && end_code <  k)
#endif
				end_code = k; 
			if(end_data < k) end_data = k; 
			k = elf_ppnt->p_vaddr + elf_ppnt->p_memsz;
			if(k > elf_brk) elf_brk = k;		     
		      }
		elf_ppnt++;
	}
	set_fs(old_fs);
	
	kfree(elf_phdata);
	
	if(interpreter_type != INTERPRETER_AOUT) SYS(close)(elf_exec_fileno);
	if (current->personality == PER_LINUX && ibcs2_interpreter)
		current->personality = PER_SVR4;
	if (current->exec_domain && current->exec_domain->module)
		__MOD_DEC_USE_COUNT(current->exec_domain->module);
	if (current->binfmt && current->binfmt->module)
		__MOD_DEC_USE_COUNT(current->binfmt->module);
	current->exec_domain = lookup_exec_domain(current->personality);
	current->binfmt = &ibcs_elf_format;
	if (current->exec_domain && current->exec_domain->module)
		_MOD_INC_USE_COUNT(current->exec_domain->module);
	if (current->binfmt && current->binfmt->module)
		__MOD_INC_USE_COUNT(current->binfmt->module);
#ifdef LOW_ELF_STACK
	current->start_stack = bprm->p = elf_stack - 4;
#endif
	bprm->p = (unsigned long) 
	  create_elf_tables((char *)bprm->p,
			bprm->argc,
			bprm->envc,
			(interpreter_type == INTERPRETER_ELF ? &elf_ex : NULL),
			load_addr,    
			interp_load_addr,
			(interpreter_type == INTERPRETER_AOUT ? 0 : 1));
	if(interpreter_type == INTERPRETER_AOUT)
	  current->mm->arg_start += strlen(passed_fileno) + 1;
	current->mm->start_brk = current->mm->brk = elf_brk;
	current->mm->end_code = end_code;
	current->mm->start_code = start_code;
	current->mm->end_data = end_data;
	current->mm->start_stack = bprm->p;
	compute_creds(bprm);
	current->flags &= ~PF_FORKNOEXEC;

	/* Calling sys_brk effectively mmaps the pages that we need for the bss and break
	   sections */
	current->mm->brk = (elf_bss + 0xfff) & 0xfffff000;
	SYS(brk)((elf_brk + 0xfff) & 0xfffff000);

	padzero(elf_bss);

	if (current->personality != PER_LINUX) {
		/* Why this, you ask???  Well SVr4 maps page 0 as read-only,
		   and some applications "depend" upon this behavior.
		   Since we do not have the power to recompile these, we
		   emulate the SVr4 behavior.  Sigh.  */
		error = do_mmap(NULL, 0, 4096, PROT_READ | PROT_EXEC,
				MAP_FIXED | MAP_PRIVATE, 0);
	}

	/* SVR4/i386 ABI (pages 3-31, 3-32) says that when the program
	   starts %edx contains a pointer to a function which might be
	   registered using `atexit'.  This provides a mean for the
	   dynamic linker to call DT_FINI functions for shared libraries
	   that have been loaded before the code runs.

	   A value of 0 tells we have no such handler.  */
	regs->edx = 0;

	start_thread(regs, elf_entry, bprm->p);
	if (current->flags & PF_PTRACED)
		send_sig(SIGTRAP, current, 0);
	MOD_DEC_USE_COUNT;
	return 0;
}


#ifdef ELF_PARSE_COMMENTS
/*
 *  Parse a comments section looking for clues as to the system this
 *  was compiled on so we can get the system call interface right.
 */
static int
parse_comments(struct linux_binprm *exe_bprm, Elf32_Shdr * sect)
{
	unsigned long offset, nbytes;
	char *buffer;
	mm_segment_t old_fs;
	int status;

	/* Fetch the size of the section. There must be something in there
	 * or the section wouldn't exist at all. We only bother with the
	 * first 8192 characters though. There isn't any point getting too
	 * carried away!
	 */
	if ((nbytes = sect->sh_size) > 8192)
		nbytes = 8192;

	if (!(buffer = (char *)__get_free_page(GFP_KERNEL)))
		return 0;

	offset = sect->sh_offset;
	while (nbytes > 0) {
		char *p;
		unsigned long count, str_start;

		status = read_exec(exe_bprm->dentry, offset, buffer,
				nbytes > PAGE_SIZE ? PAGE_SIZE : nbytes, 1);
#if 0
		if (ibcs_trace & TRACE_ELF_LD)
			printk(KERN_DEBUG
				"ELF: read %d bytes, offset %d, got %d\n",
				nbytes, offset, status);
#endif

		if (status <= 0) {
			free_page((unsigned long)buffer);
			return 0;
		}

		p = buffer;
		str_start = 0;
		for (count=0; count<status; count++) {
			struct elf_clue *clue;
			char c = *(buffer + PAGE_SIZE - 1);
			*(buffer + PAGE_SIZE - 1) = '\0';
#if 0
			if (ibcs_trace & TRACE_ELF_LD)
				printk(KERN_DEBUG "ELF: testing %s\n", p);
#endif
			for (clue=elf_clues; clue->len; clue++) {
				if ((clue->len < 0 && strstr(p, clue->text))
				|| (clue->len > 0 && !strncmp(p, clue->text, clue->len))) {
#ifdef ELF_TRACE
					if (ibcs_trace & TRACE_ELF_LD) {
						printk(KERN_DEBUG
							"ELF: testing %s\n",
							p);
						printk(KERN_DEBUG
							"ELF:    with %s\n",
							clue->text);
						printk(KERN_DEBUG
							"ELF:  giving 0x%08lx\n",
							clue->personality);
					}
#endif
					current->personality = clue->personality;
					free_page((unsigned long)buffer);
					return (1);
				}
			}
			*(buffer + PAGE_SIZE - 1) = c;

			while (*p && count < status) p++,count++;
			if (count < status) {
				p++;
				count++;
				str_start = count;
			}
		}

		/* If we didn't find an end ofstring at all this page
		 * probably isn't useful string data.
		 */
		if (str_start == 0)
			str_start = status;

		nbytes -= str_start;
		offset += str_start;
	}

	free_page((unsigned long)buffer);
	return (0);
}
#endif /* ELF_PARSE_COMMENTS */


/* This is really simpleminded and specialized - we are loading an
   a.out library that is given an ELF header. */

static int load_elf_library(int fd){
        struct file * file;
	struct elfhdr elf_ex;
	struct elf_phdr *elf_phdata  =  NULL;
	struct  inode * inode;
	unsigned int len;
	int elf_bss;
	int retval;
	unsigned int bss;
	int error;
	int i,j, k;
	loff_t offset = 0;

	MOD_INC_USE_COUNT;

	len = 0;
	file = fget(fd);
	inode = file->f_dentry->d_inode;
	elf_bss = 0;
	
	set_fs(KERNEL_DS);
	error = file->f_op->read(file, (char *) &elf_ex, sizeof(elf_ex), &offset);
	set_fs(USER_DS);
	if (error != sizeof(elf_ex)) {
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}

	if (elf_ex.e_ident[0] != 0x7f ||
	    strncmp(&elf_ex.e_ident[1], "ELF",3) != 0) {
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}
	
	/* First of all, some simple consistency checks */
	if(elf_ex.e_type != ET_EXEC || elf_ex.e_phnum > 2 ||
	   (elf_ex.e_machine != EM_386 && elf_ex.e_machine != EM_486) ||
	   (!inode->i_op || !inode->i_op->default_file_ops->mmap)){
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}
	
	/* Now read in all of the header information */
	
	if(sizeof(struct elf_phdr) * elf_ex.e_phnum > PAGE_SIZE) {
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}
	
	elf_phdata =  (struct elf_phdr *) 
		kmalloc(sizeof(struct elf_phdr) * elf_ex.e_phnum, GFP_KERNEL);
	
	retval = read_exec(inode, elf_ex.e_phoff, (char *) elf_phdata,
			   sizeof(struct elf_phdr) * elf_ex.e_phnum, 1);
	
	j = 0;
	for(i=0; i<elf_ex.e_phnum; i++)
		if((elf_phdata + i)->p_type == PT_LOAD) j++;
	
	if(j != 1)  {
		kfree(elf_phdata);
		MOD_DEC_USE_COUNT;
		return -ENOEXEC;
	}
	
	while(elf_phdata->p_type != PT_LOAD) elf_phdata++;
	
	/* Now use mmap to map the library into memory. */
	error = do_mmap(file,
			elf_phdata->p_vaddr & 0xfffff000,
			elf_phdata->p_filesz + (elf_phdata->p_vaddr & 0xfff),
			PROT_READ | PROT_WRITE | PROT_EXEC,
			MAP_FIXED | MAP_PRIVATE | MAP_DENYWRITE | MAP_EXECUTABLE,
			elf_phdata->p_offset & 0xfffff000);

	k = elf_phdata->p_vaddr + elf_phdata->p_filesz;
	if(k > elf_bss) elf_bss = k;
	
	SYS(close)(fd);
	if (error != (elf_phdata->p_vaddr & 0xfffff000)) {
	        kfree(elf_phdata);
		MOD_DEC_USE_COUNT;
		return error;
	}

	padzero(elf_bss);

	len = (elf_phdata->p_filesz + elf_phdata->p_vaddr+ 0xfff) & 0xfffff000;
	bss = elf_phdata->p_memsz + elf_phdata->p_vaddr;
	if (bss > len)
	  do_mmap(NULL, len, bss-len,
		  PROT_READ|PROT_WRITE|PROT_EXEC,
		  MAP_FIXED|MAP_PRIVATE, 0);
	kfree(elf_phdata);
	MOD_DEC_USE_COUNT;
	return 0;
}
