/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipicker.h"


#include "iactor.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "iprobefilter.h"
#include "iviewmodule.h"
#include "iviewsubjectobserver.h"
#include "ivtk.h"

#include <vtkCellArray.h>
#include <vtkCellPicker.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPointLocator.h>
#include <vtkPointPicker.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkProp.h>
#include <vtkProperty.h>
#include <vtkPropPicker.h>
#include <vtkSphereSource.h>
#include <vtkStructuredPoints.h>

//
//  templates
//
#include "iarraytemplate.h"


//
//  Picker must be an object so that it can be created by the ObjectFactory
//
IOBJECT_DEFINE_TYPE(iPicker,Picker,pi,iObjectType::_Helper);  // helper type

IOBJECT_DEFINE_KEY(iPicker,Accuracy,a,Float,1);
IOBJECT_DEFINE_KEY(iPicker,PickMethod,m,Int,1);
IOBJECT_DEFINE_KEY(iPicker,PointSize,ps,Float,1);


namespace iPicker_Private
{
	class CellPicker : public vtkCellPicker
	{

	public:

		static CellPicker* New()
		{ 
			return new CellPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class PointPicker : public vtkPointPicker
	{

	public:

		static PointPicker* New()
		{ 
			return new PointPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class ObjectPicker : public vtkPropPicker
	{

	public:

		static ObjectPicker* New()
		{ 
			return new ObjectPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};
};


using namespace iParameter;
using namespace iPicker_Private;

	
iPicker* iPicker::New(iViewModule *vm)
{
	iPicker *tmp = new iPicker(vm); IERROR_ASSERT(tmp);
	iExtensionFactory::InstallExtensions(tmp);
	return tmp;
}


iPicker::iPicker(iViewModule *vm) : iDataFormatter(vm), mPos(vm), mNumPickMethods(3)
{
	int i;

	mObjName = "";
	mObjType = _ObjectTypeUndefined;
	mDataTypePointer = 0;
	mPickMethod = _PickMethodObject;

	//
	//  Pickers
	//
	mDevices[0] = CellPicker::New(); IERROR_ASSERT(mDevices[0]);
	mDevices[1] = PointPicker::New(); IERROR_ASSERT(mDevices[1]);
	mDevices[2] = ObjectPicker::New(); IERROR_ASSERT(mDevices[2]);

	this->SetAccuracy(0.02f);

	mObserver = iPickEventObserver::New(this->GetViewModule()); IERROR_ASSERT(mObserver);
	this->AddObserver(vtkCommand::StartPickEvent,mObserver);
	this->AddObserver(vtkCommand::EndPickEvent,mObserver);
	for(i=0; i<mNumPickMethods; i++)
	{
		mDevices[i]->AddObserver(vtkCommand::AbortCheckEvent,this->GetViewModule()->GetAbortRenderEventObserver());
	}

	//
	//  Graphical representation
	//
	mPointActor = iActor::New(); IERROR_ASSERT(mPointActor);
	mPointSource = vtkSphereSource::New(); IERROR_ASSERT(mPointSource);

	mPointSize = 0.1;
	this->SetPointSize(mPointSize);
	mPointSource->SetRadius(0.5);
	mPointActor->GetProperty()->SetColor(0.9,0.9,0.9);
	mPointActor->VisibilityOff();
	mPointActor->PickableOff();

	this->GetViewModule()->AddObject(mPointActor);

	//
	//  PolyData inout for a probe filter
	//
	mProbeInput = vtkPolyData::New(); IERROR_ASSERT(mProbeInput);
	vtkCellArray *v = vtkCellArray::New(); IERROR_ASSERT(v);
	v->InsertNextCell(1);
	v->InsertCellPoint(0);
	mProbeInput->SetVerts(v);
	v->Delete();
	vtkPoints *p = vtkPoints::New(VTK_DOUBLE); IERROR_ASSERT(p);
	p->SetNumberOfPoints(1);
	p->SetPoint(0,0.0,0.0,0.0);
	mProbeInput->SetPoints(p);
	p->Delete();
}


iPicker::~iPicker()
{
	int i;
	mObserver->Delete();
	for(i=0; i<mNumPickMethods; i++) mDevices[i]->Delete();
	this->GetViewModule()->RemoveObject(mPointActor);
	mPointActor->Delete();
	mPointSource->Delete();
	mProbeInput->Delete();
}


int iPicker::Pick(double x, double y, double z, vtkRenderer *ren)
{
	this->InvokeEvent(vtkCommand::StartPickEvent,0);
	int ret = mDevices[mPickMethod]->Pick(x,y,z,ren);
	this->InvokeEvent(vtkCommand::EndPickEvent,0);
	return ret;
}


void iPicker::Modified()
{
	int i;
	for(i=0; i<mNumPickMethods; i++) mDevices[i]->Modified();
	vtkAbstractPropPicker::Modified();
	this->ClearCache();
}


void iPicker::UpdateReport()
{
	mObjName = "Unknown";
	mPos = mDevices[mPickMethod]->GetPickPosition();

	this->SetPointSize(mPointSize);
	mPointActor->SetPosition(mPos);

	this->ClearReport();
	this->UpdateReportBody(); // this can replace mPointObject via its extensions

	mPointActor->SetInput(mPointSource->GetOutput());
	mPointActor->SetScaled(true);
	mPointActor->SetPosition(mPos);

	//
	//  Check if extensions want to modify the point actor
	//
	int i;
	for(i=0; i<mExtensions.Size(); i++) if(iPickerExtension::RequiredCast(mExtensions[i])->IsUsingData(mDataTypePointer))
	{
		iPickerExtension::RequiredCast(mExtensions[i])->ModifyActor(mPointActor); // the last installed extension prevails
	}

	this->ShowPickedPoint(true);
}


void iPicker::UpdateReportBody()
{
	int j;
	iString s1, s2;

	//
	//  Get the object info
	//
#ifdef IVTK_4
	vtkProp *obj = mDevices[mPickMethod]->GetProp();
#else
	vtkProp *obj = mDevices[mPickMethod]->GetViewProp();
#endif
	if(obj == 0) 
	{
		mObjName = "";
		mPointActor->VisibilityOff(); 
		return;
	}
	iViewSubjectObserver *obs = iPointerCast<iViewSubjectObserver,vtkCommand>(obj->GetCommand(1));

	if(obs != 0)
	{
		mObjType = obs->GetObjectType();
		mObjName = obs->GetObjectName();
		mDataTypePointer = &obs->GetDataType();
	}
	else
	{
		mObjType = _ObjectTypeUndefined;
		mObjName = "Unknown";
		mDataTypePointer = 0;
	}

	//
	//  Get the data info
	//
	if(mDataTypePointer == 0) return;

	iDataLimits *lim = this->GetViewModule()->GetReader()->GetLimits(*mDataTypePointer);

	if(mDataTypePointer->IsOfClass(iData::ParticleSet()))
	{
		//
		//  Do particles separately using a PointLocator - a faster approach
		//
		vtkDataSet *ds = this->GetViewModule()->GetReader()->GetOutput(*mDataTypePointer);
		if(ds->IsA("vtkPolyData"))
		{
			vtkPolyData *data = vtkPolyData::SafeDownCast(ds);
			vtkPointLocator *locator = vtkPointLocator::New(); IERROR_ASSERT(locator);
			locator->SetDataSet(data);
			vtkIdType pid = locator->FindClosestPoint(mDevices[mPickMethod]->GetPickPosition());
			vtkFloatArray *s = vtkFloatArray::SafeDownCast(data->GetPointData()->GetScalars());
			if(s != 0)
			{
				float *p = s->GetPointer(pid);
				int natt = s->GetNumberOfComponents();
				if(p!=0 && natt>0 && pid>=0 && pid<data->GetNumberOfPoints())
				{
					this->FormatScalarData(lim,natt,p);
				}
			}
			locator->Delete();
		}
		else
		{
			IERROR_REPORT("ParticleSet data is not vtkPolyData - cannot pick.");
		}
	}
	else
	{
		//
		//  For a field, use the ProbeFilter will to find attribute values
		//
		iDataSubject *subject = this->GetViewModule()->GetReader()->GetSubject(*mDataTypePointer);

		mProbeInput->GetPoints()->SetPoint(0,mPos);
		iProbeFilter *probe = subject->CreateProbeFilter(obs->GetSubject());
		probe->SetInput(mProbeInput);
		probe->SetSource(subject->GetData());
		probe->Update();
		vtkPolyData *out = probe->GetPolyDataOutput();
		
		if(out==0 || out->GetPointData()==0)
		{
			IERROR_REPORT("Unable to probe data for the picked object");
			return;
		}

		vtkDataArray *d = 0;
		int i, nvar = 0;
		switch(mDataTypePointer->GetRank())
		{
		case 0U:
			{
				d = out->GetPointData()->GetScalars();
				nvar = lim->GetNumVars();
				break;
			}
		case 1U:
			{
				d = out->GetPointData()->GetVectors();
				nvar = 3;
				break;
			}
		case 2U:
			{
				d = out->GetPointData()->GetTensors();
				nvar = 9;
				break;
			}
		}

		if(d==0 || nvar<1 || d->GetNumberOfComponents()!=nvar || !d->IsA("vtkFloatArray"))
		{
			IERROR_REPORT("Unable to probe data for the picked object");
			return;
		}

		float *v = (float *)d->GetVoidPointer(0);

		//
		//  Make the correction for the surface
		//
		if(mObjType == _ObjectTypeSurface)
		{
			j = obs->GetIsoSurfaceData().Var;
			if(j>=0 && j<nvar) v[j] = obs->GetIsoSurfaceData().Level;
		}

		//
		//  Ask extensions to provide extra information
		//
		for(i=0; i<mExtensions.Size(); i++) if(iPickerExtension::RequiredCast(mExtensions[i])->IsUsingData(mDataTypePointer))
		{
			iPickerExtension::RequiredCast(mExtensions[i])->AddInfo(probe);
		}

		switch(mDataTypePointer->GetRank())
		{
		case 0U:
			{
				this->FormatScalarData(lim,nvar,v);
				break;
			}
		case 1U:
			{
				this->FormatVectorData(lim,-1,v);
				break;
			}
		case 2U:
			{
				this->FormatTensorData(lim,-1,v);
				break;
			}
		}

		probe->Delete();
	}
}


void iPicker::SetPointSize(float s)
{
	if(s > 0.0)
	{
		mPointSize = s;
		mPointActor->SetScaled(true);
		mPointActor->SetBasicScale(mPointSize);
		this->Modified();
	}
}


void iPicker::ShowPickedPoint(bool s)
{
	if(!mObjName.IsEmpty() && s)
	{
		mPointActor->VisibilityOn();
	}
	if(!s)
	{
		mPointActor->VisibilityOff();
	}
	this->GetViewModule()->Render();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iPicker::ExtendableObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyPickMethod(),mPickMethod);
	this->PackValue(s,KeyPointSize(),mPointSize);
	this->PackValue(s,KeyAccuracy(),mAccuracy);
}


void iPicker::ExtendableObjectUnPackStateBody(const iString &s)
{
	int i; float f;

	if(this->UnPackValue(s,KeyPickMethod(),i)) this->SetPickMethod(i);
	if(this->UnPackValue(s,KeyPointSize(),f)) this->SetPointSize(f);
	if(this->UnPackValue(s,KeyAccuracy(),f)) this->SetAccuracy(f);
}


void iPicker::SetPickMethod(int s)
{
	if(s>=0 && s<mNumPickMethods)
	{
		mPickMethod = s;
		this->ClearCache();
	}
}


void iPicker::SetAccuracy(float s)
{
	if(s > 0.0f)
	{
		mAccuracy = s;
		
		int j;
		vtkPicker *p;
		for(j=0; j<mNumPickMethods; j++)
		{
			p = dynamic_cast<vtkPicker*>(mDevices[j]);
			if(p != 0) p->SetTolerance(s);
		}
		this->ClearCache();
	}
}


//
//  Extension class
//
IOBJECTEXTENSION_DEFINE_ABSTRACT(iPicker);

void iPickerExtension::Define()
{
}

