/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "itensorfieldpipeline.h"


#include "idatalimits.h"
#include "ierror.h"
#include "ireplicatedpolydata.h"
#include "iresampleimagedatafilter.h"
#include "isolidobjectsource.h"
#include "itensorglyphfilter.h"
#include "itensorfieldviewsubject.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"
#include "iviewsubjectpipelinetemplate.h"


//
// iTensorFieldPipeline class
//
iTensorFieldPipeline::iTensorFieldPipeline(iTensorFieldViewSubject *s) : iViewSubjectPipeline(s,1)
{
	mParent = s;

	//
	//  Do VTK stuff
	//	
	mGlyph = this->CreateFilter<iTensorGlyphFilter>();
	mDataReplicated = this->CreateFilter<iReplicatedPolyData>();
	mFilter = this->CreateFilter<iResampleImageDataFilter>();
	mGlyphSphere = this->CreateFilter<iSphereSource>(); 

	//  mGlyph->SetInput(mFilter->GetOutput());  Filter only works with vtkImageData, so we need to connect them at run-time
	mGlyph->SetSource(mGlyphSphere->GetOutput());
	mGlyph->ScalingOn();

	mDataReplicated->SetInput(mGlyph->GetOutput());

	this->UpdateContents(_All);
}


iTensorFieldPipeline::~iTensorFieldPipeline()
{
}


void iTensorFieldPipeline::ProduceOutput()
{
	vtkDataSet *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	if(mFilter!=0 && input->IsA("vtkImageData"))
	{
		vtkImageData *inp = vtkImageData::SafeDownCast(input);
		if(mFilter->GetInput() != inp)
		{
			mFilter->SetInput(inp);
			mGlyph->SetInput(mFilter->GetOutput());
		}
	}
	else
	{
		if(mGlyph->GetInput() != input)
		{
			mGlyph->SetInput(input);
		}
	}

	//
	//  Needed so that the glyph size is correct
	//
	this->UpdateSize();
	//
	//  Update the pipeline
	//
	mDataReplicated->Update();
	output->ShallowCopy(mDataReplicated->GetOutput());
}


float iTensorFieldPipeline::GetContentsMemorySize() const
{
	float s = 0.0f;
	s += mGlyph->GetMemorySize();
	s += mDataReplicated->GetMemorySize();
	s += mFilter->GetOutput()->GetActualMemorySize();
	s += mGlyphSphere->GetOutput()->GetActualMemorySize();
	return s;
}


void iTensorFieldPipeline::UpdateContents(int n, int)
{
	if(n==_All || n==_Replicas) this->UpdateReplicas();
	if(n==_All || n==_SampleRate) this->UpdateSampleRate();
	if(n==_All || n==_ScalingOn) this->UpdateScalingOn();
	if(n==_All || n==_Size) this->UpdateSize();
}


void iTensorFieldPipeline::UpdateSize()
{
	//
	//  Do not update unless necessary
	//
	if(mParent->CanBeShown())
	{
		double ns = 10.0*mParent->GetGlyphSize()*2.0/this->GetLimits()->GetUpperLimit(mParent->GetDataOffset()); // 2.0 is the box size
		mGlyph->SetScaleFactor(ns);
		this->Modified();
	}
}


void iTensorFieldPipeline::UpdateScalingOn()
{
	if(mParent->GetScalingOn()) mGlyph->ScalingOn(); else mGlyph->ScalingOff();
	this->Modified();
}


void iTensorFieldPipeline::UpdateSampleRate()
{ 
	mFilter->SetSampleRate(mParent->GetGlyphSampleRate(),mParent->GetGlyphSampleRate(),mParent->GetGlyphSampleRate());
	this->Modified();
}


void iTensorFieldPipeline::UpdateReplicas()
{
	mDataReplicated->Modified();
	this->Modified();
}


void iTensorFieldPipeline::SyncWithData(const iDataSyncRequest &)
{
	this->UpdateSize();
}

