/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iviewobject.h"


#include "idata.h"
#include "idatareader.h"
#include "ierror.h"
#include "iextensionfactory.h"
#include "iviewmodule.h"
#include "iviewsubject.h"

//
//  templates
//
#include "iarraytemplate.h"


iViewObject* iViewObject::New(iViewObjectParent *parent)
{
	return iExtensionFactory::CreateViewObject(parent);
}


iViewObject::iViewObject(iViewObjectParent *parent, bool synced, bool single) : iMultiTypeDataConsumer(parent->GetViewModule(),iData::UniformScalars()), mSubjects(0)
{
	mParent = parent; IERROR_ASSERT(parent);
	mIsSynced = synced;
	mIsSingle = single;

	mPrimaryDataInfo->Erase(); // clear in expectation of ExtensionFactory completing the construction
}


void iViewObject::AddSubject(iViewSubject *subject)
{
	if(mParent->GetDataType().IsNull() || mSubjects.Size()==0)
	{
		IERROR_ASSERT(subject);
		//
		//  We configure the ViewSubject here. Thus, ViewSubject(s) will not work if they are created outside this class -  with
		//  the sole exception of iMarker that does not use a pipeline and so needs no configuring.
		//
		subject->Configure();
		mSubjects.Add(subject);
		*mPrimaryDataInfo += subject->GetDataType();
		if(!mParent->GetDataType().IsNull() && mParent->GetDataType()!=mPrimaryDataInfo->Type(0))
		{
			IERROR_FINISH("Incorrectly configured iViewObject.");
		}
	}
	else
	{
		if(subject != 0) subject->Delete();
	}
}


void iViewObject::Verify()
{
	if(mSubjects.Size() == 0)
	{
		IERROR_FINISH("Incorrectly configured iViewObject.");
	}
}


iViewObject::~iViewObject()
{
	while(mSubjects.Size() > 0) mSubjects.RemoveLast()->Delete();
}


iViewSubject* iViewObject::GetViewSubject(int obs) const
{
#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif

	if(obs == -1) obs = mActiveDataTypeIndex;
	if(obs>=0 && obs<mSubjects.Size()) return mSubjects[obs]; else
	{
		IERROR_FINISH("Requesting non-existing ViewSubject.");
		return 0;
	}
}


iViewSubject* iViewObject::GetViewSubject(const iDataType &type) const
{
	int i;
	for(i=0; i<mSubjects.Size(); i++) if(mSubjects[i]->IsUsingData(type,true)) return mSubjects[i];
	return 0;
}


void iViewObject::Show(bool s)
{
	int i;

#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif

	if(mIsSingle) for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->Show(false);
	mSubjects[mActiveDataTypeIndex]->Show(s);
}


void iViewObject::Reset()
{
	int i;
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->Reset();
}


void iViewObject::ShowClipPlane(bool s)
{
	int i;
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->ShowClipPlane(s);
}


float iViewObject::GetMemorySize() const
{
	int i;
	float s = 0.0f;

	for(i=0; i<mSubjects.Size(); i++) s += mSubjects[i]->GetMemorySize();

	return s;
}

	
void iViewObject::PackStateBody(iString &s) const
{
#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif
	this->PackValue(s,iViewSubject::KeyDataType(),mActiveDataTypeIndex);

	mSubjects[mActiveDataTypeIndex]->PackState(s);
}


void iViewObject::UnPackStateBody(const iString &s)
{
	int i;

#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif
	if(this->UnPackValue(s,iViewSubject::KeyDataType(),i)) this->SetActiveDataTypeIndex(i);

	if(mIsSynced)
	{
		for(i=0; i<mSubjects.Size(); i++)
		{
			mSubjects[i]->UnPackState(s);
			if(mSubjects[i]->mUnPackedSomething) mUnPackedSomething = true;
		}
	}
	else
	{
		mSubjects[mActiveDataTypeIndex]->UnPackState(s);
		if(mSubjects[mActiveDataTypeIndex]->mUnPackedSomething) mUnPackedSomething = true;
	}
}


void iViewObject::PackCompleteState(iString &s) const
{
	int i;
	iString s1;

#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif
	s.Init(10000);
	this->PackValue(s,iViewSubject::KeyDataType(),mActiveDataTypeIndex);

	//
	//  Pack all objects for state saving
	//
	for(i=0; i<mSubjects.Size(); i++)
	{
		mSubjects[i]->PackCompleteState(s1);
		s += iObjectKey::SpecialSeparator() + mSubjects[i]->GetDataType().GetName() + iObjectKey::Delimiter() + s1;
	}
	mCache = s;
	mCacheInvalid = false;
}


void iViewObject::UnPackCompleteState(const iString &s)
{
	int i, j, n;
	iString s1, name;

#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif
	if(this->UnPackValue(s,iViewSubject::KeyDataType(),i)) this->SetActiveDataTypeIndex(i);

	//
	//  UnPack all objects for state restoring
	//
	n = s.Contains(iObjectKey::SpecialSeparator());
	for(j=0; j<n; j++)
	{
		s1 = s.Section(iObjectKey::SpecialSeparator(),j+1,j+1);
		name = s1.Section(iObjectKey::Delimiter(),0,0);
		for(i=0; i<mSubjects.Size(); i++) if(mSubjects[i]->GetDataType().IsOfType(name))
		{
			mSubjects[i]->UnPackCompleteState(s1);
		}
	}
}


void iViewObject::UpdateDataType(int prev)
{
#ifdef I_CHECK1
	if(mActiveDataTypeIndex<0 || mActiveDataTypeIndex>mSubjects.MaxIndex()) IERROR_FINISH("iViewObject is misconfigured.");
#endif

	if(this->mIsSingle)
	{
		bool vis = false;
		if(prev>=0 && prev<mSubjects.Size())
		{
			vis = mSubjects[prev]->IsVisible();
			mSubjects[prev]->Show(false);
		}
		else IERROR_REPORT("Invalid previous data type index. Hiding the object.");
		mSubjects[mActiveDataTypeIndex]->Show(vis);
	}
	this->ClearCache();
}


void iViewObject::SyncWithDataBody(const iDataSyncRequest &r)
{
	int i;
	//
	//  If we are not visible, automatically switch to the first data type that has data
	//
	if(!this->GetViewSubject()->IsVisible() && !this->GetViewSubject()->IsDataPresent())
	{
		int i;
		for(i=0; i<mPrimaryDataInfo->Count(); i++)
		{
			if(this->GetViewModule()->GetReader()->IsThereData(mPrimaryDataInfo->Type(i)))
			{
				this->SetActiveDataType(mPrimaryDataInfo->Type(i));
				break;
			}
		}
	}
	//
	//  Sync all subjects
	//
	for(i=0; i<mSubjects.Size(); i++)
	{
		mSubjects[i]->SyncWithData(r);
	}
}

