/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iviewsubject.h"


#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "iviewmodule.h"
#include "iviewsubjectobserver.h"
#include "iviewsubjectparallelpipeline.h"
#include "iviewsubjectreplicator.h"

#include "iarraytemplate.h"


#include "iparameter.h"
using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iViewSubject,ViewSubject,-vo);  // helper type

IOBJECT_DEFINE_KEY(iViewSubject,DataType,dt,Int,1);
IOBJECT_DEFINE_KEY(iViewSubject,ClipPlane,cp,Bool,1);
IOBJECT_DEFINE_KEY(iViewSubject,DataPresent,dp,Bool,1);
IOBJECT_DEFINE_KEY(iViewSubject,Ambient,ma,Float,1);
IOBJECT_DEFINE_KEY(iViewSubject,Diffuse,md,Float,1);
IOBJECT_DEFINE_KEY(iViewSubject,SpecularPower,mp,Float,1);
IOBJECT_DEFINE_KEY(iViewSubject,Specular,ms,Float,1);
IOBJECT_DEFINE_KEY(iViewSubject,Shading,mw,Bool,1);
IOBJECT_DEFINE_KEY(iViewSubject,NumReplicas,nr,Int,6);
IOBJECT_DEFINE_KEY(iViewSubject,Ready,r,Bool,1);
IOBJECT_DEFINE_KEY(iViewSubject,Visible,vis,Bool,1);


iViewSubject::iViewSubject(iViewModule *vm, const iDataType &type, const iString &name, unsigned int flags) : iDataConsumer(vm,type), iReplicated(0,false), mObjectName(name), mPosition(vm), mAlwaysShaded((flags&_ViewSubjectAlwaysShaded)!=0U), mNoReplicating((flags&_ViewSubjectNoReplicating)!=0U), mPipelines(0)
{
	mObjectType = _ObjectTypeUndefined;

	mIsInitialized = mIsConfigured = false;

	mClipPlaneOn = mCreatingMainPipeline = false;

	mShading = mAlwaysShaded;
	mAmbient = 0.3f;
	mDiffuse = 0.4f;
	mSpecular = 0.2f;
	mSpecularPower = 50.0;

	mAttachedMarker = 0;
	mAttachedMarkerIndex = -1;

	mObjectObserver = iViewSubjectObserver::New(this); IERROR_ASSERT(mObjectObserver);
	if(mNoReplicating)
	{
		mReplicator = 0;
	}
	else
	{
		mReplicator = iViewSubjectReplicator::New(this); IERROR_ASSERT(mReplicator);
	}

	mDefaultColor = iColor(0,150,200);
}


iViewSubject::~iViewSubject()
{
	if(mAttachedMarker != 0) mAttachedMarker->DetachViewSubject(this);
	mObjectObserver->Delete();
	while(mPipelines.Size() > 0) mPipelines.RemoveLast()->Delete();
	if(mReplicator != 0) mReplicator->Delete();
}


void iViewSubject::Delete()
{
	this->ShowColorBars(false);
	iDeletable::Delete();
}

	
void iViewSubject::Configure()
{
	if(!mIsConfigured)
	{
		this->ConfigureBody();
		mIsConfigured = true;
		//
		//  We need to reset the object, as this attaches inputs to pipelines
		//
		this->Reset();
	}
	else
	{
		IERROR_FINISH("ViewSubject has been already configured.");
	}
}


iViewSubjectPipeline* iViewSubject::CreateMainPipeline(int numInputs, int id)
{
	iViewSubjectPipeline *p = iViewSubjectParallelPipeline::New(this,numInputs,id);
	this->ConfigureMainPipeline(p,id);
	return p;
}


void iViewSubject::AddMainPipeline(int numInputs)
{
	if(numInputs > 0)
	{
		mCreatingMainPipeline = true;
		iViewSubjectPipeline *p = this->CreateMainPipeline(numInputs,mPipelines.Size()); IERROR_ASSERT(p);
		mCreatingMainPipeline = false;
		mPipelines.Add(p);
	}
}


void iViewSubject::ConfigureMainPipeline(iViewSubjectPipeline *, int)
{
}


void iViewSubject::RemoveInternalData()
{
	int i;
	//
	//  remove everything, including the pipeline outputs
	//
	for(i=0; i<mPipelines.Size(); i++)
	{
		mPipelines[i]->RemoveInternalData(false);
	}
	this->RemoveInternalDataForExtraComponents();
}


void iViewSubject::RemoveInternalDataForExtraComponents()
{
	//
	//  By default we have no extra components
	//
}


iDataLimits* iViewSubject::GetLimits() const
{
	return this->GetViewModule()->GetReader()->GetLimits(this->GetDataType());
}


void iViewSubject::SetShading(bool s)
{
	if(!mAlwaysShaded)
	{
		mShading = s;
		this->UpdateMaterialProperties();
		this->ClearCache();
	}
}


void iViewSubject::SetAmbient(float v)
{
	mAmbient = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}

void iViewSubject::SetDiffuse(float v)
{
	mDiffuse = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}


void iViewSubject::SetSpecular(float v)
{
	mSpecular = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}


void iViewSubject::SetSpecularPower(float v)
{
	mSpecularPower = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}


void iViewSubject::UpdateAfterMarkerDelete(iMarker *m)
{
	if(m==0 || mAttachedMarker==0) return;

	if(mAttachedMarker == m)
	{
		mAttachedMarker = 0;
		mAttachedMarkerIndex = -1;
	}
	else 
	{
		int i;
		for(i=0; i<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex(); i++)
		{
			if(mAttachedMarker == this->GetViewModule()->GetMarkerFamily()->GetMember(i))
			{
				mAttachedMarkerIndex = i;
				return;
			}
		}
#ifdef I_CHECK1
		IERROR_REPORT_BUG;
#endif
		mAttachedMarker = 0;
		mAttachedMarkerIndex = -1;
	}
}


void iViewSubject::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyClipPlane(),mClipPlaneOn);
	if(!mAlwaysShaded)
	{
		this->PackValue(s,KeyShading(),mShading);
	}
	this->PackValue(s,KeyAmbient(),mAmbient);
	this->PackValue(s,KeyDiffuse(),mDiffuse);
	this->PackValue(s,KeySpecular(),mSpecular);
	this->PackValue(s,KeySpecularPower(),mSpecularPower);
	if(!mNoReplicating)
	{
		this->PackValue(s,KeyNumReplicas(),mNumReplicas,6);
	}
	this->PackValue(s,KeyVisible(),this->IsVisible());
	this->PackValue(s,KeyDataPresent(),this->IsDataPresent());
	this->PackValue(s,KeyReady(),this->IsReady());

	this->ViewSubjectPackStateBody(s);
}


void iViewSubject::UnPackStateBody(const iString &s)
{
	int i, d[6]; bool b; float f;

	if(this->UnPackValue(s,KeyClipPlane(),b)) this->ShowClipPlane(b);
	if(!mAlwaysShaded)
	{
		if(this->UnPackValue(s,KeyShading(),b)) this->SetShading(b);
	}
	if(this->UnPackValue(s,KeyAmbient(),f)) this->SetAmbient(f);
	if(this->UnPackValue(s,KeyDiffuse(),f)) this->SetDiffuse(f);
	if(this->UnPackValue(s,KeySpecular(),f)) this->SetSpecular(f);
	if(this->UnPackValue(s,KeySpecularPower(),f)) this->SetSpecularPower(f);

	if(!mNoReplicating)
	{
		for(i=0; i<6; i++) d[i] = mNumReplicas[i];
		if(this->UnPackValue(s,KeyNumReplicas(),d,6)) this->SetNumReplicas(d);
	}

	this->ViewSubjectUnPackStateBody(s);
}


void iViewSubject::SyncWithDataBody(const iDataSyncRequest &r)
{
	if(this->IsUsingData(r.Data(),true))
	{
		if(this->IsVisible())
		{
			//
			//  If we are visible, but data disappeared, hide us.
			//
			if(!this->IsReady()) this->ShowBody(false);
		}
	}
	
	if(this->IsUsingData(r.Data(),false))
	{
		this->ClearCache();
		this->ViewSubjectSyncWithData(r);
	}
}


void iViewSubject::Show(bool s)
{
	if(!mIsConfigured)
	{
		IERROR_FINISH("ViewSubject has not been configured.");
		return;
	}
	if(s && !mIsInitialized)
	{
		this->FinishInitialization();
		mIsInitialized = true;
	}
	if(mIsInitialized)
	{
		if(s && this->CanBeShown())
		{
			this->SyncWithData(this->RequestAll());
			this->ShowColorBars(true);
			this->ShowBody(true);
		}
		else
		{
			this->ShowColorBars(false);
			this->ShowBody(false);
		}
		this->ClearCache();
	}
}


bool iViewSubject::IsReady() const
{
	return this->CanBeShown() && this->IsDataPresent();
}


void iViewSubject::UpdateReplicas()
{
	mReplicator->Update();

	//
	//  We modify objects inside pipelines, but pipelines themselves would not execute if they remain unmodified.
	//
	int i;
	for(i=0; i<mPipelines.Size(); i++) mPipelines[i]->UpdateContents(iViewSubjectPipeline::_Replicas);

	this->ClearCache();
}


float iViewSubject::GetMemorySize() const
{
	int i;
	float s = 0.0f;

	for(i=0; i<mPipelines.Size(); i++) s += mPipelines[i]->GetMemorySize();
	
	s += this->GetExtraMemorySize();

	return s;
}


float iViewSubject::GetExtraMemorySize() const
{
	return 0.0f;
}


iViewSubjectPipeline* iViewSubject::CreatePipeline(int)
{
	return 0;
}

