/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggwidgetkeyselectionbox.h"


#include "idatastretch.h"
#include "istring.h"

#include "iggframe.h"

#include "ibgwidgetselectionboxsubject.h"

#include <limits.h>

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;

//
//  templates
//
#include "iggwidgetkeyhandlertemplate.h"


//
//******************************************
//
//  SelectionBox class
//
//******************************************
//
iggWidgetKeySelectionBox::iggWidgetKeySelectionBox(int offset, const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyHandler<int>(_WidgetTypeOther,key,_RenderModeImmediate,parent,index,indkey)
{
	mOffset = offset;
}


void iggWidgetKeySelectionBox::OnInt1Body(int)
{
	this->ExecuteControl(true);
}


void iggWidgetKeySelectionBox::SetOffset(int o)
{
	mOffset = o;
}


//
//******************************************
//
//  ItemizedSelectionBox
//
//******************************************
//
iggWidgetKeyItemizedSelectionBox::iggWidgetKeyItemizedSelectionBox(int offset, const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeySelectionBox(offset,key,parent,index,indkey)
{
}


//
//******************************************
//
//  SpinBox
//
//******************************************
//
iggWidgetKeySpinBox::iggWidgetKeySpinBox(int min, int max, const iString &title, int offset, const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeySelectionBox(offset,key,parent,index,indkey)
{
	mSubject = iggSubjectFactory::CreateWidgetSpinBoxSubject(this,min,max,title,1);
}


void iggWidgetKeySpinBox::QueryValue(int &v) const
{
	v = mSubject->GetValue() - mOffset;
}


void iggWidgetKeySpinBox::UpdateValue(int v)
{
	mSubject->SetValue(v+mOffset);
}


void iggWidgetKeySpinBox::SetFirstEntryText(const iString &text)
{
	mSubject->SetFirstEntryText(text);
}


void iggWidgetKeySpinBox::SetRange(int min, int max)
{
	mSubject->SetRange(min,max);
}


void iggWidgetKeySpinBox::SetStretch(int title, int box)
{
	mSubject->SetStretch(title,box);
}


int iggWidgetKeySpinBox::Count()
{
	return mSubject->Count();
}

//
//******************************************
//
//  ComboBox
//
//******************************************
//
iggWidgetKeyComboBox::iggWidgetKeyComboBox(const iString &title, int offset, const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyItemizedSelectionBox(offset,key,parent,index,indkey)
{
	mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,title);
}


void iggWidgetKeyComboBox::QueryValue(int &v) const
{
	v = mSubject->GetValue() - mOffset;
}


void iggWidgetKeyComboBox::UpdateValue(int v)
{
	mSubject->SetValue(v+mOffset);
}


void iggWidgetKeyComboBox::InsertItem(const iString &text, int ind)
{
	mSubject->InsertItem(text,ind);
}


void iggWidgetKeyComboBox::SetItem(const iString &text, int ind, bool vis)
{
	mSubject->SetItem(text,ind,vis);
}


void iggWidgetKeyComboBox::RemoveItem(int index)
{
	if(index>=0 && index<this->Count())	mSubject->RemoveItem(index);
}


int iggWidgetKeyComboBox::Count()
{
	return mSubject->Count();
}


void iggWidgetKeyComboBox::Clear()
{
	mSubject->Clear(); 
}


void iggWidgetKeyComboBox::SetStretch(int title, int box)
{
	mSubject->SetStretch(title,box);
}


//
//******************************************
//
//  RadioBox
//
//******************************************
//
iggWidgetKeyRadioBox::iggWidgetKeyRadioBox(int cols, const iString &title, int offset, const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyItemizedSelectionBox(offset,key,parent,index,indkey)
{
	mSubject = iggSubjectFactory::CreateWidgetRadioBoxSubject(this,cols,title);
}


void iggWidgetKeyRadioBox::QueryValue(int &v) const
{
	v = mSubject->GetValue() - mOffset; 
}


void iggWidgetKeyRadioBox::UpdateValue(int v)
{
	mSubject->SetValue(v+mOffset);
}


void iggWidgetKeyRadioBox::InsertItem(const iString &text, int index)
{
	mSubject->InsertItem(text,index);
}


void iggWidgetKeyRadioBox::SetItem(const iString &text, int index, bool vis)
{
	mSubject->SetItem(text,index,vis);
}


void iggWidgetKeyRadioBox::RemoveItem(int index)
{
	if(index>=0 && index<this->Count())	mSubject->RemoveItem(index);
}


int iggWidgetKeyRadioBox::Count()
{
	return mSubject->Count();
}


//
//******************************************
//
//  TextComboBox
//
//******************************************
//
iggWidgetKeyTextComboBox::iggWidgetKeyTextComboBox(int section, const iString &title, int offset, const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyComboBox(title,offset,key,parent,index,indkey)
{
	if(section < 0)
	{
		IERROR_REPORT("Negative sections do not exist.");
		section = 0;
	}
	mSection = section;
	mInvalidValue = iMath::_IntMax;
}


void iggWidgetKeyTextComboBox::SetInvalidValue(int v)
{
	mInvalidValue = v;
}


void iggWidgetKeyTextComboBox::QueryValue(int &v) const
{
	if(!this->ConvertFromText(mSubject->GetText(),v)) v = 0;
}


void iggWidgetKeyTextComboBox::UpdateValue(int v)
{
	int i, k;

	for(i=0; i<mSubject->Count(); i++)
	{
		if(this->ConvertFromText(mSubject->GetText(i),k) && v==k) break;
	}
	if(i < mSubject->Count())
	{
		mSubject->SetValue(i);
	}
	else
	{
		mSubject->SetValue(0);
		IERROR_REPORT("Invalid value in ComboBox widget.");
	}
}


bool iggWidgetKeyTextComboBox::ConvertFromText(const iString &text, int &val) const
{
	bool ok;
	iString ws = text;

	val = ws.Section(" ",mSection,mSection).ToInt(ok);
	if(!ok && mInvalidValue<iMath::_IntMax)
	{
		val = mInvalidValue;
		ok = true;
	}
	return ok;
}


//
//******************************************
//
//  StretchComboBox
//
//******************************************
//
iggWidgetKeyStretchComboBox::iggWidgetKeyStretchComboBox(const iObjectKey &key, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyComboBox("Stretch",0,key,parent,index,indkey)
{
	if(key.UnprefixedFullName().Contains("Stretch") < 0)
	{
		IERROR_FINISH("Invalid key in iggWidgetKeyStretchComboBox.");
	}

	int i, n;
	const iString &s = iDataStretch::GetStretchNames();
	n = s.Contains(',') + 1;
	for(i=0; i<n; i++)
	{
		this->InsertItem(s.Section(",",i,i));
	}
}

