/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idataexplorer.h"


#include "idata.h"
#include "idatahelper.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "iextensionfactory.h"
#include "ihistogram.h"
#include "ihistogrammaker.h"
#include "isurfaceviewsubject.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


//
//  DataExplorer must be an object so that it can be created by the ObjectFactory
//
IOBJECT_DEFINE_TYPE(iDataExplorer,DataExplorer,-de,iObjectType::_Helper);  // helper type


iDataExplorer* iDataExplorer::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iDataExplorer(vm);
}


iDataExplorer::iDataExplorer(iViewModule *vm) : iMultiTypeDataConsumer(vm,iDataInfo::Any())
{
	mHistogramMaker = 0;
	mMTime = 0;
	mComputeRangeOnly = false;
	
	mComponent = 0;
	mStretch = 0;
}


iDataExplorer::~iDataExplorer()
{
	if(mHistogramMaker != 0) mHistogramMaker->Delete();
}


void iDataExplorer::SyncWithDataBody(const iDataSyncRequest &)
{
	if(mComponent >= this->GetViewModule()->GetReader()->GetLimits(this->GetDataType())->GetNumVars())
	{
		mComponent = 0;
		mMTime = 0;
	}
}


void iDataExplorer::UpdateDataType(int prev)
{
	if(prev != mActiveDataTypeIndex)
	{
		if(mHistogramMaker != 0)
		{
			mHistogramMaker->Delete();
			mHistogramMaker = 0;
		}
		mComponent = 0;
		mMTime = 0;
	}
}


const iDataExplorer::Info& iDataExplorer::GetInfo(bool rangeonly)
{
	if(mHistogramMaker == 0)
	{
		mHistogramMaker = this->GetViewModule()->GetReader()->GetSubject(this->GetDataType())->CreateHistogramMaker();
		IERROR_ASSERT(mHistogramMaker);
		mHistogramMaker->SetFullResolution(true);
	}

	if(this->IsThereData())
	{
		iDataHelper h(this->GetData());
		mHistogramMaker->SetDataRank(h.GetDataRank());
		mHistogramMaker->SetInput(this->GetData(),mComponent);
		mHistogramMaker->SetStretch(mStretch);
		mHistogramMaker->GetInput()->Update();
		if(mHistogramMaker->GetInput()->GetMTime() > mMTime)
		{
			mComputeRangeOnly = rangeonly;
			this->Execute();
		}
	}
	return mInfo;
}


void iDataExplorer::SetStretch(int v)
{
	if(mStretch != v)
	{
		mStretch = v;
		mMTime = 0;
	}
}


void iDataExplorer::SetInputComponent(int c)
{
	if(c>=0 && c<this->GetNumberOfInputComponents() && c!=mComponent)
	{
		mComponent = c;
		mMTime = 0;
	}
}


int iDataExplorer::GetNumberOfInputComponents() const
{
	return this->GetViewModule()->GetReader()->GetLimits(this->GetDataType())->GetNumVars();
}


void iDataExplorer::Execute()
{
	int j;

	if(mHistogramMaker == 0)
	{
		vtkErrorMacro("iDataExplorer has no input.");
		return;
	}

	if(mComputeRangeOnly)
	{
		mHistogramMaker->GetHistogramRange(mInfo.Minimum,mInfo.CellMin,mInfo.PosMin,mInfo.Maximum,mInfo.CellMax,mInfo.PosMax);
		mInfo.Histogram = 0;
		mInfo.Median = 0.0;
		mInfo.Average = 0.0;
		mInfo.Dispersion = 0.0;
		return;
	}

	const iHistogram *h = mHistogramMaker->GetHistogram(); // dereference for better performance

	if(h == 0)
	{
		vtkErrorMacro("iDataExplorer has null histogram.");
#ifdef I_DEBUG
		h = mHistogramMaker->GetHistogram();
#endif
		mInfo.Histogram = 0;
		mInfo.Maximum = iMath::_LargeFloat;
		mInfo.Minimum = -iMath::_LargeFloat;
		mInfo.Median = 0.0;
		mInfo.Average = 0.0;
		mInfo.Dispersion = 0.0;
		return;
	}

	mMTime = mHistogramMaker->GetInput()->GetMTime();

	float max = mHistogramMaker->ApplyStretch(h->GetMaxValue(),true);
	float min = mHistogramMaker->ApplyStretch(h->GetMinValue(),false);
	float avg, dis, med;

	if(max > min) 
	{
		double sum0 = 0.0, sum1 = 0.0, sum2 = 0.0;

		for(j=0; j<h->N(); j++)
		{
			sum0 += h->Y(j);
			sum1 += h->Y(j)*h->X(j);
			sum2 += h->Y(j)*h->X(j)*h->X(j);
		}

		if(sum0 > 0.0)
		{
			avg = min + sum1/sum0*(max-min);
			dis = sqrt(sum2/sum0-pow(sum1/sum0,2.0))*(max-min);
			med = 0.0;
			sum1 = 0.0;
			for(j=0; j<h->N(); j++)
			{
				if(sum1<0.5*sum0 && (sum1+=h->Y(j))>=0.5*sum0)
				{
					med = min + h->X(j)*(max-min);
					break;
				}
			}
		}
		else
		{
			avg = dis = med = 0.5*(max+min);
		}
	}
	else
	{
		avg = med = min = max = 0.5*(max+min);
		dis = 0.0;
	}

	mInfo.Histogram = h;
	mInfo.Maximum = h->GetMaxValue();
	mInfo.Minimum = h->GetMinValue();
	mInfo.Median = mHistogramMaker->ResetStretch(med);
	mInfo.Average = mHistogramMaker->ResetStretch(avg);
	mInfo.Dispersion = mHistogramMaker->ResetStretch(dis);

	this->Modified();
}
