/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef ICALCULATORBASE_H
#define ICALCULATORBASE_H


#include "iarray.h"
#include "ibuffer.h"
#include "istring.h"

class iCalculatorParser;
namespace iCalculatorKit
{
	class Handle;
	class Variable;
	class Function;
};


class iCalculatorBase
{

public:
	
	iCalculatorBase(const iCalculatorBase *driver = 0);
	virtual ~iCalculatorBase();

	//
	//  If set, the result is always copied into it. Calculator does not own it, so it is not destroyed when the calculator is destroyed.
	//  If this is set, the calculator needs to be re-linked. The usage and dimension of the variable should correspond to the usage and
	//  dimension of the result; if they not, an error will be generated (i.e. calculator will not re-format the variable).	Use zero to unset it.
	//
	void SetResultVariable(iCalculatorKit::Variable *var);
	void SetResultVariable(iCalculatorKit::Variable &var);

	//
	//  Do all at once
	//
	bool Evaluate(const iString &expr);

	//
	//  Step-by-step work
	//
	bool Compile(const iString &expr);
	bool Link();
	bool Execute();

	//
	//  Get the list of used variables
	//
	inline const iString& ListUsedVariables() const { return mUsedVars; }

	//
	//  Useful for script work; has no effect if the driver argument was specified in the constructor
	//
	void CopyVariables(const iCalculatorBase *other);

	//
	//  Floating-point exceptions
	//
	void SetTrapFPE(bool s);
	inline bool GetTrapFPE() const { return mTrapFPE; }

	//
	//  Get the error message and position
	//
	inline const iString& GetErrorMessage() const { return mErrorMessage; }
	inline int GetErrorPosition() const { return mErrorPosition; }

	//
	//  Remove a variable with a given name. It removes the last added variable with this name, thus uncovering the previous one.
	//  This allows to implement name resolution: adding a (local) variable with the same name as an already existing one will hide
	//  the existing variable until the local variable is removed. Variables are not actually deleted, just removed from the calculator list.
	//
	void RemoveVariable(const iString &name);

	//
	//  Reset to the empty state
	//
	void Reset();

	//
	//  Parser punctuation symbols
	//
	const iString& SymbolSeparator() const;
	const iString& SymbolOpeningBraket() const;
	const iString& SymbolClosingBraket() const;
	const iString& SymbolOpeningParenthesis() const;
	const iString& SymbolClosingParenthesis() const;
	bool IsValidVariableName(const iString &name) const;

protected:

	inline iCalculatorKit::Variable* Result() const { return mResult; }

	//
	//  Add a variable to use in the expression. Calculator does not own variables, they must be created/destroyed outside this class.
	//  We keep this protected so that a child must replace it - otherwise it would be possible to add variables of one data type to a
	//  calculator of a different data type.
	//
	void AddVariable(const iCalculatorKit::Variable *var);

	//
	//  Set the error message; also allow for extending this call.
	//
	void SetError(const iString &message, int pos);
	virtual void SetErrorExtra(const iString &message, int pos);
	virtual iString ReportRunTimeErrorExtra(int ierr, const iCalculatorKit::Function *caller);

	//
	//  Inheritable methods for extending
	//
	virtual iCalculatorParser* CreateParser() const;
	virtual iCalculatorKit::Variable* CreateLiteral(const iString &str) const = 0;
	virtual iCalculatorKit::Function* CreateFunction(const iString &str, int loc) const = 0;

#ifdef I_DEBUG
	virtual void PrintStack(int iptr, int narg, int jmax, iCalculatorKit::Variable** stack) = 0;
#endif

	bool mAllowUndeterministicCalls;

private:

	void ReportRunTimeError(int ierr, const iCalculatorKit::Function *caller);

	bool CompileToken(int &index);
	bool CheckFunction(int index, int &stackptr);

	int mErrorPosition;
	iString mErrorMessage;

	bool mTrapFPE;
	iString mCode, mUsedVars;
	mutable iCalculatorParser *mParser;
	
	int mStatus;
	iCalculatorKit::Variable *mResult, *mResultVariable;
	iArray<iCalculatorKit::Handle*> mDataStack;
	iArray<iCalculatorKit::Function*> mCallStack;

	iBuffer<iCalculatorKit::Variable*> mBuffer, mSavedBuffer; // pre-allocated buffers to avoid memory allocation during execution

	iArray<const iCalculatorKit::Variable*> mVariableData;

	iArray<iCalculatorKit::Variable*> mLiterals;
	const iArray<const iCalculatorKit::Variable*> &mVariables;
};

#endif  // ICALCULATORBASE_H
