/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the Varese program. 
 * Varese is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
/* iiwu_oss.c
 *
 * Drivers for the Open (?) Sound System
 */

#include "iiwu_auport.h"
#include "iiwu_midi.h"

#if OSS_SUPPORT

#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/poll.h>

#define IIWU_OSS_DEFAULT_AUDIO_DEVICE     "/dev/dsp"
#define IIWU_OSS_DEFAULT_MIDI_DEVICE     "/dev/midi"

#define BUFFER_LENGTH 512

/** iiwu_oss_audio_driver_t
 * 
 * This structure should not be accessed directly. Use audio port
 * functions instead.  
 */
typedef struct {
  iiwu_auport_t* auport;
  void* buffer;
  pthread_t thread;
  int cont;
  int dspfd;
  int buffer_size;
  int buffer_byte_size;
  int bigendian;
  int formats;
  int format;
  int caps;
} iiwu_oss_audio_driver_t;

int delete_iiwu_oss_audio_driver(iiwu_audio_driver_t* p);

/* local utilities */
static int iiwu_oss_get_caps(iiwu_oss_audio_driver_t* dev);
static int iiwu_oss_set_queue_size(iiwu_oss_audio_driver_t* dev, int ss, int ch, int qs, int bs);
static int iiwu_oss_get_sample_formats(iiwu_oss_audio_driver_t* dev);
static void* iiwu_oss_audio_run(void* d);
static int iiwu_oss_audio_start(iiwu_oss_audio_driver_t* d);


typedef struct {
  int fd;
  pthread_t thread;
  int status;
  unsigned char buffer[BUFFER_LENGTH];
  iiwu_midi_parser_t* parser;
  iiwu_midi_handler_t* handler;
} iiwu_oss_midi_driver_t;

iiwu_midi_driver_t* new_iiwu_oss_midi_driver(iiwu_midi_handler_t* handler);
int delete_iiwu_oss_midi_driver(iiwu_midi_driver_t* p);
int iiwu_oss_midi_driver_join(iiwu_midi_driver_t* p);
int iiwu_oss_midi_driver_status(iiwu_midi_driver_t* p);
static void* iiwu_oss_midi_run(void* d);

/*
 * new_iiwu_oss_audio_driver
 */
iiwu_audio_driver_t* new_iiwu_oss_audio_driver(iiwu_auport_t* port)
{
  char* devname;
  iiwu_oss_audio_driver_t* dev = NULL;
  iiwu_pcm_data_t* dev_format;
  int format, channels, sample_rate;
  dev_format = iiwu_auport_get_dev_format(port);
  dev = IIWU_NEW(iiwu_oss_audio_driver_t);
  if (dev == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;    
  }
  IIWU_MEMSET(dev, 0, sizeof(iiwu_oss_audio_driver_t));
  dev->dspfd = -1;
  dev->auport = port;
  dev->cont = 1;
  dev->buffer_size = iiwu_auport_get_buffer_size(port);
  dev->buffer_byte_size = dev->buffer_size * iiwu_pcm_data_framesize(dev_format);
  dev->buffer = IIWU_MALLOC(dev->buffer_byte_size);
  if (dev->buffer == NULL) {
    IIWU_LOG(ERR, "new_iiwu_codec: Out of memory");
    goto error_recovery;
  }
  devname = iiwu_auport_get_device_name(port);
  if ((devname == NULL) || (IIWU_STRCMP(devname, "default") == 0)) {
    iiwu_auport_set_device_name(port, IIWU_OSS_DEFAULT_AUDIO_DEVICE);
    devname = IIWU_OSS_DEFAULT_AUDIO_DEVICE;
  }
  dev->dspfd = open(devname, O_WRONLY, 0);
  if (dev->dspfd < 0) {
    perror(devname);
    goto error_recovery;
  }
  iiwu_oss_get_sample_formats(dev);
  iiwu_oss_get_caps(dev);
  if (iiwu_oss_set_queue_size(dev, iiwu_pcm_data_get_bps(dev_format), 
			      iiwu_pcm_data_get_channels(dev_format), 
			      iiwu_auport_get_queue_size(port), 
			      iiwu_auport_get_buffer_size(port)) < 0) {
    IIWU_LOG(ERR, "Can't set device buffer size");
    goto error_recovery;
  }
  dev->format = 0;
  switch (iiwu_pcm_data_get_format(dev_format)) {
  case IIWU_SAMPLE_S8: dev->format = AFMT_S8; break;
  case IIWU_SAMPLE_U8: dev->format = AFMT_U8; break;
  case IIWU_SAMPLE_S16HE: dev->format = (dev->bigendian) ? AFMT_S16_BE : AFMT_S16_LE; break;
  case IIWU_SAMPLE_S16BE: dev->format = AFMT_S16_BE; break;
  case IIWU_SAMPLE_S16LE: dev->format = AFMT_S16_LE; break;
  case IIWU_SAMPLE_U16HE: dev->format = (dev->bigendian) ? AFMT_U16_BE : AFMT_U16_LE; break;
  case IIWU_SAMPLE_U16BE: dev->format = AFMT_U16_BE; break;
  case IIWU_SAMPLE_U16LE: dev->format = AFMT_U16_LE; break;
  }
  if (dev->format == 0) {
    IIWU_LOG(ERR, "Unknown sample format");
    goto error_recovery;
  }
  format = dev->format;
  if (ioctl(dev->dspfd, SNDCTL_DSP_SETFMT, &format) < 0) {
    IIWU_LOG(ERR, "Can't set the sample format");
    goto error_recovery;
  }
  if (dev->format != format) {
    IIWU_LOG(ERR, "Can't set the sample format");
    goto error_recovery;
  }
  channels = iiwu_pcm_data_get_channels(dev_format);
  if (ioctl(dev->dspfd, SOUND_PCM_WRITE_CHANNELS, &channels) < 0){
    IIWU_LOG(ERR, "Can't set the number of channels");
    goto error_recovery;
  }
  if (channels != iiwu_pcm_data_get_channels(dev_format)) {
    IIWU_LOG(ERR, "Can't set the number of channels");
    goto error_recovery;
  }
  sample_rate = iiwu_pcm_data_get_sample_rate(dev_format);
  if (ioctl(dev->dspfd, SNDCTL_DSP_SPEED, &sample_rate) < 0){
    IIWU_LOG(ERR, "Can't set the sample rate");
    goto error_recovery;
  }
  if ((sample_rate < 0.95 * iiwu_pcm_data_get_sample_rate(dev_format)) || 
      (sample_rate > 1.05 * iiwu_pcm_data_get_sample_rate(dev_format))) {
    IIWU_LOG(ERR, "Can't set the sample rate");
    goto error_recovery;
  }
  if (iiwu_oss_audio_start(dev)) {
    IIWU_LOG(ERR, "Can't start the audio thread");    
    goto error_recovery;
  }
  return (iiwu_audio_driver_t*) dev;
error_recovery:
  delete_iiwu_oss_audio_driver((iiwu_audio_driver_t*) dev);
  return NULL;  
}

/*
 * delete_iiwu_oss_audio_driver
 */
int delete_iiwu_oss_audio_driver(iiwu_audio_driver_t* p) 
{
  iiwu_oss_audio_driver_t* dev = (iiwu_oss_audio_driver_t*) p;  
  if (dev == NULL) {
    return IIWU_OK;
  }
  dev->cont = 0;
  if (dev->thread) {
    if (pthread_join(dev->thread, NULL)) {
      IIWU_LOG(ERR, "Failed to join the audio thread");
      return IIWU_FAILED;
    }
  }
  if (dev->dspfd >= 0) {
    close(dev->dspfd);
  }
  if (dev->buffer != NULL) {
    IIWU_FREE(dev->buffer);
  }
  IIWU_FREE(dev);
  return IIWU_OK;
}

/*
 * iiwu_oss_get_sample_formats
 */
int iiwu_oss_get_sample_formats(iiwu_oss_audio_driver_t* dev) 
{
  int mask;
  unsigned short U16 = 1;
  unsigned char* U8 = (unsigned char*) &U16;
  dev->formats = 0;
  dev->bigendian = 0;
  if (ioctl(dev->dspfd, SNDCTL_DSP_GETFMTS, &mask) == -1) {
    return -1;
  }
  dev->formats = mask;
  if (U8[1] == 1) {
    IIWU_LOG(DBG, "Machine is big endian.");
    dev->bigendian = 1;
  }
  if (U8[0] == 1) {
    IIWU_LOG(DBG, "Machine is little endian.");
    dev->bigendian = 0;
  }
  IIWU_LOG(DBG, "The sound device supports the following audio formats:");
  if (mask & AFMT_U8)        { IIWU_LOG(DBG, "  U8"); }
  if (mask & AFMT_S8)        { IIWU_LOG(DBG, "  S8"); }
  if (mask & AFMT_U16_LE)    { IIWU_LOG(DBG, "  U16LE"); }
  if (mask & AFMT_U16_BE)    { IIWU_LOG(DBG, "  U16BE"); }
  if (mask & AFMT_S16_LE)    { IIWU_LOG(DBG, "  S16LE"); }
  if (mask & AFMT_S16_BE)    { IIWU_LOG(DBG, "  S16BE"); }
  if (mask & AFMT_MU_LAW)    { IIWU_LOG(DBG, "  mu-law"); }
  if (mask & AFMT_A_LAW)     { IIWU_LOG(DBG, "  a-law"); }
  if (mask & AFMT_IMA_ADPCM) { IIWU_LOG(DBG, "  ima-adpcm"); }
  if (mask & AFMT_MPEG)      { IIWU_LOG(DBG, "  mpeg"); }
  return 0;
}

/**
 *  iiwu_oss_get_caps
 *
 *  Get the audio capacities of the sound card.
 */
int iiwu_oss_get_caps(iiwu_oss_audio_driver_t* dev)
{
  int caps;
  dev->caps = 0;
  if (ioctl(dev->dspfd, SNDCTL_DSP_GETCAPS, &caps) < 0) {
    return -1;
  }
  dev->caps = caps;
  IIWU_LOG(DBG, "The sound device has the following capabilities:");
  if (caps & DSP_CAP_DUPLEX)   { 
    IIWU_LOG(DBG, "  Duplex:    simultaneous playing and recording possible") ;
  } else { 
    IIWU_LOG(DBG, "  Duplex:    simultaneous playing and recording not possible"); 
  }
  if (caps & DSP_CAP_REALTIME) { 
    IIWU_LOG(DBG, "  Real-time: precise reporting of output pointer possible"); 
  } else { 
    IIWU_LOG(DBG, "  Real-time: precise reporting of output pointer not possible"); 
  }
  if (caps & DSP_CAP_BATCH) { 
    IIWU_LOG(DBG, "  Batch:     local storage for recording and/or playback"); 
  } else { 
    IIWU_LOG(DBG, "  Batch:     no local storage for recording and/or playback"); 
  }
  if (caps & DSP_CAP_TRIGGER) { 
    IIWU_LOG(DBG, "  Trigger:   triggering of recording/playback possible"); 
  } else { 
    IIWU_LOG(DBG, "  Trigger:   triggering of recording/playback not possible"); 
  }
  if (caps & DSP_CAP_MMAP) { 
    IIWU_LOG(DBG, "  Mmap:      direct access to the hardware level buffer possible"); 
  } else { 
    IIWU_LOG(DBG, "  Mmap:      direct access to the hardware level buffer not possible"); 
  }
  return 0;
}

/** 
 *  iiwu_oss_set_queue_size
 *
 *  Set the internal buffersize of the output device.
 *
 *  @param ss Sample size in bits
 *  @param ch Number of channels
 *  @param qs The queue size in frames
 *  @param bs The synthesis buffer size in frames
 */
int iiwu_oss_set_queue_size(iiwu_oss_audio_driver_t* dev, int ss, int ch, int qs, int bs)
{
  unsigned int fragmentSize;
  unsigned int fragSizePower;
  unsigned int fragments;
  unsigned int fragmentsPower;
  fragmentSize = (unsigned int) (bs * ch * ss / 8);
  fragSizePower = 0;
  while (0 < fragmentSize) {
    fragmentSize = (fragmentSize >> 1);
    fragSizePower++;
  }
  fragSizePower--;
  fragments = (unsigned int) (qs / bs);
  if (fragments < 2) {
    fragments = 2;
  }
  /* make sure fragments is a power of 2 */
  fragmentsPower = 0;
  while (0 < fragments) {
    fragments = (fragments >> 1);
    fragmentsPower++;
  }
  fragmentsPower--;
  fragments = (1 << fragmentsPower);
  fragments = (fragments << 16) + fragSizePower;
  return ioctl(dev->dspfd, SNDCTL_DSP_SETFRAGMENT, &fragments);
}

/*
 * iiwu_oss_audio_run
 */
void* iiwu_oss_audio_run(void* d)
{
  iiwu_oss_audio_driver_t* dev = (iiwu_oss_audio_driver_t*) d;
  iiwu_auport_t* port = dev->auport;
  void* buffer = dev->buffer;
  int len = dev->buffer_size;

  /* it's as simple as that: */
  while ((iiwu_auport_get_state(port) == IIWU_AUPORT_PLAYING) && dev->cont) {
    iiwu_auport_write(port, buffer, len);
    write(dev->dspfd, buffer, dev->buffer_byte_size);
  }

  IIWU_LOG(DBG, "Audio thread finished");

  pthread_exit(NULL);

  return 0; /* not reached */
}

/*
 * iiwu_oss_audio_start
 */
int iiwu_oss_audio_start(iiwu_oss_audio_driver_t* d) 
{
  pthread_attr_t attr;
  int err;
  int sched = SCHED_FIFO;
  if (pthread_attr_init(&attr)) {
    IIWU_LOG(ERR, "Couldn't initialize audio thread attributes");
    return IIWU_FAILED;
  }
  /* the pthread_create man page explains that
     pthread_attr_setschedpolicy returns an error if the user is not
     permitted the set SCHED_FIFO. it seems however that no error is
     returned but pthread_create fails instead. that's why i try to
     create the thread twice in a while loop. */
  while (1) {
    err = pthread_attr_setschedpolicy(&attr, sched);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the audio output");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(ERR, "Couldn't set scheduling policy.");
	return err;
      }
    }
    err = pthread_create(&d->thread, &attr, iiwu_oss_audio_run, (void*) d);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the audio output");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(PANIC, "Couldn't create the audio thread.");
	return err;
      }
    }
    return 0;
  }
}

/*
 * new_iiwu_oss_midi_driver
 */
iiwu_midi_driver_t* new_iiwu_oss_midi_driver(iiwu_midi_handler_t* handler)
{
  int err;
  iiwu_oss_midi_driver_t* dev;
  pthread_attr_t attr;
  int sched = SCHED_FIFO;
  char* device;
  
  /* not much use doing anything */
  if (handler == NULL) {
    IIWU_LOG(ERR, "Invalid argument");
    return NULL;
  }

  /* allocate the device */
  dev = IIWU_NEW(iiwu_oss_midi_driver_t);
  if (dev == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  IIWU_MEMSET(dev, 0, sizeof(iiwu_oss_midi_driver_t));
  dev->fd = -1;
  dev->handler = handler;

  /* allocate one event to store the input data */
  dev->parser = new_iiwu_midi_parser();
  if (dev->parser == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }

  /* get the device name. if none is specified, use the default device. */
  device = iiwu_midi_handler_get_device_name(handler);
  if (device == NULL) {
    iiwu_midi_handler_set_device_name(handler, IIWU_OSS_DEFAULT_MIDI_DEVICE);
    device = iiwu_midi_handler_get_device_name(handler);
  }

  /* open the default hardware device. only use midi in. */
  dev->fd = open(device, O_RDONLY, 0);
  if (dev->fd < 0) {
    perror(device);
    goto error_recovery;
  }

  dev->status = IIWU_MIDI_READY;

  /* create the midi thread */
  if (pthread_attr_init(&attr)) {
    IIWU_LOG(ERR, "Couldn't initialize midi thread attributes");
    goto error_recovery;
  }
  /* use fifo scheduling. if it fails, use default scheduling. */
  while (1) {
    err = pthread_attr_setschedpolicy(&attr, sched);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the MIDI input");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(ERR, "Couldn't set scheduling policy");
	goto error_recovery;
      }
    }
    err = pthread_create(&dev->thread, &attr, iiwu_oss_midi_run, (void*) dev);
    if (err) {
      IIWU_LOG(WARN, "Couldn't set high priority scheduling for the MIDI input");
      if (sched == SCHED_FIFO) {
	sched = SCHED_OTHER;
	continue;
      } else {
	IIWU_LOG(PANIC, "Couldn't create the midi thread.");
	goto error_recovery;
      }
    }
    break;
  }  
  return (iiwu_midi_driver_t*) dev;

 error_recovery:
  delete_iiwu_oss_midi_driver((iiwu_midi_driver_t*) dev);
  return NULL;
}

/*
 * delete_iiwu_oss_midi_driver
 */
int delete_iiwu_oss_midi_driver(iiwu_midi_driver_t* p)
{
  int err;
  iiwu_oss_midi_driver_t* dev;

  dev = (iiwu_oss_midi_driver_t*) p;
  if (dev == NULL) {
    return IIWU_OK;
  }

  dev->status = IIWU_MIDI_DONE;

  /* cancel the thread and wait for it before cleaning up */
  if (dev->thread) {
    err = pthread_cancel(dev->thread);
    if (err) {
      IIWU_LOG(ERR, "Failed to cancel the midi thread");
      return IIWU_FAILED;
    }
    if (pthread_join(dev->thread, NULL)) {
      IIWU_LOG(ERR, "Failed to join the midi thread");
      return IIWU_FAILED;
    }
  }
  if (dev->fd >= 0) {
    close(dev->fd);
  }
  if (dev->parser != NULL) {
    delete_iiwu_midi_parser(dev->parser);
  }
  IIWU_FREE(dev);
  return IIWU_OK;
}

/*
 * iiwu_oss_midi_run
 */
void* iiwu_oss_midi_run(void* d)
{
  int n, i;
  iiwu_midi_event_t* evt;
  iiwu_oss_midi_driver_t* dev = (iiwu_oss_midi_driver_t*) d;

  /* make sure the other threads can cancel this thread any time */
  if (pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL)) {
    IIWU_LOG(ERR, "Failed to set the cancel state of the midi thread");
    pthread_exit(NULL);
  }
  if (pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL)) {
    IIWU_LOG(ERR, "Failed to set the cancel state of the midi thread");
    pthread_exit(NULL);
  }

  /* go into a loop until someone tells us to stop */
  dev->status = IIWU_MIDI_LISTENING;

  while (dev->status == IIWU_MIDI_LISTENING) {

    /* read new data */
    n = read(dev->fd, dev->buffer, BUFFER_LENGTH);
    if (n < 0) {
      perror("read");
      IIWU_LOG(ERR, "Failed to read the midi input");
      dev->status = IIWU_MIDI_DONE;
    }
    
    /* let the parser convert the data into events */
    for (i = 0; i < n; i++) {
      evt = iiwu_midi_parser_parse(dev->parser, dev->buffer[i]);
      /* send the events to the synthesizer */
      if (evt != NULL) {
	iiwu_midi_handler_send_event(dev->handler, evt);
      }
    }

  }
  pthread_exit(NULL);
}

/*
 * iiwu_oss_midi_driver_join
 */
int iiwu_oss_midi_driver_join(iiwu_midi_driver_t* d)
{
  iiwu_oss_midi_driver_t* dev = (iiwu_oss_midi_driver_t*) d;
  if (pthread_join(dev->thread, NULL)) {
    IIWU_LOG(ERR, "Failed to join the midi thread");
    return IIWU_FAILED;
  }
  return IIWU_OK;
}

int iiwu_oss_midi_driver_status(iiwu_midi_driver_t* p)
{
  iiwu_oss_midi_driver_t* dev = (iiwu_oss_midi_driver_t*) p;
  return dev->status;
}

#endif /*#if OSS_SUPPORT */
