/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
#include "iiwu_synth.h"

#ifndef DISABLE_AUPORT
#include "iiwu_auport.h"
#endif

/***************************************************************
 *
 *                         GLOBAL
 */
static int iiwu_synth_initialized = 0;        /* has the synth module been initialized? */
char iiwu_synth_errbuf[512];                  /* buffer for error message */
int iiwu_synth_errno = 0;                     /* error number, not used yet */

/* the (inversed) priority in which oscillators are stolen */
unsigned char chan_priority[] = { 15, 14, 13, 12, 11, 10, 8, 7, 6, 5, 4, 3, 2, 1, 0, 9 };


/* conversion tables */
iiwu_real_t iiwu_ct2hz_tab[IIWU_CENTS_HZ_SIZE];
iiwu_real_t iiwu_vel2cb_tab[IIWU_VEL_CB_SIZE];
iiwu_real_t iiwu_cb2amp_tab[IIWU_CB_AMP_SIZE];
iiwu_real_t iiwu_posbp_tab[128];
iiwu_real_t iiwu_concave_tab[128];
iiwu_real_t iiwu_convex_tab[128];
iiwu_real_t iiwu_pan_tab[IIWU_PAN_SIZE];
iiwu_interp_coeff_t interp_coeff[IIWU_INTERP_MAX];


/* default modulators */
iiwu_mod_t default_pan_mod;
iiwu_mod_t default_att_mod;
iiwu_mod_t default_pitch_bend_mod;

/* reverb presets */
iiwu_revmodel_presets_t revmodel_preset[] = {
  { "Test 1", 0.0f, 0.0f, 0.0f, 0.5f, 0.0f }, 
  { "Test 2", 0.5f, 0.0f, 0.0f, 0.5f, 0.0f }, 
  { "Test 3", 0.5f, 0.0f, 0.0f, 0.5f, 1.0f }, 
  { "Test 4", 0.8f, 0.0f, 0.0f, 0.5f, 1.0f }, 
  { "Test 5", 0.8f, 1.0f, 1.0f, 1.0f, 1.0f }, 
  { NULL, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f }
};

/***************************************************************
 *
 *               INITIALIZATION & UTILITIES
 */

/*
 * void iiwu_synth_init
 *
 * Does all the initialization for this module.
 */
static void 
iiwu_synth_init()
{
  int i;
  double x;
  iiwu_synth_initialized++;
  sfont_init_chunks();  
  for (i = 0; i < IIWU_CENTS_HZ_SIZE; i++) {
    iiwu_ct2hz_tab[i] = (iiwu_real_t) pow(2.0, (double) i / 1200.0);
  }

/*    for (i = 0; i < IIWU_VEL_CB_SIZE; i++) { */
/*      x = (iiwu_real_t) (127.0 - i) / 128.0;  */
/*      iiwu_vel2cb_tab[i] = 960.0 * x * x * x * x * x; */
/*    } */

  for (i = 0; i < IIWU_VEL_CB_SIZE; i++) {
    x = (iiwu_real_t) (127.0 - i) / 127.0; 
    iiwu_vel2cb_tab[i] = (iiwu_real_t) (360.0 * x);
  }
  
  /* centibels to amplitude conversion */
  for (i = 0; i < IIWU_CB_AMP_SIZE; i++) {
    iiwu_cb2amp_tab[i] = (iiwu_real_t) pow(10.0, (double) i / -200.0);  
  }

  /* initialize the conversion tables */
/*    for (i = 0; i < 129; i++) { */
/*      iiwu_concave_tab[i] = (iiwu_real_t) pow(2.0, (double) i / 128.0) - 1.0; */
/*    } */
  /* concave transform curve */
  iiwu_concave_tab[0] = 0.0;
  iiwu_concave_tab[127] = 1.0;
  x = log10(128.0 / 127.0);
  for (i = 1; i < 127; i++) {
    iiwu_concave_tab[i] = (iiwu_real_t) (1.0 - x / log10(127.0 / (127.0 - i)));
  }

  /* convex transform curve */
  iiwu_convex_tab[0] = 0;
  iiwu_convex_tab[127] = 1.0;
  x = log10(128.0 / 127.0);
  for (i = 1; i < 127; i++) {
    iiwu_convex_tab[i] = (iiwu_real_t) (x / log10(127.0 / i));
  }
  /* initialize the pan conversion table */
  x = PI / 2.0 / (IIWU_PAN_SIZE - 1.0);
  for (i = 0; i < IIWU_PAN_SIZE; i++) {
    iiwu_pan_tab[i] = (iiwu_real_t) sin(i * x);
  }

#if defined(USE_TRUNCATION)
  /* this is a stupid */
  for (i = 0; i < IIWU_INTERP_MAX; i++) {
    interp_coeff[i] = 0;
  }
#elif defined(USE_LINEAR_INTERPOLATION)
  /* this is a bit stupid */
  for (i = 0; i < IIWU_INTERP_MAX; i++) {
    x = (double) i / (double) IIWU_INTERP_MAX;
    interp_coeff[i].a0 = (iiwu_real_t) (1.0 - x);
    interp_coeff[i].a1 = (iiwu_real_t) x;
  }
#elif defined(USE_CUBIC_INTERPOLATION)
  /* this is cool */
  /* Initialize the coefficients for the interpolation. The math comes
   * from a mail, posted by Olli Niemitalo to the music-dsp mailing
   * list (I found it in the music-dsp archives
   * http://www.smartelectronix.com/musicdsp/).  */
  for (i = 0; i < IIWU_INTERP_MAX; i++) {
    x = (double) i / (double) IIWU_INTERP_MAX;
    interp_coeff[i].a0 = (iiwu_real_t) (x * (-0.5 + x * (1 - 0.5 * x)));
    interp_coeff[i].a1 = (iiwu_real_t) (1.0 + x * x * (1.5 * x - 2.5));
    interp_coeff[i].a2 = (iiwu_real_t) (x * (0.5 + x * (2.0 - 1.5 * x)));
    interp_coeff[i].a3 = (iiwu_real_t) (0.5 * x * x * (x - 1.0));
  }
#endif

  /* initialize the default modulators */
  /* pan */
  iiwu_mod_set_source1(&default_pan_mod, 10, 
		       IIWU_MOD_CC | IIWU_MOD_LINEAR | IIWU_MOD_BIPOLAR | IIWU_MOD_POSITIVE);
  iiwu_mod_set_source2(&default_pan_mod, 0, 0);
  iiwu_mod_set_dest(&default_pan_mod, GEN_PAN);
  iiwu_mod_set_amount(&default_pan_mod, 1000.0);

  /* initial attenuation */
  iiwu_mod_set_source1(&default_att_mod, 7, 
		       IIWU_MOD_CC | IIWU_MOD_CONCAVE | IIWU_MOD_UNIPOLAR | IIWU_MOD_NEGATIVE);
  iiwu_mod_set_source2(&default_att_mod, 0, 0);
  iiwu_mod_set_dest(&default_att_mod, GEN_ATTENUATION);
  iiwu_mod_set_amount(&default_att_mod, 960.0);
  
  /* pitch bend */
  iiwu_mod_set_source1(&default_pitch_bend_mod, IIWU_MOD_PITCHWHEEL, 
		       IIWU_MOD_GC | IIWU_MOD_LINEAR | IIWU_MOD_BIPOLAR | IIWU_MOD_POSITIVE);
  iiwu_mod_set_source2(&default_pitch_bend_mod, IIWU_MOD_PITCHWHEELSENS, 
		       IIWU_MOD_GC | IIWU_MOD_LINEAR | IIWU_MOD_UNIPOLAR | IIWU_MOD_POSITIVE);
  iiwu_mod_set_dest(&default_pitch_bend_mod, GEN_PITCH);
  iiwu_mod_set_amount(&default_pitch_bend_mod, 12700.0);
}

/*
 * iiwu_log
 */
int 
iiwu_log(int level, char* fmt, ...)
{
  va_list args; 
  va_start (args, fmt); 
  vsprintf(iiwu_synth_errbuf, fmt, args); 
  va_end (args); 
  switch (level) {
  case PANIC: 
    IIWU_PRINTF("panic: %s\n", iiwu_synth_errbuf);
    break;
  case ERR: 
    IIWU_PRINTF("error: %s\n", iiwu_synth_errbuf);
    break;
  case WARN: 
    IIWU_PRINTF("warning: %s\n", iiwu_synth_errbuf);
    break;
  case DBG: 
#if DEBUG
    IIWU_PRINTF("debug: %s\n", iiwu_synth_errbuf);
#endif
    break;
  default:
    IIWU_PRINTF("%s\n", iiwu_synth_errbuf);
    break;
  }
  return IIWU_FAILED; 
}

/*
 * iiwu_ct2hz
 */
static iiwu_real_t 
iiwu_ct2hz(iiwu_real_t cents) 
{
  if ((cents >= 6900) && (cents < 8100)) {
    return (iiwu_real_t) 440.0 * iiwu_ct2hz_tab[(int) (cents - 6900)];
  } else if ((cents >= 8100) && (cents < 9300)) {
    return (iiwu_real_t) 880.0 * iiwu_ct2hz_tab[(int) (cents - 8100)];
  } else if ((cents >= 5700) && (cents < 6900)) {
    return (iiwu_real_t) 220.0 * iiwu_ct2hz_tab[(int) (cents - 5700)];
  } else if ((cents >= 9300) && (cents < 10500)) {
    return (iiwu_real_t) 1760.0 * iiwu_ct2hz_tab[(int) (cents - 9300)];
  } else if ((cents >= 4500) && (cents < 5700)) {
    return (iiwu_real_t) 110.0 * iiwu_ct2hz_tab[(int) (cents - 4500)];
  } else if ((cents >= 10500) && (cents < 11700)) {
    return (iiwu_real_t) 3520.0 * iiwu_ct2hz_tab[(int) (cents - 10500)];
  } else if ((cents >= 3300) && (cents < 4500)) {
    return (iiwu_real_t) 55.0 * iiwu_ct2hz_tab[(int) (cents - 3300)];
  } else if ((cents >= 11700) && (cents < 12900)) {
    return (iiwu_real_t) 7040.0 * iiwu_ct2hz_tab[(int) (cents - 11700)];
  } else if ((cents >= 2100) && (cents < 3300)) {
    return (iiwu_real_t) 27.5 * iiwu_ct2hz_tab[(int) (cents - 2100)];
  } else if ((cents >= 12900) && (cents < 14100)) {
    return (iiwu_real_t) 14080.0 * iiwu_ct2hz_tab[(int) (cents - 12900)];
  } else if ((cents >= 900) && (cents < 2100)) {
    return (iiwu_real_t) 13.75 * iiwu_ct2hz_tab[(int) (cents - 900)];
  } else {
    return (iiwu_real_t) 1.0; /* some loony trying to make you deaf */
  }
}

/*
 * iiwu_vel2cb
 */
static iiwu_real_t 
iiwu_vel2cb(int vel)
{
  return iiwu_vel2cb_tab[vel];
}

/*
 * iiwu_cb2amp
 *
 * in: a value between 0 and 961, 0 is no attenuation
 * out: a value between 1 and 0
 */
static iiwu_real_t 
iiwu_cb2amp(iiwu_real_t cb)
{
  if (cb < 0) return 1;
  if (cb > 960) return 0;
  return iiwu_cb2amp_tab[(int) cb]; 
}

/*
 * iiwu_tc2sec
 */
static iiwu_real_t 
iiwu_tc2sec(iiwu_real_t tc)
{
  return (tc == -32768) ? (iiwu_real_t) 0.0 : (iiwu_real_t) pow(2.0, (double) tc / 1200.0);
}

/*
 * iiwu_act2hz
 *
 * Convert from absolute cents to Hertz
 */
static iiwu_real_t 
iiwu_act2hz(iiwu_real_t c)
{
  return (iiwu_real_t) (8.176 * pow(2.0, (double) c / 1200.0));
}

/*
 * iiwu_hz2ct
 *
 * Convert from Hertz to cents
 */
static iiwu_real_t 
iiwu_hz2ct(iiwu_real_t f)
{
  return (iiwu_real_t) (6900 + 1200 * log(f / 440.0) / log(2.0));
}

/*
 * iiwu_pan
 */
static iiwu_real_t 
iiwu_pan(iiwu_real_t c, int left)
{
  if (left) {
    c = -c;
  }
  if (c < -500) {
    return (iiwu_real_t) 0.0;
  } else if (c > 500) {
    return (iiwu_real_t) 1.0;
  } else {
    return iiwu_pan_tab[(int) (c + 500)];
  }
}

/*
 * iiwu_concave
 */
static iiwu_real_t 
iiwu_concave(iiwu_real_t val)
{
  if (val < 0) {
    return 0;
  } else if (val > 127) {
    return 1;
  }
  return iiwu_concave_tab[(int) val];
}

/*
 * iiwu_convex
 */
static iiwu_real_t 
iiwu_convex(iiwu_real_t val)
{
  if (val < 0) {
    return 0;
  } else if (val > 127) {
    return 1;
  }
  return iiwu_convex_tab[(int) val];
}

/***************************************************************
 *
 *                      IIWU SYNTH
 */

/*
 * new_iiwu_synth
 */
iiwu_synth_t* 
new_iiwu_synth(int poly, int auport, char* driver, char* dev)
{
  int i;
  iiwu_synth_t* synth;

  /* initialize all the conversion tables and other stuff */
  if (iiwu_synth_initialized == 0) {
    iiwu_synth_init();
  }

  /* allocate a new synthesizer object */
  synth = IIWU_NEW(iiwu_synth_t);
  if (synth == NULL) {
    IIWU_LOG(ERR, "Out of memory"); 
    return NULL;
  }
  IIWU_MEMSET(synth, 0, sizeof(synth));

  /* as soon as the synth is created it starts playing. */
  synth->state = IIWU_SYNTH_PLAYING;

  /* allocate all channel objects */
  synth->channel = IIWU_ARRAY(iiwu_channel_t*, IIWU_NUM_CHANNELS);
  if (synth->channel == NULL) {
    IIWU_LOG(ERR, "Out of memory"); 
    goto error_recovery;
  }
  for (i = 0; i < IIWU_NUM_CHANNELS; i++) {
    synth->channel[i] = new_iiwu_channel(i);
    if (synth->channel[i] == NULL) {
      goto error_recovery;
    }
  }

  /* allocate all synthesis processes */
  synth->nsp = poly;
  synth->sp = IIWU_ARRAY(iiwu_sp_t*, poly);
  if (synth->sp == NULL) {
    goto error_recovery;
  }
  for (i = 0; i < synth->nsp; i++) {
    synth->sp[i] = new_iiwu_sp();
    if (synth->sp[i] == NULL) {
      goto error_recovery;
    }
  }

  /* allocate the sample buffers */
  synth->mono_buf = IIWU_ARRAY(iiwu_real_t, IIWU_BUFSIZE);
  synth->left_buf = IIWU_ARRAY(iiwu_real_t, IIWU_BUFSIZE);
  synth->right_buf = IIWU_ARRAY(iiwu_real_t, IIWU_BUFSIZE);
  synth->reverb_buf = IIWU_ARRAY(iiwu_real_t, IIWU_BUFSIZE);
  synth->chorus_buf = IIWU_ARRAY(iiwu_real_t, IIWU_BUFSIZE);
  if ((synth->mono_buf == NULL) || (synth->left_buf == NULL) || (synth->right_buf == NULL) 
      || (synth->reverb_buf == NULL) || (synth->chorus_buf == NULL)) {
    IIWU_LOG(ERR, "Out of memory"); 
    goto error_recovery;
  } 

  /* create the reverb module */
  synth->reverb = new_iiwu_revmodel();
  if (synth->reverb == NULL) {
    IIWU_LOG(ERR, "Out of memory"); 
    goto error_recovery;
  }
  iiwu_revmodel_setmode(synth->reverb, 0.0);
  iiwu_synth_set_reverb(synth, 0);

#ifndef DISABLE_AUPORT
  /* open the direct audio output if requested */
  if (auport) {
    synth->auport = new_iiwu_auport(driver, dev, IIWU_SAMPLE_FORMAT, 
				    IIWU_SAMPLE_S16HE, 44100, 2, 
				    IIWU_BUFSIZE, IIWU_QUEUESIZE, 
				    iiwu_synth_write, (void*) synth);
    if (synth->auport == NULL) {
      goto error_recovery;
    }
  }
#endif

  return synth;

 error_recovery:
  delete_iiwu_synth(synth);
  return NULL;
}

char* iiwu_auport_get_devname(iiwu_auport_t* p);

/*
 * delete_iiwu_synth
 */
int 
delete_iiwu_synth(iiwu_synth_t* synth) 
{
  int i;
  if (synth == NULL) {
    return IIWU_OK;
  }
  synth->state = IIWU_SYNTH_STOPPED;
#ifndef DISABLE_AUPORT
  if (synth->auport != NULL) {
    delete_iiwu_auport(synth->auport);
    synth->auport = NULL;
  }
#endif
  if (synth->sfont != NULL) {
    delete_iiwu_sfont(synth->sfont);
  }
  if (synth->channel != NULL) {
    for (i = 0; i < IIWU_NUM_CHANNELS; i++) {
      if (synth->channel[i] != NULL) {
	delete_iiwu_channel(synth->channel[i]);
      }
    }
    IIWU_FREE(synth->channel);
  }
  if (synth->sp != NULL) {
    for (i = 0; i < synth->nsp; i++) {
      if (synth->sp[i] != NULL) {
	delete_iiwu_sp(synth->sp[i]);
      }
    }
    IIWU_FREE(synth->sp);
  }

  /* free all the sample buffer */
  if (synth->mono_buf != NULL) {
    IIWU_FREE(synth->mono_buf);
  }
  if (synth->left_buf != NULL) {
    IIWU_FREE(synth->left_buf);
  }
  if (synth->right_buf != NULL) {
    IIWU_FREE(synth->right_buf);
  }
  if (synth->reverb_buf != NULL) {
    IIWU_FREE(synth->reverb_buf);
  }
  if (synth->chorus_buf != NULL) {
    IIWU_FREE(synth->chorus_buf);
  }

  /* release the reverb module */
  if (synth->reverb != NULL) {
    delete_iiwu_revmodel(synth->reverb);
  }

  IIWU_FREE(synth);

  return IIWU_OK;
}

/*
 * iiwu_synth_error
 */
char* 
iiwu_synth_error(iiwu_synth_t* synth)
{
  return iiwu_synth_errbuf;
}

/*
 * iiwu_synth_noteon
 */
int 
iiwu_synth_noteon(iiwu_synth_t* synth, int chan, int key, int vel) 
{
  /* check the ranges of the arguments */
  if ((chan < 0) || (chan >= IIWU_NUM_CHANNELS)) {
    IIWU_LOG(WARN, "Channel out of range");
    return IIWU_FAILED;     
  }
  if ((key < 0) || (key >= 128)) {
    IIWU_LOG(WARN, "Key out of range");
    return IIWU_FAILED;     
  }
  if ((vel < 0) || (vel >= 128)) {
    IIWU_LOG(WARN, "Velocity out of range");
    return IIWU_FAILED;     
  }

  /* notes with velocity equal zero go to noteoff  */
  if (vel == 0) {
    return iiwu_synth_noteoff(synth, chan, key);
  }

/*    printf("note %d %d %d\n", chan, key, vel); */
  
  /* let the channel handle the noteon */
  return iiwu_channel_noteon(synth->channel[chan], synth, key, vel);
}

/*
 * iiwu_synth_noteoff
 */
int 
iiwu_synth_noteoff(iiwu_synth_t* synth, int chan, int key)
{
  int i;
  iiwu_sp_t* sp;
  int status = IIWU_FAILED;

  /* search all synthesis processes with this channel and key pair */
  for (i = 0; i < synth->nsp; i++) {

    sp = synth->sp[i];

    if ((sp->chan == chan) && (sp->key == key)) {
      /* turn it off */
      iiwu_sp_noteoff(sp);
      status = IIWU_OK;
    }
  }
  return status;
}

/*
 * iiwu_synth_cc
 */
int 
iiwu_synth_cc(iiwu_synth_t* synth, int chan, int num, int val)
{
  int i;
  iiwu_sp_t* sp;

  /* check the ranges of the arguments */
  if ((chan < 0) || (chan >= IIWU_NUM_CHANNELS)) {
    IIWU_LOG(WARN, "Channel out of range");
    return IIWU_FAILED;     
  }
  if ((num < 0) || (num >= 128)) {
    IIWU_LOG(WARN, "Ctrl out of range");
    return IIWU_FAILED;     
  }

  /* set the controller value in the channel */
  iiwu_channel_cc(synth->channel[chan], num, val);

  /* tell all synthesis processes on this channel to update their
     synthesis parameters */
  for (i = 0; i < synth->nsp; i++) {
    sp = synth->sp[i];
    if (sp->chan == chan) {
      iiwu_sp_modulate(sp, 1, num, val);
    }
  }
  return IIWU_OK;
}

/*
 * iiwu_synth_pitch_bend
 */
int 
iiwu_synth_pitch_bend(iiwu_synth_t* synth, int chan, int val)
{
  int i;
  iiwu_sp_t* sp;

  /* check the ranges of the arguments */
  if ((chan < 0) || (chan >= IIWU_NUM_CHANNELS)) {
    IIWU_LOG(WARN, "Channel out of range");
    return IIWU_FAILED;     
  }

  /* set the pitch-bend value in the channel */
  iiwu_channel_pitch_bend(synth->channel[chan], val);

  /* tell all synthesis processes on this channel to update their
     synthesis parameters */
  for (i = 0; i < synth->nsp; i++) {
    sp = synth->sp[i];
    if (sp->chan == chan) {
      iiwu_sp_modulate(sp, 0, IIWU_MOD_PITCHWHEEL, val);
    }
  }
  return IIWU_OK;
}

/*
 * iiwu_synth_program_change
 */
int 
iiwu_synth_program_change(iiwu_synth_t* synth, int chan, int prognum)
{
  iiwu_preset_t* preset = NULL;
  iiwu_channel_t* channel;
  unsigned int banknum;

  if (synth->sfont == NULL) {
    return IIWU_OK;
  }

  if ((prognum >= 0) && (prognum < IIWU_NUM_PROGRAMS) &&
      (chan >= 0) && (chan < IIWU_NUM_CHANNELS)) {

    channel = synth->channel[chan];
    banknum = iiwu_channel_get_banknum(channel);

    /* inform the channel of the new program number */
    iiwu_channel_set_prognum(channel, prognum);

    IIWU_LOG(DBG, "program change: chan=%d, bank=%d, prog=%d", chan, banknum, prognum);

    /* special handling of channel 10 (or 9 counting from 0). channel
       10 is the percussion chnnel.  */
    if (channel->channum == 9) {

      /* try to search the drum instrument first */
      preset = iiwu_sfont_get_preset(synth->sfont, banknum | DRUM_INST_MASK, prognum);

      /* if that fails try to search the melodic instrument */
      if (preset == NULL) {
	preset = iiwu_sfont_get_preset(synth->sfont, banknum, prognum);
      }

      iiwu_channel_set_preset(channel, preset);

    } else {
      preset = iiwu_sfont_get_preset(synth->sfont, banknum, prognum);
      iiwu_channel_set_preset(channel, preset);
    }
    return IIWU_OK;
  }

  IIWU_LOG(ERR, "Index out of range");
  return IIWU_FAILED;
}

/*
 * iiwu_synth_bank_select
 */
int 
iiwu_synth_bank_select(iiwu_synth_t* synth, int chan, unsigned int bank)
{
  if ((chan >= 0) && (chan < IIWU_NUM_CHANNELS)) {
    iiwu_channel_set_banknum(synth->channel[chan], bank);
  }
  return IIWU_OK;
}

/*
 * iiwu_synth_program_reset
 *
 * Resend a bank select and a program change for every channel. This
 * function is called mainly after a soundfont has been loaded,
 * unloaded or reloaded.  */
int 
iiwu_synth_program_reset(iiwu_synth_t* synth)
{ 
  int i;
  /* try to set the correct presets */
  for (i = 0; i < 16; i++){
    iiwu_synth_program_change(synth, i, iiwu_channel_get_prognum(synth->channel[i]));
  }
  return IIWU_OK;
}

/*
 * iiwu_synth_set_reverb
 */
int iiwu_synth_set_reverb(iiwu_synth_t* synth, int num)
{
  int i = 0;
  while (revmodel_preset[i].name != NULL) {
    if (i == num) {
      iiwu_revmodel_setroomsize(synth->reverb, revmodel_preset[i].roomsize);
      iiwu_revmodel_setdamp(synth->reverb, revmodel_preset[i].damp);
      iiwu_revmodel_setwidth(synth->reverb, revmodel_preset[i].width);
      iiwu_revmodel_setwet(synth->reverb, revmodel_preset[i].wet);
      iiwu_revmodel_setdry(synth->reverb, revmodel_preset[i].dry);
      return IIWU_OK;
    }
    i++;
  }
  return IIWU_FAILED;
}

/*
 * iiwu_synth_write
 */
int 
iiwu_synth_write(void* data, void* out, int len)
{
  return iiwu_synth_write_lr(data, len, out, 0, 2, out, 1, 2); 
}

/******************************************************

#define COMPRESS      1
#define COMPRESS_X1   4.0
#define COMPRESS_Y1   0.6
#define COMPRESS_X2   10.0
#define COMPRESS_Y2   1.0

  len2 = 2 * len;
  alpha1 = COMPRESS_Y1 / COMPRESS_X1;
  alpha2 = (COMPRESS_Y2 - COMPRESS_Y1) / (COMPRESS_X2 - COMPRESS_X1);
  if (COMPRESS_X1 == COMPRESS_Y1) {
    for (j = 0; j < len2; j++) {
      if (buf[j] > COMPRESS_X1) {
	if (buf[j] > COMPRESS_X2) {
	  buf[j] = COMPRESS_Y2;
	} else {
	  buf[j] = COMPRESS_Y1 + alpha2 * (buf[j] - COMPRESS_X1);
	}
      } else if (buf[j] < -COMPRESS_X1) {
	if (buf[j] < -COMPRESS_X2) {
	  buf[j] = -COMPRESS_Y2;
	} else {
	  buf[j] = -COMPRESS_Y1 + alpha2 * (buf[j] + COMPRESS_X1);
	}
      }
    }
  } else {
    for (j = 0; j < len2; j++) {
      if ((buf[j] >= -COMPRESS_X1) && (buf[j] <= COMPRESS_X1)) {
	buf[j] *= alpha1;
      } else if (buf[j] > COMPRESS_X1) {
	if (buf[j] > COMPRESS_X2) {
	  buf[j] = COMPRESS_Y2;
	} else {
	  buf[j] = COMPRESS_Y1 + alpha2 * (buf[j] - COMPRESS_X1);
	}
      } else {
	if (buf[j] < -COMPRESS_X2) {
	  buf[j] = -COMPRESS_Y2;
	} else {
	  buf[j] = -COMPRESS_Y1 + alpha2 * (buf[j] + COMPRESS_X1);
	}
      }
    }
  }

***************************************************/



/* 
 *  iiwu_synth_write_lr
 */
int 
iiwu_synth_write_lr(void* data, int len, 
		    void* out1, int loff, int lincr, 
		    void* out2, int roff, int rincr)
{
  int i, j, k;
  iiwu_real_t reverb_send, chorus_send; 

  iiwu_synth_t* synth = (iiwu_synth_t*) data;
  iiwu_sp_t* sp;

  /* the output buffers */
  iiwu_real_t* left_out = (iiwu_real_t*) out1;
  iiwu_real_t* right_out = (iiwu_real_t*) out2;

  /* the local, temporary buffers */
  iiwu_real_t* mono_buf = synth->mono_buf;
  iiwu_real_t* left_buf = synth->left_buf;
  iiwu_real_t* right_buf = synth->right_buf;
  iiwu_real_t* reverb_buf = synth->reverb_buf;
  iiwu_real_t* chorus_buf = synth->chorus_buf;

  /* make sure we're playing */
  if (synth->state != IIWU_SYNTH_PLAYING) {
    return 0;
  }

  /* clean the audio buffers */
  IIWU_MEMSET(synth->left_buf, 0, len * sizeof(iiwu_real_t));
  IIWU_MEMSET(synth->right_buf, 0, len * sizeof(iiwu_real_t));
  IIWU_MEMSET(synth->reverb_buf, 0, len * sizeof(iiwu_real_t));
  IIWU_MEMSET(synth->chorus_buf, 0, len * sizeof(iiwu_real_t));

  /* reverb_send and chorus_send test whether any of the synthesis
     processes sends sound to the reverb or chorus module */
  reverb_send = 0;
  chorus_send = 0;

  /* call all playing synthesis processes */
  for (i = 0; i < synth->nsp; i++) {
    sp = synth->sp[i];
    if (_PLAYING(sp)) {
      iiwu_sp_write(sp, len, mono_buf, left_buf, right_buf, reverb_buf, chorus_buf);
      reverb_send += sp->reverb_send;
      chorus_send += sp->chorus_send;
    }
  }

  /* send to reverb */
  if (reverb_send > 0) {
    iiwu_revmodel_processmix(synth->reverb, reverb_buf, left_buf, right_buf, len);
  }

  /* send to chorus */
  if (chorus_send > 0) {
    /* FIXME */
  }

  /* mix all buffers to the final output */
  for (i = 0, j = loff, k = roff; i < len; i++, j += lincr, k += rincr) {
    left_out[j] = (iiwu_real_t) 0.10 * left_buf[i];
    right_out[k] = (iiwu_real_t) 0.10 * right_buf[i];
  }

  return 0;  
}


/*
 * iiwu_synth_alloc_sp
 */
iiwu_sp_t* 
iiwu_synth_alloc_sp(iiwu_synth_t* synth, iiwu_channel_t* chan, int key, int vel)
{
  int i, k;
  unsigned char ch;

  /* check if there's an available synthesis process */
  for (i = 0; i < synth->nsp; i++) {
    if (_AVAILABLE(synth->sp[i])) {
      synth->sp[i]->chan = chan->channum;
      synth->sp[i]->key = (unsigned char) key;
      synth->sp[i]->vel = (unsigned char) vel;
      synth->sp[i]->channel = chan;
      return synth->sp[i];
    }
  }

  /* No synthesis process available: steal one. */
  for (k = 0; k < IIWU_NUM_CHANNELS; k++) {

    /* We steal a voice according to the priorities of the
       channels. (DLS specifications) */
    ch = chan_priority[k];

    /* first, try to find a key that's been released */
    for (i = 0; i < synth->nsp; i++) {
      if ((synth->sp[i]->chan == ch) && _RELEASED(synth->sp[i])) {
	iiwu_sp_off(synth->sp[i]);
	synth->sp[i]->chan = chan->channum;
	synth->sp[i]->key = (unsigned char) key;
	synth->sp[i]->vel = (unsigned char) vel;
	synth->sp[i]->channel = chan;
	return synth->sp[i];
      }
    }

    /* second, go for any key on this channel */
    for (i = 0; i < synth->nsp; i++) {
      if (synth->sp[i]->chan == ch) {
	iiwu_sp_off(synth->sp[i]);
	synth->sp[i]->chan = chan->channum;
	synth->sp[i]->key = (unsigned char) key;
	synth->sp[i]->vel = (unsigned char) vel;
	synth->sp[i]->channel = chan;
	return synth->sp[i];
      }
    }
  }
  IIWU_LOG(WARN, "Max polyphony reached (chan=%d,key=%d)", chan->channum, key);

  return NULL;
}


/*
 * iiwu_synth_sfload
 */
int 
iiwu_synth_sfload(iiwu_synth_t* synth, const char* filename)
{
  iiwu_sfont_t* sfont;

  if (filename == NULL) {
    return IIWU_FAILED;
  }

  /* create the object */
  sfont = new_iiwu_sfont();
  if (sfont == NULL) {
    return IIWU_FAILED;
  }

  /* and load the sounfont file */
  if (iiwu_sfont_load(sfont, (char*) filename) != 0) {
    delete_iiwu_sfont(sfont);
    return IIWU_FAILED;    
  }

  /* insert the sfont as the first on the list */
  if (synth->sfont == NULL) {
    synth->sfont = sfont;
  } else {
    sfont->next = synth->sfont;
    synth->sfont = sfont;
  }

  /* reset the presets for all channels */
  iiwu_synth_program_reset(synth);
    
  return IIWU_OK;
}


/* iiwu_synth_sfcount
 *
 * Returns the number of loaded SoundFonts 
 */
int 
iiwu_synth_sfcount(iiwu_synth_t* synth)
{
  int count = 0;
  iiwu_sfont_t* sfont = synth->sfont;
  while (sfont != NULL) {
    count++;
    sfont = sfont->next;
  }
  return count;
}

/* iiwu_synth_get_font
 *
 * Returns SoundFont num 
 */
iiwu_sfont_t* 
iiwu_synth_get_font(iiwu_synth_t* synth, int num)
{
  int count = 0;
  iiwu_sfont_t* sfont = synth->sfont;
  while (sfont != NULL) {
    if (count == num) {
      return sfont;
    }
    count++;
    sfont = sfont->next;
  }
  return NULL;  
}

/* iiwu_synth_sfname
 *
 * Returns the name of the SoundFont 
 */
char* 
iiwu_synth_sfname(iiwu_synth_t* synth, int num)
{
  int count = 0;
  iiwu_sfont_t* sfont = iiwu_synth_get_font(synth, num);
  if (sfont == NULL) {
    return NULL;
  }
  IIWU_SPRINTF(synth->outbuf, "%s", sfont->file);
  return synth->outbuf;
}


/*
 * iiwu_synth_first_preset
 */
char* 
iiwu_synth_first_preset(iiwu_synth_t* synth, int num)
{
  iiwu_sfont_t* sfont;
  iiwu_preset_t* preset;
  sfont = iiwu_synth_get_font(synth, num);

  if (sfont == NULL) {
    return NULL;
  }
  preset = iiwu_sfont_first_preset(sfont);
  if (preset == NULL) {
    return NULL;
  }
  IIWU_SPRINTF(synth->outbuf, "%03d-%03d %s", preset->bank, preset->num, preset->name);
  return synth->outbuf;
}


/*
 * iiwu_synth_next_preset
 */
char* 
iiwu_synth_next_preset(iiwu_synth_t* synth, int num)
{
  iiwu_sfont_t* sfont;
  iiwu_preset_t* preset;
  sfont = iiwu_synth_get_font(synth, num);
  if (sfont == NULL) {
    return NULL;
  }
  preset = iiwu_sfont_next_preset(sfont);
  if (preset == NULL) {
    return NULL;
  }
  IIWU_SPRINTF(synth->outbuf, "%03d-%03d %s", preset->bank, preset->num, preset->name);
  return synth->outbuf;
}


/*
 * iiwu_synth_get_channel_preset
 */
char* 
iiwu_synth_get_channel_preset(iiwu_synth_t* synth, int chan)
{
  iiwu_preset_t* preset = NULL;
  if ((chan >= 0) && (chan < IIWU_NUM_CHANNELS)) {
    preset = iiwu_channel_get_preset(synth->channel[chan]);
  }
  if (preset == NULL) {
    return NULL;
  }
  IIWU_SPRINTF(synth->outbuf, "%03d-%03d %s", preset->bank, preset->num, preset->name);
  return synth->outbuf;
}


/***************************************************************
 *
 *                           CHANNEL
 */

/*
 * new_iiwu_channel
 */
iiwu_channel_t* 
new_iiwu_channel(int num)
{
#define SETCC(n,v)  chan->cc[n] = v
  int i;
  iiwu_channel_t* chan;
  chan = IIWU_NEW(iiwu_channel_t);
  if (chan == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  chan->channum = num;
  chan->prognum = (num == 9)? 0 : num;
  chan->banknum = (num == 9)? 128 : 0;
  chan->preset = NULL;
  for (i = 0; i < 128; i++) {
    SETCC(i, 0);
  }
  chan->key_pressure = 0;
  chan->channel_pressure = 0;
  chan->pitch_bend = 0;
  chan->pitch_wheel_sensitivity = 2; /* two semi-tones */
  SETCC(7, 127);  /* channel volume / initial attenuation */
  SETCC(10, 64);  /* pan controller */  
  return chan;
}

/*
 * delete_iiwu_channel
 */
int 
delete_iiwu_channel(iiwu_channel_t* chan)
{
  IIWU_FREE(chan);
  return IIWU_OK;
}

/*
 * iiwu_channel_set_preset
 */
int 
iiwu_channel_set_preset(iiwu_channel_t* chan, iiwu_preset_t* preset)
{
  chan->preset = preset;
  return IIWU_OK;
}

/*
 * iiwu_channel_get_preset
 */
iiwu_preset_t* 
iiwu_channel_get_preset(iiwu_channel_t* chan)
{
  return chan->preset;
}

/*
 * iiwu_channel_get_banknum
 */
unsigned int 
iiwu_channel_get_banknum(iiwu_channel_t* chan)
{
  return chan->banknum;
}

/*
 * iiwu_channel_set_prognum
 */
int 
iiwu_channel_set_prognum(iiwu_channel_t* chan, int prognum)
{
  chan->prognum = prognum;
  return IIWU_OK;
}

/*
 * iiwu_channel_get_prognum
 */
int 
iiwu_channel_get_prognum(iiwu_channel_t* chan)
{
  return chan->prognum;
}

/*
 * iiwu_channel_set_banknum
 */
int 
iiwu_channel_set_banknum(iiwu_channel_t* chan, unsigned int banknum)
{
  chan->banknum = banknum;
  return IIWU_OK;
}
/*
 * iiwu_channel_noteon
 */
int 
iiwu_channel_noteon(iiwu_channel_t* chan, iiwu_synth_t* synth, int key, int vel) 
{
  iiwu_preset_t* preset;
  iiwu_preset_zone_t *preset_zone, *global_preset_zone, *p;
  iiwu_inst_t* inst;
  iiwu_inst_zone_t *inst_zone, *global_inst_zone, *z;
  iiwu_sample_t* sample;
  iiwu_sp_t* sp;
  int i;

  /* make sure this channel has a preset */
  preset = chan->preset;
  if (preset == NULL) {
    IIWU_LOG(DBG, "No preset on channel %d", chan->channum);
    return IIWU_FAILED; 
  } 

  global_preset_zone = iiwu_preset_get_global_zone(preset);

  /* run thru all the zones of this preset */
  preset_zone = iiwu_preset_get_zone(preset);
  while (preset_zone != NULL) {

    /* check if the note falls into the key and velocity range of this
       preset */
    if (iiwu_preset_zone_inside_range(preset_zone, key, vel)) {

      inst = iiwu_preset_zone_get_inst(preset_zone);
      global_inst_zone = iiwu_inst_get_global_zone(inst);

      /* run thru all the zones of this instrument */
      inst_zone = iiwu_inst_get_zone(inst);
      while (inst_zone != NULL) {

	/* make sure this instrument zone has a valid sample */
	sample = iiwu_inst_zone_get_sample(inst_zone);
	if (iiwu_sample_in_rom(sample) || (sample == NULL)) {
	  inst_zone = iiwu_inst_zone_next(inst_zone);
	  continue;
	}

	/* check if the note falls into the key and velocity range of this
	   instrument */
	if (iiwu_inst_zone_inside_range(inst_zone, key, vel) && (sample != NULL)) {

	  /* this is a good zone. allocate a new synthesis process and
             initialize it */
	  sp = iiwu_synth_alloc_sp(synth, chan, key, vel);
	  if (sp == NULL) {
	    return IIWU_FAILED; 
	  }
	  if (iiwu_sp_init(sp, chan, sample) != IIWU_OK) {
	    IIWU_LOG(WARN, "Failed to initialize sp");
	    return IIWU_FAILED;
	  }

	  /* set the generators of the synthesis process. the
             generators of the global instrument zone are taken as
             default values. */
	  if (global_inst_zone != NULL) {
	    z = global_inst_zone;
	    for (i = 0; i < GEN_LAST; i++) {
	      if (z->gen[i].flags) {
		sp->gen[i].val = z->gen[i].val;
		sp->gen[i].flags = GEN_SET;
	      }
	    }
	  }

	  /* the generators of the instrument zone have authority and
             erase the default values. */
	  z = inst_zone;
	  for (i = 0; i < GEN_LAST; i++) {
	    if (z->gen[i].flags) {
	      sp->gen[i].val = z->gen[i].val;
	      sp->gen[i].flags = GEN_SET;
	    }
	  }

	  /* the generators of the global preset zone add to the
             existing values. */
	  if (global_preset_zone != NULL) {
	    p = global_preset_zone;
	    for (i = 0; i < GEN_LAST; i++) {
	      /* FIXEM: some generators should not be tested at the
                 preset level, like the "override root key" */
	      if (p->gen[i].flags) {
		sp->gen[i].val += p->gen[i].val;
		sp->gen[i].flags = GEN_SET;
	      }
	    }
	  }

	  /* the generators of the preset zone add to the existing
             values. */
	  p = preset_zone;
	  for (i = 0; i < GEN_LAST; i++) {
	    if (p->gen[i].flags) {
	      sp->gen[i].val += p->gen[i].val;
	      sp->gen[i].flags = GEN_SET;
	    }
	  }

	  /* add the default modulators to the synthesis process. */
	  iiwu_sp_add_mod(sp, &default_pan_mod);
	  iiwu_sp_add_mod(sp, &default_att_mod);
	  iiwu_sp_add_mod(sp, &default_pitch_bend_mod);

	  /* calculate all the runtime synthesis parameters. */
	  iiwu_sp_optimize(sp);

	  /* add the synthesis process to the synthesis loop. */
	  iiwu_sp_start(sp);
	}
	inst_zone = iiwu_inst_zone_next(inst_zone);
      }
    }
    preset_zone = iiwu_preset_zone_next(preset_zone);
  }
  return IIWU_OK;
}

/*
 * iiwu_channel_cc
 */
int 
iiwu_channel_cc(iiwu_channel_t* chan, int num, int value)
{
  if ((num >= 0) && (num < 128)) {
    chan->cc[num] = value;
  }
  return IIWU_OK;
}

/*
 * iiwu_channel_get_cc
 */
int 
iiwu_channel_get_cc(iiwu_channel_t* chan, int num)
{
  return ((num >= 0) && (num < 128))? chan->cc[num] : 0;
}

/*
 * iiwu_channel_pitch_bend
 */
int 
iiwu_channel_pitch_bend(iiwu_channel_t* chan, int val)
{
    chan->pitch_bend = val;
    return IIWU_OK;
}

/*
 * iiwu_channel_get_num
 */
int 
iiwu_channel_get_num(iiwu_channel_t* chan)
{
  return chan->channum;
}

/***************************************************************
 *
 *                             SP
 */

/*
 * new_iiwu_sp
 */ 
iiwu_sp_t* 
new_iiwu_sp()
{
  int i;
  iiwu_sp_t* sp;
  sp = IIWU_NEW(iiwu_sp_t);
  if (sp == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  sp->status = IIWU_SP_CLEAN;
  sp->chan = NO_CHANNEL;
  sp->key = 0;
  sp->vel = 0;
  sp->channel = NULL;
  sp->sample = NULL;
/*    sp->monobuf = IIWU_ARRAY(iiwu_real_t, IIWU_BUFSIZE); */
/*    if (sp->monobuf == NULL) { */
/*      IIWU_LOG(ERR, "Out of memory");      */
/*      IIWU_FREE(sp); */
/*      return NULL; */
/*    } */
/*    for (i = 0; i < IIWU_BUFSIZE; i++) { */
/*      sp->monobuf[i] = 0; */
/*    } */
/*    sp->rev = new_iiwu_revmodel(); */
/*    if (sp->rev == NULL) { */
/*      IIWU_FREE(sp->monobuf); */
/*      IIWU_FREE(sp); */
/*      return NULL; */
/*    } */
  iiwu_sp_init(sp, NULL, NULL);
  return sp;
}

/*
 * delete_iiwu_sp
 */
int 
delete_iiwu_sp(iiwu_sp_t* sp)
{
  if (sp == NULL) {
    return IIWU_OK;
  }
/*    if (sp->monobuf != NULL) { */
/*      IIWU_FREE(sp->monobuf); */
/*    } */
/*    if (sp->rev != NULL) { */
/*      delete_iiwu_revmodel(sp->rev); */
/*    } */
  IIWU_FREE(sp);
  return IIWU_OK;
}

/* iiwu_sp_init
 *
 * Initialize the synthesis process and sets all its generators to
 * their initial value.
 */
int 
iiwu_sp_init(iiwu_sp_t* sp, iiwu_channel_t* channel, iiwu_sample_t* sample)
{
  sp->attenuation = 0;
  sp->pitch = 6900;
  sp->root_pitch = 440.0f;
  sp->channel = channel;
  sp->mod_count = 0;
  sp->sample = sample;
  sp->ticks = 0;
  sp->debug = 0;

  iiwu_phase_set_int(sp->phase, (sample != NULL)? sample->start : 0);

  /* sample and loop start and end points */
  sp->start_offset = 0;
  sp->end_offset = 0;
  sp->loop_start_offset = 0;
  sp->loop_end_offset = 0;

  /* vol env */
  sp->volenv_state = IIWU_ENV_SKIP;
  sp->volenv_val = 0.0f;
  sp->volenv_delay = 0;
  sp->volenv_attack = 0;
  sp->volenv_hold = 0;
  sp->volenv_decay = 0;
  sp->volenv_sustain = 0;
  sp->volenv_release = 0;

  /* mod env */
  sp->modenv_state = IIWU_ENV_SKIP;
  sp->modenv_val = 0.0f;
  sp->modenv_delay = 0;
  sp->modenv_attack = 0;
  sp->modenv_hold = 0;
  sp->modenv_decay = 0;
  sp->modenv_sustain = 0;
  sp->modenv_release = 0;
  sp->modenv_to_fc = 0;
  sp->modenv_to_pitch = 0;

  /* mod lfo */
  sp->modlfo_val = 0.0;
  sp->modlfo_delay = 0;
  sp->modlfo_incr = (iiwu_real_t) (4.0 * IIWU_BUFSIZE * 8.176 / 44100.0); /* 8.176 Hz, by default */
  sp->modlfo_to_fc = 0;
  sp->modlfo_to_pitch = 0;
  sp->modlfo_to_vol = 0;

  /* vib lfo */
  sp->viblfo_val = 0.0f;
  sp->viblfo_to_pitch = 0;
  sp->viblfo_delay = 0;
  sp->viblfo_incr = (iiwu_real_t) (4.0 * IIWU_BUFSIZE * 8.176 / 44100.0); /* 8.176 Hz, by default */

  /* filter */
  sp->fres = 20001.0f;
  sp->q = 0.0f;
  sp->w1 = 0.0f;
  sp->w2 = 0.0f; 

  /* pan */
  sp->pan = 0;

  /* reverb */
  sp->reverb_send = 1.0;

  /* chorus */
  sp->chorus_send = 0;

  /* set all the ganarators to their initial value */
  iiwu_gen_init_array(&sp->gen[0]);

  return IIWU_OK;
}

/*  #define USE_TRUNCATION */
/*  #define USE_LINEAR_INTERPOLATION */
#define USE_CUBIC_INTERPOLATION

#if defined(USE_TRUNCATION)
#define INTERPOL(_o,_a,_b,_i,_c)   _o=_a*_b[_i]
#elif defined(USE_LINEAR_INTERPOLATION)
#define INTERPOL(_o,_a,_b,_i,_c)      { \
  _o = _a * (_c->a0 * _b[_i] + _c->a1 * _b[_i+1]); \
}
#elif defined(USE_CUBIC_INTERPOLATION)
#define INTERPOL(_o,_a,_b,_i,_c) { \
  _o = _a * (_c->a0 * _b[_i-1] + _c->a1 * _b[_i] + _c->a2 * _b[_i+1] + _c->a3 * _b[_i+2]); \
}
#else
#error No interpolation defined
#endif




/*
 * iiwu_sp_write
 */
int 
iiwu_sp_write(iiwu_sp_t* sp, int len, iiwu_real_t* mono, 
	      iiwu_real_t* left, iiwu_real_t* right, 
	      iiwu_real_t* reverb_buf, iiwu_real_t* chorus_buf)
{
  int i, j, k;
  iiwu_sample_t* sample;
  short* data;
  iiwu_phase_t phase, start, end, loopend, loopstart;
  iiwu_phase_t phase_incr, end_phase;
  iiwu_real_t incr;
  iiwu_real_t amp_left, amp_right;
  iiwu_real_t amp;
  iiwu_interp_coeff_t* coeff = NULL;
  int index;
  iiwu_real_t fract;
  double fres;

  /* make sure we're playing and that we have sample data */
  if (!_PLAYING(sp)) {
    return IIWU_OK;
  }
  sample = sp->sample;
  if (sample == NULL) {
    sp->status = IIWU_SP_OFF;
    return IIWU_OK;
  }
  data = sample->data;

  /* start, end and loop points */
  iiwu_phase_set_int(start, sp->start_offset);
  iiwu_phase_set_int(end, sp->end_offset);
  iiwu_phase_set_int(loopstart, sp->loop_start_offset);
  iiwu_phase_set_int(loopend, sp->loop_end_offset);

  /* amplitude envelope */
  if (sp->volenv_state == IIWU_ENV_SKIP) {
    sp->volenv_val = 1.0;
  } else if (sp->volenv_state == IIWU_ENV_SUSTAIN) {
    sp->volenv_val = sp->volenv_sustain;
  } else if (sp->volenv_state == IIWU_ENV_RELEASE) {
    sp->volenv_val += sp->volenv_release;
    if (sp->volenv_val <= 0.0) {
      sp->volenv_val = 0.0;
      sp->volenv_state = IIWU_ENV_OFF;
      sp->status = IIWU_SP_OFF;
    }
  } else {
    if ((sp->volenv_state == IIWU_ENV_DELAY) && (sp->ticks >= sp->volenv_delay)) {
      sp->volenv_state = IIWU_ENV_ATTACK;
    }
    if (sp->volenv_state == IIWU_ENV_ATTACK) {
      sp->volenv_val += sp->volenv_attack;
      if (sp->volenv_val >= 1.0) {
	sp->volenv_val = 1.0;
	sp->volenv_state = IIWU_ENV_HOLD;
      }
    }
    if ((sp->volenv_state == IIWU_ENV_HOLD) && (sp->ticks >= sp->volenv_hold)) {
      sp->volenv_state = IIWU_ENV_DECAY;
    }
    if (sp->volenv_state == IIWU_ENV_DECAY) {
      sp->volenv_val += sp->volenv_decay;
      if (sp->volenv_val <= sp->volenv_sustain) {
	sp->volenv_val = sp->volenv_sustain;
	sp->volenv_state = IIWU_ENV_SUSTAIN;
      }
    }
  }

  /* no need to go further if the final amplitude is zero */
  if (sp->volenv_val == 0.0) {
    goto post_process;
  }
  
  /* modulation envelope */
  if (sp->modenv_state == IIWU_ENV_SKIP) {
    sp->modenv_val = 1.0;
  } else if (sp->modenv_state == IIWU_ENV_SUSTAIN) {
    sp->modenv_val = sp->modenv_sustain;
  } else if (sp->modenv_state == IIWU_ENV_RELEASE) {
    sp->modenv_val += sp->modenv_release;
    if (sp->modenv_val <= 0.0) {
      sp->modenv_val = 0.0;
      sp->modenv_state = IIWU_ENV_OFF;
    }
  } else {
    if ((sp->modenv_state == IIWU_ENV_DELAY) && (sp->ticks >= sp->modenv_delay)) {
      sp->modenv_state = IIWU_ENV_ATTACK;
    }
    if (sp->modenv_state == IIWU_ENV_ATTACK) {
      sp->modenv_val += sp->modenv_attack;
      if (sp->modenv_val >= 1.0) {
	sp->modenv_val = 1.0;
	sp->modenv_state = IIWU_ENV_HOLD;
      }
    }
    if ((sp->modenv_state == IIWU_ENV_HOLD) && (sp->ticks >= sp->modenv_hold)) {
      sp->modenv_state = IIWU_ENV_DECAY;
    }
    if (sp->modenv_state == IIWU_ENV_DECAY) {
      sp->modenv_val += sp->modenv_decay;
      if (sp->modenv_val <= sp->modenv_sustain) {
	sp->modenv_val = sp->modenv_sustain;
	sp->modenv_state = IIWU_ENV_SUSTAIN;
      }
    }
  }
  
  /* mod lfo */
  if (sp->ticks >= sp->modlfo_delay) {
    sp->modlfo_val += sp->modlfo_incr;
    if (sp->modlfo_val > 1.0) {
      sp->modlfo_incr = -sp->modlfo_incr;
      sp->modlfo_val = (iiwu_real_t) 2.0 - sp->modlfo_val;
    } else if (sp->modlfo_val < -1.0) {
      sp->modlfo_incr = -sp->modlfo_incr;
      sp->modlfo_val = (iiwu_real_t) -2.0 - sp->modlfo_val;
    }
  }
  
  /* vib lfo */
  if (sp->ticks >= sp->viblfo_delay) {
    sp->viblfo_val += sp->viblfo_incr;
    if (sp->viblfo_val > (iiwu_real_t) 1.0) {
      sp->viblfo_incr = -sp->viblfo_incr;
      sp->viblfo_val = (iiwu_real_t) 2.0 - sp->viblfo_val;
    } else if (sp->viblfo_val < -1.0) {
      sp->viblfo_incr = -sp->viblfo_incr;
      sp->viblfo_val = (iiwu_real_t) -2.0 - sp->viblfo_val;
    }
  }
  
  /* calculate final amplitude 
   * - initial gain
   * - amplitude envelope
   * - dividing by 32768 to scale samples down to [-1, 1] interval 
   */
  if (sp->volenv_state == IIWU_ENV_ATTACK) {
    amp = iiwu_cb2amp(sp->attenuation + sp->modlfo_val * sp->modlfo_to_vol) * sp->volenv_val / 32768.0;
  } else {
    amp = iiwu_cb2amp(sp->attenuation
		      + 960 * (1.0 - sp->volenv_val)
		      + sp->modlfo_val * sp->modlfo_to_vol) / 32768.0;
  }
  
  /* calculate final pitch, or phase increment */
  incr = iiwu_ct2hz(sp->pitch
		    + sp->modlfo_val * sp->modlfo_to_pitch
		    + sp->viblfo_val * sp->viblfo_to_pitch
		    + sp->modenv_val * sp->modenv_to_pitch) / sp->root_pitch;

  iiwu_phase_set(phase, sp->phase);
  iiwu_phase_set_float(phase_incr, incr);
  
  /* calculate the phase at the end of the loop */
  iiwu_phase_set_float(end_phase, incr * len);
  iiwu_phase_incr(end_phase, phase);

  /* to loop or not to loop */
  if ((_SAMPLEMODE(sp) == IIWU_LOOP_DURING_RELEASE)
      || ((_SAMPLEMODE(sp) == IIWU_LOOP) 
	  && (sp->volenv_state < IIWU_ENV_RELEASE))) {

    if (iiwu_phase_lt(end_phase, loopend)) { 

      /* we're not close to the loop point, so don't check */ 
      for (i = 0; i < len; i++) {
	index = iiwu_phase_index(phase);
#ifndef USE_TRUNCATION
	coeff = &interp_coeff[iiwu_phase_interp_index(phase)];
#endif
	INTERPOL(mono[i], amp, data, index, coeff);
	iiwu_phase_incr(phase, phase_incr);
      }

    } else { 

      /* might be looping, so check */
      for (i = 0; i < len; i++) {
	index = iiwu_phase_index(phase);
#ifndef USE_TRUNCATION
	coeff = &interp_coeff[iiwu_phase_interp_index(phase)];
#endif
	INTERPOL(mono[i], amp, data, index, coeff);
	iiwu_phase_incr(phase, phase_incr);

	/* if we passed the loopend, wrap around as
           phase += loopstart - loopend */
	if (iiwu_phase_gt(phase, loopend)) {
	  iiwu_phase_decr(phase, loopend);
	  iiwu_phase_incr(phase, loopstart);
	}
      }
    } 

  } else {

    if (iiwu_phase_lt(end_phase, end)) { 
      /* no fear of end of buffer */
      for (i = 0; i < len; i++) {
	index = iiwu_phase_index(phase);
#ifndef USE_TRUNCATION
	coeff = &interp_coeff[iiwu_phase_interp_index(phase)];
#endif
	INTERPOL(mono[i], amp, data, index, coeff);
	iiwu_phase_incr(phase, phase_incr);
      }

    } else {

      /* check for end of buffer */
      for (i = 0; i < len; i++) {

	if (iiwu_phase_lt(phase, end)) {
	  index = iiwu_phase_index(phase);
#ifndef USE_TRUNCATION
	  coeff = &interp_coeff[iiwu_phase_interp_index(phase)];
#endif
	  INTERPOL(mono[i], amp, data, index, coeff);
	  iiwu_phase_incr(phase, phase_incr);

	} else {
	  /* if it's passed the end, turn the synthesis process off */
	  if (sp->status != IIWU_SP_OFF) {
	    sp->volenv_state = IIWU_ENV_OFF;
	    sp->status = IIWU_SP_OFF;
	  }
	  mono[i] = 0;
	}
      }
    }
  } 

  /* copy back the phase for the next cycle */
  iiwu_phase_set(sp->phase, phase);
  
  /* pass the sound thru the resonant filter */
  fres = iiwu_ct2hz(sp->fres
		    + sp->modlfo_val * sp->modlfo_to_fc
		    + sp->modenv_val * sp->modenv_to_fc);

  /* I'm using 19900 as upper frequency because the default 13500
     cents returns 19912 Hz */
  if ((fres <= 19900.0) && (sp->q > 0.0)) {
    /* calculate filter coefficients
     *
     * FIXME: 1) i've picked up an old algorithm i've been using for 8
     * (?)  years. i'm not sure that the q-factor in soundfont units
     * corresponds to what this calculation is expecting. i'll use it
     * till i have something better. 2) the DLS specification is a
     * little more precise about the filter characteristics but they
     * use a not-so-standard-to-me definition of fc.
     * */
    double x, kb, q;
    iiwu_real_t w, w1, w2, y, b1, b2, gain;
    q = (double) sp->q;
    x = PI * fres / 44100.0;  
    kb = cos(x) / sin(x);   /* is there a way to avoid the cos and sin functions? */
    x = 1.0 + kb * kb + kb / q;
    b2 = (iiwu_real_t) ((1.0 + kb * kb - kb / q) / x);
    b1 = (iiwu_real_t) (2.0 * (1.0 - kb * kb) / x);
    gain = (iiwu_real_t) (0.25 * (1.0 + b1 + b2));
    w1 = sp->w1;
    w2 = sp->w2;
    /* do the actual filtering */
    for (i = 0; i < len; i++) {
      w = gain * mono[i] - b1 * w1 - b2 * w2;  
      y = w + w1 + w1 + w2;  
      w2 = w1;  
      w1 = w;  
      mono[i] = y; 
    }
    sp->w1 = w1;
    sp->w2 = w2;
  }

  /* copy the final samples in the left, right, reverb and chorus
     buffers */
  
  /* panning */
  amp_left = iiwu_pan(sp->pan, 1);
  amp_right = iiwu_pan(sp->pan, 0);
  for (i = 0; i < len; i++) {
    left[i] += amp_left * mono[i];
    right[i] += amp_right * mono[i];
  }

  /* reverb */
  if (sp->reverb_send > 0) {
    for (i = 0; i < len; i++) {
      reverb_buf[i] += sp->reverb_send * mono[i];
    }
  }

  /* chorus */
  if (sp->chorus_send > 0) {
    for (i = 0; i < len; i++) {
      chorus_buf[i] += sp->chorus_send * mono[i];
    }
  }

 post_process:
  sp->ticks += len;
  return IIWU_OK;
}




/*
 * iiwu_sp_get_channel
 */
iiwu_channel_t* 
iiwu_sp_get_channel(iiwu_sp_t* sp)
{
  return sp->channel;
}

/*
 * iiwu_sp_start
 */
int iiwu_sp_start(iiwu_sp_t* sp)
{
  sp->status = IIWU_SP_ON;
  return IIWU_OK;
}

/*
 * iiwu_sp_optimize
 *
 * in this function we calculate the values of all the parameters. the
 * parameters are converted to their most useful unit for the DSP
 * algorithm, for exemple, number of samples instead of
 * timecents. Some parameters keep their "perceptual" unit and
 * conversion will be done in the DSP function. This is the case, for
 * exemple, for the pitch since it is modulated by the controllers in
 * cents. */
int iiwu_sp_optimize(iiwu_sp_t* sp)
{
  float x;
  int bufsize;
  iiwu_sample_t* sample;
  sample = sp->sample;
  bufsize = IIWU_BUFSIZE;
  
  /* volume envelope
   *
   * - delay and hold times are converted to absolute number of samples
   * - sustain is converted to its absolute value
   * - attack, decay and release are converted to their increment per sample
   */
  sp->volenv_sustain = iiwu_cb2amp(sp->gen[GEN_VOLENVSUSTAIN].val);

  /* first, calculate relative sample times */
  sp->volenv_delay = 44100 * iiwu_tc2sec(sp->gen[GEN_VOLENVDELAY].val);
  sp->volenv_attack = 44100 * iiwu_tc2sec(sp->gen[GEN_VOLENVATTACK].val);
  sp->volenv_hold = 44100 * iiwu_tc2sec(sp->gen[GEN_VOLENVHOLD].val);
  sp->volenv_decay = 44100 * iiwu_tc2sec(sp->gen[GEN_VOLENVDECAY].val);
  sp->volenv_release = 44100 * iiwu_tc2sec(sp->gen[GEN_VOLENVRELEASE].val);

  /* calc absolute time for hold */
  sp->volenv_hold += sp->volenv_delay + sp->volenv_attack;

  /* calc increment for attack, decay and release over the span of a buffer */
  if (sp->volenv_attack > 0) {
    sp->volenv_attack = (iiwu_real_t) bufsize / sp->volenv_attack;
  } else {
    sp->volenv_attack = 1.0;
  }
  if (sp->volenv_decay > 0) {
    sp->volenv_decay = (iiwu_real_t) -bufsize / sp->volenv_decay;
  } else {
    sp->volenv_decay = -1.0;
  }
  if (sp->volenv_release > 0) {
    sp->volenv_release = (iiwu_real_t) -bufsize / sp->volenv_release;
  } else {
    sp->volenv_release = -1.0;
  }

  /* do we have to calculate an envelope? */
  sp->volenv_state = (sp->gen[GEN_VOLENVDELAY].flags || 
		      sp->gen[GEN_VOLENVATTACK].flags ||
		      sp->gen[GEN_VOLENVHOLD].flags ||
		      sp->gen[GEN_VOLENVDECAY].flags ||
		      sp->gen[GEN_VOLENVSUSTAIN].flags ||
		      sp->gen[GEN_VOLENVRELEASE].flags) ? IIWU_ENV_DELAY : IIWU_ENV_SKIP;
  
  /* modulation envelope
   *
   * - delay and hold times are converted to absolute number of samples
   * - sustain is converted to its absolute value
   * - attack, decay and release are converted to their increment per sample
   */
  sp->modenv_sustain = 1.0 - 0.001 * sp->gen[GEN_MODENVSUSTAIN].val;
  if (sp->modenv_sustain < 0) {
    sp->modenv_sustain = 0;
  } else if (sp->modenv_sustain > 1) {
    sp->modenv_sustain = 1;
  }

  /* first, calculate relative sample times */
  sp->modenv_delay = 44100 * iiwu_tc2sec(sp->gen[GEN_MODENVDELAY].val);
  sp->modenv_attack = 44100 * iiwu_tc2sec(sp->gen[GEN_MODENVATTACK].val);
  sp->modenv_hold = 44100 * iiwu_tc2sec(sp->gen[GEN_MODENVHOLD].val);
  sp->modenv_decay = 44100 * iiwu_tc2sec(sp->gen[GEN_MODENVDECAY].val);
  sp->modenv_release = 44100 * iiwu_tc2sec(sp->gen[GEN_MODENVRELEASE].val);

  /* calc absolute time for hold */
  sp->modenv_hold += sp->modenv_delay + sp->modenv_attack;

  /* calc increment for attack, decay and release over the span of a buffer */
  if (sp->modenv_attack > 0) {
    sp->modenv_attack = (iiwu_real_t) bufsize / sp->modenv_attack;
  } else {
    sp->modenv_attack = 1.0;
  }
  if (sp->modenv_decay > 0) {
    sp->modenv_decay = (iiwu_real_t) -bufsize / sp->modenv_decay;
  } else {
    sp->modenv_decay = -1.0;
  }
  if (sp->modenv_release > 0) {
    sp->modenv_release = (iiwu_real_t) -bufsize / sp->modenv_release;
  } else {
    sp->modenv_release = -1.0;
  }

  /* do we have to calculate an envelope? */
  sp->modenv_state = (sp->gen[GEN_MODENVDELAY].flags || 
		      sp->gen[GEN_MODENVATTACK].flags ||
		      sp->gen[GEN_MODENVHOLD].flags ||
		      sp->gen[GEN_MODENVDECAY].flags ||
		      sp->gen[GEN_MODENVSUSTAIN].flags ||
		      sp->gen[GEN_MODENVRELEASE].flags) ? IIWU_ENV_DELAY : IIWU_ENV_SKIP;
  sp->modenv_to_pitch = sp->gen[GEN_MODENVTOPITCH].val;
  sp->modenv_to_fc = sp->gen[GEN_MODENVTOFILTERFC].val;
  
  /* pitch */
  /* the GEN_PITCH is a hack to fit the pitch bend controller into the
     modulator paradigm */
  sp->gen[GEN_PITCH].val = 100 * sp->key;
/* FIXME   sp->pitch = _GEN(sp, GEN_PITCH) + 100 * _GEN(sp, GEN_COARSETUNE) + _GEN(sp, GEN_FINETUNE); */
  sp->pitch = _GEN(sp, GEN_PITCH);
  
  /* root key */
  if (sp->gen[GEN_OVERRIDEROOTKEY].val > -1) {
    sp->root_pitch = 100 * sp->gen[GEN_OVERRIDEROOTKEY].val;
  } else {
    sp->root_pitch = sp->sample->origpitch * 100 + sp->sample->pitchadj;
  }
  sp->root_pitch = iiwu_ct2hz(sp->root_pitch);
  if (sample != NULL) {
    sp->root_pitch *= (iiwu_real_t) 44100.0 / sample->samplerate;
  }

  x = iiwu_ct2hz(sp->pitch) / sp->root_pitch; 

/*    printf("key=%d, coarse tune=%f, fine tune=%f, pitch=%f, root=%f, freq=%f\n",  */
/*  	 sp->key, _GEN(sp, GEN_COARSETUNE), _GEN(sp, GEN_FINETUNE), sp->pitch, sp->root_pitch, x); */
  
  /* sample start and ends points */
  if (sample != NULL) {
    sp->start_offset = (sample->start
			+ sp->gen[GEN_STARTADDROFS].val
			+ 32768 * sp->gen[GEN_STARTADDRCOARSEOFS].val);

    sp->end_offset = (sample->end
		      + sp->gen[GEN_ENDADDROFS].val
		      + 32768 * sp->gen[GEN_ENDADDRCOARSEOFS].val);

    sp->loop_start_offset = (sample->loopstart
			     + sp->gen[GEN_STARTLOOPADDROFS].val
			     + 32768 * sp->gen[GEN_STARTLOOPADDRCOARSEOFS].val);

    sp->loop_end_offset = (sample->loopend
			   + sp->gen[GEN_ENDLOOPADDROFS].val
			   + 32768 * sp->gen[GEN_ENDLOOPADDRCOARSEOFS].val);
  }
  
  /* attenuation */
  if (sp->gen[GEN_VELOCITY].flags) {
    sp->attenuation = iiwu_vel2cb((int) sp->gen[GEN_VELOCITY].val);
  } else {
    sp->attenuation = iiwu_vel2cb(sp->vel);
    if (sp->gen[GEN_ATTENUATION].flags) {
      sp->attenuation += sp->gen[GEN_ATTENUATION].val;
    }
  }
  
  /* pan */
  sp->pan = sp->gen[GEN_PAN].val;

  /* reverb */
  sp->reverb_send = sp->gen[GEN_REVERBSEND].val / 1000.0;

  /* chorus */
  sp->chorus_send = sp->gen[GEN_CHORUSSEND].val / 1000.0;
  
  /* mod lfo 
   *
   * - the frequency is converted into a delta value, per buffer of IIWU_BUFSIZE samples
   * - the delay into a sample delay
   */
  sp->modlfo_incr = 4.0 * IIWU_BUFSIZE * iiwu_act2hz(sp->gen[GEN_MODLFOFREQ].val) / 44100.0;
  sp->modlfo_delay = 44100 * iiwu_tc2sec(sp->gen[GEN_MODLFODELAY].val);
  sp->modlfo_to_pitch = sp->gen[GEN_MODLFOTOPITCH].val;
  sp->modlfo_to_vol = sp->gen[GEN_MODLFOTOVOL].val;
  sp->modlfo_to_fc = sp->gen[GEN_MODLFOTOFILTERFC].val;
  
  /* vib lfo 
   *
   * - the frequency is converted into a delta value, per buffer of IIWU_BUFSIZE samples
   * - the delay into a sample delay
   */
  sp->viblfo_incr = 4.0 * IIWU_BUFSIZE * iiwu_act2hz(sp->gen[GEN_VIBLFOFREQ].val) / 44100.0;
  sp->viblfo_delay = 44100 * iiwu_tc2sec(sp->gen[GEN_VIBLFODELAY].val);
  sp->viblfo_to_pitch = sp->gen[GEN_VIBLFOTOPITCH].val;
  
  /* filter 
   *
   * - the resonance frequency is converted from absolute cents to midicents
   */
  sp->fres = iiwu_hz2ct(iiwu_act2hz(sp->gen[GEN_FILTERFC].val));
  sp->q = sp->gen[GEN_FILTERQ].val / 10.0;

  return IIWU_OK;
}

/*
 * iiwu_sp_update_param
 */
void 
iiwu_sp_update_param(iiwu_sp_t* sp, int gen)
{
  switch (gen) {
  case GEN_PAN:
    sp->pan = _GEN(sp, GEN_PAN);
    break;
  case GEN_ATTENUATION:
    if (sp->gen[GEN_VELOCITY].flags) {
      sp->attenuation = iiwu_vel2cb((int) sp->gen[GEN_VELOCITY].val);
    } else {
      sp->attenuation = iiwu_vel2cb(sp->vel);
      if (sp->gen[GEN_ATTENUATION].flags) {
	sp->attenuation += sp->gen[GEN_ATTENUATION].val;
      }
    }
    break;
  case GEN_PITCH:
/*  FIXME    sp->pitch = _GEN(sp, GEN_PITCH) + 100 * _GEN(sp, GEN_COARSETUNE) + _GEN(sp, GEN_FINETUNE); */
    sp->pitch = _GEN(sp, GEN_PITCH);
    break;
  }
}

/*
 * iiwu_sp_modulate
 *
 * In this implementation, I want to make sure that all controllers
 * are event based: the parameter values of the DSP algorithm should
 * only be updates when a controller event arrived and not at every
 * iteration of the audio cycle. 
 *
 * The update is done in three steps:
 *
 * - first, we look for all the modulators that have the changed
 * controller as a source. This will yield a list of generators that
 * will be changed because of the controller event.
 *
 * - For every changed generator, calculate its new value. This is the
 * sum of its original value plus the values of al the attached
 * modulators.
 *
 * - For every changed generator, convert its value to the correct
 * unit of the corresponding DSP parameter */
int iiwu_sp_modulate(iiwu_sp_t* sp, int cc, int ctrl, int val)
{
  int i, k;
  iiwu_mod_t* mod;
  int gen;
  iiwu_real_t modval;
  
/*    printf("Chan=%d, CC=%d, Src=%d, Val=%d\n", sp->channel->channum, cc, ctrl, val); */
  
  for (i = 0; i < sp->mod_count; i++) {
    mod = &sp->mod[i];
    /* step 1: find all the modules that have the changed controller
     * as input source. */
    if (iiwu_mod_has_source(mod, cc, ctrl)) {
      gen = iiwu_mod_get_dest(mod);
      modval = 0.0;

      /* step 2: for every changed modulator, calculate the modulation
       * value of its associated generator */
      for (k = 0; k < sp->mod_count; k++) {
	if (iiwu_mod_has_dest(&sp->mod[k], gen)) {
	  modval += iiwu_mod_get_value(&sp->mod[k], sp->channel, sp);
	}
      }

      iiwu_gen_set_mod(&sp->gen[gen], modval);
      /* step 3: now that we have the new value of the generator,
       * recalculate the parameter values that are derived from the
       * generator */
      iiwu_sp_update_param(sp, gen);
      
/*        if (ctrl == 10) { */
/*  	printf("CC=%d, Pan = %5.2f\n", iiwu_channel_get_cc(sp->channel, 10), sp->pan); */
/*        } */
/*        if (ctrl == 7) { */
/*  	printf("CC=%d, Att = %5.2f\n", iiwu_channel_get_cc(chan, 7), sp->param[PAR_ATTENUATION].val); */
/*        } */
    }
  }
  return IIWU_OK;  
}

/*
 * iiwu_sp_noteoff
 */
int 
iiwu_sp_noteoff(iiwu_sp_t* sp)
{
  sp->chan = NO_CHANNEL;
  sp->volenv_state = IIWU_ENV_RELEASE;
  sp->modenv_state = IIWU_ENV_RELEASE;
  return IIWU_OK;
}

/*
 * iiwu_sp_off
 */
int 
iiwu_sp_off(iiwu_sp_t* sp)
{
  sp->volenv_state = IIWU_ENV_OFF;
  sp->modenv_state = IIWU_ENV_OFF;
  sp->status = IIWU_SP_OFF;
  return IIWU_OK;
}

/*
 * iiwu_sp_add_mod
 */
void 
iiwu_sp_add_mod(iiwu_sp_t* sp, iiwu_mod_t* mod)
{
  if (sp->mod_count < IIWU_NUM_MOD) {
    iiwu_mod_clone(&sp->mod[sp->mod_count++], mod);
  }
}

/***************************************************************
 *
 *                           GEN
 */

/* iiwu_gen_init_array
 *
 * Set an array of generators to their initial value 
 */
int 
iiwu_gen_init_array(iiwu_gen_t* gen)
{
  int i;
  for (i = 0; i < GEN_LAST; i++) {
    gen[i].flags = GEN_UNUSED;
    gen[i].mod = 0.0;
  }
  gen[GEN_STARTADDROFS].val = 0;
  gen[GEN_ENDADDROFS].val = 0;
  gen[GEN_STARTLOOPADDROFS].val = 0;
  gen[GEN_ENDLOOPADDROFS].val = 0;
  gen[GEN_STARTADDRCOARSEOFS].val = 0;
  gen[GEN_MODLFOTOPITCH].val = 0;
  gen[GEN_VIBLFOTOPITCH].val = 0;
  gen[GEN_MODENVTOPITCH].val = 0;
  gen[GEN_FILTERFC].val = 13500;
  gen[GEN_FILTERQ].val = 0;
  gen[GEN_MODLFOTOFILTERFC].val = 0;
  gen[GEN_MODENVTOFILTERFC].val = 0;
  gen[GEN_ENDADDRCOARSEOFS].val = 0;
  gen[GEN_MODLFOTOVOL].val = 0;
  gen[GEN_CHORUSSEND].val = 0;
  gen[GEN_REVERBSEND].val = 0;
  gen[GEN_PAN].val = 0;
  gen[GEN_MODLFODELAY].val = -12000;
  gen[GEN_MODLFOFREQ].val = 0;
  gen[GEN_VIBLFODELAY].val = -12000;
  gen[GEN_VIBLFOFREQ].val = 0;
  gen[GEN_MODENVDELAY].val = -12000;
  gen[GEN_MODENVATTACK].val = -12000;
  gen[GEN_MODENVHOLD].val = -12000;
  gen[GEN_MODENVDECAY].val = -12000;
  gen[GEN_MODENVSUSTAIN].val = 0;
  gen[GEN_MODENVRELEASE].val = -12000;
  gen[GEN_KEYTOMODENVHOLD].val = 0;
  gen[GEN_KEYTOMODENVDECAY].val = 0;
  gen[GEN_VOLENVDELAY].val = -12000;
  gen[GEN_VOLENVATTACK].val = -12000;
  gen[GEN_VOLENVHOLD].val = -12000;
  gen[GEN_VOLENVDECAY].val = -12000;
  gen[GEN_VOLENVSUSTAIN].val = 0;
  gen[GEN_VOLENVRELEASE].val = -12000;
  gen[GEN_KEYTOVOLENVHOLD].val = 0;
  gen[GEN_KEYTOVOLENVDECAY].val = 0;
  gen[GEN_STARTLOOPADDRCOARSEOFS].val = 0;
  gen[GEN_KEYNUM].val = -1;
  gen[GEN_VELOCITY].val = -1;
  gen[GEN_ATTENUATION].val = 0;
  gen[GEN_ENDLOOPADDRCOARSEOFS].val = 0;
  gen[GEN_COARSETUNE].val = 0;
  gen[GEN_FINETUNE].val = 0;
  gen[GEN_SAMPLEID].val = 0;
  gen[GEN_SAMPLEMODE].val = 0;
  gen[GEN_SCALETUNE].val = 100;
  gen[GEN_EXCLUSIVECLASS].val = 0;
  gen[GEN_OVERRIDEROOTKEY].val = -1;
  gen[GEN_PITCH].val = 0;
  return IIWU_OK;
}

/***************************************************************
 *
 *                           MOD
 */

/*
 * iiwu_mod_clone
 */
void 
iiwu_mod_clone(iiwu_mod_t* mod, iiwu_mod_t* src)
{
  mod->dest = src->dest;
  mod->src1 = src->src1; 
  mod->flags1 = src->flags1; 
  mod->src2 = src->src2; 
  mod->flags2 = src->flags2; 
  mod->amount = src->amount; 
}

/*
 * iiwu_mod_set_source1
 */
void 
iiwu_mod_set_source1(iiwu_mod_t* mod, int src, int flags)
{
  mod->src1 = src; 
  mod->flags1 = flags; 
}

/*
 * iiwu_mod_set_source2
 */
void 
iiwu_mod_set_source2(iiwu_mod_t* mod, int src, int flags)
{
  mod->src2 = src; 
  mod->flags2 = flags; 
}

/*
 * iiwu_mod_set_dest
 */
void 
iiwu_mod_set_dest(iiwu_mod_t* mod, int dest)
{
  mod->dest = dest; 
}

/*
 * iiwu_mod_set_amount
 */
void 
iiwu_mod_set_amount(iiwu_mod_t* mod, iiwu_real_t amount)
{
  mod->amount = amount;
}

/*
 * iiwu_mod_get_value
 */
iiwu_real_t 
iiwu_mod_get_value(iiwu_mod_t* mod, iiwu_channel_t* chan, iiwu_sp_t* sp)
{
  iiwu_real_t v1 = 0.0, v2 = 1.0;
  iiwu_real_t range1 = 128.0, range2 = 128.0;

  /* get the initial value of the first source */
  if (mod->src1 > 0) {
    if (mod->flags1 & IIWU_MOD_CC) {
      v1 = iiwu_channel_get_cc(chan, mod->src1);
    } else {  /* source 1 is one of the direct controllers */
      switch (mod->src1) {
      case IIWU_MOD_VELOCITY: 
	v1 = sp->vel;
	break;
      case IIWU_MOD_KEY: 
	v1 = sp->key;
	break;
      case IIWU_MOD_KEYPRESSURE: 
	v1 = chan->key_pressure;
	break;
      case IIWU_MOD_CHANNELPRESSURE: 
	v1 = chan->channel_pressure; 
	break;
      case IIWU_MOD_PITCHWHEEL: 
	v1 = chan->pitch_bend;
	range1 = 0x4000;
	break;
      case IIWU_MOD_PITCHWHEELSENS: 
	v1 = chan->pitch_wheel_sensitivity;
	break;
      default: 
	v1 = 0.0; 	
      }
    }

    /* transform the input value */
    switch (mod->flags1 & 0x0f) {
    case 0: /* linear, unipolar, positive */
      v1 /= range1;
      break;
    case 1: /* linear, unipolar, negative */
      v1 = 1.0 - v1 / range1;	  	  
      break;
    case 2: /* linear, bipolar, positive */
      v1 = -1.0 + 2.0 * v1 / range1;
      break;
    case 3: /* linear, bipolar, negative */
      v1 = -1.0 + 2.0 * v1 / range1;
      break;
    case 4: /* concave, unipolar, positive */
      v1 = iiwu_concave(v1);
      break;
    case 5: /* concave, unipolar, negative */
      v1 = iiwu_concave(128 - v1);
      break;
    case 6: /* concave, bipolar, positive */
      v1 = (v1 > 64)? iiwu_concave(2 * (v1 - 64)) : -iiwu_concave(2 * (64 - v1));
      break;
    case 7: /* concave, bipolar, negative */
      v1 = (v1 > 64)? -iiwu_concave(2 * (v1 - 64)) : iiwu_concave(2 * (64 - v1));
      break;
    case 8: /* convex, unipolar, positive */
      v1 = iiwu_convex(v1);
      break;
    case 9: /* convex, unipolar, negative */
      v1 = 1.0 - iiwu_convex(v1);
      break;
    case 10: /* convex, bipolar, positive */
      v1 = (v1 > 64)? -iiwu_convex(2 * (v1 - 64)) : iiwu_convex(2 * (64 - v1));
      break;
    case 11: /* convex, bipolar, negative */
      v1 = (v1 > 64)? -iiwu_convex(2 * (v1 - 64)) : iiwu_convex(2 * (64 - v1));
      break;
    case 12: /* switch, unipolar, positive */
      v1 = (v1 >= 64)? 1.0 : 0.0;
      break;
    case 13: /* switch, unipolar, negative */
      v1 = (v1 >= 64)? 0.0 : 1.0;
      break;
    case 14: /* switch, bipolar, positive */
      v1 = (v1 >= 64)? 1.0 : -1.0;
      break;
    case 15: /* switch, bipolar, negative */
      v1 = (v1 >= 64)? -1.0 : 1.0;
      break;
    }
  } else {
    return 0.0;
  }

  /* no need to go further */
  if (v1 == 0.0) {
    return 0.0;
  }

  /* get the second input source */
  if (mod->src2 > 0) {
    if (mod->flags2 & IIWU_MOD_CC) {
      v2 = iiwu_channel_get_cc(chan, mod->src2);
    } else {
      switch (mod->src2) {
      case IIWU_MOD_VELOCITY: 
	v2 = sp->vel;
	break;
      case IIWU_MOD_KEY: 
	v2 = sp->key;
	break;
      case IIWU_MOD_KEYPRESSURE: 
	v2 = chan->key_pressure;
	break;
      case IIWU_MOD_CHANNELPRESSURE: 
	v2 = chan->channel_pressure; 
	break;
      case IIWU_MOD_PITCHWHEEL: 
	v2 = chan->pitch_bend;
	break;
      case IIWU_MOD_PITCHWHEELSENS: 
	v2 = chan->pitch_wheel_sensitivity;
	break;
      default: 
	v1 = 0.0; 	
      }
    }

    /* transform the second input value */
    switch (mod->flags2 & 0x0f) {
    case 0: /* linear, unipolar, positive */
      v2 /= range2;
      break;
    case 1: /* linear, unipolar, negative */
      v2 = 1.0 - v2 / range2;	  	  
      break;
    case 2: /* linear, bipolar, positive */
      v2 = -1.0 + 2.0 * v2 / range2;
      break;
    case 3: /* linear, bipolar, negative */
      v2 = -1.0 + 2.0 * v2 / range2;
      break;
    case 4: /* concave, unipolar, positive */
      v2 = iiwu_concave(v2);
      break;
    case 5: /* concave, unipolar, negative */
      v2 = iiwu_concave(128 - v2);
      break;
    case 6: /* concave, bipolar, positive */
      v2 = (v2 > 64)? iiwu_concave(2 * (v2 - 64)) : -iiwu_concave(2 * (64 - v2));
      break;
    case 7: /* concave, bipolar, negative */
      v2 = (v2 > 64)? -iiwu_concave(2 * (v2 - 64)) : iiwu_concave(2 * (64 - v2));
      break;
    case 8: /* convex, unipolar, positive */
      v2 = iiwu_convex(v2);
      break;
    case 9: /* convex, unipolar, negative */
      v2 = 1.0 - iiwu_convex(v2);
      break;
    case 10: /* convex, bipolar, positive */
      v2 = (v2 > 64)? -iiwu_convex(2 * (v2 - 64)) : iiwu_convex(2 * (64 - v2));
      break;
    case 11: /* convex, bipolar, negative */
      v2 = (v2 > 64)? -iiwu_convex(2 * (v2 - 64)) : iiwu_convex(2 * (64 - v2));
      break;
    case 12: /* switch, unipolar, positive */
      v2 = (v2 >= 64)? 1.0 : 0.0;
      break;
    case 13: /* switch, unipolar, negative */
      v2 = (v2 >= 64)? 0.0 : 1.0;
      break;
    case 14: /* switch, bipolar, positive */
      v2 = (v2 >= 64)? 1.0 : -1.0;
      break;
    case 15: /* switch, bipolar, negative */
      v2 = (v2 >= 64)? -1.0 : 1.0;
      break;
    }
  } else {
    v2 = 1.0;
  }

  /* it's as simple as that: */
  return mod->amount * v1 * v2;
}
 
/***************************************************************
 *
 *                           SFONT
 */

/*
 * new_iiwu_sfont
 */
iiwu_sfont_t* 
new_iiwu_sfont()
{
  iiwu_sfont_t* sfont;
  sfont = IIWU_NEW(iiwu_sfont_t);
  if (sfont == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  sfont->next = NULL;
  sfont->file = NULL;
  sfont->samplepos = 0;
  sfont->samplesize = 0;
  sfont->sample = NULL;
  sfont->sampledata = NULL;
  sfont->preset = NULL;
  return sfont;
}

/*
 * delete_iiwu_sfont
 */
int 
delete_iiwu_sfont(iiwu_sfont_t* sfont)
{
  iiwu_sample_t* sample;
  iiwu_preset_t* preset;
  if (sfont->file != NULL) {
    IIWU_FREE(sfont->file);
  }
  sample = sfont->sample;
  while (sample != NULL) {
    sfont->sample = sample->next;
    delete_iiwu_sample(sample);
    sample = sfont->sample;
  }
  if (sfont->sampledata != NULL) {
    IIWU_FREE(sfont->sampledata);
  }
  preset = sfont->preset;
  while (preset != NULL) {
    sfont->preset = preset->next;
    delete_iiwu_preset(preset);
    preset = sfont->preset;
  }
  return IIWU_OK;
}

/*
 * iiwu_sfont_load
 */
int 
iiwu_sfont_load(iiwu_sfont_t* sfont, char* file)
{
  SFData* sfdata;
  GSList *p;
  SFPreset* sfpreset;
  SFSample* sfsample;
  iiwu_sample_t* sample;
  iiwu_preset_t* preset;

  sfont->file = IIWU_MALLOC(1 + IIWU_STRLEN(file));
  if (sfont->file == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return IIWU_FAILED;
  }
  IIWU_STRCPY(sfont->file, file);

  /* The actual loading is done in the sfont and sffile files */
  sfdata = sfload_file(file);
  if (sfdata == NULL) {
    IIWU_LOG(ERR, "Couldn't load soundfont file");
    return IIWU_FAILED;
  }

  /* Keep track of the position and size of the sample data because
     it's loaded separately (and might be unoaded/reloaded in future) */
  sfont->samplepos = sfdata->samplepos;
  sfont->samplesize = sfdata->samplesize;

  /* load sample data in one block */
  if (iiwu_sfont_load_sampledata(sfont) != IIWU_OK) {
    return IIWU_FAILED;
  }

  /* Create all the sample headers */
  p = sfdata->sample;
  while (p != NULL) {
    sfsample = (SFSample *) p->data;
    sample = new_iiwu_sample();
    if (sample == NULL) {
      return IIWU_FAILED;
    }
    if (iiwu_sample_import_sfont(sample, sfsample, sfont) != IIWU_OK) {
      return IIWU_FAILED;
    }
    iiwu_sfont_add_sample(sfont, sample);
    p = g_slist_next(p);
  }

  /* Load all the presets */ 
  p = sfdata->preset;
  while (p != NULL) {
    sfpreset = (SFPreset *) p->data;
    preset = new_iiwu_preset(sfont);
    if (preset == NULL) {
      return IIWU_FAILED;
    }
    if (iiwu_preset_import_sfont(preset, sfpreset, sfont) != IIWU_OK) {
      return IIWU_FAILED;
    }
    iiwu_sfont_add_preset(sfont, preset);
    p = g_slist_next(p);
  }
  sfont_free_data(sfdata);
  return IIWU_OK;
}

/* iiwu_sfont_add_sample
 *
 * Add a sample to the SoundFont
 */
int 
iiwu_sfont_add_sample(iiwu_sfont_t* sfont, iiwu_sample_t* sample) 
{
  if (sfont->sample == NULL) {
    sample->next = NULL;
    sfont->sample = sample;
  } else {
    sample->next = sfont->sample;
    sfont->sample = sample;
  }
  return IIWU_OK;
}

/* iiwu_sfont_add_preset
 *
 * Add a preset to the SoundFont
 */
int 
iiwu_sfont_add_preset(iiwu_sfont_t* sfont, iiwu_preset_t* preset) 
{
  iiwu_preset_t *cur, *prev;
  if (sfont->preset == NULL) {
    preset->next = NULL;
    sfont->preset = preset;
  } else {
    /* sort them as we go along. very basic sorting trick. */
    cur = sfont->preset;
    prev = NULL;
    while (cur != NULL) {
      if ((preset->bank < cur->bank) 
	  || ((preset->bank == cur->bank) && (preset->num < cur->num))) {
	if (prev == NULL) {
	  preset->next = cur;
	  sfont->preset = preset;
	} else {
	  preset->next = cur;
	  prev->next = preset;
	}
	return IIWU_OK;
      }
      prev = cur;
      cur = cur->next;
    }
    preset->next = NULL;
    prev->next = preset;
  }
  return IIWU_OK;
}

/*
 * iiwu_sfont_load_sampledata
 */
int 
iiwu_sfont_load_sampledata(iiwu_sfont_t* sfont) 
{
  iiwu_file fd;
  unsigned short endian;
  fd = IIWU_FOPEN(sfont->file, "rb");
  if (fd == NULL) {
    IIWU_LOG(ERR, "Can't open soundfont file"); 
    return IIWU_FAILED;
  }
  if (IIWU_FSEEK(fd, sfont->samplepos, SEEK_SET) == -1) {
    perror("error");
    IIWU_LOG(ERR, "Failed to seek position in data file"); 
    return IIWU_FAILED;
  }
  sfont->sampledata = (short*) IIWU_MALLOC(sfont->samplesize);
  if (sfont->sampledata == NULL) {
    IIWU_LOG(ERR, "Out of memory"); 
    return IIWU_FAILED;
  }
  if (IIWU_FREAD(sfont->sampledata, 1, sfont->samplesize, fd) < sfont->samplesize) {
    IIWU_LOG(ERR, "Failed to read sample data"); 
    return IIWU_FAILED;
  }
  IIWU_FCLOSE(fd);
  
  /* I'm not sure this endian test is waterproof...  */
  endian = 0x0100;

  /* If this machine is big endian, the sample have to byte swapped  */
  if (((char *) &endian)[0]) {
    unsigned char* cbuf;
    unsigned char hi, lo;
    unsigned int i, j;
    short s;
    cbuf = (unsigned char*) sfont->sampledata;
    for (i = 0, j = 0; j < sfont->samplesize; i++) {
      lo = cbuf[j++];
      hi = cbuf[j++];
      s = (hi << 8) | lo;
      sfont->sampledata[i] = s;
    }
  }
  return IIWU_OK;
}

/*
 * iiwu_sfont_get_sample
 */
iiwu_sample_t* 
iiwu_sfont_get_sample(iiwu_sfont_t* sfont, char *s) 
{
  iiwu_sample_t* sample = sfont->sample;
  while (sample != NULL) {
    if (IIWU_STRCMP(sample->name, s) == 0) {
      return sample;
    }
    sample = sample->next;
  }
  return NULL;
}

/*
 * iiwu_sfont_get_preset
 */
iiwu_preset_t* 
iiwu_sfont_get_preset(iiwu_sfont_t* sfont, unsigned int bank, unsigned int num) 
{
  iiwu_preset_t* preset = sfont->preset;
  while (preset != NULL) {
    if ((preset->bank == bank) && ((preset->num == num))) {
      return preset;
    }
    preset = preset->next;
  }
  if (sfont->next != NULL) {
    return iiwu_sfont_get_preset(sfont->next, bank, num);
  }
  return NULL;
}

/*
 * iiwu_sfont_append_sfont
 */
int 
iiwu_sfont_append_sfont(iiwu_sfont_t* sfont, iiwu_sfont_t* sfont2) 
{
  if (sfont2 == NULL) {
    return IIWU_OK;
  }
  if (sfont->next != NULL) {
    return iiwu_sfont_append_sfont(sfont->next, sfont2);
  } else {
    sfont2->next = NULL;
    sfont->next = sfont2;
  }
  return IIWU_OK;
}

/*
 * iiwu_sfont_first_preset
 */
iiwu_preset_t* 
iiwu_sfont_first_preset(iiwu_sfont_t* sfont)
{
  sfont->cur = sfont->preset;
  return sfont->cur;
}

/*
 * iiwu_sfont_next_preset
 */
iiwu_preset_t* 
iiwu_sfont_next_preset(iiwu_sfont_t* sfont)
{
  if (sfont->cur == NULL) {
    return NULL;
  }
  sfont->cur = iiwu_preset_next(sfont->cur);
  return sfont->cur;
}

/***************************************************************
 *
 *                           PRESET
 */

/*
 * new_iiwu_preset
 */
iiwu_preset_t* 
new_iiwu_preset(iiwu_sfont_t* sfont)
{
  iiwu_preset_t* preset = IIWU_NEW(iiwu_preset_t);
  if (preset == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  preset->next = NULL;
  preset->sfont = sfont;
  preset->name[0] = 0;
  preset->bank = 0;
  preset->num = 0;
  preset->global_zone = NULL;
  preset->zone = NULL;
  return preset;
}

/*
 * delete_iiwu_preset
 */
int 
delete_iiwu_preset(iiwu_preset_t* preset)
{
  int err = IIWU_OK;
  iiwu_preset_zone_t* zone;
  if (preset->global_zone != NULL) {
    if (delete_iiwu_preset_zone(preset->global_zone) != IIWU_OK) {
      err = IIWU_FAILED;
    }
    preset->global_zone = NULL;
  }
  zone = preset->zone;
  while (zone != NULL) {
    preset->zone = zone->next;
    if (delete_iiwu_preset_zone(zone) != IIWU_OK) {
      err = IIWU_FAILED;
    }
    zone = preset->zone;
  }
  IIWU_FREE(preset);
  return err;
}

/*
 * iiwu_preset_next
 */
iiwu_preset_t* 
iiwu_preset_next(iiwu_preset_t* preset)
{
  return preset->next;
}

/*
 * iiwu_preset_set_global_zone
 */
int 
iiwu_preset_set_global_zone(iiwu_preset_t* preset, iiwu_preset_zone_t* zone)
{
  preset->global_zone = zone;
  return IIWU_OK;
}

/*
 * iiwu_preset_import_sfont
 */
int 
iiwu_preset_import_sfont(iiwu_preset_t* preset, SFPreset* sfpreset, iiwu_sfont_t* sfont)
{
  GSList *p;
  SFZone* sfzone;
  iiwu_preset_zone_t* zone;
  int count;
  char zone_name[256];
  if ((sfpreset->name != NULL) && (IIWU_STRLEN(sfpreset->name) > 0)) {
    IIWU_STRCPY(preset->name, sfpreset->name);
  } else {
    IIWU_SPRINTF(preset->name, "Bank%d,Preset%d", sfpreset->bank, sfpreset->prenum);
  }
  preset->bank = sfpreset->bank;
  preset->num = sfpreset->prenum;
  p = sfpreset->zone;
  count = 0;
  while (p != NULL) {
    sfzone = (SFZone *) p->data;
    IIWU_SPRINTF(zone_name, "%s/%d", preset->name, count);
    zone = new_iiwu_preset_zone(zone_name);
    if (zone == NULL) {
      return IIWU_FAILED;
    }
    if (iiwu_preset_zone_import_sfont(zone, sfzone, sfont) != IIWU_OK) {
      return IIWU_FAILED;
    }
    if ((count == 0) && (iiwu_preset_zone_get_inst(zone) == NULL)) {
      iiwu_preset_set_global_zone(preset, zone);
    } else if (iiwu_preset_add_zone(preset, zone) != IIWU_OK) {
      return IIWU_FAILED;
    }
    p = g_slist_next(p);
    count++;
  }
  return IIWU_OK;
}

/*
 * iiwu_preset_add_zone
 */
int 
iiwu_preset_add_zone(iiwu_preset_t* preset, iiwu_preset_zone_t* zone)
{
  if (preset->zone == NULL) {
    zone->next = NULL;
    preset->zone = zone;
  } else {
    zone->next = preset->zone;
    preset->zone = zone;
  }
  return IIWU_OK;
}

/*
 * iiwu_preset_get_zone
 */
iiwu_preset_zone_t* 
iiwu_preset_get_zone(iiwu_preset_t* preset)
{
  return preset->zone;
}

/*
 * iiwu_preset_get_global_zone
 */
iiwu_preset_zone_t* 
iiwu_preset_get_global_zone(iiwu_preset_t* preset)
{
  return preset->global_zone;  
}

/*
 * iiwu_preset_zone_next
 */
iiwu_preset_zone_t* 
iiwu_preset_zone_next(iiwu_preset_zone_t* preset)
{
  return preset->next;
}

/*
 * new_iiwu_preset_zone
 */
iiwu_preset_zone_t* 
new_iiwu_preset_zone(char *name)
{
  int size;
  iiwu_preset_zone_t* zone = NULL;
  zone = IIWU_NEW(iiwu_preset_zone_t);
  if (zone == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  zone->next = NULL;
  size = 1 + IIWU_STRLEN(name);
  zone->name = IIWU_MALLOC(size);
  if (zone->name == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    IIWU_FREE(zone);
    return NULL;
  }
  IIWU_STRCPY(zone->name, name);
  zone->inst = NULL;
  zone->keylo = 0;
  zone->keyhi = 128;
  zone->vello = 0;
  zone->velhi = 128;
  iiwu_gen_init_array(&zone->gen[0]);
  return zone;
}

/***************************************************************
 *
 *                           PRESET_ZONE
 */

/*
 * delete_iiwu_preset_zone
 */
int 
delete_iiwu_preset_zone(iiwu_preset_zone_t* zone)
{
  IIWU_FREE(zone);
  return IIWU_OK;
}

/*
 * iiwu_preset_zone_import_sfont
 */
int 
iiwu_preset_zone_import_sfont(iiwu_preset_zone_t* zone, SFZone *sfzone, iiwu_sfont_t* sfont)
{
  GSList *r;
  SFGen* sfgen;
  int count;
  for (count = 0, r = sfzone->gen; r != NULL; count++) {
    sfgen = (SFGen *) r->data;
    switch (sfgen->id) {
    case GEN_KEYRANGE: 
      zone->keylo = (int) sfgen->amount.range.lo;
      zone->keyhi = (int) sfgen->amount.range.hi;
      break;
    case GEN_VELRANGE: 
      zone->vello = (int) sfgen->amount.range.lo;
      zone->velhi = (int) sfgen->amount.range.hi;
      break;
    default:
      /* FIXME: some generators have an unsigne word amount value but i don't know which ones */
      zone->gen[sfgen->id].val = (iiwu_real_t) sfgen->amount.sword;
      zone->gen[sfgen->id].flags = GEN_SET;
      break;
    }
    r = g_slist_next(r);
  }
  if ((sfzone->instsamp != NULL) && (sfzone->instsamp->data != NULL)) {
    zone->inst = (iiwu_inst_t*) new_iiwu_inst();
    if (zone->inst == NULL) {
      IIWU_LOG(ERR, "Out of memory");
      return IIWU_FAILED;
    }
    if (iiwu_inst_import_sfont(zone->inst, (SFInst *) sfzone->instsamp->data, sfont) != IIWU_OK) {
      return IIWU_FAILED;    
    }
  }  
  return IIWU_OK;
}

/*
 * iiwu_preset_zone_get_inst
 */
iiwu_inst_t* 
iiwu_preset_zone_get_inst(iiwu_preset_zone_t* zone)
{
  return zone->inst;
}

/*
 * iiwu_preset_zone_inside_range
 */
int 
iiwu_preset_zone_inside_range(iiwu_preset_zone_t* zone, int key, int vel)
{
  return ((zone->keylo <= key) &&
	  (zone->keyhi >= key) &&
	  (zone->vello <= vel) &&
	  (zone->velhi >= vel));
}

/***************************************************************
 *
 *                           INST
 */

/*
 * new_iiwu_inst
 */
iiwu_inst_t* 
new_iiwu_inst()
{
  iiwu_inst_t* inst = IIWU_NEW(iiwu_inst_t);
  if (inst == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  inst->name[0] = 0;
  inst->global_zone = NULL;
  inst->zone = NULL;
  return inst;
}

/*
 * delete_iiwu_inst
 */
int 
delete_iiwu_inst(iiwu_inst_t* inst)
{
  iiwu_inst_zone_t* zone;
  int err = IIWU_OK;
  if (inst->global_zone != NULL) {
    if (delete_iiwu_inst_zone(inst->global_zone) != IIWU_OK) {
      err = IIWU_FAILED;
    }
    inst->global_zone = NULL;
  }
  zone = inst->zone;
  while (zone != NULL) {
    inst->zone = zone->next;
    if (delete_iiwu_inst_zone(zone) != IIWU_OK) {
      err = IIWU_FAILED;
    }
    zone = inst->zone;    
  }
  IIWU_FREE(inst);
  return err;
}

/*
 * iiwu_inst_set_global_zone
 */
int 
iiwu_inst_set_global_zone(iiwu_inst_t* inst, iiwu_inst_zone_t* zone)
{
  inst->global_zone = zone;
  return IIWU_OK;
}

/*
 * iiwu_inst_import_sfont
 */
int 
iiwu_inst_import_sfont(iiwu_inst_t* inst, SFInst *sfinst, iiwu_sfont_t* sfont)
{
  GSList *p;
  SFZone* sfzone;
  iiwu_inst_zone_t* zone;
  char zone_name[256];
  int count;
  p = sfinst->zone;
  if ((sfinst->name != NULL) && (IIWU_STRLEN(sfinst->name) > 0)) {
    IIWU_STRCPY(inst->name, sfinst->name);    
  } else {
    IIWU_STRCPY(inst->name, "<untitled>");
  }
  count = 0;
  while (p != NULL) {
    sfzone = (SFZone *) p->data;
    IIWU_SPRINTF(zone_name, "%s/%d", inst->name, count);
    zone = new_iiwu_inst_zone(zone_name);
    if (zone == NULL) {
      return IIWU_FAILED;
    }
    if (iiwu_inst_zone_import_sfont(zone, sfzone, sfont) != IIWU_OK) {
      return IIWU_FAILED;
    }
    if ((count == 0) && (iiwu_inst_zone_get_sample(zone) == NULL)) {
      iiwu_inst_set_global_zone(inst, zone);
    } else if (iiwu_inst_add_zone(inst, zone) != IIWU_OK) {
      return IIWU_FAILED;
    }
    p = g_slist_next(p);
    count++;
  }
  return IIWU_OK;
}

/*
 * iiwu_inst_add_zone
 */
int 
iiwu_inst_add_zone(iiwu_inst_t* inst, iiwu_inst_zone_t* zone)
{
  if (inst->zone == NULL) {
    zone->next = NULL;
    inst->zone = zone;
  } else {
    zone->next = inst->zone;
    inst->zone = zone;
  }
  return IIWU_OK;
}

/*
 * iiwu_inst_get_zone
 */
iiwu_inst_zone_t* 
iiwu_inst_get_zone(iiwu_inst_t* inst)
{
  return inst->zone;
}

/*
 * iiwu_inst_get_global_zone
 */
iiwu_inst_zone_t* 
iiwu_inst_get_global_zone(iiwu_inst_t* inst)
{
  return inst->global_zone;  
}

/***************************************************************
 *
 *                           INST_ZONE
 */

/*
 * new_iiwu_inst_zone
 */
iiwu_inst_zone_t* 
new_iiwu_inst_zone(char* name)
{
  int size;
  iiwu_inst_zone_t* zone = NULL;
  zone = IIWU_NEW(iiwu_inst_zone_t);
  if (zone == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  zone->next = NULL;
  size = 1 + IIWU_STRLEN(name);
  zone->name = IIWU_MALLOC(size);
  if (zone->name == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    IIWU_FREE(zone);
    return NULL;
  }
  IIWU_STRCPY(zone->name, name);
  zone->sample = NULL;
  zone->keyhi = 128;
  zone->vello = 0;
  zone->velhi = 128;
  iiwu_gen_init_array(&zone->gen[0]);
  return zone;
}

/*
 * delete_iiwu_inst_zone
 */
int 
delete_iiwu_inst_zone(iiwu_inst_zone_t* zone)
{
  IIWU_FREE(zone);
  return IIWU_OK;
}

/*
 * iiwu_inst_zone_next
 */
iiwu_inst_zone_t* 
iiwu_inst_zone_next(iiwu_inst_zone_t* zone)
{
  return zone->next;
}

/*
 * iiwu_inst_zone_import_sfont
 */
int 
iiwu_inst_zone_import_sfont(iiwu_inst_zone_t* zone, SFZone *sfzone, iiwu_sfont_t* sfont)
{
  GSList *r;
  SFGen* sfgen;
  SFMod* sfmod;
  int count;
  for (count = 0, r = sfzone->gen; r != NULL; count++) {
    sfgen = (SFGen *) r->data;
    switch (sfgen->id) {
    case GEN_KEYRANGE: 
      zone->keylo = (int) sfgen->amount.range.lo;
      zone->keyhi = (int) sfgen->amount.range.hi;
      break;
    case GEN_VELRANGE: 
      zone->vello = (int) sfgen->amount.range.lo;
      zone->velhi = (int) sfgen->amount.range.hi;
      break;
    default:
      /* FIXME: some generators have an unsigned word amount value but i don't know which ones */
      zone->gen[sfgen->id].val = (iiwu_real_t) sfgen->amount.sword;
      zone->gen[sfgen->id].flags = GEN_SET;
      break;
    }
    r = g_slist_next(r);
  }
  for (count = 0, r = sfzone->gen; r != NULL; count++) {
    sfmod = (SFMod *) r->data;
    r = g_slist_next(r);
  }
  if ((sfzone->instsamp != NULL) && (sfzone->instsamp->data != NULL)) {
    zone->sample = iiwu_sfont_get_sample(sfont, ((SFSample *) sfzone->instsamp->data)->name);
    if (zone->sample == NULL) {
      IIWU_LOG(ERR, "Couldnt fins sample name");
      return IIWU_FAILED;
    }
  }
  return IIWU_OK;
}

/*
 * iiwu_inst_zone_get_sample
 */
iiwu_sample_t* 
iiwu_inst_zone_get_sample(iiwu_inst_zone_t* zone)
{
  return zone->sample;
}

/*
 * iiwu_inst_zone_inside_range
 */
int 
iiwu_inst_zone_inside_range(iiwu_inst_zone_t* zone, int key, int vel)
{
  return ((zone->keylo <= key) &&
	  (zone->keyhi >= key) &&
	  (zone->vello <= vel) &&
	  (zone->velhi >= vel));
}

/***************************************************************
 *
 *                           SAMPLE
 */

/*
 * new_iiwu_sample
 */
iiwu_sample_t* 
new_iiwu_sample()
{
  iiwu_sample_t* sample = NULL;
  sample = IIWU_NEW(iiwu_sample_t);
  if (sample == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  sample->next = NULL;
  sample->data = NULL;
  sample->start = 0;
  sample->end = 0;
  sample->loopstart = 0;
  sample->loopend = 0;
  sample->samplerate = 0;
  sample->origpitch = 0;
  sample->pitchadj = 0;
  return sample;
}

/*
 * delete_iiwu_sample
 */
int 
delete_iiwu_sample(iiwu_sample_t* sample)
{
  IIWU_FREE(sample);
  return IIWU_OK;
}

/*
 * iiwu_sample_import_sfont
 */
int 
iiwu_sample_import_sfont(iiwu_sample_t* sample, SFSample* sfsample, iiwu_sfont_t* sfont)
{
  IIWU_STRCPY(sample->name, sfsample->name);
  sample->data = sfont->sampledata;
  sample->start = sfsample->start;
  sample->end = sfsample->start + sfsample->end;
  sample->loopstart = sfsample->start + sfsample->loopstart;
  sample->loopend = sfsample->start + sfsample->loopend;
  sample->samplerate = sfsample->samplerate;
  sample->origpitch = sfsample->origpitch;
  sample->pitchadj = sfsample->pitchadj;
  sample->sampletype = sfsample->sampletype;
  if (sample->sampletype & SF_SAMPLETYPE_ROM) {
    IIWU_LOG(WARN, "Ignoring ROM sample");
  }
  return IIWU_OK;
}

/*
 * iiwu_sample_in_rom
 */
int 
iiwu_sample_in_rom(iiwu_sample_t* sample)
{
  return (sample->sampletype & SF_SAMPLETYPE_ROM);
}

/***************************************************************
 *
 *                           REVERB
 */
/*
 Written by Jezar at Dreampoint, June 2000
 http://www.dreampoint.co.uk
 This code is public domain

 Translated to C by Peter Hanappe, Mai 2001
*/

/*
 Macro for killing denormalled numbers

 Written by Jezar at Dreampoint, June 2000
 http://www.dreampoint.co.uk
 Based on IS_DENORMAL macro by Jon Watte
 This code is public domain
*/
#if defined(WITH_FLOAT)
#define undenormalise(sample) if(((*(unsigned int*)&sample)&0x7f800000)==0) sample=0.0f
#else
/* FIXME: translate for double */
#define undenormalise(sample)
#endif

typedef struct _iiwu_allpass iiwu_allpass;
typedef struct _iiwu_comb iiwu_comb;

struct _iiwu_allpass {
  iiwu_real_t feedback;
  iiwu_real_t *buffer;
  int bufsize;
  int bufidx;
};

void 
iiwu_allpass_setbuffer(iiwu_allpass* allpass, iiwu_real_t *buf, int size)
{
  allpass->bufidx = 0;
  allpass->buffer = buf; 
  allpass->bufsize = size;
}

void 
iiwu_allpass_mute(iiwu_allpass* allpass)
{
  int i;
  int len = allpass->bufsize;
  iiwu_real_t* buf = allpass->buffer;
  for (i = 0; i < len; i++) {
    buf[i] = 0;
  }
}

void 
iiwu_allpass_setfeedback(iiwu_allpass* allpass, iiwu_real_t val)
{
  allpass->feedback = val;
}

iiwu_real_t 
iiwu_allpass_getfeedback(iiwu_allpass* allpass)
{
  return allpass->feedback;
}

#define iiwu_allpass_process(_allpass, _input) \
{ \
  iiwu_real_t output; \
  iiwu_real_t bufout; \
  bufout = _allpass.buffer[_allpass.bufidx]; \
  undenormalise(bufout); \
  output = -_input + bufout; \
  _allpass.buffer[_allpass.bufidx] = _input + (bufout * _allpass.feedback); \
  if (++_allpass.bufidx >= _allpass.bufsize) { \
    _allpass.bufidx = 0; \
  } \
  _input = output; \
}

/*  iiwu_real_t iiwu_allpass_process(iiwu_allpass* allpass, iiwu_real_t input) */
/*  { */
/*    iiwu_real_t output; */
/*    iiwu_real_t bufout; */
/*    bufout = allpass->buffer[allpass->bufidx]; */
/*    undenormalise(bufout); */
/*    output = -input + bufout; */
/*    allpass->buffer[allpass->bufidx] = input + (bufout * allpass->feedback); */
/*    if (++allpass->bufidx >= allpass->bufsize) { */
/*      allpass->bufidx = 0; */
/*    } */
/*    return output; */
/*  } */

struct _iiwu_comb {
  iiwu_real_t feedback;
  iiwu_real_t filterstore;
  iiwu_real_t damp1;
  iiwu_real_t damp2;
  iiwu_real_t *buffer;
  int bufsize;
  int bufidx;
};

void 
iiwu_comb_setbuffer(iiwu_comb* comb, iiwu_real_t *buf, int size) 
{
  comb->filterstore = 0;
  comb->bufidx = 0;
  comb->buffer = buf; 
  comb->bufsize = size;
}

void 
iiwu_comb_mute(iiwu_comb* comb)
{
  int i;
  iiwu_real_t* buf = comb->buffer;
  int len = comb->bufsize;
  for (i = 0; i < len; i++)
    buf[i]=0;
}

void 
iiwu_comb_setdamp(iiwu_comb* comb, iiwu_real_t val) 
{
  comb->damp1 = val; 
  comb->damp2 = 1 - val;
}

iiwu_real_t 
iiwu_comb_getdamp(iiwu_comb* comb) 
{
  return comb->damp1;
}

void 
iiwu_comb_setfeedback(iiwu_comb* comb, iiwu_real_t val) 
{
  comb->feedback = val;
}

iiwu_real_t 
iiwu_comb_getfeedback(iiwu_comb* comb) 
{
  return comb->feedback;
}

#define iiwu_comb_process(_comb, _input, _output) \
{ \
  iiwu_real_t _tmp = _comb.buffer[_comb.bufidx]; \
  undenormalise(_tmp); \
  _comb.filterstore = (_tmp * _comb.damp2) + (_comb.filterstore * _comb.damp1); \
  undenormalise(_comb.filterstore); \
  _comb.buffer[_comb.bufidx] = _input + (_comb.filterstore * _comb.feedback); \
  if (++_comb.bufidx >= _comb.bufsize) { \
    _comb.bufidx = 0; \
  } \
  _output += _tmp; \
}

/*  iiwu_real_t iiwu_comb_process(iiwu_comb* comb, iiwu_real_t input) */
/*  { */
/*    iiwu_real_t output; */
/*    iiwu_real_t output = comb->buffer[comb->bufidx]; */
/*    undenormalise(output); */
/*    comb->filterstore = (output * comb->damp2) + (comb->filterstore * comb->damp1); */
/*    undenormalise(comb->filterstore); */
/*    comb->buffer[comb->bufidx] = input + (comb->filterstore * comb->feedback); */
/*    if (++comb->bufidx >= comb->bufsize) { */
/*      comb->bufidx = 0; */
/*    } */
/*    return output; */
/*  } */

#define numcombs 8
#define numallpasses 4
#define muted 0
#define fixedgain 0.015f
#define scalewet 3
#define scaledry 2
#define scaledamp 0.4f
#define scaleroom 0.28f
#define offsetroom 0.7f
#define initialroom 0.5f
#define initialdamp 0.5f
#define initialwet 1 / scalewet
#define initialdry 0
#define initialwidth 1
#define initialmode 0
#define freezemode 0.5f
#define stereospread 23

/*
 These values assume 44.1KHz sample rate
 they will probably be OK for 48KHz sample rate
 but would need scaling for 96KHz (or other) sample rates.
 The values were obtained by listening tests.
*/
#define combtuningL1 1116
#define combtuningR1 1116 + stereospread
#define combtuningL2 1188
#define combtuningR2 1188 + stereospread
#define combtuningL3 1277
#define combtuningR3 1277 + stereospread
#define combtuningL4 1356
#define combtuningR4 1356 + stereospread
#define combtuningL5 1422
#define combtuningR5 1422 + stereospread
#define combtuningL6 1491
#define combtuningR6 1491 + stereospread
#define combtuningL7 1557
#define combtuningR7 1557 + stereospread
#define combtuningL8 1617
#define combtuningR8 1617 + stereospread
#define allpasstuningL1 556
#define allpasstuningR1 556 + stereospread
#define allpasstuningL2 441
#define allpasstuningR2 441 + stereospread
#define allpasstuningL3 341
#define allpasstuningR3 341 + stereospread
#define allpasstuningL4 225
#define allpasstuningR4 225 + stereospread

struct _iiwu_revmodel_t {
  iiwu_real_t gain;
  iiwu_real_t roomsize,roomsize1;
  iiwu_real_t damp,damp1;
  iiwu_real_t wet,wet1,wet2;
  iiwu_real_t dry;
  iiwu_real_t width;
  iiwu_real_t mode;
  /*
   The following are all declared inline 
   to remove the need for dynamic allocation
   with its subsequent error-checking messiness
  */
  /* Comb filters */
  iiwu_comb combL[numcombs];
  iiwu_comb combR[numcombs];
  /* Allpass filters */
  iiwu_allpass allpassL[numallpasses];
  iiwu_allpass allpassR[numallpasses];
  /* Buffers for the combs */
  iiwu_real_t bufcombL1[combtuningL1];
  iiwu_real_t bufcombR1[combtuningR1];
  iiwu_real_t bufcombL2[combtuningL2];
  iiwu_real_t bufcombR2[combtuningR2];
  iiwu_real_t bufcombL3[combtuningL3];
  iiwu_real_t bufcombR3[combtuningR3];
  iiwu_real_t bufcombL4[combtuningL4];
  iiwu_real_t bufcombR4[combtuningR4];
  iiwu_real_t bufcombL5[combtuningL5];
  iiwu_real_t bufcombR5[combtuningR5];
  iiwu_real_t bufcombL6[combtuningL6];
  iiwu_real_t bufcombR6[combtuningR6];
  iiwu_real_t bufcombL7[combtuningL7];
  iiwu_real_t bufcombR7[combtuningR7];
  iiwu_real_t bufcombL8[combtuningL8];
  iiwu_real_t bufcombR8[combtuningR8];
  /* Buffers for the allpasses */
  iiwu_real_t bufallpassL1[allpasstuningL1];
  iiwu_real_t bufallpassR1[allpasstuningR1];
  iiwu_real_t bufallpassL2[allpasstuningL2];
  iiwu_real_t bufallpassR2[allpasstuningR2];
  iiwu_real_t bufallpassL3[allpasstuningL3];
  iiwu_real_t bufallpassR3[allpasstuningR3];
  iiwu_real_t bufallpassL4[allpasstuningL4];
  iiwu_real_t bufallpassR4[allpasstuningR4];
};

void iiwu_revmodel_mute(iiwu_revmodel_t* rev);
iiwu_real_t iiwu_revmodel_getmode(iiwu_revmodel_t* rev);

iiwu_revmodel_t* 
new_iiwu_revmodel()
{
  iiwu_revmodel_t* rev;
  rev = IIWU_NEW(iiwu_revmodel_t);
  if (rev == NULL) {
    return NULL;
  }
  /* Tie the components to their buffers */
  iiwu_comb_setbuffer(&rev->combL[0], rev->bufcombL1, combtuningL1);
  iiwu_comb_setbuffer(&rev->combR[0], rev->bufcombR1, combtuningR1);
  iiwu_comb_setbuffer(&rev->combL[1], rev->bufcombL2, combtuningL2);
  iiwu_comb_setbuffer(&rev->combR[1], rev->bufcombR2, combtuningR2);
  iiwu_comb_setbuffer(&rev->combL[2], rev->bufcombL3, combtuningL3);
  iiwu_comb_setbuffer(&rev->combR[2], rev->bufcombR3, combtuningR3);
  iiwu_comb_setbuffer(&rev->combL[3], rev->bufcombL4, combtuningL4);
  iiwu_comb_setbuffer(&rev->combR[3], rev->bufcombR4, combtuningR4);
  iiwu_comb_setbuffer(&rev->combL[4], rev->bufcombL5, combtuningL5);
  iiwu_comb_setbuffer(&rev->combR[4], rev->bufcombR5, combtuningR5);
  iiwu_comb_setbuffer(&rev->combL[5], rev->bufcombL6, combtuningL6);
  iiwu_comb_setbuffer(&rev->combR[5], rev->bufcombR6, combtuningR6);
  iiwu_comb_setbuffer(&rev->combL[6], rev->bufcombL7, combtuningL7);
  iiwu_comb_setbuffer(&rev->combR[6], rev->bufcombR7, combtuningR7);
  iiwu_comb_setbuffer(&rev->combL[7], rev->bufcombL8, combtuningL8);
  iiwu_comb_setbuffer(&rev->combR[7], rev->bufcombR8, combtuningR8);
  iiwu_allpass_setbuffer(&rev->allpassL[0], rev->bufallpassL1, allpasstuningL1);
  iiwu_allpass_setbuffer(&rev->allpassR[0], rev->bufallpassR1, allpasstuningR1);
  iiwu_allpass_setbuffer(&rev->allpassL[1], rev->bufallpassL2, allpasstuningL2);
  iiwu_allpass_setbuffer(&rev->allpassR[1], rev->bufallpassR2, allpasstuningR2);
  iiwu_allpass_setbuffer(&rev->allpassL[2], rev->bufallpassL3, allpasstuningL3);
  iiwu_allpass_setbuffer(&rev->allpassR[2], rev->bufallpassR3, allpasstuningR3);
  iiwu_allpass_setbuffer(&rev->allpassL[3], rev->bufallpassL4, allpasstuningL4);
  iiwu_allpass_setbuffer(&rev->allpassR[3], rev->bufallpassR4, allpasstuningR4);
  /* Set default values */
  iiwu_allpass_setfeedback(&rev->allpassL[0], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassR[0], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassL[1], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassR[1], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassL[2], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassR[2], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassL[3], 0.5f);
  iiwu_allpass_setfeedback(&rev->allpassR[3], 0.5f);
  iiwu_revmodel_setwet(rev, initialwet);
  iiwu_revmodel_setroomsize(rev, initialroom);
  iiwu_revmodel_setdry(rev, initialdry);
  iiwu_revmodel_setdamp(rev, initialdamp);
  iiwu_revmodel_setwidth(rev, initialwidth);
  iiwu_revmodel_setmode(rev, initialmode);
  /* Buffer will be full of rubbish - so we MUST mute them */
  iiwu_revmodel_mute(rev);
  return rev;
}

void 
delete_iiwu_revmodel(iiwu_revmodel_t* rev)
{
  IIWU_FREE(rev);
}

void 
iiwu_revmodel_mute(iiwu_revmodel_t* rev)
{
  int i;
  if (iiwu_revmodel_getmode(rev) >= freezemode) {
    return;
  }
  for (i = 0; i < numcombs;i++) {
    iiwu_comb_mute(&rev->combL[i]);
    iiwu_comb_mute(&rev->combR[i]);
  }
  for (i = 0; i < numallpasses; i++) {
    iiwu_allpass_mute(&rev->allpassL[i]);
    iiwu_allpass_mute(&rev->allpassR[i]);
  }
}

/*
  Changes: 
  - uses a mono input signal
 */
void 
iiwu_revmodel_processreplace(iiwu_revmodel_t* rev, iiwu_real_t *in, 
			     iiwu_real_t *left_out, iiwu_real_t *right_out,
			     long numsamples)
{
  int i, k = 0;
  iiwu_real_t outL, outR, input;

  while (numsamples-- > 0) {
    outL = outR = 0;
    input = in[k] * rev->gain;

    /* Accumulate comb filters in parallel */
    for (i = 0; i < numcombs; i++) {
      iiwu_comb_process(rev->combL[i], input, outL);
      iiwu_comb_process(rev->combR[i], input, outR);
/*        outL += iiwu_comb_process(&rev->combL[i], input); */
/*        outR += iiwu_comb_process(&rev->combR[i], input); */
    }
    /* Feed through allpasses in series */
    for (i = 0; i < numallpasses; i++) {
      iiwu_allpass_process(rev->allpassL[i], outL);
      iiwu_allpass_process(rev->allpassR[i], outR);
/*        outL = iiwu_allpass_process(&rev->allpassL[i], outL); */
/*        outR = iiwu_allpass_process(&rev->allpassR[i], outR); */
    }
    /* Calculate output REPLACING anything already there */
    left_out[k] = outL * rev->wet1 + outR * rev->wet2 + in[k] * rev->dry;
    right_out[k] = outR * rev->wet1 + outL * rev->wet2 + in[k] * rev->dry;
    k++;
  }
}

/*
  Changes: 
  - now uses a mono input signal
 */
void 
iiwu_revmodel_processmix(iiwu_revmodel_t* rev, iiwu_real_t *in, 
			 iiwu_real_t *left_out, iiwu_real_t *right_out,
			 long numsamples)
{
  int i, k = 0;
  iiwu_real_t outL, outR, input;

  while (numsamples-- > 0) {
    outL = outR = 0;
    input = in[k] * rev->gain;

    /* Accumulate comb filters in parallel */
    for (i = 0; i < numcombs; i++) {
      iiwu_comb_process(rev->combL[i], input, outL);
      iiwu_comb_process(rev->combR[i], input, outR);
/*        outL += iiwu_comb_process(&rev->combL[i], input); */
/*        outR += iiwu_comb_process(&rev->combR[i], input); */
    }
    /* Feed through allpasses in series */
    for (i = 0; i < numallpasses; i++) {
      iiwu_allpass_process(rev->allpassL[i], outL);
      iiwu_allpass_process(rev->allpassR[i], outR);
/*        outL = iiwu_allpass_process(&rev->allpassL[i], outL); */
/*        outR = iiwu_allpass_process(&rev->allpassR[i], outR); */
    }
    /* Calculate output MIXING with anything already there */
    left_out[k] += outL * rev->wet1 + outR * rev->wet2 + in[k] * rev->dry;
    right_out[k] += outR * rev->wet1 + outL * rev->wet2 + in[k] * rev->dry;
    k++;
  }
}

void 
iiwu_revmodel_update(iiwu_revmodel_t* rev)
{
  /* Recalculate internal values after parameter change */
  int i;
  rev->wet1 = rev->wet * (rev->width / 2 + 0.5f);
  rev->wet2 = rev->wet * ((1 - rev->width) / 2);

  if (rev->mode >= freezemode) {
      rev->roomsize1 = 1;
      rev->damp1 = 0;
      rev->gain = muted;
  } else {
    rev->roomsize1 = rev->roomsize;
    rev->damp1 = rev->damp;
    rev->gain = fixedgain;
  }
  for (i = 0; i < numcombs; i++) {
    iiwu_comb_setfeedback(&rev->combL[i], rev->roomsize1);
    iiwu_comb_setfeedback(&rev->combR[i], rev->roomsize1);
  }
  for (i = 0; i < numcombs; i++) {
    iiwu_comb_setdamp(&rev->combL[i], rev->damp1);
    iiwu_comb_setdamp(&rev->combR[i], rev->damp1);
  }
}

/*
 The following get/set functions are not inlined, because
 speed is never an issue when calling them, and also
 because as you develop the reverb model, you may
 wish to take dynamic action when they are called.
*/
void 
iiwu_revmodel_setroomsize(iiwu_revmodel_t* rev, iiwu_real_t value)
{
  rev->roomsize = (value * scaleroom) + offsetroom;
  iiwu_revmodel_update(rev);
}

iiwu_real_t 
iiwu_revmodel_getroomsize(iiwu_revmodel_t* rev)
{
  return (rev->roomsize - offsetroom) / scaleroom;
}

void 
iiwu_revmodel_setdamp(iiwu_revmodel_t* rev, iiwu_real_t value)
{
  rev->damp = value * scaledamp;
  iiwu_revmodel_update(rev);
}

iiwu_real_t 
iiwu_revmodel_getdamp(iiwu_revmodel_t* rev)
{
  return rev->damp / scaledamp;
}

void 
iiwu_revmodel_setwet(iiwu_revmodel_t* rev, iiwu_real_t value)
{
  rev->wet = value * scalewet;
  iiwu_revmodel_update(rev);
}

iiwu_real_t 
iiwu_revmodel_getwet(iiwu_revmodel_t* rev)
{
  return rev->wet / scalewet;
}

void 
iiwu_revmodel_setdry(iiwu_revmodel_t* rev, iiwu_real_t value)
{
  rev->dry = value * scaledry;
}

iiwu_real_t 
iiwu_revmodel_getdry(iiwu_revmodel_t* rev)
{
  return rev->dry / scaledry;
}

void 
iiwu_revmodel_setwidth(iiwu_revmodel_t* rev, iiwu_real_t value)
{
  rev->width = value;
  iiwu_revmodel_update(rev);
}

iiwu_real_t 
iiwu_revmodel_getwidth(iiwu_revmodel_t* rev)
{
  return rev->width;
}

void 
iiwu_revmodel_setmode(iiwu_revmodel_t* rev, iiwu_real_t value)
{
  rev->mode = value;
  iiwu_revmodel_update(rev);
}

iiwu_real_t 
iiwu_revmodel_getmode(iiwu_revmodel_t* rev)
{
  if (rev->mode >= freezemode) {
    return 1;
  } else {
    return 0;
  }
}
