/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */

#ifndef _IIWU_SYNTH_H
#define _IIWU_SYNTH_H

/***************************************************************
 *
 *                         INCLUDES
 */

#include "iiwusynth_priv.h"
#include "smurf.h"

#if !defined(WIN32) && !defined(MACINTOSH)
#include <sys/types.h>
#else
typedef int int32_t;
typedef unsigned int u_int32_t;
#endif

/***************************************************************
 *
 *                         DEFINES
 */
#define IIWU_NUM_CHANNELS      16
#define IIWU_NUM_PROGRAMS      129
#define IIWU_NUM_MOD           64
#define IIWU_CENTS_HZ_SIZE     1200
#define IIWU_VEL_CB_SIZE       128
#define IIWU_CB_AMP_SIZE       961
#define IIWU_PAN_SIZE          1002

#ifdef WIN32
#define IIWU_BUFSIZE           512
#define IIWU_QUEUESIZE         8192
#else
#define IIWU_BUFSIZE           128
#define IIWU_QUEUESIZE         1024
#endif

#define NO_CHANNEL             0xff
#define DRUM_INST_MASK         ((unsigned int)0x80000000)

#ifndef PI
#define PI                     3.141592654
#endif

/* Choose the type of interpolation */
/*  #define USE_TRUNCATION */
/*  #define USE_LINEAR_INTERPOLATION */
#define USE_CUBIC_INTERPOLATION


/***************************************************************
 *
 *                         ENUM
 */
enum iiwu_loop {
  IIWU_UNLOOPED,
  IIWU_LOOP_DURING_RELEASE,
  IIWU_NOTUSED,
  IIWU_LOOP
};

enum iiwu_mod_flags
{
  IIWU_MOD_POSITIVE = 0,
  IIWU_MOD_NEGATIVE = 1,
  IIWU_MOD_UNIPOLAR = 0,
  IIWU_MOD_BIPOLAR = 2,
  IIWU_MOD_LINEAR = 0,
  IIWU_MOD_CONCAVE = 4,
  IIWU_MOD_CONVEX = 8,
  IIWU_MOD_SWITCH = 12,
  IIWU_MOD_GC = 0,
  IIWU_MOD_CC = 16
};

enum iiwu_mod_src
{
  IIWU_MOD_NONE = 0,
  IIWU_MOD_VELOCITY = 2,
  IIWU_MOD_KEY = 3,
  IIWU_MOD_KEYPRESSURE = 10,
  IIWU_MOD_CHANNELPRESSURE = 13,
  IIWU_MOD_PITCHWHEEL = 14,
  IIWU_MOD_PITCHWHEELSENS = 16
};

enum iiwu_gen_type
{
  GEN_STARTADDROFS,
  GEN_ENDADDROFS,
  GEN_STARTLOOPADDROFS,
  GEN_ENDLOOPADDROFS,
  GEN_STARTADDRCOARSEOFS,
  GEN_MODLFOTOPITCH,
  GEN_VIBLFOTOPITCH,
  GEN_MODENVTOPITCH,
  GEN_FILTERFC,
  GEN_FILTERQ,
  GEN_MODLFOTOFILTERFC,
  GEN_MODENVTOFILTERFC,
  GEN_ENDADDRCOARSEOFS,
  GEN_MODLFOTOVOL,
  GEN_UNUSED1,
  GEN_CHORUSSEND,
  GEN_REVERBSEND,
  GEN_PAN,
  GEN_UNUSED2,
  GEN_UNUSED3,
  GEN_UNUSED4,
  GEN_MODLFODELAY,
  GEN_MODLFOFREQ,
  GEN_VIBLFODELAY,
  GEN_VIBLFOFREQ,
  GEN_MODENVDELAY,
  GEN_MODENVATTACK,
  GEN_MODENVHOLD,
  GEN_MODENVDECAY,
  GEN_MODENVSUSTAIN,
  GEN_MODENVRELEASE,
  GEN_KEYTOMODENVHOLD,
  GEN_KEYTOMODENVDECAY,
  GEN_VOLENVDELAY,
  GEN_VOLENVATTACK,
  GEN_VOLENVHOLD,
  GEN_VOLENVDECAY,
  GEN_VOLENVSUSTAIN,
  GEN_VOLENVRELEASE,
  GEN_KEYTOVOLENVHOLD,
  GEN_KEYTOVOLENVDECAY,
  GEN_INSTRUMENT,
  GEN_RESERVED1,
  GEN_KEYRANGE,
  GEN_VELRANGE,
  GEN_STARTLOOPADDRCOARSEOFS,
  GEN_KEYNUM,
  GEN_VELOCITY,
  GEN_ATTENUATION,
  GEN_RESERVED2,
  GEN_ENDLOOPADDRCOARSEOFS,
  GEN_COARSETUNE,
  GEN_FINETUNE,
  GEN_SAMPLEID,
  GEN_SAMPLEMODE,
  GEN_RESERVED3,
  GEN_SCALETUNE,
  GEN_EXCLUSIVECLASS,
  GEN_OVERRIDEROOTKEY,
  GEN_PITCH,     /* the initial pitch is not a "standard"
		  * generator. It is not mentioned in the list of
		  * generator in the SF2 specifications. It is used,
		  * however, as the destination for the default pitch
		  * wheel modulator. */
  GEN_LAST };

enum iiwu_gen_flags
{
  GEN_UNUSED,
  GEN_SET,
  GEN_CHANGED
};

enum iiwu_env_status
{
  IIWU_ENV_SKIP,
  IIWU_ENV_DELAY,
  IIWU_ENV_ATTACK,
  IIWU_ENV_HOLD,
  IIWU_ENV_DECAY,
  IIWU_ENV_SUSTAIN,
  IIWU_ENV_RELEASE,
  IIWU_ENV_OFF
};

enum iiwu_sp_status
{
  IIWU_SP_CLEAN,
  IIWU_SP_ON,
  IIWU_SP_RELEASED,
  IIWU_SP_OFF
};

enum iiwu_synth_status
{
  IIWU_SYNTH_CLEAN,
  IIWU_SYNTH_PLAYING,
  IIWU_SYNTH_QUIET,
  IIWU_SYNTH_STOPPED
};

/***************************************************************
 *
 *       FORWARD DECLARATIONS 
 */
typedef struct _iiwu_sfont_t iiwu_sfont_t;
typedef struct _iiwu_preset_t iiwu_preset_t;
typedef struct _iiwu_preset_zone_t iiwu_preset_zone_t;
typedef struct _iiwu_inst_t iiwu_inst_t;
typedef struct _iiwu_inst_zone_t iiwu_inst_zone_t;
typedef struct _iiwu_sample_t iiwu_sample_t;
typedef struct _iiwu_mod_t iiwu_mod_t;
typedef struct _iiwu_gen_t iiwu_gen_t;
typedef struct _iiwu_sp_t iiwu_sp_t;
typedef struct _iiwu_channel_t iiwu_channel_t;
typedef struct _iiwu_revmodel_t iiwu_revmodel_t;

/***************************************************************
 *
 *         TYPE DEFINITIONS & FUNCTION DECLARATIONS
 */

#if defined(WITH_FLOAT)
typedef float iiwu_real_t;
#define IIWU_SAMPLE_FORMAT IIWU_SAMPLE_FLOAT
#else
typedef double iiwu_real_t;
#define IIWU_SAMPLE_FORMAT IIWU_SAMPLE_DOUBLE
#endif


#if defined(USE_TRUNCATION)
typedef char iiwu_interp_coeff_t;

#elif defined(USE_LINEAR_INTERPOLATION)
typedef struct {
  iiwu_real_t a0, a1;
} iiwu_interp_coeff_t;

#elif defined(USE_CUBIC_INTERPOLATION)
typedef struct {
  iiwu_real_t a0, a1, a2, a3;
} iiwu_interp_coeff_t;

#else
#error No interpolation defined
#endif


/*
 *  phase
 */

#define IIWU_INTERP_BITS        8
#define IIWU_INTERP_BITS_MASK   0xff000000
#define IIWU_INTERP_BITS_SHIFT  24
#define IIWU_INTERP_MAX         256

#define IIWU_FRACT_MAX ((double)4294967296.0)

typedef struct {
  int32_t index;
  u_int32_t fract;
} iiwu_phase_t;

#define iiwu_phase_set(a, b) { \
  (a).fract = (b).fract; \
  (a).index = (b).index; \
}

#define iiwu_phase_set_int(a, b)   { \
  (a).index = (int32_t) (b); \
  (a).fract = 0; \
}

#define iiwu_phase_set_float(a, b)   { \
  (a).index = (int32_t) (b); \
  (a).fract = (u_int32_t) (((double)(b) - (double)((a).index)) * IIWU_FRACT_MAX); \
}

#define iiwu_phase_fract(_x) \
  ((iiwu_real_t)((double)((_x).fract) / IIWU_FRACT_MAX))

#define iiwu_phase_index(_x) \
  ((int)(_x).index)

#define iiwu_phase_interp_index(_x) \
  ((int)(((_x).fract & IIWU_INTERP_BITS_MASK) >> IIWU_INTERP_BITS_SHIFT))

#define iiwu_phase_double(_x) \
  ((_x).index + ((double)((_x).fract) / IIWU_FRACT_MAX))

/* The idea to use (a).index += (b).index + ((a).fract < (b).fract) to
   handle wrap-arounds comes from Mozilla's macros to handle 64-bit
   integer on 32-bit platforms. Header prlong.h in the NSPR
   library. www.mozilla.org. */
#define iiwu_phase_incr(a, b)  { \
  (a).fract += (b).fract; \
  (a).index += (b).index + ((a).fract < (b).fract); \
}

#define iiwu_phase_decr(a, b) { \
  (a).fract -= b.fract; \
  (a).index -= b.index - ((a).fract < (b).fract); \
}

#define iiwu_phase_lt(a, b) \
  (((a).index < (b).index) || (((a).index == (b).index) && ((a).fract < (b).fract)))

#define iiwu_phase_gt(a, b) \
  (((a).index > (b).index) || (((a).index == (b).index) && ((a).fract > (b).fract)))

#define iiwu_phase_eq(a, b) \
  (((a).index == (b).index) && ((a).fract == (b).fract))

/*
 *  iiwu_mod_t
 */
struct _iiwu_mod_t
{
  unsigned char dest;
  unsigned char src1;
  unsigned char flags1;
  unsigned char src2;
  unsigned char flags2;
  iiwu_real_t amount;
};

void iiwu_mod_clone(iiwu_mod_t* mod, iiwu_mod_t* src);
void iiwu_mod_set_source1(iiwu_mod_t* mod, int src, int flags); 
void iiwu_mod_set_source2(iiwu_mod_t* mod, int src, int flags); 
void iiwu_mod_set_dest(iiwu_mod_t* mod, int dst); 
void iiwu_mod_set_amount(iiwu_mod_t* mod, iiwu_real_t amount); 
iiwu_real_t iiwu_mod_get_value(iiwu_mod_t* mod, iiwu_channel_t* chan, iiwu_sp_t* sp); 

#define iiwu_mod_has_source(mod,cc,ctrl)  \
( ((((mod)->src1 == ctrl) && (((mod)->flags1 & IIWU_MOD_CC) != 0) && (cc != 0)) \
   || ((((mod)->src1 == ctrl) && (((mod)->flags1 & IIWU_MOD_CC) == 0) && (cc == 0)))) \
|| ((((mod)->src2 == ctrl) && (((mod)->flags2 & IIWU_MOD_CC) != 0) && (cc != 0)) \
    || ((((mod)->src2 == ctrl) && (((mod)->flags2 & IIWU_MOD_CC) == 0) && (cc == 0)))))

#define iiwu_mod_has_dest(mod,gen)  ((mod)->dest == gen)
#define iiwu_mod_get_dest(mod)  ((mod)->dest)

/*
 *  iiwu_gen_t
 */
struct _iiwu_gen_t
{
  unsigned char flags;
  iiwu_real_t val;
  iiwu_real_t mod;
};

int iiwu_gen_init_array(iiwu_gen_t* gen);
#define iiwu_gen_set_mod(gen,val)  (gen)->mod = val

/*
 * iiwu_sp_t
 */
struct _iiwu_sp_t
{
  unsigned char status;
  unsigned char chan;               /* the channel number, quick access for channel messages */
  unsigned char key;                /* the key, quick acces for noteoff */
  unsigned char vel;                /* the velocity */
  iiwu_channel_t* channel;
  iiwu_gen_t gen[GEN_LAST];
  iiwu_mod_t mod[IIWU_NUM_MOD];
  int mod_count;
  iiwu_sample_t* sample;
  unsigned int ticks;
/*    iiwu_real_t* monobuf; */
  iiwu_phase_t phase;                  /* the phase of the sample wave */
  /* basic parameters */
  iiwu_real_t pitch;                      /* the pitch in midicents */
  iiwu_real_t attenuation;                /* the attenuation in centibels */
  iiwu_real_t root_pitch;
  /* sample and loop start and end points */
  int start_offset;
  int end_offset;
  int loop_start_offset;
  int loop_end_offset;
  /* vol env */
  char volenv_state;                /* the state of the volume envelope */
  iiwu_real_t volenv_val;             /* the value of the volume envelope */
  unsigned int volenv_delay;
  iiwu_real_t volenv_attack;
  iiwu_real_t volenv_hold;
  iiwu_real_t volenv_decay;
  iiwu_real_t volenv_sustain;
  iiwu_real_t volenv_release;
  /* mod env */
  char modenv_state;                /* the state of the modulation envelope */
  iiwu_real_t modenv_val;             /* the value of the modulation envelope */
  unsigned int modenv_delay;
  iiwu_real_t modenv_attack;
  iiwu_real_t modenv_hold;
  iiwu_real_t modenv_decay;
  iiwu_real_t modenv_sustain;
  iiwu_real_t modenv_release;
  iiwu_real_t modenv_to_fc; 
  iiwu_real_t modenv_to_pitch; 
  /* mod lfo */
  iiwu_real_t modlfo_val;             /* the value of the modulation LFO */
  unsigned int modlfo_delay;        /* the delay of the lfo in samples */
  iiwu_real_t modlfo_incr;            /* the lfo frequency is converted to a per-buffer increment */ 
  iiwu_real_t modlfo_to_fc; 
  iiwu_real_t modlfo_to_pitch; 
  iiwu_real_t modlfo_to_vol; 
  /* vib lfo */
  iiwu_real_t viblfo_val;             /* the value of the vibrato LFO */
  unsigned int viblfo_delay;           /* the delay of the lfo in samples */
  iiwu_real_t viblfo_incr;            /* the lfo frequency is converted to a per-buffer increment */ 
  iiwu_real_t viblfo_to_pitch; 
  /* for the resonant filter */
  iiwu_real_t fres;                   /* the resonance frequency, in cents (not absolute cents) */
  iiwu_real_t q;                      /* the q-factor */
  iiwu_real_t w1, w2;                 /* delayed samples */
  /* pan */
  iiwu_real_t pan;
  /* reverb */
  iiwu_real_t reverb_send;
/*    iiwu_revmodel_t* rev; */
  /* chorus */
  iiwu_real_t chorus_send;
  /* for debugging purposeses */
  int debug;
};

iiwu_sp_t* new_iiwu_sp(void);
int delete_iiwu_sp(iiwu_sp_t* sp);
int iiwu_sp_start(iiwu_sp_t* sp);

int iiwu_sp_write(iiwu_sp_t* sp, int len, iiwu_real_t* mono, 
		  iiwu_real_t* left, iiwu_real_t* right, 
		  iiwu_real_t* reverb_buf, iiwu_real_t* chorus_buf);

int iiwu_sp_init(iiwu_sp_t* sp, iiwu_channel_t* channel, iiwu_sample_t* sample);
int iiwu_sp_modulate(iiwu_sp_t* sp, int cc, int ctrl, int val);
int iiwu_sp_set_param(iiwu_sp_t* sp, int code, iiwu_real_t value);
iiwu_real_t iiwu_sp_calc_param(iiwu_sp_t* sp, int code);
int iiwu_sp_noteoff(iiwu_sp_t* sp);
int iiwu_sp_off(iiwu_sp_t* sp);
int iiwu_sp_optimize(iiwu_sp_t* sp);
void iiwu_sp_add_mod(iiwu_sp_t* sp, iiwu_mod_t* mod);

/* Updates all the synthesis parameters associated with generator
 * gen. The calculation may involve several generors.  */
void iiwu_sp_update_param(iiwu_sp_t* sp, int gen);

#define _PLAYING(sp)  (((sp)->status == IIWU_SP_ON) || ((sp)->status == IIWU_SP_RELEASED))
#define _AVAILABLE(sp)  (((sp)->status == IIWU_SP_CLEAN) || ((sp)->status == IIWU_SP_OFF))
#define _RELEASED(sp)  ((sp)->chan == NO_CHANNEL)
#define _SAMPLEMODE(sp) ((int)(sp)->gen[GEN_SAMPLEMODE].val)
#define _GEN(_sp,_n)  ((_sp)->gen[_n].val + (_sp)->gen[_n].mod)

/*
 * iiwu_channel_t
 */
struct _iiwu_channel_t
{
  int channum;
  unsigned int banknum;
  int prognum;
  iiwu_preset_t* preset;
  short key_pressure;
  short channel_pressure;
  short pitch_bend;
  short pitch_wheel_sensitivity;
  short cc[128];
};

iiwu_channel_t* new_iiwu_channel(int num);
int delete_iiwu_channel(iiwu_channel_t* chan);
int iiwu_channel_set_preset(iiwu_channel_t* chan, iiwu_preset_t* preset);
iiwu_preset_t* iiwu_channel_get_preset(iiwu_channel_t* chan);
unsigned int iiwu_channel_get_banknum(iiwu_channel_t* chan);
int iiwu_channel_set_banknum(iiwu_channel_t* chan, unsigned int bank);
int iiwu_channel_set_prognum(iiwu_channel_t* chan, int prognum);
int iiwu_channel_get_prognum(iiwu_channel_t* chan);
int iiwu_channel_noteon(iiwu_channel_t* chan, iiwu_synth_t* synth, int key, int vel);
int iiwu_channel_cc(iiwu_channel_t* chan, int ctrl, int val);
int iiwu_channel_pitch_bend(iiwu_channel_t* chan, int val);

/*
 * iiwu_synth_t
 */
struct _iiwu_synth_t
{
  int state;                         /* the synthesizer state */
  iiwu_sfont_t* sfont;               /* the loaded soundfont */
  iiwu_channel_t** channel;          /* the channels */
  int nsp;                           /* the length of the synthesis process array */
  iiwu_sp_t** sp;                    /* the synthesis processes */
  iiwu_real_t* mono_buf;
  iiwu_real_t* left_buf;
  iiwu_real_t* right_buf;
  iiwu_real_t* reverb_buf;
  iiwu_real_t* chorus_buf;
#ifndef DISABLE_AUPORT
  iiwu_auport_t* auport;             /* the audio port */
#endif
  iiwu_revmodel_t* reverb;
  char outbuf[256];                  /* buffer for message output */
};

iiwu_sp_t* iiwu_synth_alloc_sp(iiwu_synth_t* synth, iiwu_channel_t* chan, int key, int vel);
int iiwu_synth_program_reset(iiwu_synth_t* synth);
int iiwu_synth_set_reverb(iiwu_synth_t* synth, int num);
iiwu_sfont_t* iiwu_synth_get_font(iiwu_synth_t* synth, int num);

/*
 * iiwu_sfont_t
 */
struct _iiwu_sfont_t
{
  iiwu_sfont_t* next;
  char* file;               /* the filename of this soundfont */
  unsigned int samplepos;   /* the position in the file at which the sample data starts */
  unsigned int samplesize;  /* the size of the sample data */
  short* sampledata;        /* the sample data, loaded in ram */
  iiwu_sample_t* sample;    /* the samples in this soundfont */
  iiwu_preset_t* preset;    /* the presets of this soundfont */
  iiwu_preset_t* cur;       /* the current preset in the enumeration */
};

iiwu_sfont_t* new_iiwu_sfont(void);
int delete_iiwu_sfont(iiwu_sfont_t* sfont);
int iiwu_sfont_load(iiwu_sfont_t* sfont, char* file);
int iiwu_sfont_load_sampledata(iiwu_sfont_t* sfont);
int iiwu_sfont_add_sample(iiwu_sfont_t* sfont, iiwu_sample_t* sample);
int iiwu_sfont_add_preset(iiwu_sfont_t* sfont, iiwu_preset_t* preset);
iiwu_sample_t* iiwu_sfont_get_sample(iiwu_sfont_t* sfont, char *s);
iiwu_preset_t* iiwu_sfont_get_preset(iiwu_sfont_t* sfont, unsigned int bank, unsigned int prenum);
iiwu_preset_t* iiwu_sfont_first_preset(iiwu_sfont_t* sfont);
iiwu_preset_t* iiwu_sfont_next_preset(iiwu_sfont_t* sfont);
int iiwu_sfont_append_sfont(iiwu_sfont_t* sfont, iiwu_sfont_t* sfont2);

/*
 * iiwu_preset_t
 */
struct _iiwu_preset_t
{
  iiwu_preset_t* next;
  iiwu_sfont_t* sfont;                  /* the soundfont this preset belongs to */
  char name[21];                        /* the name of the preset */
  unsigned int bank;                    /* the bank number */
  unsigned int num;                     /* the preset number */
  iiwu_preset_zone_t* global_zone;        /* the global zone of the preset */
  iiwu_preset_zone_t* zone;               /* the chained list of preset zones */
};

iiwu_preset_t* new_iiwu_preset(iiwu_sfont_t* sfont);
int delete_iiwu_preset(iiwu_preset_t* preset);
iiwu_preset_t* iiwu_preset_next(iiwu_preset_t* preset);
int iiwu_preset_import_sfont(iiwu_preset_t* preset, SFPreset* sfpreset, iiwu_sfont_t* sfont);
int iiwu_preset_set_global_zone(iiwu_preset_t* preset, iiwu_preset_zone_t* zone);
int iiwu_preset_add_zone(iiwu_preset_t* preset, iiwu_preset_zone_t* zone);
iiwu_preset_zone_t* iiwu_preset_get_zone(iiwu_preset_t* preset);
iiwu_preset_zone_t* iiwu_preset_get_global_zone(iiwu_preset_t* preset);

/*
 * iiwu_preset_zone
 */
struct _iiwu_preset_zone_t
{
  iiwu_preset_zone_t* next;
  char* name;
  iiwu_inst_t* inst;
  int keylo;
  int keyhi;
  int vello;
  int velhi;
  iiwu_gen_t gen[GEN_LAST];
};

iiwu_preset_zone_t* new_iiwu_preset_zone(char* name);
int delete_iiwu_preset_zone(iiwu_preset_zone_t* zone);
iiwu_preset_zone_t* iiwu_preset_zone_next(iiwu_preset_zone_t* preset);
int iiwu_preset_zone_import_sfont(iiwu_preset_zone_t* zone, SFZone* sfzone, iiwu_sfont_t* sfont);
int iiwu_preset_zone_inside_range(iiwu_preset_zone_t* zone, int key, int vel);
iiwu_inst_t* iiwu_preset_zone_get_inst(iiwu_preset_zone_t* zone);

/*
 * iiwu_inst_t
 */
struct _iiwu_inst_t
{
  char name[21];
  iiwu_inst_zone_t* global_zone;
  iiwu_inst_zone_t* zone;
};

iiwu_inst_t* new_iiwu_inst(void);
int delete_iiwu_inst(iiwu_inst_t* inst);
int iiwu_inst_import_sfont(iiwu_inst_t* inst, SFInst *sfinst, iiwu_sfont_t* sfont);
int iiwu_inst_set_global_zone(iiwu_inst_t* inst, iiwu_inst_zone_t* zone);
int iiwu_inst_add_zone(iiwu_inst_t* inst, iiwu_inst_zone_t* zone);
iiwu_inst_zone_t* iiwu_inst_get_zone(iiwu_inst_t* inst);
iiwu_inst_zone_t* iiwu_inst_get_global_zone(iiwu_inst_t* inst);

/*
 * iiwu_inst_zone_t
 */
struct _iiwu_inst_zone_t
{
  iiwu_inst_zone_t* next;
  char* name;
  iiwu_sample_t* sample;
  int keylo;
  int keyhi;
  int vello;
  int velhi;
  iiwu_gen_t gen[GEN_LAST];
};

iiwu_inst_zone_t* new_iiwu_inst_zone(char* name);
int delete_iiwu_inst_zone(iiwu_inst_zone_t* zone);
iiwu_inst_zone_t* iiwu_inst_zone_next(iiwu_inst_zone_t* zone);
int iiwu_inst_zone_import_sfont(iiwu_inst_zone_t* zone, SFZone *sfzone, iiwu_sfont_t* sfont);
int iiwu_inst_zone_inside_range(iiwu_inst_zone_t* zone, int key, int vel);
iiwu_sample_t* iiwu_inst_zone_get_sample(iiwu_inst_zone_t* zone);

/*
 * iiwu_sample_t
 */
struct _iiwu_sample_t
{
  iiwu_sample_t* next;           /* the next sample in the chained list */
  char name[21];
  unsigned int start;
  unsigned int end;
  unsigned int loopstart;
  unsigned int loopend;
  unsigned int samplerate;
  int origpitch;
  int pitchadj;
  int cents;
  int sampletype;
  short* data;
};

iiwu_sample_t* new_iiwu_sample(void);
int delete_iiwu_sample(iiwu_sample_t* sample);
int iiwu_sample_import_sfont(iiwu_sample_t* sample, SFSample* sfsample, iiwu_sfont_t* sfont);
int iiwu_sample_in_rom(iiwu_sample_t* sample);

/*
 * reverb
 */
iiwu_revmodel_t* new_iiwu_revmodel(void);
void delete_iiwu_revmodel(iiwu_revmodel_t* rev);

void 
iiwu_revmodel_processmix(iiwu_revmodel_t* rev, iiwu_real_t *in, 
			 iiwu_real_t *left_out, iiwu_real_t *right_out,
			 long numsamples);

void 
iiwu_revmodel_processreplace(iiwu_revmodel_t* rev, iiwu_real_t *in, 
			     iiwu_real_t *left_out, iiwu_real_t *right_out,
			     long numsamples);

void iiwu_revmodel_setroomsize(iiwu_revmodel_t* rev, iiwu_real_t value);
void iiwu_revmodel_setdamp(iiwu_revmodel_t* rev, iiwu_real_t value);
void iiwu_revmodel_setwet(iiwu_revmodel_t* rev, iiwu_real_t value);
void iiwu_revmodel_setdry(iiwu_revmodel_t* rev, iiwu_real_t value);
void iiwu_revmodel_setwidth(iiwu_revmodel_t* rev, iiwu_real_t value);
void iiwu_revmodel_setmode(iiwu_revmodel_t* rev, iiwu_real_t value);

/*
 * reverb preset
 */
typedef struct _iiwu_revmodel_presets_t {
  char* name;
  iiwu_real_t roomsize;
  iiwu_real_t damp;
  iiwu_real_t width;
  iiwu_real_t wet;
  iiwu_real_t dry;
} iiwu_revmodel_presets_t;


/*
 * misc
 */
static void iiwu_synth_init(void);
static iiwu_real_t iiwu_ct2hz(iiwu_real_t cents);
static iiwu_real_t iiwu_vel2cb(int vel);
static iiwu_real_t iiwu_cb2amp(iiwu_real_t cb);
static iiwu_real_t iiwu_tc2sec(iiwu_real_t tc);
static iiwu_real_t iiwu_act2hz(iiwu_real_t c);
static iiwu_real_t iiwu_hz2ct(iiwu_real_t c);
static iiwu_real_t iiwu_pan(iiwu_real_t c, int left);
static iiwu_real_t iiwu_concave(iiwu_real_t val);
static iiwu_real_t iiwu_convex(iiwu_real_t val);


#endif  /* _IIWU_SYNTH_H */
