/*==================================================================
 * sfont.h - Header file for Sound font routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2000 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#ifndef SMURF_H
#define SMURF_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
/*
include <glib.h>
include "i18n.h"
*/

/*-----------------------------------glib.h----------------------------*/

/* Provide definitions for some commonly used macros.
 *  Some of them are only provided if they haven't already
 *  been defined. It is assumed that if they are already
 *  defined then the current definition is correct.
 */
#ifndef	FALSE
#define	FALSE	(0)
#endif

#ifndef	TRUE
#define	TRUE	(!FALSE)
#endif

/* Provide type definitions for commonly used types.
 *  These are useful because a "gint8" can be adjusted
 *  to be 1 byte (8 bits) on all platforms. Similarly and
 *  more importantly, "gint32" can be adjusted to be
 *  4 bytes (32 bits) on all platforms.
 */
typedef char   gchar;
typedef short  gshort;
typedef long   glong;
typedef int    gint;
typedef gint   gboolean;

typedef unsigned char	guchar;
typedef unsigned short	gushort;
typedef unsigned long	gulong;
typedef unsigned int	guint;

typedef signed char gint8;
typedef unsigned char guint8;
typedef signed short gint16;
typedef unsigned short guint16;
typedef signed int gint32;
typedef unsigned int guint32;
typedef void* gpointer;
typedef const void *gconstpointer;

#define GPOINTER_TO_INT(p)	((gint)   (p))

typedef struct _GSList		GSList;
typedef struct _GMemChunk	GMemChunk;

typedef gint		(*GCompareFunc)		(gconstpointer	a,
						 gconstpointer	b);
struct _GSList
{
  gpointer data;
  GSList *next;
};

GSList* g_slist_alloc		(void);
void	g_slist_free		(GSList		*list);
void	g_slist_free_1		(GSList		*list);
GSList*  g_slist_sort           (GSList          *list,
		                 GCompareFunc    compare_func);
GSList* g_slist_append		(GSList		*list,
				 gpointer	 data);
GSList* g_slist_prepend		(GSList		*list,
				 gpointer	 data);
GSList* g_slist_remove		(GSList		*list,
				 gpointer	 data);
GSList* g_slist_remove_link	(GSList		*list,
				 GSList		*llink);
GSList* g_slist_nth		(GSList		*list,
				 guint		 n);
GSList* g_slist_last		(GSList		*list);

#define g_slist_next(slist)	((slist) ? (((GSList *)(slist))->next) : NULL)

gchar*	 g_strdup		(const gchar *str);

#  define g_new(type, count)	  \
      ((type *) IIWU_MALLOC ((unsigned) sizeof (type) * (count)))

#define G_ALLOC_ONLY	  1
#define G_ALLOC_AND_FREE  2

GMemChunk* g_mem_chunk_new     (gchar	  *name,
				gint	   atom_size,
				gulong	   area_size,
				gint	   type);
void	   g_mem_chunk_destroy (GMemChunk *mem_chunk);
gpointer   g_mem_chunk_alloc   (GMemChunk *mem_chunk);
void	   g_mem_chunk_free    (GMemChunk *mem_chunk,
				gpointer   mem);

#define g_mem_chunk_create(type, pre_alloc, alloc_type)	( \
  g_mem_chunk_new (#type " mem chunks (" #pre_alloc ")", \
		   sizeof (type), \
		   sizeof (type) * (pre_alloc), \
		   (alloc_type)) \
)
#define g_chunk_new(type, chunk)	( \
  (type *) g_mem_chunk_alloc (chunk) \
)

/* Provide simple macro statement wrappers (adapted from Perl):
 *  G_STMT_START { statements; } G_STMT_END;
 *  can be used as a single statement, as in
 *  if (x) G_STMT_START { ... } G_STMT_END; else ...
 *
 *  For gcc we will wrap the statements within `({' and `})' braces.
 *  For SunOS they will be wrapped within `if (1)' and `else (void) 0',
 *  and otherwise within `do' and `while (0)'.
 */
#if !(defined (G_STMT_START) && defined (G_STMT_END))
#  if defined (__GNUC__) && !defined (__STRICT_ANSI__) && !defined (__cplusplus)
#    define G_STMT_START	(void)(
#    define G_STMT_END		)
#  else
#    if (defined (sun) || defined (__sun__))
#      define G_STMT_START	if (1)
#      define G_STMT_END	else (void)0
#    else
#      define G_STMT_START	do
#      define G_STMT_END	while (0)
#    endif
#  endif
#endif


/* Basic bit swapping functions
 */
#define GUINT16_SWAP_LE_BE_CONSTANT(val)	((guint16) ( \
    (((guint16) (val) & (guint16) 0x00ffU) << 8) | \
    (((guint16) (val) & (guint16) 0xff00U) >> 8)))
#define GUINT32_SWAP_LE_BE_CONSTANT(val)	((guint32) ( \
    (((guint32) (val) & (guint32) 0x000000ffU) << 24) | \
    (((guint32) (val) & (guint32) 0x0000ff00U) <<  8) | \
    (((guint32) (val) & (guint32) 0x00ff0000U) >>  8) | \
    (((guint32) (val) & (guint32) 0xff000000U) >> 24)))

#define GUINT16_SWAP_LE_BE(val) (GUINT16_SWAP_LE_BE_CONSTANT (val))
#define GUINT32_SWAP_LE_BE(val) (GUINT32_SWAP_LE_BE_CONSTANT (val))

#define GINT16_TO_LE(val)	((gint16) (val))
#define GUINT16_TO_LE(val)	((guint16) (val))
#define GINT16_TO_BE(val)	((gint16) GUINT16_SWAP_LE_BE (val))
#define GUINT16_TO_BE(val)	(GUINT16_SWAP_LE_BE (val))
#define GINT32_TO_LE(val)	((gint32) (val))
#define GUINT32_TO_LE(val)	((guint32) (val))
#define GINT32_TO_BE(val)	((gint32) GUINT32_SWAP_LE_BE (val))
#define GUINT32_TO_BE(val)	(GUINT32_SWAP_LE_BE (val))

/* The G*_TO_?E() macros are defined in glibconfig.h.
 * The transformation is symmetric, so the FROM just maps to the TO.
 */
#define GINT16_FROM_LE(val)	(GINT16_TO_LE (val))
#define GUINT16_FROM_LE(val)	(GUINT16_TO_LE (val))
#define GINT16_FROM_BE(val)	(GINT16_TO_BE (val))
#define GUINT16_FROM_BE(val)	(GUINT16_TO_BE (val))
#define GINT32_FROM_LE(val)	(GINT32_TO_LE (val))
#define GUINT32_FROM_LE(val)	(GUINT32_TO_LE (val))
#define GINT32_FROM_BE(val)	(GINT32_TO_BE (val))
#define GUINT32_FROM_BE(val)	(GUINT32_TO_BE (val))

/*-----------------------------------i18n.h----------------------------*/
#define _(s) s

/*-----------------------------------sfont.h----------------------------*/

#define SF_SAMPLETYPE_MONO	1
#define SF_SAMPLETYPE_RIGHT	2
#define SF_SAMPLETYPE_LEFT	4
#define SF_SAMPLETYPE_LINKED	8
#define SF_SAMPLETYPE_ROM	0x8000

#define SF_SAMPMODES_LOOP	1
#define SF_SAMPMODES_UNROLL	2

#define SF_MIN_SAMPLERATE	400
#define SF_MAX_SAMPLERATE	50000

#define SF_MIN_SAMPLE_LENGTH	32

/* Sound Font structure defines */

typedef struct _SFVersion
{				/* version structure */
  guint16 major;
  guint16 minor;
}
SFVersion;

typedef struct _SFMod
{				/* Modulator structure */
  guint16 src;			/* source modulator */
  guint16 dest;			/* destination generator */
  gint16 amount;		/* signed, degree of modulation */
  guint16 amtsrc;		/* second source controls amnt of first */
  guint16 trans;		/* transform applied to source */
}
SFMod;

typedef union _SFGenAmount
{				/* Generator amount structure */
  gint16 sword;			/* signed 16 bit value */
  guint16 uword;		/* unsigned 16 bit value */
  struct
  {
    guint8 lo;			/* low value for ranges */
    guint8 hi;			/* high value for ranges */
  }
  range;
}
SFGenAmount;

typedef struct _SFGen
{				/* Generator structure */
  guint16 id;			/* generator ID */
  SFGenAmount amount;		/* generator value */
}
SFGen;

typedef struct _SFZone
{				/* Sample/instrument zone structure */
  GSList *instsamp;		/* instrument/sample pointer for zone */
  GSList *gen;			/* list of generators */
  GSList *mod;			/* list of modulators */
}
SFZone;

typedef struct _SFSample
{				/* Sample structure */
  gchar name[21];		/* Name of sample */
  guint8 samfile:1;		/* Loaded sfont/sample buffer = 0/1 */
  guint32 start;		/* Offset in sample area to start of sample */
  guint32 end;			/* Offset from start to end of sample,
				   this is the last point of the
				   sample, the SF spec has this as the
				   1st point after, corrected on
				   load/save */
  guint32 loopstart;		/* Offset from start to start of loop */
  guint32 loopend;		/* Offset from start to end of loop,
				   marks the first point after loop,
				   whose sample value is ideally
				   equivalent to loopstart */
  guint32 samplerate;		/* Sample rate recorded at */
  guint8 origpitch;		/* root midi key number */
  gint8 pitchadj;		/* pitch correction in cents */
  guint16 sampletype;		/* 1 mono,2 right,4 left,linked 8,0x8000=ROM */
}
SFSample;

typedef struct _SFInst
{				/* Instrument structure */
  gchar name[21];		/* Name of instrument */
  GSList *zone;			/* list of instrument zones */
}
SFInst;

typedef struct _SFPreset
{				/* Preset structure */
  gchar name[21];		/* preset name */
  guint16 prenum;		/* preset number */
  guint16 bank;			/* bank number */
  guint32 libr;			/* Not used (preserved) */
  guint32 genre;		/* Not used (preserved) */
  guint32 morph;		/* Not used (preserved) */
  GSList *zone;			/* list of preset zones */
}
SFPreset;

/* NOTE: sffd is also used to determine if sound font is new (NULL) */
typedef struct _SFData
{				/* Sound font data structure */
  SFVersion version;		/* sound font version */
  SFVersion romver;		/* ROM version */
  guint32 samplepos;		/* position within sffd of the sample chunk */
  guint32 samplesize;		/* length within sffd of the sample chunk */
  gchar *fname;			/* file name */
  FILE *sffd;			/* loaded sfont file descriptor */
  GSList *info;		     /* linked list of info strings (1st byte is ID) */
  GSList *preset;		/* linked list of preset info */
  GSList *inst;			/* linked list of instrument info */
  GSList *sample;		/* linked list of sample info */
  gboolean up2date;		/* saved sound font is up to date? */
  gboolean beensaved;		/* been saved since open/creation? */
}
SFData;

/* sf file chunk IDs */
enum
{ UNKN_ID, RIFF_ID, LIST_ID, SFBK_ID,
  INFO_ID, SDTA_ID, PDTA_ID,	/* info/sample/preset */

  IFIL_ID, ISNG_ID, INAM_ID, IROM_ID, /* info ids (1st byte of info strings) */
  IVER_ID, ICRD_ID, IENG_ID, IPRD_ID,	/* more info ids */
  ICOP_ID, ICMT_ID, ISFT_ID,	/* and yet more info ids */

  SNAM_ID, SMPL_ID,		/* sample ids */
  PHDR_ID, PBAG_ID, PMOD_ID, PGEN_ID,	/* preset ids */
  IHDR_ID, IBAG_ID, IMOD_ID, IGEN_ID,	/* instrument ids */
  SHDR_ID			/* sample info */
};

/* generator types */
enum
{ Gen_StartAddrOfs, Gen_EndAddrOfs, Gen_StartLoopAddrOfs,
  Gen_EndLoopAddrOfs, Gen_StartAddrCoarseOfs, Gen_ModLFO2Pitch,
  Gen_VibLFO2Pitch, Gen_ModEnv2Pitch, Gen_FilterFc, Gen_FilterQ,
  Gen_ModLFO2FilterFc, Gen_ModEnv2FilterFc, Gen_EndAddrCoarseOfs,
  Gen_ModLFO2Vol, Gen_Unused1, Gen_ChorusSend, Gen_ReverbSend, Gen_Pan,
  Gen_Unused2, Gen_Unused3, Gen_Unused4,
  Gen_ModLFODelay, Gen_ModLFOFreq, Gen_VibLFODelay, Gen_VibLFOFreq,
  Gen_ModEnvDelay, Gen_ModEnvAttack, Gen_ModEnvHold, Gen_ModEnvDecay,
  Gen_ModEnvSustain, Gen_ModEnvRelease, Gen_Key2ModEnvHold,
  Gen_Key2ModEnvDecay, Gen_VolEnvDelay, Gen_VolEnvAttack,
  Gen_VolEnvHold, Gen_VolEnvDecay, Gen_VolEnvSustain, Gen_VolEnvRelease,
  Gen_Key2VolEnvHold, Gen_Key2VolEnvDecay, Gen_Instrument,
  Gen_Reserved1, Gen_KeyRange, Gen_VelRange,
  Gen_StartLoopAddrCoarseOfs, Gen_Keynum, Gen_Velocity,
  Gen_Attenuation, Gen_Reserved2, Gen_EndLoopAddrCoarseOfs,
  Gen_CoarseTune, Gen_FineTune, Gen_SampleId, Gen_SampleModes,
  Gen_Reserved3, Gen_ScaleTune, Gen_ExclusiveClass, Gen_OverrideRootKey,
  Gen_Dummy
}
Gen_Type;

#define Gen_MaxValid 	Gen_Dummy - 1	/* maximum valid generator */
#define Gen_Count	Gen_Dummy	/* count of generators */
#define GenArrSize sizeof(SFGenAmount)*Gen_Count	/* gen array size */

/* generator unit type */
enum
{
  None,				/* No unit type */
  Unit_Smpls,			/* in samples */
  Unit_32kSmpls,		/* in 32k samples */
  Unit_Cent,			/* in cents (1/100th of a semitone) */
  Unit_HzCent,			/* in Hz Cents */
  Unit_TCent,			/* in Time Cents */
  Unit_cB,			/* in centibels (1/100th of a decibel) */
  Unit_Percent,			/* in percentage */
  Unit_Semitone,		/* in semitones */
  Unit_Range			/* a range of values */
}
Gen_Unit;

/* global data */

extern guint16 badgen[]; 	/* list of bad generators */
extern guint16 badpgen[]; 	/* list of bad preset generators */

/* Memory chunk pointers */
GMemChunk *chunk_preset;
GMemChunk *chunk_inst;
GMemChunk *chunk_sample;
GMemChunk *chunk_zone;
GMemChunk *chunk_mod;
GMemChunk *chunk_gen;

/* functions */
void sfont_init_chunks (void);

void sfont_close (SFData * sf);
void sfont_free_data (SFData * sf);
void sfont_free_zone (SFZone * zone);
gint sfont_preset_compare_func (gconstpointer a, gconstpointer b);

void sfont_zone_delete (SFData * sf, GSList ** zlist, SFZone * zone);

GSList *gen_inlist (gint gen, GSList * genlist);
gint gen_valid (gint gen);
gint gen_validp (gint gen);


/*-----------------------------------sffile.h----------------------------*/
/* 
   File structures and routines (used to be in sffile.h) 
*/

#define CHNKIDSTR(id)           &idlist[(id - 1) * 4]

/* sfont file chunk sizes */
#define SFPHDRSIZE	38
#define SFBAGSIZE	4
#define SFMODSIZE	10
#define SFGENSIZE	4
#define SFIHDRSIZE	22
#define SFSHDRSIZE	46

/* sfont file data structures */
typedef struct _SFChunk
{				/* RIFF file chunk structure */
  guint32 id;			/* chunk id */
  guint32 size;			/* size of the following chunk */
}
SFChunk;

typedef struct _SFPhdr
{
  guint8 name[20];		/* preset name */
  guint16 preset;		/* preset number */
  guint16 bank;			/* bank number */
  guint16 pbagndx;		/* index into preset bag */
  guint32 library;		/* just for preserving them */
  guint32 genre;		/* Not used */
  guint32 morphology;		/* Not used */
}
SFPhdr;

typedef struct _SFBag
{
  guint16 genndx;		/* index into generator list */
  guint16 modndx;		/* index into modulator list */
}
SFBag;

typedef struct _SFIhdr
{
  gchar name[20];		/* Name of instrument */
  guint16 ibagndx;		/* Instrument bag index */
}
SFIhdr;

typedef struct _SFShdr
{				/* Sample header loading struct */
  gchar name[20];		/* Sample name */
  guint32 start;		/* Offset to start of sample */
  guint32 end;			/* Offset to end of sample */
  guint32 loopstart;		/* Offset to start of loop */
  guint32 loopend;		/* Offset to end of loop */
  guint32 samplerate;		/* Sample rate recorded at */
  guint8 origpitch;		/* root midi key number */
  gint8 pitchadj;		/* pitch correction in cents */
  guint16 samplelink;		/* Not used */
  guint16 sampletype;		/* 1 mono,2 right,4 left,linked 8,0x8000=ROM */
}
SFShdr;

/* data */
extern gchar idlist[];

/* functions */
SFData *sfload_file (gchar * fname);


/*-----------------------------------util.h----------------------------*/
/*
  Utility functions (formerly in util.h)
 */
#define FAIL	0
#define OK	1

enum
{ ErrWarn, ErrFatal, ErrStatus, ErrCorr, ErrEof, ErrMem, Errno,
  ErrRead, ErrWrite
};

#define ErrMax		ErrWrite
#define ErrnoStart	Errno
#define ErrnoEnd	ErrWrite

gint gerr (gint ev, gchar * fmt, ...);
void *safe_malloc (size_t size);
gint safe_fread (void *buf, gint count, FILE * fd);
gint safe_fwrite (void *buf, gint count, FILE * fd);
gint safe_fseek (FILE * fd, long ofs, gint whence);

#endif
