/***************************************
  $Header: /cvsroot/petscgraphics/petsc.c,v 1.4 2002/03/21 21:01:12 hazelsct Exp $

  This is the petsc.c main file.  It has all of the PETSc-dependent functions.
***************************************/


#include <petscda.h>
#include "config.h" /* esp. for inline */
#include "illuminator.h" /* Just to make sure the interface is "right" */

#undef __FUNCT__
#define __FUNCT__ "DATriangulate"

/*++++++++++++++++++++++++++++++++++++++
  Calculate vertices of isoquant triangles in a 3-D distributed array.  This
  takes a PETSc DA object, does some sanity checks, calculates array sizes, and
  then gets the local part of the array and passes it to Draw3DBlock for
  triangulation.

  int DATriangulate Returns 0 or an error code.

  DA theda The PETSc distributed array object.

  Vec X PETSc vector object with data we'd like to graph.

  int this Index of the field we'd like to draw.

  PetscScalar *minmax Position of block corners: xmin, xmax, ymin, ymax, zmin, zmax.

  int n_quants Number of isoquant surfaces to draw (isoquant values), or
  PETSC_DECIDE to use red, yellow, green, blue at 0.2, 0.4, 0.6 and 0.8 between
  the vector's minimum and maximum values.

  PetscScalar *isoquants Array of function values at which to draw isoquants, or
  PETSC_NULL if n_quants=PETSC_DECIDE.

  PetscScalar *colors Array of color R,G,B,A quads for each isoquant, or PETSC_NULL
  if n_quants=PETSC_DECIDE.
  ++++++++++++++++++++++++++++++++++++++*/

int DATriangulate
(DA theda, Vec X, int this, PetscScalar *minmax, int n_quants,
 PetscScalar *isoquants, PetscScalar *colors)
{
  Vec localX;
  PetscScalar *x, isoquantdefaults[4], localminmax[6],
    colordefaults[16] = { 1,0,0,.5, 1,1,0,.5, 0,1,0,.5, 0,0,1,.5 };
  PetscReal themax, themin;
  int i, ierr, fields, xw,yw,zw, xs,ys,zs, xm,ym,zm, gxs,gys,gzs, gxm,gym,gzm;

  /* Get global and local grid boundaries, should check periodicity */
  ierr = DAGetInfo (theda, &i, &xw,&yw,&zw, PETSC_NULL,PETSC_NULL,PETSC_NULL,
		    &fields, PETSC_NULL, PETSC_NULL, PETSC_NULL);CHKERRQ(ierr);
  if (i!=3)
    SETERRQ (PETSC_ERR_ARG_WRONGSTATE, "DA must be 3-dimensional");
  ierr = DAGetCorners (theda, &xs,&ys,&zs, &xm,&ym,&zm); CHKERRQ (ierr);
  ierr = DAGetGhostCorners (theda, &gxs,&gys,&gzs, &gxm,&gym,&gzm);
  CHKERRQ (ierr);

  /* Default isoquants */
  if (n_quants == PETSC_DECIDE) {
    ierr = VecStrideMin (X, this, PETSC_NULL, &themin); CHKERRQ (ierr);
    ierr = VecStrideMax (X, this, PETSC_NULL, &themax); CHKERRQ (ierr);
    /* Red, yellow, green, blue at 0.2, 0.4, 0.6, 0.8, all with alpha=0.5 */
    n_quants = 4;
    isoquantdefaults[0] = themin + 0.2*(themax-themin);
    isoquantdefaults[1] = themin + 0.4*(themax-themin);
    isoquantdefaults[2] = themin + 0.6*(themax-themin);
    isoquantdefaults[3] = themin + 0.8*(themax-themin);
    isoquants = isoquantdefaults;
    colors = colordefaults;
  }

  /* Get the local array of points, with ghosts */
  ierr = DACreateLocalVector (theda, &localX); CHKERRQ (ierr);
  ierr = DAGlobalToLocalBegin (theda, X, INSERT_VALUES, localX); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd (theda, X, INSERT_VALUES, localX); CHKERRQ (ierr);
  ierr = VecGetArray (localX, &x); CHKERRQ (ierr);

  /* Calculate local physical size */
  localminmax[0] = minmax[0] + minmax[1]*xs/xw;
  localminmax[1] = minmax[0] + minmax[1]*(xs+xm)/xw;
  localminmax[2] = minmax[2] + minmax[3]*ys/yw;
  localminmax[3] = minmax[2] + minmax[3]*(ys+ym)/yw;
  localminmax[4] = minmax[4] + minmax[5]*zs/zw;
  localminmax[5] = minmax[4] + minmax[5]*(zs+zm)/zw;

  /* If the array is too big, cut it down to size */
  if (gxm <= xs-gxs+xm)
    xm = gxm-xs-gxs-1;
  if (gym <= ys-gys+ym)
    ym = gym-ys-gys-1;
  if (gzm <= zs-gzs+zm)
    zm = gzm-zs-gzs-1;

  /* Let 'er rip! */
  ierr = Draw3DBlock (gxm,gym,gzm, xs-gxs,ys-gys,zs-gzs, xm,ym,zm, localminmax,
		      x+this, fields, n_quants, isoquants, colors);
  CHKERRQ (ierr);
  VecRestoreArray (localX, &x);
  VecDestroy (localX); CHKERRQ (ierr);

  return 0;
}


#undef __FUNCT__
#define __FUNCT__ "IllErrorHandler"

/*++++++++++++++++++++++++++++++++++++++
  Handle errors, in this case the PETSc way.

  int IllErrorHandler Returns the error code supplied.

  int id Index of the error, defined in petscerror.h.

  char *message Text of the error message.
  ++++++++++++++++++++++++++++++++++++++*/

int IllErrorHandler (int id, char *message)
{
  SETERRQ (id, message);
  exit (1);
}
