#include <config.h>
#include <stdio.h>
#ifdef HAVE_UNIX_SOCKET
#include <sys/types.h>
#include <unistd.h>
#endif
#include "IMSvrCfgFile.hh"
#include "IMLog.hh"

#ifdef HAVE_TLS
#include "IMTLS.hh"
#endif

static const int config_file_line_max = 4096;
static const char *config_delimiter = " \t";

void
IMSvrCfgFile::initialize()
{
    adddir("ListenAddress", &IMSvrCfgFile::add_listen_address, 1);
    adddir("ListenFile", &IMSvrCfgFile::add_listen_file, 1);
    adddir("DefaultPermission", &IMSvrCfgFile::set_default_permission, 1);
    adddir("AccessControl", &IMSvrCfgFile::set_access_control, 2);
    adddir("AllowSystemUser", &IMSvrCfgFile::set_allow_system_user, 1);
    adddir("UserPermission", &IMSvrCfgFile::set_user_permission, -1);
    adddir("SSLListenFile", &IMSvrCfgFile::add_ssl_listen_file, 1);
    adddir("SSLListenAddress", &IMSvrCfgFile::add_ssl_listen_address, 1);
    adddir("SSLCertificateFile", &IMSvrCfgFile::add_certificate_file, 1);
    adddir("SSLCertificateKeyFile", &IMSvrCfgFile::add_certificate_keyfile, 1);
    adddir("SSLCACertificateFile", &IMSvrCfgFile::add_cacertificate_file, 1);
    adddir("SSLCACertificatePath", &IMSvrCfgFile::add_cacertificate_path, 1);
    adddir("SSLVerifyClient", &IMSvrCfgFile::set_verify_client, 1);
    adddir("SSLVerifyDepth", &IMSvrCfgFile::set_verify_depth, 1);
}

bool
IMSvrCfgFile::
set_verify_depth(DirArgs &args)
{
#ifdef HAVE_TLS
   string value;
   DirArgs::iterator it1 = args.begin();

   value  = *it1;

   return IMTLS::get_instance()->set_verify_depth(value);
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
set_verify_client(DirArgs &args)
{
#ifdef HAVE_TLS
    string value;
    DirArgs::iterator it1 = args.begin();

    value = *it1;

    return IMTLS::get_instance()->set_verify_client(value);
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_cacertificate_path(DirArgs &args)
{
#ifdef HAVE_TLS
    string dirname;
    DirArgs::iterator it1 = args.begin();

    filename = *it1;

    return IMTLS::get_instance()->set_cacertificate_path(dirname);
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_cacertificate_file(DirArgs &args)
{
#ifdef HAVE_TLS
    string filename;
    DirArgs::iterator it1 = args.begin();

    filename = *it1;

    return IMTLS::get_instance()->set_cacertificate_file(filename);
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_certificate_keyfile(DirArgs &args)
{
#ifdef HAVE_TLS
    string filename;
    DirArgs::iterator it1 = args.begin();

    filename = *it1;

    return IMTLS::get_instance()->set_certificate_key_file(filename);
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_certificate_file(DirArgs &args)
{
#ifdef HAVE_TLS
    string filename;
    DirArgs::iterator it1 = args.begin();

    filename = *it1;

    return IMTLS::get_instance()->set_certificate_file(filename);
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_ssl_listen_address(DirArgs &args)
{
#ifdef HAVE_TLS
    string address, service;
    string::size_type fpos;
    DirArgs::iterator it1 = args.begin();

    if (get_boolval(IMSvrCfg::USER))  {
      return true;
    }
    fpos = it1->rfind(':');
    if (fpos == string::npos) {
	address = *it1;
	service = get_strval(IMSvrCfg::PORT);
    } else {
	address = it1->substr(0, fpos);
	service = it1->substr(fpos + 1);
    }
    if (address.size() == 0) {
	LOG_ERROR("Invalid AddSSLListenAddress:%s", it1->c_str());
	return false;
    }

    listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::INET, IMSocketAddress::TLS, address, service));

    LOG_DEBUG("SSLAddListenAddress:%s", it1->c_str());
    return true;
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_ssl_listen_file(DirArgs &args)
{
#if defined(HAVE_TLS) && defined(HAVE_UNIX_SOCKET)
    string service;
    string::size_type fpos;
    DirArgs::iterator it1 = args.begin();

    service = *it1;

    if (service.size() == 0) {
	LOG_ERROR("Invalid AddListenFile:%s", it1->c_str());
    }

    if (get_boolval(IMSvrCfg::USER)) {
        string user_name;
        if (get_process_user(user_name)) {
            listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::UNIX_DOMAIN_PER_USER,
                                    IMSocketAddress::TLS, 
	         		    string("/tmp/.iiimp-unix-") + user_name, service));
        }
    } else {
        listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::UNIX_DOMAIN, 
                                IMSocketAddress::TLS,
                                "/tmp/.iiimp-unix", service));
    }

    LOG_DEBUG("SSLAddListenFile:%s", it1->c_str());
    return true;
#else
    LOG_WARNING("htt_server was compiled without TLS support. ignored");
    return true;
#endif
}

bool
IMSvrCfgFile::
add_listen_file(DirArgs &args)
{
#ifdef HAVE_UNIX_SOCKET
    string service;
    string::size_type fpos;
    DirArgs::iterator it1 = args.begin();

    service = *it1;

    if (service.size() == 0) {
	LOG_ERROR("Invalid AddListenFile:%s", it1->c_str());
    }

    if (get_boolval(IMSvrCfg::USER)) {
        string user_name;
        if (get_process_user(user_name))
            listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::UNIX_DOMAIN_PER_USER, 
				    string("/tmp/.iiimp-unix-") + user_name, service));
    } else {
        listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::UNIX_DOMAIN, "/tmp/.iiimp-unix", service));
    }

    LOG_DEBUG("AddListenFile:%s", it1->c_str());
#endif
    return true;
}

bool
IMSvrCfgFile::
add_listen_address(DirArgs &args)
{
    string address, service;
    string::size_type fpos;
    DirArgs::iterator it1 = args.begin();

    // when  the htt_server invokes as a per-user daemon,
    // doesn't listen to TCP ports.
    if (get_boolval(USER))  {
        return true;
    }
    fpos = it1->rfind(':');
    if (fpos == string::npos) {
	address = *it1;
	service = get_strval(IMSvrCfg::PORT);
    } else {
	address = it1->substr(0, fpos);
	service = it1->substr(fpos + 1);
    }
    if (address.size() == 0) {
	LOG_ERROR("Invalid AddListenAddress:%s", it1->c_str());
	return false;
    }

    listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::INET, address, service));

    LOG_DEBUG("AddListenAddress:%s", it1->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_access_control(DirArgs &args)
{
    DirArgs::iterator it1 = args.begin();
    DirArgs::iterator it2 = it1;
    it2++;
    IMAuth::access_type at = get_access_type(*it1);

    if (at == IMAuth::UNKNOWN) return false;
    get_usermgr(ptarget)->set_entry(it2->c_str(), at);

    LOG_DEBUG("AccessControl(%s):%s", it1->c_str(), it2->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_default_permission(DirArgs &args)
{
    DirArgs::iterator it1 = args.begin();
    IMAuth::access_type at = get_access_type(*it1);

    if (at == IMAuth::UNKNOWN) return false;
    get_usermgr(ptarget)->set_default_entry(at);

    LOG_DEBUG("SetDefaultPermission(%s)", it1->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_allow_system_user(DirArgs &args)
{
    IMAuth::access_type at = get_access_type(*(args.begin()));
    if (at == IMAuth::UNKNOWN) return false;
    get_usermgr(ptarget)->set_systemuser_permission(at);
    LOG_DEBUG("AllowSystemUser(%s)", args.begin()->c_str());
    return true;
}

bool
IMSvrCfgFile::
set_user_permission(DirArgs &args)
{
    DirArgs::iterator it;
    IMAuth::access_type at;
    string::size_type pos;
    string username, password;

    it = args.begin();
    if (it == args.end()) return false;
    at = get_access_type(*it);
    if (at == IMAuth::UNKNOWN) return false;
    LOG_DEBUG("UserPermission(%s)", it->c_str());
    it++;
    if (it == args.end()) return false;
    for (; it != args.end(); it++) {
	username = *it;
	LOG_DEBUG("  add user:%s", username.c_str());
	pos = username.find(':');
	if (pos == string::npos) {
	    get_usermgr(ptarget)->add_user(username.c_str(),
					   NULL, at);
	} else {
	    password = username.substr(pos + 1);
	    username = username.substr(0, pos);
	    get_usermgr(ptarget)->add_user(username.c_str(),
					   password.c_str(), at);
	}
    }
    return true;
}

IMAuth::access_type
IMSvrCfgFile::
get_access_type(
    string &key
)
{
    if (key.compare("deny") == 0)
	return IMAuth::DENY;
    else if (key.compare("permit") == 0)
	return IMAuth::PERMIT;
    else if (key.compare("checkuser") == 0)
	return IMAuth::CHECKUSER;
    else if (key.compare("password") == 0)
	return IMAuth::PASSWORD;
    else
	return IMAuth::UNKNOWN;
}

bool
IMSvrCfgFile::
readconfig()
{
    FILE *fp;
    char buf[config_file_line_max];
    char token[config_file_line_max];
    char *p;
    int i, argsnum, lineno;
    bool flag = true;
    DirectiveMap::iterator it;

    fp = fopen(filename.c_str(), "rw");
    if (!fp) return true;
    lineno = 0;
    while (fgets(buf, sizeof(buf), fp))	{
	lineno++;
	if (p = strchr(buf, '\n')) *p = '\0';
	if (buf[0] == '#') continue;
	p = strtok(buf, config_delimiter);
	if (!p) continue;

	it = dirmap.find(p);
	if (it == dirmap.end()) {
	    LOG_ERROR("Unknown directive (%d):%s\n", lineno, p);
	    flag = false;
	    break;
	}
	argsnum = it->second.argsnum;
	DirArgs args;
	if (argsnum >= 0) {
	    for (i = 0; i < argsnum; i++) {
		p = strtok(NULL, config_delimiter);
		if (!p) {
		    LOG_ERROR("too few arguments in directive (%d):%s\n",
			      lineno, it->first);
		}
		args.push_back(string(p));
	    }
	} else {
	    while (p = strtok(NULL, config_delimiter)) {
		args.push_back(string(p));
	    }
	}
	DirHandler h = it->second.h;
	if (!(this->*h)(args)) {
	    LOG_ERROR("Directive error (%d):%s", lineno, it->first);
	    flag = false;
	    break;
	}
    }

    fclose(fp);
    return flag;
}

void
IMSvrCfgFile::
adddir(
    const char *dir,
    DirHandler h,
    int argsnum
)
{
    Directive d;
    d.argsnum = argsnum;
    d.h = h;
    dirmap[dir] = d;
}

bool
IMSvrCfgFile::configure(
    IMSvr *pimsvr
)
{
    ptarget = pimsvr;
    /* ...configuration ... */
    if (!readconfig()) return false;
#ifdef HAVE_TLS
    IMTLS::get_instance()->setup();
#endif
    /* config listen address if it has not been configured yet. */
    if (listenaddrvec.empty()) {
        if (get_boolval(USER)) {
            // when htt_server run as a per-user daemon.
            // listens to /tmp/.iiimp-unix-${USER}/${PORT}
            string user_name;
            if (get_process_user(user_name))
                listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::UNIX_DOMAIN_PER_USER,
                                                  string("/tmp/.iiimp-unix-") + user_name,
                                                  get_strval(IMSvrCfg::PORT)));
        } else {
            // by default, htt_server listens to
            // localhost:9010, /tmp/.iiimp-unix/${PORT}
	    listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::INET,
                                              get_strval(IMSvrCfg::HOSTNAME),
					      get_strval(IMSvrCfg::PORT)));
            listenaddrvec.push_back(IMSocketAddress(IMSocketAddress::UNIX_DOMAIN,
                                              string("/tmp/.iiimp-unix"),
                                              get_strval(IMSvrCfg::PORT)));
        }
    }
    config_listenaddress(pimsvr, listenaddrvec);


    return true;
}

IMSvrCfgFile::IMSvrCfgFile(
    IMSvrCfg *pbase,
    const char *x_filename
) :
    IMSvrCfg(*pbase), filename(x_filename)
{
    initialize();
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
