//
// (c) Polar Pyramid                                                                                                             
// suggestions to stolken@kabelfoon.nl and/or stolk@xs4all.nl
//

#include <math.h>
#include <assert.h>
#include <iostream.h>

#include "vector3.h"

/////////////////////////////////////////////////////////////////////////////
//
// Default Constructor
//

 Vector3::Vector3()

{
	x = 0;
	y = 0;
	z = 0;

	interp = true;
}

/////////////////////////////////////////////////////////////////////////////
//
// Copy Constructor
//

 Vector3::Vector3(const Vector3& v)
  : x(v.x), y(v.y), z(v.z)
{
	interp = v.interp;
}

/////////////////////////////////////////////////////////////////////////////
//
// Element Constructor
//

 Vector3::Vector3(float ix, float iy, float iz)
  : x(ix), y(iy), z(iz)
{
	interp = true;
}

///////////////////////////////////////////////////////////////////
//
// Interpolation properties
//
 bool Vector3::isInterpolatable(){
	return interp;
}

 void Vector3::setInterpolatable(bool flag)
{
	interp = flag;
}

/////////////////////////////////////////////////////////////////////////////
//
// Array Access
//

 float& Vector3::operator [](int i)
{
  switch (i)
  {
  case 0: return x;
  case 1: return y;
  case 2: return z;
  }
  assert(false);
  return x;
}

 const float& Vector3::operator [](int i) const
{
  switch (i)
  {
  case 0: return x;
  case 1: return y;
  case 2: return z;
  }
  assert(false);
  return x;
}

/////////////////////////////////////////////////////////////////////////////
//
// Assignment
//

 Vector3& Vector3::assign(float ix, float iy, float iz)
{
  x = ix;
  y = iy;
  z = iz;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
// Math Operators
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// - Vector3
//

 Vector3 Vector3::operator -() const
{
  return Vector3(-x, -y, -z);
}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 += Vector3
//

 Vector3& Vector3::operator += (const Vector3& v)
{
  x += v.x;
  y += v.y;
  z += v.z;
  return *this;
}


/////////////////////////////////////////////////////////////////////////////
//
// Vector3 -= Vector3
//

 Vector3& Vector3::operator -= (const Vector3& v)
{
  x -= v.x;
  y -= v.y;
  z -= v.z;
  return *this;
}


/////////////////////////////////////////////////////////////////////////////
//
// Vector3 *= float
//


 Vector3& Vector3::operator *= (float c)
{
  x *= c;
  y *= c;
  z *= c;
  return *this;
}
////////////////////////////////////////////////////////////////
//
// Scale the vector
//
 void Vector3::scale(float _x, float _y, float _z)
{
	x *= _x;
	y *= _y;
	z *= _z;

}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 /= float
//



 Vector3& Vector3::operator /= (float c)
{
  assert(c != 0.0f);
  float inv = 1.0f/c;
  x *= inv;
  y *= inv;
  z *= inv;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 + Vector3
//


 Vector3 Vector3::operator + (const Vector3& v) const
{
  return Vector3(x + v.x,
		 y + v.y,
		 z + v.z);
}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 - Vector3
//


 Vector3 Vector3::operator - (const Vector3& v) const
{
  return Vector3(x - v.x,
		 y - v.y,
		 z - v.z);
}


/////////////////////////////////////////////////////////////////////////////
//
// Vector3 / float
//


 Vector3 Vector3::operator / (float c) const
{
  assert(c != 0.0f);
  return Vector3(x/c,
		 y/c,
		 z/c);
}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 == Vector3
//


 bool Vector3::operator == (const Vector3& v) const
{
  return ((x == v.x) &&
          (y == v.y) &&
          (z == v.z));
}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 != Vector3
//


 bool Vector3::operator != (const Vector3& v) const
{
  return (!(*this == v));
}

/////////////////////////////////////////////////////////////////////////////
//
// float * Vector3
//


 Vector3 operator * (float c, Vector3 const &v)
{
  return Vector3(v.x*c,
		 v.y*c,
		 v.z*c);
}

/////////////////////////////////////////////////////////////////////////////
//
// Vector3 * float
//


 Vector3 operator * (Vector3 const &v, float c)
{
  return Vector3(v.x*c,
		 v.y*c,
		 v.z*c);
}

 void Vector3::Dump(void) const
{
  cout << "Vector3 " << x << "," << y << "," << z << "\n";
}

/////////////////////////////////////////////////////////////////////////////
//
// length : returns the length of the vector3 as a float, note that
//          this has to return a float because it uses the sqrt function
//

 float Vector3::length() const
{
  return sqrt(lengthSqr());
}

/////////////////////////////////////////////////////////////////////////////
//
// lengthSqr : returns the squared length of the vector3
//


 float Vector3::lengthSqr() const
{
  return x*x + y*y + z*z;
}

/////////////////////////////////////////////////////////////////////////////
//
// length2d : returns the length of the vector3 as a float, note that
//          this has to return a float because it uses the sqrt function
//


 float Vector3::length2d() const
{
  return sqrt(lengthSqr2d());
}

/////////////////////////////////////////////////////////////////////////////
//
// lengthSqr3d : returns the squared 2d length of the vector3
//

 float Vector3::lengthSqr2d() const
{
  return x*x + y*y;
}

/////////////////////////////////////////////////////////////////////////////
//
// normalize : will normalize the vector3, make it of length 1
//

 void Vector3::normalize(void)
{
  float len = length();
  if (len != 0.0)
    *this /= ((float) len);      // avoid SIGFPE
}

/////////////////////////////////////////////////////////////////////////////
//
// dot : returns the dot product of 2 vectors
//

 float Vector3::dot(const Vector3& v2) const
{
  return x*v2.x + y*v2.y + z*v2.z;
}

/////////////////////////////////////////////////////////////////////////////
//
// dot2d : returns the dot product of 2 vector3s for 2d only
//

 float Vector3::dot2d(const Vector3& v2) const
{
  return x*v2.x + y*v2.y;
}

/////////////////////////////////////////////////////////////////////////////
//
// * operator : returns the cross product of 2 vector3s
//

 Vector3 Vector3::operator *(const Vector3& v) const
{
  return Vector3(y*v.z - z*v.y,
				 -x*v.z + z*v.x,
				 x*v.y - y*v.x);
}

/////////////////////////////////////////////////////////////////////////////
//
// distanceSqr : return the squared distance between 2 given points which
//               avoids taking an expensive sqrt call. Usefull when comparing
//               different distances.
//

 float Vector3::distanceSqr(const Vector3& v2) const
{
  return (x - v2.x) * (x - v2.x) +
	 (y - v2.y) * (y - v2.y) +
	 (z - v2.z) * (z - v2.z);
}

/////////////////////////////////////////////////////////////////////////////
//
// distance : return the distance between 2 given points, note that
//            this has to return a float because it uses the sqrt function
//

 float Vector3::distance(const Vector3& v2) const
{
  return sqrt(distanceSqr(v2));
}

/////////////////////////////////////////////////////////////////////////////
//
// distanceSqr2d : return the squared distance between 2 given points in 2d
//                 only (z is ignored)
//

 float Vector3::distanceSqr2d(const Vector3& v2) const
{
  return (x - v2.x) * (x - v2.x) +
	 (y - v2.y) * (y - v2.y);
}

/////////////////////////////////////////////////////////////////////////////
//
// distance : return the distance between 2 given points, note that
//            this has to return a float because it uses the sqrt function
//

 float Vector3::distance2d(const Vector3& v2) const
{
  return sqrt(distanceSqr2d(v2));
}




