/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkPathToImageFilter.h,v $
  Language:  C++
  Date:      $Date: 2005-07-27 15:21:11 $
  Version:   $Revision: 1.3 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkPathToImageFilter_h
#define __itkPathToImageFilter_h

#include "itkImageSource.h"
#include "itkConceptChecking.h"

namespace itk
{
  
/** \class PathToImageFilter
 * \brief Base class for filters that take a Path as input and produce an image as output.
 * Base class for filters that take a Path as input and produce an image as
 * output. By default, if the user does not specify the size of the output
 * image, the maximum size of the path's bounding box is used.  The default
 * spacing of the image is given by the spacing of the input  path (currently
 * assumed internally to be 1.0).
 */
template <class TInputPath, class TOutputImage>
class ITK_EXPORT PathToImageFilter : public ImageSource<TOutputImage>
{
public:
  /** Standard class typedefs. */
  typedef PathToImageFilter  Self;
  typedef ImageSource<TOutputImage>  Superclass;
  typedef SmartPointer<Self>  Pointer;
  typedef SmartPointer<const Self>  ConstPointer;
  
  /** Method for creation through the object factory. */
  itkNewMacro(Self);
  
  /** Run-time type information (and related methods). */
  itkTypeMacro(PathToImageFilter,ImageSource);
  
  /** Some convenient typedefs. */
  typedef typename Superclass::OutputImageRegionType  OutputImageRegionType;    
  typedef          TInputPath                         InputPathType;
  typedef typename InputPathType::Pointer             InputPathPointer;
  typedef typename InputPathType::ConstPointer        InputPathConstPointer;
  typedef          TOutputImage                       OutputImageType;          
  typedef typename OutputImageType::Pointer           OutputImagePointer;       
  typedef typename OutputImageType::SizeType          SizeType;                 
  typedef typename OutputImageType::ValueType         ValueType;                

  /** ImageDimension constants */
  itkStaticConstMacro(OutputImageDimension, unsigned int,
                      TOutputImage::ImageDimension);

  /** Set/Get the path input of this process object.  */
  virtual void SetInput( const InputPathType *path);
  virtual void SetInput( unsigned int, const TInputPath * path);
  const InputPathType * GetInput(void);
  const InputPathType * GetInput(unsigned int idx);

  /** Spacing (size of a pixel) of the output image. The
   * spacing is the geometric distance between image samples.
   * It is stored internally as double, but may be set from
   * float. \sa GetSpacing() */
  virtual void SetSpacing( const double* spacing);
  virtual void SetSpacing( const float* spacing);
  virtual const double* GetSpacing() const;

  /** Set/Get the value for pixels on and off the path. 
  * By default, this filter will return a "0" image with path pixels set to 1 */
  itkSetMacro(PathValue, ValueType);
  itkGetMacro(PathValue, ValueType);
  itkSetMacro(BackgroundValue, ValueType);
  itkGetMacro(BackgroundValue, ValueType);

  /** The origin of the output image. The origin is the geometric
   * coordinates of the index (0,0,...,0).  It is stored internally
   * as double but may be set from float.
   * \sa GetOrigin() */
  virtual void SetOrigin( const double* origin);
  virtual void SetOrigin( const float* origin);
  virtual const double * GetOrigin() const;

  /** Set/Get Size */
  itkSetMacro(Size,SizeType);
  itkGetMacro(Size,SizeType);

protected:
  PathToImageFilter();
  ~PathToImageFilter();

  virtual void GenerateOutputInformation(){}; // do nothing
  virtual void GenerateData();


  SizeType     m_Size;
  double       m_Spacing[OutputImageDimension];
  double       m_Origin[OutputImageDimension];
  ValueType    m_PathValue;
  ValueType    m_BackgroundValue;

  virtual void PrintSelf(std::ostream& os, Indent indent) const;

private:
  PathToImageFilter(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

};

} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkPathToImageFilter.txx"
#endif

#endif
