/*  Color management example.
 *
 *  color.cc
 *
 *  Inti Translation of the GtkGLExt color management example
 *  written by Naofumi Yasufuku <naofumi@users.sourceforge.net>.
 */

#include "color.h"
#include <inti/gdk/color.h>
#include <inti/gdk/gl/drawable.h>
#include <inti/gdk/gl/query.h>
#include <inti/gtk/button.h>
#include <inti/gtk/gl/init.h>
#include <gdk/gdkgltokens.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <iostream>

namespace {

const int TIMEOUT_INTERVAL = 10;

} // namespace

/*  GLDrawingArea
 */

GLDrawingArea::GLDrawingArea(Gdk::GL::Config *glconfig)
{
	set_size_request(200, 200);
	set_gl_capability(*glconfig, true, glconfig->is_rgba() ? GDK_GL_RGBA_TYPE : GDK_GL_COLOR_INDEX_TYPE);

	examine_attrib(*glconfig);

	colors.push_back(Gdk::Color()); // black
	colors.push_back(Gdk::Color(65535, 0, 0)); // red
	colors.push_back(Gdk::Color(0, 65535, 0)); // green
	colors.push_back(Gdk::Color(0, 0, 65535)); // blue

	if (!glconfig->is_rgba())
	{
		using namespace std;

		cout <<  "\nAllocate colors.\n";

		// Allocate writable color cells.
		Gdk::Colormap *colormap = glconfig->get_colormap();
		vector<bool> successes;
		int not_allocated = colormap->alloc_colors(colors, successes, colors.size(), true);
		cout << "Not allocated = " << not_allocated << endl;

		for (unsigned i = 0; i < colors.size(); i++)
		{
			cout << "colors[" << i << "] = {" << colors[i].get_pixel() << ", " << colors[i].get_red();
			cout << ", " << colors[i].get_green() << ", " << colors[i].get_blue() << "}" << endl;
		}

		cout << "\nQuery colors.\n";

		Gdk::Color color;
		for (unsigned i = 0; i < colors.size(); i++)
		{
			color.set_pixel(colors[i].get_pixel());
			colormap->query_color(colors[i].get_pixel(), color);
			cout << "colors[" << i << "] = {" << colors[i].get_pixel() << ", " << color.get_red();
			cout << ", " << color.get_green() << ", " << color.get_blue() << "}" << endl;
		}
		cout << endl;
	}
}

GLDrawingArea::~GLDrawingArea()
{
}

bool
GLDrawingArea::on_configure_event(const Gdk::EventConfigure& event)
{
	if (!gl_begin()) // OpenGL begin
		return false;

	glViewport(0, 0, get_allocation().width(), get_allocation().height());

	gl_end(); // OpenGL end
	return true;
}

bool
GLDrawingArea::on_expose_event(const Gdk::EventExpose& event)
{
	if (!gl_begin()) // OpenGL begin
		return false;

	glClear(GL_COLOR_BUFFER_BIT);

	glBegin(GL_TRIANGLES);
	glIndexi(colors[1].get_pixel());
	glColor3f(1.0, 0.0, 0.0);
	glVertex2i(0, 1);
	glIndexi(colors[2].get_pixel());
	glColor3f(0.0, 1.0, 0.0);
	glVertex2i(-1, -1);
	glIndexi(colors[3].get_pixel());
	glColor3f(0.0, 0.0, 1.0);
	glVertex2i(1, -1);
	glEnd();

	if (is_double_buffered())
		swap_buffers();
	else
		glFlush();

	gl_end(); // OpenGL end
	return true;
}

void
GLDrawingArea::examine_attrib(Gdk::GL::Config& glconfig)
{
	using namespace std;

	cout << "\nOpenGL visual configurations :\n\n";
	cout << "Gdk::GL::Config::is_rgba() = " << (glconfig.is_rgba() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::is_double_buffered() = " << (glconfig.is_double_buffered() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::is_stereo() = " << (glconfig.is_stereo() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_alpha() = " << (glconfig.has_alpha() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_depth_buffer() = " << (glconfig.has_depth_buffer() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_stencil_buffer() = " << (glconfig.has_stencil_buffer() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_accum_buffer() = " << (glconfig.has_accum_buffer() ? "true" : "false") << endl;
	cout << endl;

	print_attrib(glconfig, "GDK_GL_USE_GL", GDK_GL_USE_GL, true);
	print_attrib(glconfig, "GDK_GL_BUFFER_SIZE", GDK_GL_BUFFER_SIZE, false);
	print_attrib(glconfig, "GDK_GL_LEVEL", GDK_GL_LEVEL, false);
	print_attrib(glconfig, "GDK_GL_RGBA", GDK_GL_RGBA, true);
	print_attrib(glconfig, "GDK_GL_DOUBLEBUFFER",  GDK_GL_DOUBLEBUFFER, true);
	print_attrib(glconfig, "GDK_GL_STEREO", GDK_GL_STEREO, true);
	print_attrib(glconfig, "GDK_GL_AUX_BUFFERS", GDK_GL_AUX_BUFFERS, false);
	print_attrib(glconfig, "GDK_GL_RED_SIZE", GDK_GL_RED_SIZE, false);
	print_attrib(glconfig, "GDK_GL_GREEN_SIZE", GDK_GL_GREEN_SIZE, false);
	print_attrib(glconfig, "GDK_GL_BLUE_SIZE",  GDK_GL_BLUE_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ALPHA_SIZE",  GDK_GL_ALPHA_SIZE, false);
	print_attrib(glconfig, "GDK_GL_DEPTH_SIZE",  GDK_GL_DEPTH_SIZE, false);
	print_attrib(glconfig, "GDK_GL_STENCIL_SIZE", GDK_GL_STENCIL_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_RED_SIZE", GDK_GL_ACCUM_RED_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_GREEN_SIZE", GDK_GL_ACCUM_GREEN_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_BLUE_SIZE", GDK_GL_ACCUM_BLUE_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_ALPHA_SIZE", GDK_GL_ACCUM_ALPHA_SIZE, false);

	cout << endl;
}

void
GLDrawingArea::print_attrib(Gdk::GL::Config& glconfig, const char *attrib_str, int attrib, bool is_boolean)
{
	using namespace std;

	cout << attrib_str << " = ";

	int value;
	if (glconfig.get_attrib(attrib, value))
	{
		if (is_boolean)
			cout << (value ? "true" : "false") << endl;
		else
			cout << value << endl;
	}
	else
		cout << "*** Cannot get " << attrib_str << " attribute value" << endl;
}

/*  Window
 */

Window::Window(Gdk::GL::Config *glconfig)
{
	set_title("Color Example");

	// Perform the resizes immediately.
	set_resize_mode(Gtk::RESIZE_IMMEDIATE);

	// Get automatically redrawn if any of their children changed allocation.
	set_reallocate_redraws(true);

	Gtk::VBox *vbox = new Gtk::VBox;
	add(*vbox);
	vbox->show();

	// Create an OpenGL-capable drawing area.
	GLDrawingArea *drawing_area = new GLDrawingArea(glconfig);
	vbox->pack_start(*drawing_area);
	drawing_area->show();

	// Simple quit button.
	Gtk::Button *button = new Gtk::Button("Quit");
	button->sig_clicked().connect(slot(&Main::quit));
	vbox->pack_start(*button, false, false);
	button->show();
	show();
}

Window::~Window()
{
}

int main (int argc, char *argv[])
{
	using namespace Main;

	init(&argc, &argv);

	// Initialize GtkGLExt
	Gtk::GL::init(&argc, &argv);

	// Query OpenGL extension version.
	int major, minor;
	Gdk::GL::query_version(&major, &minor);
	std::cout << "\nOpenGL extension version - " << major << "." << minor << std::endl;

	// Display mode.
	Gdk::GL::ConfigMode mode = Gdk::GL::MODE_RGB;

	for (int i = 0; i < argc; i++)
	{
		if (strcmp(argv[i], "--ci") == 0)
			mode = Gdk::GL::MODE_INDEX;
	}

	// Configure OpenGL-capable visual.
	Pointer<Gdk::GL::Config> glconfig = Gdk::GL::Config::create(mode | Gdk::GL::MODE_DOUBLE);
	if (!glconfig)
	{
		std::cout << "*** Cannot find the double-buffered visual." << std::endl;
		std::cout << "*** Trying single-buffered visual." << std::endl;

		// Try single-buffered visual
		glconfig = Gdk::GL::Config::create(mode);
		if (!glconfig)
		{
			std::cout << "*** No appropriate OpenGL-capable visual found." << std::endl;
			return 1;
		}
	}

	// Top-level window.
	Window window(glconfig);
	window.sig_destroy().connect(slot(&Inti::Main::quit));

	run();
	return 0;
}

