/*
 *
 * $Id: postgre.c, v 0.9 2001/04/06 23:22:45 kaiser Exp $
 *
 * postgresql backend to ipac
 * Copyright (C) 2001 Al Zaharov
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author can be reached via email: kaiser13@mail2000.ru, or by 
 * fido: Al_Zaharov@p88.f58.n5005.z2.fidonet.org
 *
*/

#include "ipac.h"
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/param.h>
#include <unistd.h>
#include <libpq-fe.h>

//#define DEBUG_POST
#undef DEBUG_POST

static char *pghost = NULL;
static char *pgport = NULL;
static char *pgoptions = NULL;
static char *pgtty = NULL;
static char *dbname = "ipac";

#ifdef DEBUG_POST
static char *log_file = "/post_log";
static FILE *logs;
#endif

static PGconn *conn;
static PGresult *res;

/* plain file ipac interface entries */
int postgre_ipac_open(int flag);
int postgre_ipac_store_record(const data_record_type *data);
int postgre_ipac_list_timestamps(timestamp_t start, timestamp_t end,
		timestamp_t **data, timestamp_t *just_before,
		timestamp_t *just_after);
int postgre_ipac_get_records(timestamp_t timestamp_b, timestamp_t timestamp_e, data_record_type **data);
int postgre_ipac_delete_record(timestamp_t timestamp);
int postgre_ipac_get_raw_list(char *ag_name, char *login, raw_rule_type **data);
void postgre_ipac_close();

static const storage_method_t interface_entry = {
	"postgre",
	postgre_ipac_open,
	postgre_ipac_store_record,
	postgre_ipac_list_timestamps,
	postgre_ipac_get_records,
	postgre_ipac_delete_record,
	postgre_ipac_close
};

const storage_method_t *ipac_sm_interface_postgre() {
	return &interface_entry;
}

int postgre_ipac_open(int flag)
{
	conn=PQsetdb(pghost, pgport, pgoptions, pgtty, dbname);
	if(PQstatus(conn) == CONNECTION_BAD) {
    		fprintf(stderr, "Connection to database '%s' failed.\n", dbname);
    		fprintf(stderr, "%s", PQerrorMessage(conn));
    		PQfinish(conn);
    		return(1);
	}
#ifdef DEBUG_POST
	logs=fopen(log_file, "a+");
	fprintf(logs, "%lu : postgre_ipac_open\n", time(NULL));
#endif        
	storage_opened = 1;
	return(0);
}

int postgre_ipac_store_record(const data_record_type *data)
{
    rule_type *rule, *firstrule;
    char wh_exec[380];

    firstrule = data->firstrule;

//    if (firstrule == NULL) return 0;

    res = PQexec(conn, "BEGIN");
    PQclear(res);
    for (rule = firstrule; rule; rule=rule->next)
    {
        sprintf(wh_exec, "insert into logs (rule_name, bytes, pkts, that_time) "
          "values ('%s', %Lu, %Lu, %lu)", rule->name, rule->bytes, rule->pkts, data->timestamp);
#ifdef DEBUG_POST
	fprintf(logs, "%lu : postgre_ipac_store_record\n", time(NULL));
	fprintf(logs, "%s\n", wh_exec);
	fflush(logs);
#endif	
        res = PQexec(conn, wh_exec);
	PQclear(res);
    }
    res = PQexec(conn, "COMMIT");
    PQclear(res);
    return 0;
}

int postgre_ipac_list_timestamps(timestamp_t start, timestamp_t end,
		timestamp_t **data, timestamp_t *just_before,
		timestamp_t *just_after)
{
    int i;    
    timestamp_t *ts, *ts_list=NULL;
    int ts_list_len = 0;
    int ts_list_used = 0;

    res = PQexec(conn, "BEGIN");
    PQclear(res);

    res = PQexec(conn, "declare mycursor binary cursor for select distinct that_time from logs");
    PQclear(res);

    if (just_before != NULL)
    	    *just_before = (timestamp_t)-1;
    if (just_after != NULL)
	    *just_after = (timestamp_t)-1;

    res = PQexec(conn, "FETCH ALL in mycursor");

    for (i=0; i<PQntuples(res); i++)
    {
	ts = (timestamp_t *) PQgetvalue(res, i, 0);
	if (*ts < start)
	    {
		if (just_before != NULL &&
			(*just_before==(timestamp_t)-1
				|| *just_before < *ts))
			*just_before = *ts;
		continue;
	    }
	if (*ts > end)
	{
		if (just_after != NULL &&
				(*just_after == (timestamp_t)-1
				|| *just_after > *ts))
			*just_after = *ts;
		continue;
	}

	if (ts_list_used == ts_list_len) {
		if (ts_list_len == 0)
			ts_list_len = 8;
		ts_list_len <<= 1;
		ts_list = (timestamp_t *)xrealloc(ts_list,
				ts_list_len * sizeof(timestamp_t));
	}
	ts_list[ts_list_used++] = *ts;
    }
    PQclear(res);

    res = PQexec(conn, "CLOSE mycursor");
    PQclear(res);
    res = PQexec(conn, "COMMIT");
    PQclear(res);

    qsort(ts_list, ts_list_used, sizeof(timestamp_t), compare_timestamp_t);
    *data = ts_list;

    return ts_list_used;
}


int postgre_ipac_get_records(timestamp_t timestamp_b, timestamp_t timestamp_e, 
					    data_record_type **data)
{
    rule_type *r, *r1=NULL;
    char *buf;
    int i, nr_timestamps, index;
    timestamp_t timestamp_akt;
    char wh_exec[320];
    char *pkts;
    char *bytes;

    if (!timestamp_e) timestamp_e=timestamp_b;

    res = PQexec(conn, "BEGIN");
    PQclear(res);
    
    sprintf(wh_exec, "select count(distinct that_time) from logs where "
	             "that_time between %lu and %lu", timestamp_b, timestamp_e);
    res = PQexec(conn, wh_exec);
    nr_timestamps=strtol(PQgetvalue(res,0,0),0,0);
    PQclear(res);

    /* create record_data_type. */
    *data = (data_record_type *)xmalloc(sizeof(data_record_type) * nr_timestamps);

    sprintf(wh_exec, "select rule_name, bytes, pkts, that_time "
	             "from logs where that_time between %lu and %lu order by "
		     " that_time", timestamp_b, timestamp_e);
#ifdef DEBUG_POST
    fprintf(logs, "%lu : postgre_ipac_get_record\n", time(NULL));
    fprintf(logs, "%s\n", wh_exec);
    fflush(logs);
#endif	

    res = PQexec(conn, wh_exec);
    timestamp_akt=0;index=-1;
    for (i=0; i<PQntuples(res); i++) {
        timestamp_t tstamp_new = strtoull((char *) PQgetvalue(res,i,3),NULL,0);
        if (tstamp_new != timestamp_akt) // do we have a new timestamp?
        {
                timestamp_akt=tstamp_new;
                index++;
                if (index > nr_timestamps) {
                        fprintf(stderr,"We got more records then timestamps "
			    "were reported before. This should not happen\n");
                        break;
                }
                (*data)[index].timestamp = timestamp_akt;
                (*data)[index].machine_name = xstrdup(hostname);
                (*data)[index].firstrule = NULL;
                r1 = NULL;
        }
	r = new_rule();
	if (r1 == NULL)
	    (*data)[index].firstrule = r;
	else
	    r1->next = r;
	r1 = r;
	buf = (char *) PQgetvalue(res, i, 0);
	bytes = (char *) PQgetvalue(res, i, 1);
	r->bytes = strtoull(bytes, &pkts, 0);
	pkts = (char *) PQgetvalue(res, i, 2);
	r->pkts =  strtoull(pkts, &bytes, 0);
	strncpy(r->name, buf, PQgetlength(res, i, 0));
#ifdef DEBUG_POST
	fprintf(logs, "Record: %s, bytes %s, pkts %s\n", r->name, bytes,
							    pkts);
	fflush(logs);
#endif	
	r->name[PQgetlength(res, i, 0)] = '\0';
    }
    PQclear(res);

    res = PQexec(conn, "COMMIT");
    PQclear(res);
    return index+1;
}

int postgre_ipac_delete_record(timestamp_t timestamp)
{
        char wh_exec[320];

        res = PQexec(conn, "BEGIN");
        PQclear(res);

        sprintf(wh_exec, "delete from logs where that_time=%lu", timestamp);
#ifdef DEBUG_POST
        fprintf(logs, "%lu : postgre_ipac_delete_record\n", time(NULL));
        fprintf(logs, "%s\n", wh_exec);
        fflush(logs);
#endif	
        PQexec(conn, wh_exec);
        PQclear(res);
        res = PQexec(conn, "COMMIT");
        PQclear(res);
        return 0;
}

int postgre_ipac_rule_active(char *rule_name)
{
	int tmp=1;
	char wh_exec[300];

	res = PQexec(conn, "BEGIN");
	PQclear(res);

	sprintf(wh_exec, "declare mycursor binary cursor for "
                     "SELECT active FROM active WHERE rule_name='%s'", rule_name);
	res = PQexec(conn, wh_exec);
	PQclear(res);
    
	res = PQexec(conn, "FETCH ALL in mycursor");
	if (PQntuples(res) != 0) 
	        tmp = *(int *) PQgetvalue(res, 0, 0);
	PQclear(res);

	res = PQexec(conn, "CLOSE mycursor");
	PQclear(res);
	res = PQexec(conn, "COMMIT");
	PQclear(res);
    
	return tmp;
}

int postgre_ipac_get_raw_list(char *ag_name, char *login, raw_rule_type **data)
{
        char buf[450];
        char buf1[100];
	raw_rule_type *r, *r1;
	int i;
	
	res = PQexec(conn, "BEGIN");
	PQclear(res);
	sprintf(buf, "declare mycursor binary cursor for SELECT "
		"login||' - '||serv_name, ip_addr, ip_net, sport, dport, dest, "
		"proto, iface from customers, services_names, services, nets, "
		"tariffs, nets_names "
		"where tariffs.tariff_id=services.tariff_id and "
		"tariffs.cust_id=customers.cust_id and "
		"services_names.serv_id=services.serv_id and "
		"nets.net_id=services.net_id and "
		"nets_names.net_id=nets.net_id ");
	if (*login) {
		sprintf(buf1, "and login='%s' ", login);
		strcpy(buf+strlen(buf), buf1);
	}
	if (*ag_name) {
		sprintf(buf1, "and agent='%s' ", ag_name);
		strcpy(buf+strlen(buf), buf1);
	}
	strcpy(buf+strlen(buf), "order by nets_names.glob\n");
#ifdef DEBUG_POST
	fprintf(logs,"%lu : postgre_ipac_get_raw_list\n%s\n", time(NULL), buf);
	fflush(logs);
#endif
	res = PQexec(conn, buf);
	PQclear(res);
	res = PQexec(conn, "FETCH ALL in mycursor");
	r1 = NULL;
	for (i=0; i<PQntuples(res); i++) {
		r = new_raw_rule();
		if (r1 == NULL)
			*data = r;
		else
			r1->next = r;
		r1 = r;
		strncpy(r->name, PQgetvalue(res, i, 0), PQgetlength(res, i, 0));
		r->name[PQgetlength(res, i, 0)] = '\0';

		strncpy(r->dest, PQgetvalue(res, i, 5), PQgetlength(res, i, 5));
		r->dest[PQgetlength(res, i, 5)] = '\0';

		if (!memcmp(r->dest, "out",3)) {
			strncpy(r->snet, PQgetvalue(res, i, 1), 
						    sizeof(r->snet)-1);
//			r->snet[PQgetlength(res, i, 1)] = '\0';
			strncpy(r->dnet, PQgetvalue(res, i, 2), 
						    sizeof(r->dnet)-1);
//			r->dnet[PQgetlength(res, i, 2)] = '\0';
		} else {
			strncpy(r->dnet, PQgetvalue(res, i, 1), 
						    sizeof(r->dnet)-1);
//			r->dnet[PQgetlength(res, i, 1)] = '\0';
			strncpy(r->snet, PQgetvalue(res, i, 2), 
						    sizeof(r->snet)-1);
//			r->snet[PQgetlength(res, i, 2)] = '\0';
		}
		strncpy(r->sport, PQgetvalue(res, i, 3), PQgetlength(res, i, 3));
		r->sport[PQgetlength(res, i, 3)] = '\0';
		strncpy(r->dport, PQgetvalue(res, i, 4), PQgetlength(res, i, 4));
		r->dport[PQgetlength(res, i, 4)] = '\0';
		
		strncpy(r->protocol, PQgetvalue(res, i, 6), 
						    PQgetlength(res, i, 6));
		r->protocol[PQgetlength(res, i, 6)] = '\0';
		strncpy(r->iface, PQgetvalue(res, i, 7), PQgetlength(res, i, 7));
		r->iface[PQgetlength(res, i, 7)] = '\0';

		if (memcmp(r->name, "admin", 5)) 
			strcpy(r->policy, "RETURN");
		else
			r->policy[0]='\0';
	}
	PQclear(res);
	res = PQexec(conn, "CLOSE mycursor");
	PQclear(res);
	res = PQexec(conn, "COMMIT");
	PQclear(res);
	return 0;
}

void postgre_ipac_close()
{
//    PQclear(res);
#ifdef DEBUG_POST
	fprintf(logs, "%lu : postgre_ipac_close\n", time(NULL));
	fclose(logs);
	fflush(logs);
#endif    
        PQfinish(conn);
	storage_opened = 0;
}
