// -*- C++ -*-
// --------------------------------------------------------------------
// Painter abstraction
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEPAINTER_H
#define IPEPAINTER_H

#include "ipegeo.h"
#include "ipecolor.h"
#include "ipestyle.h"
#include "ipebitmap.h"

class IpeText;

// --------------------------------------------------------------------

class IPE_EXPORT IpePainter {
public:
  IpePainter(const IpeStyleSheet *style);
  virtual ~IpePainter();

  void Transform(const IpeMatrix &m);
  void Untransform(bool direct);
  void Translate(const IpeVector &v);
  void Push();
  void Pop();

  void NewPath();
  void MoveTo(const IpeVector &v);
  void LineTo(const IpeVector &v);
  void CurveTo(const IpeVector &v1, const IpeVector &v2, const IpeVector &v3);
  inline void CurveTo(const IpeBezier &bezier);
  virtual void Rect(const IpeRect &re);
  void DrawEllipse();
  void DrawArc(double alpha);
  void ClosePath();
  void DrawPath();
  void DrawBitmap(IpeBitmap bitmap);
  void DrawText(const IpeText *text);

  void SetStroke(IpeAttribute color);
  void SetFill(IpeAttribute color);
  void SetLineWidth(IpeAttribute wid);
  void SetDashStyle(IpeAttribute dash);
  void SetLineCap(IpeAttribute cap);
  void SetLineJoin(IpeAttribute join);
  void SetWindRule(IpeAttribute rule);
  void SetTextSize(IpeAttribute size);
  void SetMarkSize(IpeAttribute size);
  void SetMarkShape(int shape);

  //! Return style sheet.
  inline const IpeStyleSheet *StyleSheet() const { return iStyleSheet; }
  //! Return current stroke color (always absolute).
  inline IpeAttribute Stroke() const { return iState.back().iStroke; }
  //! Return current fill color (always absolute).
  inline IpeAttribute Fill() const { return iState.back().iFill; }
  //! Return current transformation matrix.
  inline const IpeMatrix &Matrix() const { return iState.back().iMatrix; }
  //! Return current line width (always absolute).
  inline IpeAttribute LineWidth() const {return iState.back().iLineWidth; }
  //! Return current dash style (always absolute).
  inline IpeAttribute DashStyle() const {return iState.back().iDashStyle; }
  //! Return current line cap.
  inline IpeAttribute LineCap() const {return iState.back().iLineCap; }
  //! Return current line join.
  inline IpeAttribute LineJoin() const {return iState.back().iLineJoin; }
  //! Return current wind rule.
  inline IpeAttribute WindRule() const {return iState.back().iWindRule; }
  //! Return current text font size.
  inline IpeAttribute TextSize() const { return iState.back().iTextSize; }
  //! Return current mark size.
  inline IpeAttribute MarkSize() const { return iState.back().iMarkSize; }
  //! Return current mark shape.
  inline int MarkShape() const { return iState.back().iMarkShape; }

  //! Return repository.
  inline const IpeRepository *Repository() const {
    return StyleSheet()->Repository(); }

protected:
  virtual void DoPush();
  virtual void DoPop();
  virtual void DoNewPath();
  virtual void DoMoveTo(const IpeVector &v);
  virtual void DoLineTo(const IpeVector &v);
  virtual void DoCurveTo(const IpeVector &v1, const IpeVector &v2,
			 const IpeVector &v3);
  virtual void DoClosePath();
  virtual void DoDrawPath();
  virtual void DoDrawBitmap(IpeBitmap bitmap);
  virtual void DoDrawText(const IpeText *text);

protected:
  struct State {
    State() : iMarkShape(0) { /* nothing */ }
    IpeAttribute iStroke;
    IpeAttribute iFill;
    IpeAttribute iLineWidth;
    IpeAttribute iDashStyle;
    IpeAttribute iLineCap;
    IpeAttribute iLineJoin;
    IpeAttribute iWindRule;
    IpeAttribute iTextSize;
    IpeAttribute iMarkSize;
    int iMarkShape;
    IpeMatrix iMatrix;
  };
  typedef std::list<State> StateSeq;
  StateSeq iState;
  const IpeStyleSheet *iStyleSheet;
  int iInPath;
};

// --------------------------------------------------------------------

//! Overloaded function.
/*! Assumes current position is \c bezier.iV[0] */
inline void IpePainter::CurveTo(const IpeBezier &bezier)
{
  CurveTo(bezier.iV[1], bezier.iV[2], bezier.iV[3]);
}

// --------------------------------------------------------------------
#endif
