/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
/*
  smallest-circle.cpp: Ipelet for computing the smallest circle
  enclosing a set of points

  Author: Chris Gray <cgray@win.tue.nl>
  Date: Mon Feb 21 11:47:34 2005
  License: same as ipe.

  Description: This ipelet computes the smallest circle enclosing a
  set of points.  The algorithm is the randomized incremental
  algorithm taken from _Computational Geometry: Algorithms and
  Applications_ by M. de Berg, M. van Kreveld, M. Overmars, and
  O. Schwarzkopf.
 */

#include "ipelet.h"
#include "ipepath.h"
#include "ipepage.h"
#include "ipevisitor.h"
#include "ipemark.h"

static const char *aboutText =
"Written by Chris Gray (www.win.tue.nl/~cgray/ipelets.html)";

class MyCircle;

class SmallestCircleIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual const char *Label() const
  { return "Smallest Circle Enclosing Points"; }
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
  virtual const char *About() const { return aboutText; }
private:
  MyCircle *MiniDisc(IpeVector **points, int npoints);
  MyCircle *MiniDiscWithPoint(IpeVector **points, int npoints,
			      IpeVector *goodPoint);
  MyCircle *MiniDiscWith2Points(IpeVector **points, int npoints,
				IpeVector *goodPoint, IpeVector *goodPoint2);
  IpeVector **MyCopy(IpeVector **points, int npoints);
  IpeVector **MyPermute(IpeVector **points, int npoints);
};


class MyCircle {
private:
  IpeVector *center;
  IpeScalar radius;
public:
  MyCircle(IpeVector *p1, IpeVector *p2)
  {
    IpeVector p3 = *p1 - *p2;
    IpeVector p4 = *p1 + *p2;
    radius = p3.Len() / 2.0;
    p4 *= (double) 0.5;
    center = new IpeVector(p4);
  }
  MyCircle(IpeVector *p1, IpeVector *p2, IpeVector *p3);
  ~MyCircle() { delete center; }
  bool contains(IpeVector *p1)
  { IpeVector p2 = *center - *p1; bool ret = (p2.Len() < radius); return ret;}
  IpeVector *getCenter() { return center; }
  IpeScalar getRadius() { return radius; }
};

MyCircle::MyCircle(IpeVector *p1, IpeVector *p2, IpeVector *p3)
{
  IpeScalar x1, y1, x2, y2, x3, y3;
  IpeScalar mat[2][3];
  x1 = p1->iX;
  x2 = p2->iX;
  x3 = p3->iX;
  y1 = p1->iY;
  y2 = p2->iY;
  y3 = p3->iY;

  // Initialize the matrix
  mat[0][0] = 2 * (x2 - x1);
  mat[1][0] = 2 * (x2 - x3);
  mat[0][1] = 2 * (y2 - y1);
  mat[1][1] = 2 * (y2 - y3);
  mat[0][2] = x1 * x1 + y1 * y1 - (x2 * x2) - (y2 * y2);
  mat[1][2] = x3 * x3 + y3 * y3 - (x2 * x2) - (y2 * y2);

  // Now do poor man's row reduction
  mat[0][1] /= mat[0][0];
  mat[0][2] /= mat[0][0];
  mat[0][0] = 1.0;

  mat[1][1] -= (mat[0][1] * mat[1][0]);
  mat[1][2] -= (mat[0][2] * mat[1][0]);
  mat[1][0] = 0.0;

  mat[1][2] /= mat[1][1];
  mat[1][1] = 1.0;

  mat[0][2] -= (mat[1][2] * mat[0][1]);
  mat[0][1] = 0.0;

  center = new IpeVector(-mat[0][2], -mat[1][2]);
  radius = sqrt((x1 - center->iX) *
		(x1 - center->iX) + (y1 - center->iY) * (y1 - center->iY));
}


IpeVector **SmallestCircleIpelet::MyCopy(IpeVector **points, int npoints)
{
  int i;
  IpeVector **ret = new IpeVector*[npoints];

  for(i = 0; i < npoints; i++) {
    ret[i] = new IpeVector(*(points[i]));
  }
  return ret;
}

#ifdef WIN32
int rand_between(int start, int end)
{
  return start + (rand() % (end - start));
}
#else

static unsigned int seed = 12354;

int rand_between(int start, int end)
{
  int rand = rand_r(&seed);
  double scaled = ((double)(end - start) / (double)RAND_MAX) * rand + start;
  int ret = (int) scaled;
  if (scaled > end) {
    printf ("error! rand_between failed\n");
  }
  return ret;
}
#endif

IpeVector **SmallestCircleIpelet::MyPermute(IpeVector **points, int npoints)
{
  int i;
  IpeVector *temp;
  int toExchange;
  for (i = npoints - 1; i > 0; i--) {
    toExchange = rand_between(0, i);
    temp = points[i];
    points[i] = points[toExchange];
    points[toExchange] = temp;
  }
  return points;
}

MyCircle *SmallestCircleIpelet::MiniDisc(IpeVector **points, int npoints)
{
  int i;
  MyCircle *currentCircle;
  points = MyPermute(points, npoints);
  currentCircle = new MyCircle(points[0], points[1]);
  for(i = 2; i < npoints; i++) {
    if(currentCircle->contains(points[i])) {
    } else {
      delete currentCircle;
      currentCircle = MiniDiscWithPoint(points, i, points[i]);
    }
  }
  return currentCircle;
}

MyCircle *SmallestCircleIpelet::MiniDiscWithPoint(IpeVector **points,
						  int npoints,
						  IpeVector *goodPoint)
{
  int i;
  MyCircle *currentCircle;
  IpeVector **scratchPoints = MyCopy(points, npoints);
  scratchPoints = MyPermute(scratchPoints, npoints - 1);
  currentCircle = new MyCircle(goodPoint, scratchPoints[0]);

  for(i = 1; i < npoints; i++) {
    if(currentCircle->contains(scratchPoints[i])) {
    } else {
      delete currentCircle;
      currentCircle = MiniDiscWith2Points(scratchPoints, i, goodPoint,
					  scratchPoints[i]);
    }
  }
  for(i = 0; i < npoints; i++) {
    delete scratchPoints[i];
  }
  delete [] scratchPoints;
  return currentCircle;
}

MyCircle *SmallestCircleIpelet::MiniDiscWith2Points(IpeVector **points,
						    int npoints,
						    IpeVector *goodPoint,
						    IpeVector *goodPoint2)
{
  int i;
  MyCircle *currentCircle = new MyCircle(goodPoint, goodPoint2);

  for(i = 0; i < npoints; i++) {
    if(currentCircle->contains(points[i])) {
    } else {
      delete currentCircle;
      currentCircle = new MyCircle(points[i], goodPoint, goodPoint2);
    }
  }
  return currentCircle;
}


void SmallestCircleIpelet::Run(int, IpePage *page, IpeletHelper *helper)
{
  IpePage::iterator it;
  int length = 0;
  int i, j, k;
  MyCircle *outputCircle;
  fflush(stdout);
  for (it = page->begin(); it != page->end(); it++) {
    if (it->Select() && it->Object() && it->Object()->AsMark()) {
      length++;
    } else if (it->Select() && it->Object() && it->Object()->AsPath()) {
      for (j = 0; j < it->Object()->AsPath()->NumSubPaths(); j++) {
	const IpeSubPath *sp = it->Object()->AsPath()->SubPath(j);
	if (sp->AsSegs()) {
	  length += sp->AsSegs()->NumSegments() + 1;
	  //length += (0 && sp->Closed()) ? 0 : 1;
	}
      }
    }
  }
  if (length < 2) {
    helper->Message("Not enough selected");
    return;
  }
  fflush(stdout);
  IpeVector **points = new IpeVector*[length];
  i = 0;
  for (it = page->begin(); it != page->end(); it++) {
    if (it->Select() && it->Object()) {
      IpeMatrix m = it->Object()->Matrix();
      if (it->Object()->AsMark()) {
	points[i] = new IpeVector(m * it->Object()->AsMark()->Position());
	i++;
      } else if (it->Object()->AsPath()) {
	for (j = 0; j < it->Object()->AsPath()->NumSubPaths(); j++) {
	  const IpeSubPath *sp = it->Object()->AsPath()->SubPath(j);
	  if (sp->AsSegs()) {
	    for (k = 0; k < sp->AsSegs()->NumSegments(); k++) {
	      points[i] = new IpeVector(m * sp->AsSegs()->Segment(k).CP(0));
	      i++;
	    }
	    points[i] = new IpeVector(m * sp->AsSegs()->Segment(k - 1).CP(1));
	    i++;
	  }
	}
      }
    }

  }

  outputCircle = MiniDisc(points, length);
  IpeMatrix m(outputCircle->getRadius(), 0.0, 0.0,
	      outputCircle->getRadius(),
	      outputCircle->getCenter()->iX, outputCircle->getCenter()->iY);

  IpeEllipse *ellipse = new IpeEllipse(m);
  IpePath *path = new IpePath(helper->Attributes());
  path->AddSubPath(ellipse);
  page->push_back(IpePgObject(IpePgObject::EPrimary,
			      helper->CurrentLayer(), path));
  delete outputCircle;
  for (i = 0; i < length; i++) {
    delete points[i];
  }
  delete [] points;
  helper->Message("Created smallest enclosing circle");
}

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new SmallestCircleIpelet;
}
