/*
 * IRC - Internet Relay Chat, ircd/IPcheck.c
 * Copyright (C) 1998 Carlo Wood ( Run @ undernet.org )
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: IPcheck.c,v 1.2.2.4 2000/07/21 04:25:23 isomer Exp $
 *
 * 
 * This file should be edited in a window with a width of 141 characters
 * ick
 */
#include "IPcheck.h"
#include "client.h"
#include "ircd.h"
#include "numnicks.h"       /* NumNick, NumServ (GODMODE) */
#include "ircd_alloc.h"
#include "s_bsd.h"          /* SetIPChecked */
#include "s_debug.h"        /* Debug */
#include "s_user.h"         /* TARGET_DELAY */
#include "send.h"

#include <assert.h>
#include <stdio.h>          /* NULL ... bleah */


struct IPTargetEntry {
  int           count;
  unsigned char targets[MAXTARGETS];
};

struct IPRegistryEntry {
  struct IPRegistryEntry*  next;
  struct IPTargetEntry*    target;
  unsigned int             addr;
  int		           last_connect;
  unsigned short           connected;
  unsigned char            attempts;
};

/*
 * Hash table for IPv4 address registry
 *
 * Hash table size must be a power of 2
 * Use 64K hash table to conserve memory
 */
#define IP_REGISTRY_TABLE_SIZE 0x10000
#define MASK_16                0xffff

#define NOW ((unsigned short)(CurrentTime & MASK_16))
#define CONNECTED_SINCE(x) (NOW - (x))

#define IPCHECK_CLONE_LIMIT 4
#define IPCHECK_CLONE_PERIOD 40
#define IPCHECK_CLONE_DELAY 600


static struct IPRegistryEntry* hashTable[IP_REGISTRY_TABLE_SIZE];
static struct IPRegistryEntry* freeList = 0;

static unsigned int ip_registry_hash(unsigned int ip)
{
  return ((ip >> 16) ^ ip) & (IP_REGISTRY_TABLE_SIZE - 1);
}

static struct IPRegistryEntry* ip_registry_find(unsigned int ip)
{
  struct IPRegistryEntry* entry = hashTable[ip_registry_hash(ip)];
  for ( ; entry; entry = entry->next) {
    if (entry->addr == ip)
      break;
  }
  return entry;
}

static void ip_registry_add(struct IPRegistryEntry* entry)
{
  unsigned int bucket = ip_registry_hash(entry->addr);
  entry->next = hashTable[bucket];
  hashTable[bucket] = entry;
}
  
static void ip_registry_remove(struct IPRegistryEntry* entry)
{
  unsigned int bucket = ip_registry_hash(entry->addr);
  if (hashTable[bucket] == entry)
    hashTable[bucket] = entry->next;
  else {
    struct IPRegistryEntry* prev = hashTable[bucket];
    for ( ; prev; prev = prev->next) {
      if (prev->next == entry) {
        prev->next = entry->next;
        break;
      }
    }
  }
}
 
static struct IPRegistryEntry* ip_registry_new_entry()
{
  struct IPRegistryEntry* entry = freeList;
  if (entry)
    freeList = entry->next;
  else
    entry = (struct IPRegistryEntry*) MyMalloc(sizeof(struct IPRegistryEntry));

  assert(0 != entry);
  memset(entry, 0, sizeof(struct IPRegistryEntry));
  entry->last_connect = NOW;     /* Seconds since last connect attempt */
  entry->connected    = 1;       /* connected clients for this IP */
  entry->attempts     = 1;       /* Number attempts for this IP */
  return entry;
}

static void ip_registry_delete_entry(struct IPRegistryEntry* entry)
{
  if (entry->target)
    MyFree(entry->target);
  entry->next = freeList;
  freeList = entry;
}

static unsigned int ip_registry_update_free_targets(struct IPRegistryEntry* entry)
{
  unsigned int free_targets = STARTTARGETS;

  if (entry->target) {
    free_targets = entry->target->count + (CONNECTED_SINCE(entry->last_connect) / TARGET_DELAY);
    if (free_targets > STARTTARGETS)
      free_targets = STARTTARGETS;
    entry->target->count = free_targets;
  }
  return free_targets;
}

static void ip_registry_expire_entry(struct IPRegistryEntry* entry)
{
  /*
   * Don't touch this number, it has statistical significance
   * XXX - blah blah blah
   */
  if (CONNECTED_SINCE(entry->last_connect) > 600) {
    /*
     * expired
     */
    ip_registry_remove(entry);
    ip_registry_delete_entry(entry);
  }
  else if (CONNECTED_SINCE(entry->last_connect) > 120 && 0 != entry->target) {
    /*
     * Expire storage of targets
     */
    MyFree(entry->target);
    entry->target = 0;
  }
}

/*
 * ip_registry_expire
 */
static void ip_registry_expire()
{
  int i;
  struct IPRegistryEntry* entry;
  struct IPRegistryEntry* entry_next;

  for (i = 0; i < IP_REGISTRY_TABLE_SIZE; ++i) {
    for (entry = hashTable[i]; entry; entry = entry_next) {
      entry_next = entry->next;
      if (0 == entry->connected)
        ip_registry_expire_entry(entry);
    }
  }
}

/*
 * IPcheck_local_connect
 *
 * Event:
 *   A new connection was accept()-ed with IP number `cptr->ip.s_addr'.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Return:
 *     1 : You're allowed to connect.
 *     0 : You're not allowed to connect.
 *
 * Throttling:
 *
 * A connection should be rejected when a connection from the same IP number was
 * received IPCHECK_CLONE_LIMIT times before this connect attempt, with
 * reconnect intervals of IPCHECK_CLONE_PERIOD seconds or less.
 *
 * Free target inheritance:
 *
 * When the client is accepted, then the number of Free Targets
 * of the cptr is set to the value stored in the found IPregistry
 * structure, or left at STARTTARGETS.  This can be done by changing
 * cptr->nexttarget to be `now - (TARGET_DELAY * (FREE_TARGETS - 1))',
 * where FREE_TARGETS may range from 0 till STARTTARGETS.
 */
int ip_registry_check_local(unsigned int addr, time_t* next_target_out)
{
  struct IPRegistryEntry* entry = ip_registry_find(addr);
  unsigned int free_targets = STARTTARGETS;
 
  if (0 == entry) {
    entry       = ip_registry_new_entry();
    entry->addr = addr;    /* The IP number of registry entry */
    ip_registry_add(entry);
    return 1;
  }
  /* Note that this also connects server connects.
   * It is hard and not interesting, to change that.
   *
   * Don't allow more then 255 connects from one IP number, ever
   */
  if (0 == ++entry->connected)
    return 0;

  if (CONNECTED_SINCE(entry->last_connect) > IPCHECK_CLONE_PERIOD)
    entry->attempts = 0;

  free_targets = ip_registry_update_free_targets(entry);
  entry->last_connect = NOW;

  if (0 == ++entry->attempts)   /* Check for overflow */
    --entry->attempts;

  if (entry->attempts < IPCHECK_CLONE_LIMIT) {
    if (next_target_out)
      *next_target_out = CurrentTime - (TARGET_DELAY * free_targets - 1);
  }
  else if ((CurrentTime - me.since) > IPCHECK_CLONE_DELAY) {
    /* 
     * Don't refuse connection when we just rebooted the server
     */
#ifdef NOTHROTTLE 
    return 1;
#else
    --entry->connected;
    return 0;
#endif        
  }
  return 1;
}

/*
 * IPcheck_remote_connect
 *
 * Event:
 *   A remote client connected to Undernet, with IP number `cptr->ip.s_addr'
 *   and hostname `hostname'.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Return 0 on failure, 1 on success.
 */
int ip_registry_check_remote(struct Client* cptr, int is_burst)
{
  struct IPRegistryEntry* entry = ip_registry_find(cptr->ip.s_addr);

  /*
   * Mark that we did add/update an IPregistry entry
   */
  SetIPChecked(cptr);
  if (0 == entry) {
    entry = ip_registry_new_entry();
    entry->addr = cptr->ip.s_addr;
    if (is_burst)
      entry->attempts = 0;
    ip_registry_add(entry);
  }
  else {
    if (0 == ++entry->connected) {
      /* 
       * Don't allow more then 255 connects from one IP number, ever
       */
      return 0;
    }
    if (CONNECTED_SINCE(entry->last_connect) > IPCHECK_CLONE_PERIOD)
      entry->attempts = 0;
    if (!is_burst) {
      if (0 == ++entry->attempts) {
        /*
         * Check for overflow
         */
        --entry->attempts;
      }
      ip_registry_update_free_targets(entry);
      entry->last_connect = NOW;
    }
  }
  return 1;
}

/*
 * IPcheck_connect_fail
 *
 * Event:
 *   This local client failed to connect due to legal reasons.
 *
 * Action:
 *   Neutralize the effect of calling IPcheck_local_connect, in such
 *   a way that the client won't be penalized when trying to reconnect
 *   again.
 */
void ip_registry_connect_fail(unsigned int addr)
{
  struct IPRegistryEntry* entry = ip_registry_find(addr);
  if (entry)
    --entry->attempts;
}

/*
 * IPcheck_connect_succeeded
 *
 * Event:
 *   A client succeeded to finish the registration.
 *
 * Finish IPcheck registration of a successfully, locally connected client.
 */
void ip_registry_connect_succeeded(struct Client *cptr)
{
  const char*             tr    = "";
  unsigned int free_targets     = STARTTARGETS;
  struct IPRegistryEntry* entry = ip_registry_find(cptr->ip.s_addr);

  if (!entry) {
    Debug((DEBUG_ERROR, "Missing registry entry for: %s", cptr->sock_ip));
    return;
  }
  if (entry->target) {
    memcpy(cptr->targets, entry->target->targets, MAXTARGETS);
    free_targets = entry->target->count;
    tr = " tr";
  }
  sendto_one(cptr, ":%s NOTICE %s :on %u ca %u(%u) ft %u(%u)%s",
             me.name, cptr->name, entry->connected, entry->attempts,
             IPCHECK_CLONE_LIMIT, free_targets, STARTTARGETS, tr);
}

/*
 * IPcheck_disconnect
 *
 * Event:
 *   A local client disconnected or a remote client left Undernet.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Remove all expired IPregistry structures from the hash bucket
 *     that belongs to this clients IP number.
 */
void ip_registry_disconnect(struct Client *cptr)
{
  struct IPRegistryEntry* entry = ip_registry_find(cptr->ip.s_addr);
  if (0 == entry) {
    /*
     * trying to find an entry for a server causes this to happen,
     * servers should never have FLAGS_IPCHECK set
     */
    return;
  }
  /*
   * If this was the last one, set `last_connect' to disconnect time (used for expiration)
   */
  if (0 == --entry->connected) {
    if (CONNECTED_SINCE(entry->last_connect) > IPCHECK_CLONE_LIMIT * IPCHECK_CLONE_PERIOD) {
      /*
       * Otherwise we'd penetalize for this old value if the client reconnects within 20 seconds
       */
      entry->attempts = 0;
    }
    ip_registry_update_free_targets(entry);
    entry->last_connect = NOW;
  }
  if (MyConnect(cptr)) {
    unsigned int free_targets;
    /*
     * Copy the clients targets
     */
    if (0 == entry->target) {
      entry->target = (struct IPTargetEntry*) MyMalloc(sizeof(struct IPTargetEntry));
      entry->target->count = STARTTARGETS;
    }
    assert(0 != entry->target);

    memcpy(entry->target->targets, cptr->targets, MAXTARGETS);
    /*
     * This calculation can be pretty unfair towards large multi-user hosts, but
     * there is "nothing" we can do without also allowing spam bots to send more
     * messages or by drastically increasing the ammount of memory used in the IPregistry.
     *
     * The problem is that when a client disconnects, leaving no free targets, then
     * the next client from that IP number has to pay for it (getting no free targets).
     * But ALSO the next client, and the next client, and the next client etc - until
     * another client disconnects that DOES leave free targets.  The reason for this
     * is that if there are 10 SPAM bots, and they all disconnect at once, then they
     * ALL should get no free targets when reconnecting.  We'd need to store an entry
     * per client (instead of per IP number) to avoid this.
     */
    if (cptr->nexttarget < CurrentTime) {
        /*
         * Number of free targets
         */
      free_targets = (CurrentTime - cptr->nexttarget) / TARGET_DELAY + 1;
    }
    else
      free_targets = 0;
    /*
     * Add bonus, this is pretty fuzzy, but it will help in some cases.
     */
    if ((CurrentTime - cptr->firsttime) > 600)
      /*
       * Was longer then 10 minutes online?
       */
      free_targets += (CurrentTime - cptr->firsttime - 600) / TARGET_DELAY;
    /*
     * Finally, store smallest value for Judgement Day
     */
    if (free_targets < entry->target->count)
      entry->target->count = free_targets;
  }
}

/*
 * IPcheck_nr
 *
 * Returns number of clients with the same IP number
 */
int ip_registry_count(unsigned int addr)
{
  struct IPRegistryEntry* entry = ip_registry_find(addr);
  return (entry) ? entry->connected : 0;
}

/*
 * IPcheck_local_connect
 *
 * Event:
 *   A new connection was accept()-ed with IP number `cptr->ip.s_addr'.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Return:
 *     1 : You're allowed to connect.
 *     0 : You're not allowed to connect.
 *
 * Throttling:
 *
 * A connection should be rejected when a connection from the same IP number was
 * received IPCHECK_CLONE_LIMIT times before this connect attempt, with
 * reconnect intervals of IPCHECK_CLONE_PERIOD seconds or less.
 *
 * Free target inheritance:
 *
 * When the client is accepted, then the number of Free Targets
 * of the cptr is set to the value stored in the found IPregistry
 * structure, or left at STARTTARGETS.  This can be done by changing
 * cptr->nexttarget to be `now - (TARGET_DELAY * (FREE_TARGETS - 1))',
 * where FREE_TARGETS may range from 0 till STARTTARGETS.
 */
int IPcheck_local_connect(struct in_addr a, time_t* next_target_out)
{
  assert(0 != next_target_out);
  return ip_registry_check_local(a.s_addr, next_target_out);
}

/*
 * IPcheck_remote_connect
 *
 * Event:
 *   A remote client connected to Undernet, with IP number `cptr->ip.s_addr'
 *   and hostname `hostname'.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Return 0 on failure, 1 on success.
 */
int IPcheck_remote_connect(struct Client *cptr, int is_burst)
{
  assert(0 != cptr);
  return ip_registry_check_remote(cptr, is_burst);
}

/*
 * IPcheck_connect_fail
 *
 * Event:
 *   This local client failed to connect due to legal reasons.
 *
 * Action:
 *   Neutralize the effect of calling IPcheck_local_connect, in such
 *   a way that the client won't be penalized when trying to reconnect
 *   again.
 */
void IPcheck_connect_fail(struct in_addr a)
{
  ip_registry_connect_fail(a.s_addr);
}

/*
 * IPcheck_connect_succeeded
 *
 * Event:
 *   A client succeeded to finish the registration.
 *
 * Finish IPcheck registration of a successfully, locally connected client.
 */
void IPcheck_connect_succeeded(struct Client *cptr)
{
  assert(0 != cptr);
  ip_registry_connect_succeeded(cptr);
}

/*
 * IPcheck_disconnect
 *
 * Event:
 *   A local client disconnected or a remote client left Undernet.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Remove all expired IPregistry structures from the hash bucket
 *     that belongs to this clients IP number.
 */
void IPcheck_disconnect(struct Client *cptr)
{
  assert(0 != cptr);
  ip_registry_disconnect(cptr);
}

/*
 * IPcheck_nr
 *
 * Returns number of clients with the same IP number
 */
unsigned short IPcheck_nr(struct Client *cptr)
{
  assert(0 != cptr);
  return ip_registry_count(cptr->ip.s_addr);
}

void IPcheck_expire()
{
  static time_t next_expire = 0;
  if (next_expire < CurrentTime) {
    ip_registry_expire();
    next_expire = CurrentTime + 60;
  }
}

#if 0
struct IPregistry_vector {
  unsigned short length;
  unsigned short allocated_length;
  struct IPregistry *vector;
};

#define HASHTABSIZE 0x2000      /* Must be power of 2 */
static struct IPregistry_vector IPregistry_hashtable[HASHTABSIZE];

/*
 * Calculate a `hash' value between 0 and HASHTABSIZE, from the internet address `in_addr'.
 * Apply it immedeately to the table, effectively hiding the table itself.
 */
#define CALCULATE_HASH(in_addr) \
  struct IPregistry_vector *hash; \
  do { unsigned int ip = (in_addr).s_addr; \
       hash = &IPregistry_hashtable[((ip >> 14) + (ip >> 7) + ip) & (HASHTABSIZE - 1)]; } while(0)

/*
 * Fit `now' in an unsigned short, the advantage is that we use less memory
 * `struct IPregistry::last_connect' can be smaller while the only disadvantage 
 * is that if someone reconnects after exactly 18 hours and 12 minutes, and NOBODY with the
 * same _hash_ value for this IP-number did disconnect in the meantime, then the server
 * will think he reconnected immedeately. In other words: No disadvantage at all.
 */
#define BITMASK 0xffff          /* Same number of bits as `struct IPregistry::last_connect' */
#define HAS_TARGETS_MAGIC 15
#define HAS_TARGETS(entry) ((entry)->free_targets == HAS_TARGETS_MAGIC)

#if STARTTARGETS >= HAS_TARGETS_MAGIC
#error "That doesn't fit in 4 bits, does it?"
#endif

/* IP(entry) returns the `struct in_addr' of the IPregistry. */
#define IP(entry) (HAS_TARGETS(entry) ? (entry)->ip_targets.ptr->ip : (entry)->ip_targets.ip)
#define FREE_TARGETS(entry) (HAS_TARGETS(entry) ? (entry)->ip_targets.ptr->free_targets : (entry)->free_targets)

static unsigned short count = 10000, average_length = 4;

static struct IPregistry *IPregistry_add(struct IPregistry_vector *iprv)
{
  assert(0 != iprv);
  if (iprv->length == iprv->allocated_length)
  {
    iprv->allocated_length += 4;
    if (iprv->vector) {
      iprv->vector = 
              (struct IPregistry*) MyRealloc(iprv->vector,
                       iprv->allocated_length * sizeof(struct IPregistry));
    }
    else {
      iprv->vector = 
              (struct IPregistry*) MyMalloc(
                       iprv->allocated_length * sizeof(struct IPregistry));
    }
  }
  return &iprv->vector[iprv->length++];
}

static struct IPregistry *IPregistry_find(struct IPregistry_vector *iprv,
    struct in_addr ip)
{
  if (iprv->length > 0)
  {
    struct IPregistry *i, *end = &iprv->vector[iprv->length];
    for (i = &iprv->vector[0]; i < end; ++i)
      if (IP(i).s_addr == ip.s_addr)
        return i;
  }
  return NULL;
}

static struct IPregistry *IPregistry_find_with_expire(struct IPregistry_vector
    *iprv, struct in_addr ip)
{
  struct IPregistry *last;
  struct IPregistry *curr;
  struct IPregistry *retval = NULL;

  /*
   * if the vector is empty, IPcheck_disconnect will cause the server
   * to core when NDEBUG is defined
   */
  if (iprv->length < 1)
    return retval;

  last = &iprv->vector[iprv->length - 1];

  for (curr = &iprv->vector[0]; curr < last;)
  {
    if (IP(curr).s_addr == ip.s_addr)
      /* `curr' is element we looked for */
      retval = curr;
    else if (curr->connected == 0)
    {
      if (CONNECTED_SINCE(curr) > 600U) /* Don't touch this number, it has statistical significance */
      {
        /* `curr' expired */
        if (HAS_TARGETS(curr))
          MyFree(curr->ip_targets.ptr);
        *curr = *last--;
        iprv->length--;
        if (--count == 0)
        {
          /* Make ever 10000 disconnects an estimation of the average vector length */
          count = 10000;
          average_length =
              (UserStats.clients + UserStats.unknowns + UserStats.local_servers) / HASHTABSIZE;
        }
        /* Now check the new element (last) that was moved to this position */
        continue;
      }
      else if (CONNECTED_SINCE(curr) > 120U && HAS_TARGETS(curr))
      {
        /* Expire storage of targets */
        struct in_addr ip1 = curr->ip_targets.ptr->ip;
        curr->free_targets = curr->ip_targets.ptr->free_targets;
        MyFree(curr->ip_targets.ptr);
        curr->ip_targets.ip = ip1;
      }
    }
    /* Did not expire, check next element */
    ++curr;
  }
  /* Now check the last element in the list (curr == last) */
  if (IP(curr).s_addr == ip.s_addr)
    /* `curr' is element we looked for */
    retval = curr;
  else if (curr->connected == 0)
  {
    if (CONNECTED_SINCE(curr) > 600U)   /* Don't touch this number, it has statistical significance */
    {
      /* `curr' expired */
      if (HAS_TARGETS(curr))
        MyFree(curr->ip_targets.ptr);
      iprv->length--;
      if (--count == 0)
      {
        /* Make ever 10000 disconnects an estimation of the average vector length */
        count = 10000;
        average_length =
            (UserStats.clients + UserStats.unknowns + UserStats.local_servers) / HASHTABSIZE;
      }
    }
    else if (CONNECTED_SINCE(curr) > 120U && HAS_TARGETS(curr))
    {
      /* Expire storage of targets */
      struct in_addr ip1 = curr->ip_targets.ptr->ip;
      curr->free_targets = curr->ip_targets.ptr->free_targets;
      MyFree(curr->ip_targets.ptr);
      curr->ip_targets.ip = ip1;
    }
  }
  /* Do we need to shrink the vector? */
  if (iprv->allocated_length > average_length
      && iprv->allocated_length - iprv->length >= 4)
  {
    struct IPregistry *newpos;
    iprv->allocated_length = iprv->length;
    newpos =
        (struct IPregistry *)MyRealloc(iprv->vector,
        iprv->allocated_length * sizeof(struct IPregistry));
    if (newpos != iprv->vector) /* Is this ever true? */
    {
      retval =
          (struct IPregistry *)((char *)retval + ((char *)newpos -
          (char *)iprv->vector));
      iprv->vector = newpos;
    }
  }
  return retval;
}

static void reset_connect_time(struct IPregistry *entry)
{
  unsigned int previous_free_targets;

  /* Apply aging */
  previous_free_targets =
      FREE_TARGETS(entry) + CONNECTED_SINCE(entry) / TARGET_DELAY;
  if (previous_free_targets > STARTTARGETS)
    previous_free_targets = STARTTARGETS;
  if (HAS_TARGETS(entry))
    entry->ip_targets.ptr->free_targets = previous_free_targets;
  else
    entry->free_targets = previous_free_targets;

  entry->last_connect = NOW;
}

/*
 * IPcheck_local_connect
 *
 * Event:
 *   A new connection was accept()-ed with IP number `cptr->ip.s_addr'.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Return:
 *     1 : You're allowed to connect.
 *     0 : You're not allowed to connect.
 *
 * Throttling:
 *
 * A connection should be rejected when a connection from the same IP number was
 * received IPCHECK_CLONE_LIMIT times before this connect attempt, with
 * reconnect intervals of IPCHECK_CLONE_PERIOD seconds or less.
 *
 * Free target inheritance:
 *
 * When the client is accepted, then the number of Free Targets
 * of the cptr is set to the value stored in the found IPregistry
 * structure, or left at STARTTARGETS.  This can be done by changing
 * cptr->nexttarget to be `now - (TARGET_DELAY * (FREE_TARGETS - 1))',
 * where FREE_TARGETS may range from 0 till STARTTARGETS.
 */
int IPcheck_local_connect(struct in_addr a, time_t* next_target_out)
{
  struct IPregistry *entry;
  CALCULATE_HASH(a);
  assert(0 != next_target_out);

  if (!(entry = IPregistry_find(hash, a)))
  {
    entry = IPregistry_add(hash);
    entry->ip_targets.ip = a;          /* The IP number of registry entry */
    entry->last_connect = NOW;          /* Seconds since last connect attempt */
    entry->connected = 1;               /* connected clients for this IP */
    entry->connect_attempts = 1;        /* Number attempts for this IP */
    entry->free_targets = STARTTARGETS; /* free targets a client gets */
    return 1;
  }
  /* Note that this also connects server connects.
   * It is hard and not interesting, to change that.
   *
   * Don't allow more then 255 connects from one IP number, ever
   */
  if (0 == ++entry->connected)
    return 0;

  if (CONNECTED_SINCE(entry) > IPCHECK_CLONE_PERIOD)
    entry->connect_attempts = 0;

  reset_connect_time(entry);

  if (0 == ++entry->connect_attempts)   /* Check for overflow */
    --entry->connect_attempts;

  if (entry->connect_attempts <= IPCHECK_CLONE_LIMIT)
    *next_target_out = CurrentTime - (TARGET_DELAY * (FREE_TARGETS(entry) - 1));

  /* Don't refuse connection when we just rebooted the server */
  else if (CurrentTime - me.since > IPCHECK_CLONE_DELAY)
#ifndef NOTHROTTLE 
    return 0;
#else
    return 1;
#endif        
  return 1;
}

/*
 * IPcheck_remote_connect
 *
 * Event:
 *   A remote client connected to Undernet, with IP number `cptr->ip.s_addr'
 *   and hostname `hostname'.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Return -1 on failure, 0 on success.
 */
int IPcheck_remote_connect(struct Client *cptr, const char *hostname,
    int is_burst)
{
  struct IPregistry *entry;
  CALCULATE_HASH(cptr->ip);
  SetIPChecked(cptr);           /* Mark that we did add/update an IPregistry entry */
  if (!(entry = IPregistry_find(hash, cptr->ip)))
  {
    entry = IPregistry_add(hash);
    entry->ip_targets.ip = cptr->ip;    /* The IP number of registry entry */
    entry->last_connect = NOW;  /* Seconds since last connect (attempt) */
    entry->connected = 1;       /* Number of currently connected clients with this IP number */
    entry->connect_attempts = is_burst ? 1 : 0; /* Number of clients that connected with this IP number */
    entry->free_targets = STARTTARGETS; /* Number of free targets that a client gets on connect */
  }
  else
  {
#ifdef GODMODE
    sendto_one(cptr,
        "%s NOTICE %s%s :I saw your face before my friend (connected: %u; connect_attempts %u; free_targets %u)",
        NumServ(&me), NumNick(cptr), entry->connected, entry->connect_attempts,
        FREE_TARGETS(entry));
#endif
    if (++(entry->connected) == 0)      /* Don't allow more then 255 connects from one IP number, ever */
      return -1;
    if (CONNECTED_SINCE(entry) > IPCHECK_CLONE_PERIOD)
      entry->connect_attempts = 0;
    if (!is_burst)
    {
      if (++(entry->connect_attempts) == 0)     /* Check for overflow */
        --(entry->connect_attempts);
      reset_connect_time(entry);
    }
  }
  return 0;
}

/*
 * IPcheck_connect_fail
 *
 * Event:
 *   This local client failed to connect due to legal reasons.
 *
 * Action:
 *   Neutralize the effect of calling IPcheck_local_connect, in such
 *   a way that the client won't be penalized when trying to reconnect
 *   again.
 */
void IPcheck_connect_fail(struct in_addr a)
{
  struct IPregistry *entry;
  CALCULATE_HASH(a);
  if ((entry = IPregistry_find(hash, a)))
    --entry->connect_attempts;
}

/*
 * IPcheck_connect_succeeded
 *
 * Event:
 *   A client succeeded to finish the registration.
 *
 * Finish IPcheck registration of a successfully, locally connected client.
 */
void IPcheck_connect_succeeded(struct Client *cptr)
{
  struct IPregistry *entry;
  const char *tr = "";
  CALCULATE_HASH(cptr->ip);
  entry = IPregistry_find(hash, cptr->ip);
  if (HAS_TARGETS(entry))
  {
    memcpy(cptr->targets, entry->ip_targets.ptr->targets, MAXTARGETS);
    tr = " tr";
  }
  sendto_one(cptr, ":%s NOTICE %s :on %u ca %u(%u) ft %u(%u)%s",
      me.name, cptr->name, entry->connected, entry->connect_attempts,
      IPCHECK_CLONE_LIMIT, FREE_TARGETS(entry), STARTTARGETS, tr);
}

/*
 * IPcheck_disconnect
 *
 * Event:
 *   A local client disconnected or a remote client left Undernet.
 *
 * Action:
 *   Update the IPcheck registry.
 *   Remove all expired IPregistry structures from the hash bucket
 *     that belongs to this clients IP number.
 */
void IPcheck_disconnect(struct Client *cptr)
{
  struct IPregistry *entry;
  CALCULATE_HASH(cptr->ip);
  entry = IPregistry_find_with_expire(hash, cptr->ip);
  if (0 == entry) {
    /*
     * trying to find an entry for a server causes this to happen,
     * servers should never have FLAGS_IPCHECK set
     */
    assert(0 != entry);
    return;
  }
  /*
   * If this was the last one, set `last_connect' to disconnect time (used for expiration)
   */
  if (--(entry->connected) == 0) {
    if (CONNECTED_SINCE(entry) > IPCHECK_CLONE_LIMIT * IPCHECK_CLONE_PERIOD)
      /*
       * Otherwise we'd penetalize for this old value if the client reconnects within 20 seconds
       */
      entry->connect_attempts = 0;
    reset_connect_time(entry);
  }
  if (MyConnect(cptr)) {
    unsigned int inheritance;
    /*
     * Copy the clients targets
     */
    if (HAS_TARGETS(entry)) {
      entry->free_targets = entry->ip_targets.ptr->free_targets;
      MyFree(entry->ip_targets.ptr);
    }
    entry->ip_targets.ptr =
        (struct ip_targets_st*) MyMalloc(sizeof(struct ip_targets_st));

    assert(0 != entry->ip_targets.ptr);
    entry->ip_targets.ptr->ip = cptr->ip;
    entry->ip_targets.ptr->free_targets = entry->free_targets;
    entry->free_targets = HAS_TARGETS_MAGIC;
    memcpy(entry->ip_targets.ptr->targets, cptr->targets, MAXTARGETS);
    /*
     * This calculation can be pretty unfair towards large multi-user hosts, but
     * there is "nothing" we can do without also allowing spam bots to send more
     * messages or by drastically increasing the ammount of memory used in the IPregistry.
     *
     * The problem is that when a client disconnects, leaving no free targets, then
     * the next client from that IP number has to pay for it (getting no free targets).
     * But ALSO the next client, and the next client, and the next client etc - until
     * another client disconnects that DOES leave free targets.  The reason for this
     * is that if there are 10 SPAM bots, and they all disconnect at once, then they
     * ALL should get no free targets when reconnecting.  We'd need to store an entry
     * per client (instead of per IP number) to avoid this.
     */
    if (cptr->nexttarget <= CurrentTime)
        /*
         * Number of free targets
         */
      inheritance = (CurrentTime - cptr->nexttarget) / TARGET_DELAY + 1;
    else
      inheritance = 0;
    /*
     * Add bonus, this is pretty fuzzy, but it will help in some cases.
     */
    if (CurrentTime - cptr->firsttime > 600)
      /*
       * Was longer then 10 minutes online?
       */
      inheritance += (CurrentTime - cptr->firsttime - 600) / TARGET_DELAY;
    /*
     * Finally, store smallest value for Judgement Day
     */
    if (inheritance < entry->ip_targets.ptr->free_targets)
      entry->ip_targets.ptr->free_targets = inheritance;
  }
}

/*
 * IPcheck_nr
 *
 * Returns number of clients with the same IP number
 */
unsigned short IPcheck_nr(struct Client *cptr)
{
  struct IPregistry *entry;
  CALCULATE_HASH(cptr->ip);
  entry = IPregistry_find(hash, cptr->ip);
  return (entry ? entry->connected : 0);
}
#endif
